/*
 *  Wrapper base class for GstProps
 *  Copyright (C) 2002 Tim Jansen <tim@tjansen.de>
 *  API Documentation
 *  Copyright (C) 1999,2000 Erik Walthinsen <omega@cse.ogi.edu>
 *                     2000 Wim Taymans <wtay@chello.be>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef KDE_GST_PROPS_H
#define KDE_GST_PROPS_H

#include <kde/gst/simplewrapper.h>
#include <qvaluevector.h>

namespace KDE {
namespace GST {
	struct PropsEntry;

/**
 * Props is used to attach certain properties to a pad. Properties
 * are usually used in conjunction with @ref Caps.
 *
 * @short Properties
 * @see Caps
 */
	class Props : public SimpleWrapper {
	private:
		void *reserved;
	public:
		enum PropsType {
			PROPS_END_TYPE = 0,
			PROPS_INVALID_TYPE,
			PROPS_INT_TYPE,
			PROPS_FLOAT_TYPE,
			PROPS_FOURCC_TYPE,
			PROPS_BOOL_TYPE,
			PROPS_STRING_TYPE,
			PROPS_VAR_TYPE,   /* after this marker start the variable properties */
			PROPS_LIST_TYPE,
			PROPS_FLOAT_RANGE_TYPE,
			PROPS_INT_RANGE_TYPE,

			PROPS_LAST_TYPE = PROPS_END_TYPE + 16
		};
		
/**
 * Creates a new Props that wrapps the given GstProps.
 * You must not create a object of this class on the stack, always 
 * use new.
 * @param real the GstProps to be wrapped.
 */
	        Props(void *real);
	        virtual ~Props();

/**
 * Creates a new Props that wrapps the given object. Unlike the
 * constructor this re-used already existing objects.
 * @param real the object to be wrapped
 * @return the Props
 */
		static Props *wrap(void *real);

/**
 * Create a new empty property.
 */
		Props();

/**
 * Merge the properties of tomerge into props.
 *
 * @param tomerge the property to merge 
 * @return the new merged property 
 */
		Props* merge(Props *tomerge);

/**
 * Copy the property structure.
 *
 * @return the new property that is a copy of the original
 * one.
 */
		Props* copy ();

/**
 * Copy the property structure if the refcount is >1.
 *
 * @return A new props that can be safely written to.
 */
		Props* copyOnWrite();

/**
 * Increase the refcount of the property structure.
 */
		void ref();

/**
 * Decrease the refcount of the property structure, destroying
 * the property if the refcount is 0.
 */
		void unref();

/**
 * Checks whether two capabilities are compatible.
 *
 * @param toprops a property
 * @return TRUE if compatible, FALSE otherwise
 */
		bool checkCompatibility(Props *toprops);

/**
 * Unrolls all lists in the given Props. This is useful if you
 * want to loop over the props.
 *
 * @return A GList with the unrolled props entries.
 */
		QValueVector<PropsEntry*> normalize();

/**
 * Modifies the value of the given entry in the props struct
 * and sets it to the given integer.
 *
 * @param name the name of the entry to modify
 * @param value the value to set
 * @return the new modified property structure.
 */
		Props* set(const QString &name, int value);

/**
 * Modifies the value of the given entry in the props struct
 * and sets it to "any integer".
 *
 * @param name the name of the entry to modify
 * @return the new modified property structure.
 */
		Props* setIntAny(const QString &name);

/**
 * Modifies the value of the given entry in the props struct
 * and sets it to the given integer range.
 *
 * @param name the name of the entry to modify
 * @param valuea the min value to set
 * @param valueb the max value to set
 * @return the new modified property structure.
 */
		Props* setIntRange(const QString &name, int valuea, int valueb);

/**
 * Modifies the value of the given entry in the props struct
 * and sets it to the negative integer range.
 *
 * @param name the name of the entry to modify
 * @return the new modified property structure.
 */
		Props* setIntNegative(const QString &name);

/**
 * Modifies the value of the given entry in the props struct
 * and sets it to the given FourCC in integer form.
 *
 * @param name the name of the entry to modify
 * @param value the value to set
 * @return the new modified property structure.
 */
		Props* setFourCC(const QString &name, int value);

/**
 * Modifies the value of the given entry in the props struct
 * and sets it to the given FourCC in string form.
 *
 * @param name the name of the entry to modify
 * @param value the value to set
 * @return the new modified property structure.
 */
		Props* setFourCC(const QString &name, const QString &value);

/**
 * Modifies the value of the given entry in the props struct
 * and sets it to the given FourCC in character form.
 *
 * @param name the name of the entry to modify
 * @param a the first character of the FourCC code
 * @param b the second character of the FourCC code
 * @param c the third character of the FourCC code
 * @param d the fourth character of the FourCC code
 * @return the new modified property structure.
 */
		Props* setFourCC(const QString &name, 
				 char a, char b, char c, char d);

/**
 * Modifies the value of the given entry in the props struct
 * and sets it to the given boolean.
 *
 * @param name the name of the entry to modify
 * @param value the value to set
 * @return the new modified property structure.
 */
		Props* set(const QString &name, bool value);

/**
 * Modifies the value of the given entry in the props struct
 * and sets it to the given string.
 *
 * @param name the name of the entry to modify
 * @param value the value to set
 * @return the new modified property structure.
 */
		Props* set(const QString &name, const QString &value);

/**
 * Modifies the value of the given entry in the props struct
 * and sets it to the given float.
 *
 * @param name the name of the entry to modify
 * @param value the value to set
 * @return the new modified property structure.
 */
		Props* set(const QString &name, float value);

/**
 * Modifies the value of the given entry in the props struct
 * and sets it to the given float range.
 *
 * @param name the name of the entry to modify
 * @param valuea the minimum value to set
 * @param valueb the maximum value to set
 * @return the new modified property structure.
 */
		Props* set(const QString &name, float valuea, float valueb);

/**
 * Modifies the value of the given entry in the props struct
 * and sets it to the positive integer range.
 *
 * @param name the name of the entry to modify
 * @return the new modified property structure.
 */
		Props* setIntPositive(const QString &name);

/**
 * Retrieves the value of the property with the given
 * name. 
 * @param name the name of the entry to retrieve
 * @return the integer of the property or 0 if it does not exist
 */
		int getInt(const QString &name);

/**
 * Retrieves the value of the property with the given
 * name. 
 * @param name the name of the entry to retrieve
 * @param a the start of the range
 * @param b the end of the range
 * @return true if property exists
 */
		bool getIntRange(const QString &name, int &a, int &b);

/**
 * Retrieves the value of the property with the given
 * name. 
 * @param name the name of the entry to retrieve
 * @return the float of the property or 0 if it does not exist
 */
		float getFloat(const QString &name);

/**
 * Retrieves the value of the property with the given
 * name. 
 * @param name the name of the entry to retrieve
 * @param a the start of the range
 * @param b the end of the range
 * @return true if property exists
 */
		bool getFloatRange(const QString &name, float &a, float &b);

/**
 * Retrieves the value of the property with the given
 * name. 
 * @param name the name of the entry to retrieve
 * @return the fourcc integer of the property or 0 if it does not exist
 */
		int getFourCCInt(const QString &name);

/**
 * Retrieves the value of the property with the given
 * name. 
 * @param name the name of the entry to retrieve
 * @return the boolean of the property will be written here, undefined if 
 *         the argument does not exist
 */
		bool getBool(const QString &name);

/**
 * Retrieves the value of the property with the given
 * name. 
 * @param name the name of the entry to retrieve
 * @param result the boolean of the property will be written here
 * @return true if a property of the given name was found
 */
		bool getBool(const QString &name, bool &result);

/**
 * Checks if a given props has a property with the given name.
 *
 * @param name the name of the key to find
 * @return TRUE if the property was found, FALSE otherwise.
 */
		bool hasProperty(const QString &name);

/**
 * Checks if a given props has a property with the given name that 
 * is also fixed, ie. is not a list or a range.
 *
 * @param name the name of the key to find
 * @return TRUE if the property was found, FALSE otherwise.
 */
		bool hasFixedProperty(const QString &name);

/**
 * Checks if a given props has a property with the given name and the given type.
 *
 * @param name the name of the key to find
 * @param type the type of the required property
 * @return TRUE if the property was found, FALSE otherwise.
 */
		bool hasPropertyTyped(const QString &name, PropsType type);

/**
 * Calculates the intersection bewteen two Props.
 *
 * @param props2 another property
 * @return a Props with the intersection or NULL if the 
 * intersection is empty.
 */
		Props* intersect(Props *props2);

/**
 * Get the props entry with the geven name
 *
 * @param name the name of the entry to get
 * @return The props entry with the geven name or NULL when
 * the entry was not found.
 */
		const PropsEntry* getEntry(const QString &name);


	};

	/**
	 * Represents a property value.
	 */
	struct PropsEntry {
		QString name;
		Props::PropsType type;
		union {
			int vInt;
			float vFloat;
			unsigned int vFourcc;
			bool vBool;
			char *vString;
			QValueVector<PropsEntry*> *vList;
			struct {
				int a;
				int b;
			} vIntRange;
			struct {
				float a;
				float b;
			} vFloatRange;
		} data;

		~PropsEntry();
		static PropsEntry* wrap(void*);
	};
}
}

#endif
