/* $Id: we_logging.c,v 1.1 2007/11/24 19:14:36 ekalin Exp $ */

/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <kcconfig.h>
#endif

#include <string.h>
#include <libintl.h>
#include <gtk/gtk.h>
#include <glade/glade.h>
#include <gmodule.h>

#include "simocombobox.h"

#include "kildclient.h"
#include "perlscript.h"


/***********************
 * Function prototypes *
 ***********************/
static void we_log_start_cb(GtkWidget *widget, gpointer data);
static void we_log_stop_cb(GtkWidget *widget, gpointer data);
/* Glade signals */
G_MODULE_EXPORT void select_logfile_cb(gpointer data, GtkWidget *widget);
G_MODULE_EXPORT void revert_logformat_cb(gpointer data, GtkWidget *widget);



GtkWidget *
get_we_logging_tab(GladeXML *gladexml, World *world)
{
  GtkWidget *panel;
  GtkWidget *btnStart;
  GtkWidget *btnStop;
  GtkWidget *image;

  panel = glade_xml_get_widget(gladexml, "we_panel_logging");

  btnStart = glade_xml_get_widget(gladexml, "btnLogStart");
  image = gtk_image_new_from_stock(GTK_STOCK_MEDIA_PLAY,
                                   GTK_ICON_SIZE_BUTTON);
  gtk_button_set_image(GTK_BUTTON(btnStart), image);
  g_signal_connect(G_OBJECT(btnStart), "clicked",
                   G_CALLBACK(we_log_start_cb), world);

  btnStop = glade_xml_get_widget(gladexml, "btnLogStop");
  g_signal_connect(G_OBJECT(btnStop), "clicked",
                   G_CALLBACK(we_log_stop_cb), world);

  return panel;
}


void
fill_we_logging_tab(World *world, GladeXML *gladexml)
{
  GtkLabel        *lblLogStatus;
  GtkWidget       *btnLogStart;
  GtkWidget       *btnLogStop;
  GtkEntry        *txtLogFile;
  GtkToggleButton *chkLogAuto;
  GtkToggleButton *chkLogAddTime;
  GtkEntry        *txtLogTimeFormat;

  if (!world) {
    return;
  }

  if (!gladexml) {
  /* Called from Perl's XS_logfile. In this case there is no gladexml,
     and the World Editor might not have been created yet. */
    if (!world->dlgEditWorld) {
      return;
    }

    gladexml = glade_get_widget_tree(world->dlgEditWorld);
  }

  lblLogStatus     = GTK_LABEL(glade_xml_get_widget(gladexml, "lblLogStatus"));
  btnLogStart      = glade_xml_get_widget(gladexml, "btnLogStart");
  btnLogStop       = glade_xml_get_widget(gladexml, "btnLogStop");
  txtLogFile       = GTK_ENTRY(glade_xml_get_widget(gladexml, "txtLogFile"));
  chkLogAuto       = GTK_TOGGLE_BUTTON(glade_xml_get_widget(gladexml,
                                                            "chkLogAuto"));
  chkLogAddTime    = GTK_TOGGLE_BUTTON(glade_xml_get_widget(gladexml,
                                                            "chkLogAddTime"));
  txtLogTimeFormat = GTK_ENTRY(glade_xml_get_widget(gladexml,
                                                    "txtLogTimeFormat"));

  if (world->connected) {
    if (world->log_file) {
      gchar *msg;

      msg = g_strdup_printf(_("Logging to file %s."), world->log_actual_file);
      gtk_label_set_text(lblLogStatus, msg);
      g_free(msg);
      gtk_widget_set_sensitive(btnLogStart, FALSE);
      gtk_widget_set_sensitive(btnLogStop, TRUE);
    } else {
      gtk_label_set_text(lblLogStatus, _("Logging is disabled."));
      gtk_widget_set_sensitive(btnLogStart, TRUE);
      gtk_widget_set_sensitive(btnLogStop, FALSE);
    }
  } else {
    gtk_label_set_text(lblLogStatus, _("Not connected."));
    gtk_widget_set_sensitive(btnLogStart, FALSE);
    gtk_widget_set_sensitive(btnLogStop, FALSE);
  }

  if (world->log_file_name) {
    gtk_entry_set_text(txtLogFile, world->log_file_name);
  }

  gtk_toggle_button_set_active(chkLogAuto, world->log_autostart);
  gtk_toggle_button_set_active(chkLogAddTime, world->log_add_time);
  if (world->log_timeformat) {
    gtk_entry_set_text(txtLogTimeFormat, world->log_timeformat);
  }
}


void
update_we_logging_parameters(World *world)
{
  GladeXML        *gladexml;
  GtkEntry        *txtLogFile;
  GtkToggleButton *chkLogAuto;
  GtkToggleButton *chkLogAddTime;
  GtkEntry        *txtLogTimeFormat;
  const gchar     *filename;
  const gchar     *format;

  gladexml         = glade_get_widget_tree(world->dlgEditWorld);
  txtLogFile       = GTK_ENTRY(glade_xml_get_widget(gladexml, "txtLogFile"));
  chkLogAuto       = GTK_TOGGLE_BUTTON(glade_xml_get_widget(gladexml,
                                                            "chkLogAuto"));
  chkLogAddTime    = GTK_TOGGLE_BUTTON(glade_xml_get_widget(gladexml,
                                                            "chkLogAddTime"));
  txtLogTimeFormat = GTK_ENTRY(glade_xml_get_widget(gladexml,
                                                    "txtLogTimeFormat"));

  g_free(world->log_file_name);
  filename = gtk_entry_get_text(txtLogFile);
  if (filename && *filename) {
    world->log_file_name = g_strdup(filename);
  } else {
    world->log_file_name = NULL;
  }

  world->log_autostart = gtk_toggle_button_get_active(chkLogAuto);
  world->log_add_time  = gtk_toggle_button_get_active(chkLogAddTime);

  format = gtk_entry_get_text(txtLogTimeFormat);
  if (!world->log_timeformat || strcmp(world->log_timeformat, format) != 0) {
    g_free(world->log_timeformat);
    world->log_timeformat = g_strdup(format);
  }
}


static
void
we_log_start_cb(GtkWidget *widget, gpointer data)
{
  World *world = (World *) data;
  gchar *errmsg = NULL;

  /* Copy parameters in the World structure */
  update_we_logging_parameters(world);

  if (start_log(world, &errmsg)) {
    fill_we_logging_tab(world, NULL);
  } else {
    GtkWidget *msgdlg = gtk_message_dialog_new(GTK_WINDOW(world->dlgEditWorld),
                                               GTK_DIALOG_DESTROY_WITH_PARENT,
                                               GTK_MESSAGE_ERROR,
                                               GTK_BUTTONS_CLOSE,
                                               errmsg);
    gtk_window_set_title(GTK_WINDOW(msgdlg), _("Logging not started"));
    gtk_dialog_run(GTK_DIALOG(msgdlg));
    gtk_widget_destroy(msgdlg);
    g_free(errmsg);
  }
}


static
void
we_log_stop_cb(GtkWidget *widget, gpointer data)
{
  World *world = (World *) data;

  update_we_logging_parameters(world);
  stop_log(world);
  fill_we_logging_tab(world, NULL);
}


void
select_logfile_cb(gpointer data, GtkWidget *widget)
{
  GtkEntry    *txtLogFile = GTK_ENTRY(data);
  GladeXML    *gladexml;
  GtkWindow   *dlgEditWorld;
  GtkWidget   *filedlg;
  const gchar *filename;

  gladexml = glade_get_widget_tree(widget);
  dlgEditWorld = (GtkWindow *) g_object_get_data(G_OBJECT(gladexml),
                                                 "dlgEditWorld");

  filedlg = gtk_file_chooser_dialog_new(_("Select log file"),
                                        dlgEditWorld,
                                        GTK_FILE_CHOOSER_ACTION_SAVE,
                                        GTK_STOCK_CANCEL,
                                        GTK_RESPONSE_CANCEL,
                                        GTK_STOCK_OK,
                                        GTK_RESPONSE_OK,
                                        NULL);
  gtk_dialog_set_default_response(GTK_DIALOG(filedlg), GTK_RESPONSE_OK);
  filename = gtk_entry_get_text(txtLogFile);
  if (filename && *filename) {
    gtk_file_chooser_set_filename(GTK_FILE_CHOOSER(filedlg), filename);
  }

  if (gtk_dialog_run(GTK_DIALOG(filedlg)) == GTK_RESPONSE_OK) {
    gchar *new_file;

    new_file = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(filedlg));
    gtk_entry_set_text(txtLogFile, new_file);
    g_free(new_file);
  }

  gtk_widget_destroy(filedlg);
}


void
revert_logformat_cb(gpointer data, GtkWidget *widget)
{
  GtkEntry *txtLogTimeFormat = GTK_ENTRY(data);

  gtk_entry_set_text(txtLogTimeFormat, DEFAULT_LOGTIME_FORMAT);
}
