/***************************************************************************
                          imagemapeditor.cpp  -  description
                             -------------------
    begin                : Wed Apr 4 2001
    copyright            : (C) 2001 by Jan Schfer
    email                : j_schaef@informatik.uni-kl.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <iostream.h>
// QT
#include <qlayout.h>
#include <qlistview.h>
#include <qpushbutton.h>
#include <qvbox.h>
#include <qpixmap.h>
#include <qcombobox.h>
#include <qsplitter.h>
#include <qfileinfo.h>
#include <qmultilineedit.h>
#include <qtextstream.h>
#include <qpopupmenu.h>
#include <qdict.h>
#include <qwhatsthis.h>
#include <qtooltip.h>
#include <qpainter.h>

// KDE
#include <kcommand.h>
#include <kdebug.h>
#include <klocale.h>
#include <kaction.h>
#include <kaccel.h>
#include <kstdaction.h>
#include <kiconloader.h>
#include <kfiledialog.h>
#include <kmessagebox.h>
#include <kapp.h>
#include <kkeydialog.h>
#include <kedittoolbar.h>
#include <klineeditdlg.h>
#include <kglobal.h>
#include <kstddirs.h>
#include <kstatusbar.h>

// local
#include "kimagemapeditor.moc"
#include "drawzone.h"
#include "kimedialogs.h"
#include "kimecommands.h"
#include "qextfileinfo.h"
#include "areacreator.h"


KImageMapEditor::KImageMapEditor(QWidget* parent)
	: KMainWindow(parent)
{

	QSplitter *splitter= new QSplitter(this);
	
	QVBox *vbox= new QVBox(splitter);
	
	listView = new QListView(vbox);
	listView->addColumn(i18n("HRef"));
	listView->addColumn(i18n("Image"));
	listView->setMultiSelection(true);
	listView->setSelectionMode( QListView::Extended );
	listView->setSorting(-1); // The user can't sort by clicking on the header
	
	QWhatsThis::add( listView, i18n("<h3>Arealist</h3>The arealist shows you all areas of the map.<br>\
																	 The left column shows the link of the area, on the right column \
																	 you can see which part of the image is covered by the area.<br>\
																	 The maximum size of the preview images can be configured."));
	QToolTip::add( listView, i18n("A list of all areas"));
	
	connect( listView, SIGNAL(selectionChanged()), this, SLOT(slotSelectionChanged()));
	connect( listView, SIGNAL(doubleClicked(QListViewItem*)), this, SLOT(showTagEditor(QListViewItem*)));
	connect( listView, SIGNAL(rightButtonPressed(QListViewItem*,const QPoint &,int)), this,
		SLOT(slotShowPopupMenu(QListViewItem*,const QPoint &)));
	drawZone = new DrawZone(splitter,this);
	
	QHBox *hbox= new QHBox(vbox);
	upBtn= new QPushButton("",hbox);
	upBtn->setPixmap(SmallIcon("up"));
//	upBtn->setEnabled(false);
	
	downBtn= new QPushButton("",hbox);
	downBtn->setPixmap(SmallIcon("down"));
//	downBtn->setEnabled(false);

		
	// Shows the text:
	// "Drop an image or html file"
	QString path = KGlobal::dirs()->findResourceDir( "data", "kimagemapeditor/dropimage.png" ) + "kimagemapeditor/dropimage.png";
	drawZone->setPicture(QImage(path));
	
	splitter->setResizeMode(drawZone,QSplitter::Stretch);
	splitter->setResizeMode(vbox,QSplitter::KeepSize);
	
	areas = new AreaList();
	currentSelected= new AreaSelection();
	_currentToolType=KImageMapEditor::Selection;
	copyArea=0L;
	defaultArea=0L;
	_stdout=false;
	modified=false;
	setupActions();
	setupStatusBar();
	createGUI("kimagemapeditorui.rc");
  setCentralWidget(splitter);
	readProperties(kapp->config());
	popupMenu = static_cast<QPopupMenu *>(factory()->container("popup_main", this));
}

KImageMapEditor::~KImageMapEditor() {
	delete accel;
	delete areas;
	delete currentSelected;
	
}

void KImageMapEditor::setupActions() {
	// File Quit
	(void)KStdAction::quit(kapp, SLOT(closeAllWindows()),actionCollection());

	// File Open
  KAction *temp=KStdAction::open(this, SLOT(slotOpenFile()), actionCollection());
  QMimeSourceFactory::defaultFactory()->setPixmap( "openimage", SmallIcon("fileopen") );
	temp->setWhatsThis(i18n("<h3><img source=\"openimage\"> Open File</h3>Click this to <em>open</em> a new picture or html file."));
	temp->setToolTip(i18n("Open new picture or html file"));
	
	// File Open Recent
  recentFilesAction = KStdAction::openRecent(this, SLOT(slotOpenFile(const KURL&)),
                                      actionCollection());
	// File Save
  saveAction=KStdAction::save(this, SLOT(slotSaveFile()), actionCollection());
  QMimeSourceFactory::defaultFactory()->setPixmap( "saveimage", SmallIcon("filesave") );
	saveAction->setWhatsThis(i18n("<h3><img source=\"saveimage\"> Save Imagemap</h3>Click this to <em>save the imagemap</em>."));
	saveAction->setToolTip(i18n("Save imagemap"));
	// File Save As
  (void)KStdAction::saveAs(this, SLOT(slotSaveFileAs()), actionCollection());

	// File Close
  temp=KStdAction::close(this, SLOT(slotCloseFile()), actionCollection());
  QMimeSourceFactory::defaultFactory()->setPixmap( "closeimage", SmallIcon("fileclose") );
	temp->setWhatsThis(i18n("<h3><img source=\"closeimage\"> Close File</h3>Click this to <em>close</em> the open picture or html file."));
	temp->setToolTip(i18n("Close picture or html file"));


  // Edit Copy
  copyAction=KStdAction::copy(this, SLOT(slotCopy()), actionCollection());
  QMimeSourceFactory::defaultFactory()->setPixmap( "editcopyimage", SmallIcon("editcopy") );
	copyAction->setWhatsThis(i18n("<h3><img source=\"editcopyimage\"> Copy</h3>"
													"Click this to <em>copy</em> the selected area."));
	copyAction->setEnabled(false);													
					
	// Edit Cut																					
  cutAction=KStdAction::cut(this, SLOT(slotCut()), actionCollection());
  QMimeSourceFactory::defaultFactory()->setPixmap( "editcutimage", SmallIcon("editcut") );
	cutAction->setWhatsThis(i18n("<h3><img source=\"editcutimage\"> Cut</h3>"
													"Click this to <em>cut</em> the selected area."));
	cutAction->setEnabled(false);													
	
	// Edit Paste													
  pasteAction=KStdAction::paste(this, SLOT(slotPaste()), actionCollection());
  QMimeSourceFactory::defaultFactory()->setPixmap( "editpasteimage", SmallIcon("editpaste") );
	pasteAction->setWhatsThis(i18n("<h3><img source=\"editpasteimage\"> Paste</h3>"
													"Click this to <em>paste</em> the copied area."));
	pasteAction->setEnabled(false);													


  // Edit Delete
  deleteAction=new KAction(i18n("&Delete"), "editdelete",
              Key_Delete,this,SLOT (slotDelete()),actionCollection(), "edit_delete");
  QMimeSourceFactory::defaultFactory()->setPixmap( "editdeleteimage", SmallIcon("editdelete") );
	deleteAction->setWhatsThis(i18n("<h3><img source=\"editdeleteimage\"> Delete</h3>"
													"Click this to <em>delete</em> the selected area."));
	deleteAction->setEnabled(false);													
	
	// Edit Undo/Redo													
	_commandHistory = new KCommandHistory( actionCollection(), true);
			
	// Edit Properties
	areaPropertiesAction= new KAction(i18n("Pr&operties"),0,this,SLOT(showTagEditor()),
		actionCollection(), "edit_properties");
	areaPropertiesAction->setEnabled(false);		
													
	// View Zoom In													
  zoomInAction=KStdAction::zoomIn(this, SLOT(slotZoomIn()), actionCollection());
	// View Zoom Out													
  zoomOutAction=KStdAction::zoomOut(this, SLOT(slotZoomOut()), actionCollection());
	
	// View Zoom
  zoomAction=new KSelectAction(i18n("Zoom"), 0,this,SLOT (slotZoom()),
      actionCollection(), "view_zoom");
	zoomAction->setWhatsThis(i18n("<h3>Zoom</h3>"
													"Choose the desired zoom level."));
	zoomAction->setItems(QStringList()
		 << i18n("25 %")
		 << i18n("50 %")
		 << i18n("100 %")
		 << i18n("150 %")
		 << i18n("200 %")
		 << i18n("250 %")
		 << i18n("300 %")
		 << i18n("500 %")
		 << i18n("750 %")
		 << i18n("1000 %"));
		
	zoomAction->setCurrentItem(2);
	zoomAction->setComboWidth(50);

	highlightAreasAction = new KToggleAction(i18n("Highlight Areas"),0, this, SLOT (slotHightlightAreas()),
	    actionCollection(), "view_highlightareas");
  	
	showAltAction = new KToggleAction(i18n("Show Alt Tag"),0, this, SLOT (slotShowAltTag()),
	    actionCollection(), "view_showalt");
	
	
	temp= new KAction(i18n("Map &Name"),0,this,SLOT(slotEditMapName()),
		actionCollection(), "map_name");
	
	temp= new KAction(i18n("Edit &Default Area"),0,this,SLOT(slotEditDefaultArea()),
		actionCollection(), "map_defaultarea");
	
	temp= new KAction(i18n("Show &HTML"),0,this,SLOT(slotShowHTML()),
		actionCollection(), "map_showhtml");
	
		
	// Selection Tool
  arrowAction=new KRadioAction(i18n("&Selection"), "arrow",
             0,this,SLOT (slotDrawArrow()),
      actionCollection(), "tool_arrow");
  QMimeSourceFactory::defaultFactory()->setPixmap( "arrowimage", SmallIcon("arrow") );
	arrowAction->setWhatsThis(i18n("<h3><img source=\"arrowimage\"> Selection</h3>"
													"Click this to select areas."));
	arrowAction->setExclusiveGroup("drawing");
	arrowAction->setChecked(true);

	// Circle
  circleAction=new KRadioAction(i18n("&Circle"), "circle",
             0,this,SLOT (slotDrawCircle()),
      actionCollection(), "tool_circle");
  QMimeSourceFactory::defaultFactory()->setPixmap( "circleimage", SmallIcon("drawcircle") );
	circleAction->setWhatsThis(i18n("<h3><img source=\"circleimage\"> Circle</h3>"
													"Click this to start drawing a circle."));
	circleAction->setExclusiveGroup("drawing");
													
	// Rectangle
  rectangleAction=new KRadioAction(i18n("&Rectangle"), "rectangle",
             0,this,SLOT (slotDrawRectangle()),
      actionCollection(), "tool_rectangle");
  QMimeSourceFactory::defaultFactory()->setPixmap( "rectangleimage", SmallIcon("drawrectangle") );
	rectangleAction->setWhatsThis(i18n("<h3><img source=\"rectangleimage\"> Rectangle</h3>"
													"Click this to start drawing a rectangle."));
	rectangleAction->setExclusiveGroup("drawing");

	// Polygon
  polygonAction=new KRadioAction(i18n("&Polygon"), "polygon",
             0,this,SLOT (slotDrawPolygon()),
      actionCollection(), "tool_polygon");
  QMimeSourceFactory::defaultFactory()->setPixmap( "polygonimage", SmallIcon("drawpolygon") );
	polygonAction->setWhatsThis(i18n("<h3><img source=\"polygonimage\"> Polygon</h3>"
													"Click this to start drawing a polygon."));
	polygonAction->setExclusiveGroup("drawing");
	
	// Freehand
  freehandAction=new KRadioAction(i18n("&Freehandpolygon"), "freehand",
             0,this,SLOT (slotDrawFreehand()),
      actionCollection(), "tool_freehand");
  QMimeSourceFactory::defaultFactory()->setPixmap( "freehandimage", SmallIcon("freehand") );
	freehandAction->setWhatsThis(i18n("<h3><img source=\"freehandimage\"> Freehandpolygon</h3>"
													"Click this to start drawing a freehand polygon."));
	freehandAction->setExclusiveGroup("drawing");

	// Add Point
  addPointAction=new KRadioAction(i18n("&Add Point"), "addpoint",
             0,this,SLOT (slotDrawAddPoint()),
      actionCollection(), "tool_addpoint");
  QMimeSourceFactory::defaultFactory()->setPixmap( "addpointimage", SmallIcon("addpoint") );
	addPointAction->setWhatsThis(i18n("<h3><img source=\"addpointimage\"> Add Point</h3>"
													"Click this to add points to a polygon."));
	addPointAction->setExclusiveGroup("drawing");
		
	// Remove Point
  removePointAction=new KRadioAction(i18n("&Remove Point"), "removepoint",
             0,this,SLOT (slotDrawRemovePoint()),
      actionCollection(), "tool_removepoint");
  QMimeSourceFactory::defaultFactory()->setPixmap( "removepointimage", SmallIcon("removepoint") );
	removePointAction->setWhatsThis(i18n("<h3><img source=\"removepointimage\"> Remove Point</h3>"
													"Click this to remove points from a polygon."));
	removePointAction->setExclusiveGroup("drawing");
	
	accel = new KAccel(this);
	
	KAction *cancelAction = new KAction(i18n("Cancel Drawing"), Key_Escape, this, SLOT( slotCancelDrawing() ),
															actionCollection(), "canceldrawing" );
	cancelAction->plugAccel(accel, true);
	
	moveLeftAction = new KAction(i18n("Move Left"), Key_Left, this, SLOT( slotMoveLeft() ),
															 actionCollection() , "moveleft" );
	moveLeftAction->plugAccel(accel, true);
															
	moveRightAction = new KAction(i18n("Move Right"), Key_Right, this, SLOT( slotMoveRight() ),
															 actionCollection() , "moveright" );
	moveRightAction->plugAccel(accel, true);

	moveUpAction = new KAction(i18n("Move Up"), Key_Up, this, SLOT( slotMoveUp() ),
															 actionCollection() , "moveup" );
	moveUpAction->plugAccel(accel, true);

	moveDownAction = new KAction(i18n("Move Down"), Key_Down, this, SLOT( slotMoveDown() ),
															 actionCollection() , "movedown" );
	moveDownAction->plugAccel(accel, true);
			
	increaseWidthAction = new KAction(i18n("Increase Width"), Key_Right + SHIFT, this, SLOT( slotIncreaseWidth() ),
															 actionCollection() , "increasewidth" );
	increaseWidthAction->plugAccel(accel, true);

	decreaseWidthAction = new KAction(i18n("Decrease Width"), Key_Left + SHIFT, this, SLOT( slotDecreaseWidth() ),
															 actionCollection() , "Decreasewidth" );
	decreaseWidthAction->plugAccel(accel, true);
	
	increaseHeightAction = new KAction(i18n("Increase Height"), Key_Up + SHIFT, this, SLOT( slotIncreaseHeight() ),
															 actionCollection() , "increaseheight" );
	increaseHeightAction->plugAccel(accel, true);
	
	decreaseHeightAction = new KAction(i18n("Decrease Height"), Key_Down + SHIFT, this, SLOT( slotDecreaseHeight() ),
															 actionCollection() , "decreaseheight" );
	decreaseHeightAction->plugAccel(accel, true);

	toFrontAction = new KAction(i18n("Bring to front"), 0 , this, SLOT( slotToFront() ),
															 actionCollection() , "tofront" );
	
	toBackAction  = new KAction(i18n("To Back"), 0 , this, SLOT( slotToBack() ),
															 actionCollection() , "toback" );
	
	forwardOneAction  = new KAction(i18n("Forward One"), "raise" ,0, this, SLOT( slotForwardOne() ),
															 actionCollection() , "forwardone" );
	backOneAction = new KAction(i18n("Back One"), "lower" ,0, this, SLOT( slotBackOne() ),
															 actionCollection() , "backone" );

	forwardOneAction->plug(upBtn);
	backOneAction->plug(downBtn);			
	
	connect( upBtn, SIGNAL(pressed()), forwardOneAction, SLOT(activate()));												
	connect( downBtn, SIGNAL(pressed()), backOneAction, SLOT(activate()));												
		
			
	(void)KStdAction::showToolbar(this, SLOT(slotToggleToolbar()), actionCollection());
  (void)KStdAction::showStatusbar(this, SLOT(slotToggleStatusbar()), actionCollection());
  (void)KStdAction::keyBindings(this, SLOT(slotEditKeys()), actionCollection());
	(void)KStdAction::configureToolbars(this, SLOT(slotEditToolbars()), actionCollection());	
	(void)KStdAction::preferences(this, SLOT(slotShowPreferences()), actionCollection());
	
	updateActionAccess();
}

void KImageMapEditor::setupStatusBar() {
	statusBar()->insertItem(i18n(" Cursor")+" : x: 0 ,y: 0",STATUS_CURSOR);
	statusBar()->insertItem(i18n(" Selection")+" : - ",STATUS_SELECTION);
}

void KImageMapEditor::slotShowPopupMenu(const QPoint & pos) {
	popupMenu->popup(pos);
}

void KImageMapEditor::slotShowPopupMenu(QListViewItem* item,const QPoint & p)
{
	if (!item->isSelected())
	{
  	deselectAll();
		select(item);
	}
	slotShowPopupMenu(p);
}

void KImageMapEditor::slotChangeStatusCoords(int x,int y) {
	statusBar()->changeItem(QString(" Cursor : x: %1 ,y: %2 ").arg(x).arg(y),STATUS_CURSOR);
}

void KImageMapEditor::slotUpdateSelectionCoords() {
	if (selected()->count()>0) {
		QRect r=selected()->rect();
		statusBar()->changeItem(
		  QString(" Selection : x: %1 ,y: %2, w: %3, h: %4 ").arg(r.left()).arg(r.top()).arg(r.width()).arg(r.height())
		  ,STATUS_SELECTION);
		kapp->processEvents();
	} else
		statusBar()->changeItem(" Selection : - ",STATUS_SELECTION);
}

void KImageMapEditor::slotUpdateSelectionCoords( const QRect & r )
{
	statusBar()->changeItem(
	  QString(" Selection : x: %1 ,y: %2, w: %3, h: %4 ").arg(r.left()).arg(r.top()).arg(r.width()).arg(r.height())
	  ,STATUS_SELECTION);
	kapp->processEvents();
}

KApplication* KImageMapEditor::app() const
{
  return kapp;
}

void KImageMapEditor::addArea(Area* area) {
	if (!area) return;
	
	// Perhaps we've got a selection of areas
	// so test it and add all areas of the selection
	// nested selections are possible but doesn't exist
	AreaSelection *selection=0L;
	if ( (selection = dynamic_cast <AreaSelection*> ( area ) ) )
	{
		AreaList list = selection->getAreaList();
		
		for (Area* a = list.first(); a != 0L; a = list.next() )
		{
		 	areas->prepend(a);
  		a->setListViewItem(new QListViewItem(listView,a->attribute("href")));
	  	a->listViewItem()->setPixmap(1,makeListViewPix(*a));
		}
	}
	else
	{
  	areas->prepend(area);
  	area->setListViewItem(new QListViewItem(listView,area->attribute("href")));
  	area->listViewItem()->setPixmap(1,makeListViewPix(*area));
  }
	
	modified=true;

}

void KImageMapEditor::addAreaAndEdit(Area* s)
{
	areas->prepend(s);
	s->setListViewItem(new QListViewItem(listView,s->attribute("href")));
	s->listViewItem()->setPixmap(1,makeListViewPix(*s));
	deselectAll();
	select(s);
	if (!showTagEditor(selected())) {
		// If the user has pressed cancel
		// he undos the creation
		commandHistory()->undo();
	}
}

void KImageMapEditor::deleteArea( Area * area )
{
	if (!area) return;
	
  // only for repaint reasons
	QRect redrawRect = area->selectionRect();
	
	// Perhaps we've got a selection of areas
	// so test it and delete the whole selection
	// nested selections are possible but doesn't exist
	AreaSelection *selection=0L;
		if ( (selection = dynamic_cast <AreaSelection*> ( area ) ) )
	{
		AreaList list = selection->getAreaList();
		
		for (Area* a = list.first(); a != 0L; a = list.next() )
		{
		  currentSelected->remove(a);
			areas->remove( a );
			a->deleteListViewItem();
		}
	}
	else
	{
		deselect( area );
		areas->remove( area );  	
   	area->deleteListViewItem();
  }
	
	drawZone->repaintRect(redrawRect);
	
	
	// Only to disable cut and copy actions
	if (areas->count()==0)
		deselectAll();
		
	modified=true;
}

void KImageMapEditor::deleteSelected() {
	
	Area *a;
	AreaList list=currentSelected->getAreaList();
	
  for ( a=list.first(); a != 0; a=list.next() )	{
  	currentSelected->remove( a );
		areas->remove( a );  	
		delete a->listViewItem();
  }

	
	drawZone->repaintArea( *currentSelected );
	// Only to disable cut and copy actions
	if (areas->count()==0)
		deselectAll();
		
	modified=true;
}

void KImageMapEditor::deleteAllAreas()
{
	for (Area* a=areas->first();a!=0L;)
	{
		deselect( a );
		areas->remove( a );  	
   	a->deleteListViewItem();
		a=areas->first(); // because the current is deleted
	}

	drawZone->viewport()->repaint();
	
}

void KImageMapEditor::updateAllAreas()
{
	for (Area* a=areas->first();a!=0L;a=areas->next()) {
		a->listViewItem()->setPixmap(1,makeListViewPix(*a));
	}
}

void KImageMapEditor::updateSelection() const {
	listView->triggerUpdate();
}

AreaSelection* KImageMapEditor::selected() const {
	return currentSelected;
}

void KImageMapEditor::select(Area* a)
{
	if (!a) return;
	
	currentSelected->add(a);
	updateActionAccess();
//  slotUpdateSelectionCoords();
//	drawZone->repaintArea( *a);

}

void KImageMapEditor::selectWithoutUpdate(Area* a)
{
	if (!a) return;
	currentSelected->add(a);
}

void KImageMapEditor::slotSelectionChanged()
{
	AreaListIterator it = areaList();
	AreaList list = currentSelected->getAreaList();
	
	for ( ; it.current() != 0L; ++it)
	{
		if ( it.current()->listViewItem()->isSelected() != (list.containsRef(it.current()) > 0) )
		{
			it.current()->listViewItem()->isSelected()
				? select( it.current() )
				:	deselect( it.current() );
			
			drawZone->repaintArea( *it.current());
		}
	}
	
}

void KImageMapEditor::select( QListViewItem* item)
{
	
	AreaListIterator it = areaList();

	for ( ; it.current() != 0L; ++it)
	{
		if (it.current()->listViewItem() == item )
		{
			select( it.current() );
			drawZone->repaintArea( *it.current());
		}
	}


}

AreaListIterator KImageMapEditor::areaList() const {
	AreaListIterator it(*areas);
	return it;
}


void KImageMapEditor::slotAreaChanged(Area *area)
{
	if (!area) return;
	modified=true;

	AreaSelection *selection=0L;
	if ( (selection = dynamic_cast <AreaSelection*> ( area ) ) )
	{
		AreaListIterator it = selection->getAreaListIterator();
		
		for ( ; it.current() != 0L; ++it )
		{
			if (it.current()->listViewItem()) {
  			it.current()->listViewItem()->setText(0,it.current()->attribute("href"));
  			it.current()->listViewItem()->setPixmap(1,makeListViewPix(*it.current()));
			}
		}
		
	}
	else
	if (area->listViewItem()) {
		area->listViewItem()->setText(0,area->attribute("href"));
		area->listViewItem()->setPixmap(1,makeListViewPix(*area));
	}
	
	drawZone->repaintArea(*area);

}

void KImageMapEditor::deselect(Area* a)
{
	if (a) {
		currentSelected->remove(a);
//		drawZone->repaintArea(*a);
		updateActionAccess();	
    slotUpdateSelectionCoords();
	}
}

void KImageMapEditor::deselectWithoutUpdate(Area* a)
{
	if (a) {
		currentSelected->remove(a);
	}
}


/**
* Makes sure, that the actions cut, copy, delete and
* show properties
* can only be executed if sth. is selected.
**/
void KImageMapEditor::updateActionAccess()
{
	if ( 0 < selected()->count() )
	{
		areaPropertiesAction->setEnabled(true);
		deleteAction->setEnabled(true);
 		copyAction->setEnabled(true);													
 		cutAction->setEnabled(true);		
	  moveLeftAction->setEnabled(true);
	  moveRightAction->setEnabled(true);
	  moveUpAction->setEnabled(true);
	  moveDownAction->setEnabled(true);
 		toFrontAction->setEnabled(true);
 		toBackAction->setEnabled(true);
 		
 		if ( (selected()->count() == 1) )
 		{
 		  if (selected()->type()==Area::Polygon)
 		  {
   			increaseWidthAction->setEnabled(false);
        decreaseWidthAction->setEnabled(false);
        increaseHeightAction->setEnabled(false);
        decreaseHeightAction->setEnabled(false);
        addPointAction->setEnabled(true);
        removePointAction->setEnabled(true);
 		  }
   		else
   		{	
     		increaseWidthAction->setEnabled(true);
        decreaseWidthAction->setEnabled(true);
        increaseHeightAction->setEnabled(true);
        decreaseHeightAction->setEnabled(true);
        addPointAction->setEnabled(false);
        removePointAction->setEnabled(false);
      }

 		}
 		else
 		{
 			increaseWidthAction->setEnabled(false);
      decreaseWidthAction->setEnabled(false);
      increaseHeightAction->setEnabled(false);
      decreaseHeightAction->setEnabled(false);
      addPointAction->setEnabled(false);
      removePointAction->setEnabled(false);
 		}
 		
	}
	else
	{
		areaPropertiesAction->setEnabled(false);
		deleteAction->setEnabled(false);
 		copyAction->setEnabled(false);													
 		cutAction->setEnabled(false);		
	  moveLeftAction->setEnabled(false);
	  moveRightAction->setEnabled(false);
	  moveUpAction->setEnabled(false);
	  moveDownAction->setEnabled(false);
		increaseWidthAction->setEnabled(false);
    decreaseWidthAction->setEnabled(false);
    increaseHeightAction->setEnabled(false);
    decreaseHeightAction->setEnabled(false);
 		toFrontAction->setEnabled(false);
 		toBackAction->setEnabled(false);
    addPointAction->setEnabled(false);
    removePointAction->setEnabled(false);
 		
	}
	
	updateUpDownBtn();
}

void KImageMapEditor::updateUpDownBtn()
{
 	AreaList list = currentSelected->getAreaList();
	
 	if (list.isEmpty() || (areas->count() < 2))
 	{
 		forwardOneAction->setEnabled(false);													
		upBtn->setEnabled(false);
 		backOneAction->setEnabled(false);
 		downBtn->setEnabled(false);
	  return; 	
 	}
 	// if the first Area is in the selection can't move up	
 	if (list.find( areas->getFirst() ) == -1)
 	{
 		forwardOneAction->setEnabled(true);
 		upBtn->setEnabled(true);
 	}
 	else {
 		forwardOneAction->setEnabled(false);													
		upBtn->setEnabled(false);
 	}
		
 	drawZone->repaintArea(*currentSelected);
 		
 	// if the last Area is in the selection can't move down
 	if (list.find( areas->getLast() ) == -1)
  {
 		backOneAction->setEnabled(true);
 		downBtn->setEnabled(true);
 	}
 	else {
 		backOneAction->setEnabled(false);
 		downBtn->setEnabled(false);
 	}

}

void KImageMapEditor::deselectAll()
{
	QRect redrawRect= currentSelected->selectionRect();
	currentSelected->reset();
	drawZone->repaintRect(redrawRect);
	updateActionAccess();
}

Area* KImageMapEditor::onArea(const QPoint & p) const {
	for (Area* s=areas->first();s!=0L;s=areas->next()) {
		if (s->contains(p))
			return s;
	}
	return 0L;
}

void KImageMapEditor::slotShowPreferences() {
	PreferencesDialog *dialog = new PreferencesDialog(this);
	connect(dialog, SIGNAL(applyClicked()), this, SLOT(slotConfigChanged()));
	dialog->exec();
	delete dialog;
}

int KImageMapEditor::showTagEditor(Area *a) {
	if (!a) return 0;
	drawZone->repaintArea(*a);

	AreaDialog *dialog= new AreaDialog(this,a);
	connect (dialog, SIGNAL(areaChanged(Area*)), this, SLOT(slotAreaChanged(Area*)));
	
	int result = dialog->exec();
	
	return result;


}

int KImageMapEditor::showTagEditor(QListViewItem *item) {
	if (!item) return 0;
	for (Area* a=areas->first();a!=0L;a=areas->next()) {
		if (a->listViewItem()==item) {
			return showTagEditor(a);
		}
	}
	return 0;
}

int KImageMapEditor::showTagEditor() {
	return showTagEditor(selected());
}


QString KImageMapEditor::getHTMLImageMap() const {
	QString retStr;
	retStr+="<map "+QString("name=\"")+_mapName+"\">\n";
	
	for (Area* a=areas->first();a!=0L;a=areas->next()) {
		retStr+="  "+a->getHTMLCode()+"\n";
	}
	
	if (defaultArea && defaultArea->finished())
		retStr+="  "+defaultArea->getHTMLCode()+"\n";
	
	retStr+="</map>\n";
	return retStr;
}

QPixmap KImageMapEditor::makeListViewPix(Area & a)
{
	QPixmap pix=a.cutOut(drawZone->picture());
	
	double shrinkFactor=1;

	// picture fits into max row height ?		
	if (maxAreaPreviewHeight < pix.height())
		shrinkFactor = ( (double) maxAreaPreviewHeight / pix.height() );
		
	QPixmap pix2(pix.width()*shrinkFactor, pix.height()*shrinkFactor);
	
	// Give all pixels a defined color
	pix2.fill(Qt::white);
	
	QPainter p(&pix2);
	
	p.scale(shrinkFactor,shrinkFactor);
	p.drawPixmap(0,0,pix);
	
	return pix2;
}


void KImageMapEditor::setPicture(const KURL & url) {
	_imageUrl=url;
	setPicture(QImage(url.path()));
}

void KImageMapEditor::setPicture(const QImage & pix) {
	drawZone->setPicture(pix);
}


void KImageMapEditor::slotDrawArrow() {
	_currentToolType=KImageMapEditor::Selection;
	
}

void KImageMapEditor::slotDrawCircle() {
	_currentToolType=KImageMapEditor::Circle;
	
}

void KImageMapEditor::slotDrawRectangle() {
	_currentToolType=KImageMapEditor::Rectangle;
	
}

void KImageMapEditor::slotDrawPolygon() {
	_currentToolType=KImageMapEditor::Polygon;
}

void KImageMapEditor::slotDrawFreehand() {
	_currentToolType=KImageMapEditor::Freehand;
}

void KImageMapEditor::slotDrawAddPoint() {
	_currentToolType=KImageMapEditor::AddPoint;
}

void KImageMapEditor::slotDrawRemovePoint() {
	_currentToolType=KImageMapEditor::RemovePoint;
}


void KImageMapEditor::slotZoom() {

	int i=zoomAction->currentItem();
	switch (i) {
		case 0 : drawZone->setZoom(0.25);break;
		case 1 : drawZone->setZoom(0.5);break;
		case 2 : drawZone->setZoom(1);break;
		case 3 : drawZone->setZoom(1.5);break;
		case 4 : drawZone->setZoom(2.0);break;
		case 5 : drawZone->setZoom(2.5);break;
		case 6 : drawZone->setZoom(3);break;
		case 7 : drawZone->setZoom(5);break;
		case 8 : drawZone->setZoom(7.5);break;
		case 9 : drawZone->setZoom(10);break;
	}
	if (i<10)
  	zoomInAction->setEnabled(true);
  else
  	zoomInAction->setEnabled(false);
  	
  if (i>0)
		zoomOutAction->setEnabled(true);
	else
		zoomOutAction->setEnabled(false);
}

void KImageMapEditor::slotZoomIn() {
	if (zoomAction->currentItem()==(int)(zoomAction->items().count()-1))
		return;
		
	zoomAction->setCurrentItem(zoomAction->currentItem()+1);
	slotZoom();
}

void KImageMapEditor::slotZoomOut() {
	if (zoomAction->currentItem()==0)
		return;

	zoomAction->setCurrentItem(zoomAction->currentItem()-1);
	slotZoom();
}

void KImageMapEditor::slotEditDefaultArea()
{
	if (defaultArea)
		showTagEditor(defaultArea);
	else {
		defaultArea= new DefaultArea();
		showTagEditor(defaultArea);
	}
		
}

void KImageMapEditor::slotEditMapName()
{
	bool ok=false;
	QString input = KLineEditDlg::getText(i18n("Enter the name of the map"),_mapName,&ok,this);	
	if (ok)
		_mapName =input;
}

void KImageMapEditor::slotShowHTML()
{
	KDialogBase *dialog= new KDialogBase(this,"",true,i18n("HTML code of the map"),KDialogBase::Ok);
	QMultiLineEdit *edit = new QMultiLineEdit(dialog);
	
	edit->setText(getHTMLImageMap());
	edit->setReadOnly(true);
	dialog->setMainWidget(edit);
//  dialog->resize(dialog->calculateSize(edit->maxLineWidth(),edit->numLines()*));
//	dialog->adjustSize();
	dialog->resize(600,400);
	dialog->exec();
}


void KImageMapEditor::slotCut()
{
	if ( 0 == currentSelected->count() )
		return;
	if (copyArea)
		delete copyArea;
				
	copyArea= static_cast< AreaSelection* > (currentSelected->clone());
	pasteAction->setEnabled(true);	
	KCommand *command= new CutCommand(this,*currentSelected);
	commandHistory()->addCommand( command ,true);	
}


void KImageMapEditor::slotDelete()
{
	if ( 0 == currentSelected->count() )
		return;
	
	KCommand *command= new DeleteCommand(this,*currentSelected);
	commandHistory()->addCommand( command ,true);	
}

void KImageMapEditor::slotCopy()
{
	if (copyArea)
		delete copyArea;
	
	copyArea = static_cast< AreaSelection* > (currentSelected->clone());
	pasteAction->setEnabled(true);	
}

void KImageMapEditor::slotPaste()
{
	if (!copyArea)
		return;

	copyArea->moveBy(5,5);
  if (copyArea->rect().x()>= drawZone->getImageRect().width() ||
      copyArea->rect().y()>= drawZone->getImageRect().height())
      copyArea->moveTo(0,0);

  if (copyArea->rect().width()>drawZone->getImageRect().width() ||
      copyArea->rect().height()>drawZone->getImageRect().height())
      return;

	AreaSelection *a=static_cast< AreaSelection* > (copyArea->clone());
	commandHistory()->addCommand( new PasteCommand(this,*a),true);	
	delete a;
//	addAreaAndEdit(a);
}

void KImageMapEditor::slotCloseFile()
{
	QString path = KGlobal::dirs()->findResourceDir( "data", "kimagemapeditor/dropimage.png" ) + "kimagemapeditor/dropimage.png";
	drawZone->setPicture(QImage(path));
  deleteAllAreas();
}


void KImageMapEditor::slotOpenFile()
{
	KURL url=KFileDialog::getOpenURL(QString::null,
					"*.png *.jpg *.jpeg *.gif *.htm *.html|Web file\n"
					"*.png *.jpg *.jpeg *.gif *.bmp *.xbm *.xpm *.pnm *.mng|Image\n"
					"*.htm *.html|HTML-File\n"
					"*.png|PNG-Image\n*.jpg *.jpeg|JPEG-Image\n*.gif|GIF-Image\n*|All files"
					,this,i18n("Choose a picture to open"));
	if (!url.isEmpty()) {
    slotOpenFile(url);
    recentFilesAction->addURL(url);
	}
}

void KImageMapEditor::slotOpenFile(const KURL & url)
{
	QString ext=QFileInfo(url.path()).extension().lower();

  QFileInfo fileInfo(url.path());

  if ( !fileInfo.exists() )
  {
    	KMessageBox::information(this,
        i18n("The file <b> %1 </b> does not exists!").arg(fileInfo.fileName()),
        i18n("File does not exists!"));
      return;
  }
	
	if (ext=="png" || ext=="jpg" || ext=="jpeg" || ext=="gif" ||
			ext=="bmp" || ext=="xbm" || ext=="xpm" || ext=="mng" || ext=="pnm")
	{
		setPicture(url);
		updateAllAreas();
	} else
		openHTMLFile(url);

	drawZone->viewport()->repaint();
}

void KImageMapEditor::openHTMLFile(const KURL & url)
{	
	QFile f(url.path());
	f.open(IO_ReadOnly);
	QTextStream s(&f);
	QString str;
	QChar w;
	QDict<QString> *attr=0L;
	QList<ImageTag> *images= new QList<ImageTag>;
	MapTag *map=0L;
	QList<MapTag> *maps = new QList<MapTag>;
	
	
	bool readMap=false;
	
	while (!s.atEnd()) {
		
 		s >> w;
		if (w=="<") {
			attr=new QDict<QString>(getTagAttributes(s));
			
			if (attr->find("tagname")) {
 				
  			if (attr->find("tagname")->lower()=="img") {
  				images->append(attr);
  			} else
  			if (attr->find("tagname")->lower()=="map") {
  				map = new MapTag();
  				map->name=(*attr->find("name"));
  				readMap=true;
  			} else
  			if (attr->find("tagname")->lower()=="/map") {
  				readMap=false;
  				if (maps)
  					maps->append(map);
				} else
				if (readMap && attr->find("tagname")->lower()=="area") {
  				map->prepend(attr);
				}
				
			}
		}		
	}

	f.close();
	
	ImageMapChooseDialog *dialog = new ImageMapChooseDialog(this,maps,images,url);
	dialog->exec();
	map=dialog->currentMap;
	
	KURL imageUrl=dialog->pixUrl;
	
  // Remove old areas only if a new map is loaded
	if (map)
	   deleteAllAreas();
	
	// Only update the image if an image was choosen
	if (!imageUrl.isEmpty()) {
		setPicture(imageUrl);
	}
		
	if (map) {
		
		setMapName(map->name);
  	for (AreaTag *tag=map->first();tag!=0L;tag=map->next())
  	{
    		QString shape="rect";
    		if (tag->find("shape"))
    			shape=*tag->find("shape");
    			
    		Area::ShapeType type=Area::Rectangle;
    		if (shape=="circle")
    			type=Area::Circle;
    		else if (shape=="poly")
    			type=Area::Polygon;
    		else if (shape=="default")
    			type=Area::Default;
    			
    		Area* a=AreaCreator::create(type);

    		if (tag->find("href"))
    			a->setAttribute("href",*tag->find("href"));
  	  		
    		if (tag->find("alt"))
    			a->setAttribute("alt",*tag->find("alt"));
    		
    		if (tag->find("target"))
    			a->setAttribute("target",*tag->find("target"));
    		
    		if (tag->find("title"))
    			a->setAttribute("title",*tag->find("title"));

    		if (tag->find("onclick"))
    			a->setAttribute("onclick",*tag->find("onclick"));

    		if (tag->find("onmousedown"))
    			a->setAttribute("onmousedown",*tag->find("onmousedown"));
    			    			    			    			
    		if (tag->find("onmouseup"))
    			a->setAttribute("onmouseup",*tag->find("onmouseup"));
    		
    		if (tag->find("onmouseover"))
    			a->setAttribute("onmouseover",*tag->find("onmouseover"));
    		
    		if (tag->find("onmousemove"))
    			a->setAttribute("onmousemove",*tag->find("onmousemove"));
    		
    		if (tag->find("onmouseout"))
    			a->setAttribute("onmouseout",*tag->find("onmouseout"));
    			    			

    			    			
    		if (type==Area::Default) {
    			defaultArea=a;
    			continue;
    		}
    		
    		if (tag->find("coords"))
    			a->setCoords(*tag->find("coords"));
    			
    		a->setMoving(false);
  			addArea(a);  	
  	}
	}


	
}

QDict<QString> KImageMapEditor::getTagAttributes(QTextStream & s)
{
	QDict<QString> dict(17,false);
	// the "<" is already read
	QChar w;
	QString attr,value;
	
	// get the tagname
	while (!s.atEnd() && w!=" ") {
		s >> w;
		if (w==" " || w==">") {
			dict.insert("tagname",new QString(value));
			break;
		}
 		value+=w;
	}
	
	// do we have a comment ?
	// read the comment and return
	if (value.right(3)=="-->")
		return dict;
		
	if (value.startsWith("!--")) {
		while (!s.atEnd()) {
			s >> w;
			if (w=="-") {
				s >> w;
				if (w=="-") {
					s >> w;
					if (w==">")
						return dict;
				}
			}
		}
	}
	
	bool attrRead=true;	// currently reading an attribute ?
	bool equalSign=false; // an equalsign was read?
	bool valueRead=false; // currently reading a value ?
	bool quotation=false; // currently reading a value with quotation marks ?
	bool php=false; // currently reading a php script
	attr="";
	value="";
	
	//get the other attributes
	while (!s.atEnd() && w!=">")
	{
		s >> w;

		// End of PHP Script ?		
		if (php && (w=="?") )
		{
	    s >> w;
 		  if (valueRead)
          value+=w;		
		
	    if (w==">")
	    {
	      php = false;
		    s >> w;
	    }
		}
		
		// Wrong syntax or PHP-Skript !
		if (!php && (w=="<"))
		{
		  if (valueRead)
         value+=w;		
		  s >> w;
		  if (valueRead)
         value+=w;		
  		
  		if (w=="?")
  		{
		    php = true;
		  }
		} else
		// finished ?
		if (w==">") {
			if (valueRead)
				dict.insert(attr,new QString(value));
				return dict;
		} else
		// currently reading an attribute ?
		if (attrRead) {
			// if there is a whitespace the attributename has finished
			// possibly there isn't any value e.g. noshade
			if (w==" ")
				attrRead=false;
			else
			// an equal sign signals that the value follows
			if (w=="=") {
				attrRead=false;
				equalSign=true;
			} else
				attr+=w;
		} else
		// an equalsign was read ? delete every whitespace
		if (equalSign) {
			if (w!=" ") {
				equalSign=false;
				valueRead=true;
				if (w=="\"" || w=="'")
					quotation=true;
			}
		} else
		// currently reading the value
		if (valueRead) {
      // if php read without regarding anything
      if (php)
				 value+=w;
			// if value within quotation marks is read
			// only stop when another quotationmark is found
			else
			if (quotation) {
				if (w!="\"" && w!="'") {
					value+=w;
				} else {
					quotation=false;
					valueRead=false;
					dict.insert(attr,new QString(value));
					attr="";value="";
					
				}
			} else
		  // a whitespace indicates that the value has finished
			if (w==" ") {
				valueRead=false;
				dict.insert(attr,new QString(value));
				attr="";value="";
			}
		} else {
			if (w!=" ") {
				attrRead=true;
				attr+=w;
			}
		}
	}

	return dict;		
	
}



void KImageMapEditor::slotSaveFile()
{
	if (_mapUrl.isEmpty())
		slotSaveFileAs();
	else
		saveImageMap(_mapUrl);
}

void KImageMapEditor::slotSaveFileAs()
{
  KURL url = KFileDialog::getSaveURL(0L,"*.htm *.html|"+i18n("HTML file")+"\n*.txt|"+
  						i18n("Text file")+"\n*|"+i18n("All Files"),this);
  if (url.isEmpty() || !url.isValid()) {
    return;
  }			


  QFileInfo fileInfo(url.path());

  if(fileInfo.exists() && !fileInfo.isWritable()) {
    KMessageBox::sorry(this, i18n("You do not have write permission to this file"));
    return;
  }

  if ( fileInfo.exists() )
  {
  	if (fileInfo.extension().lower()=="html" ||
  			fileInfo.extension().lower()=="htm")
  	{
    	if (KMessageBox::warningYesNo(this,
        i18n("<b>ATTENTION!</b> The file <b> %1 </b> already exists!<br>"
        		"Even saving to a HTML-file will overwrite it.<br>"
        		"Saving to a HTML-file will be implemented later.<br>"
        		"Do you want to overwrite it?<br>").arg(fileInfo.fileName()),
        i18n("Really overwrite the HTML file ?"))==KMessageBox::No)
        return;
  	} else
  	if (KMessageBox::warningYesNo(this,
      i18n("The file <b> %1 </b> already exists!<br>Do you want to overwrite it?").arg(fileInfo.fileName()),
      i18n("Really overwrite the file ?"))==KMessageBox::No)
      return;
  }
  saveImageMap(url);
}

void KImageMapEditor::saveImageMap(const KURL & url)
{
	QFileInfo fileInfo(url.path());
	QString ext=fileInfo.extension(false).lower();
	modified=false;
  	
 	if (_mapName.isEmpty()) {
 		slotEditMapName();
 	}
 	QFile file(url.path());
 	file.open(IO_WriteOnly);
 	QTextStream t(&file);
 	t << "<html>\n"
 		<< "<head>\n"
 		<< "  <title></title>\n"
 		<< "</head>\n"
 	 	<< "<body>\n"
 	 	<< "  " << getHTMLImageMap()
 	 	<< "\n"
 	 	<< "  <img src=\"" << QExtFileInfo::toRelative(_imageUrl.path(),url.directory()) << "\""
 	 	<< " usemap=\"#" << _mapName << "\""
 	 	<< " width=\"" << drawZone->picture().width() << "\""
 	 	<< " height=\"" << drawZone->picture().height() << "\">\n"
 	 	<< "</body>\n"
 	 	<< "</html>";
		
 	file.close();
	
}

void KImageMapEditor::saveProperties(KConfig *config)
{
	saveMainWindowSettings(config);
	recentFilesAction->saveEntries(config,"Data");
  config->setGroup("Appearance");
	saveWindowSize(config);
  kapp->config()->setGroup("General");
  kapp->config()->writeEntry("highlightareas",highlightAreasAction->isChecked());
  kapp->config()->writeEntry("showalt",showAltAction->isChecked());
	
  config->sync();
}

void KImageMapEditor::readProperties(KConfig *config)
{
	applyMainWindowSettings(config);
  recentFilesAction->loadEntries(config,"Data");
  kapp->config()->setGroup("Appearance");
	restoreWindowSize(config);

  slotConfigChanged();
		
}

void KImageMapEditor::slotConfigChanged()
{
  kapp->config()->setGroup("Appearance");
	int newHeight=kapp->config()->readNumEntry("maximum-preview-height",50);
  kapp->config()->setGroup("General");
	_commandHistory->setUndoLimit(kapp->config()->readNumEntry("undo-level",20));
	_commandHistory->setRedoLimit(kapp->config()->readNumEntry("redo-level",20));
	Area::highlightArea = kapp->config()->readBoolEntry("highlightareas",true);
	highlightAreasAction->setChecked(Area::highlightArea);
	Area::showAlt = kapp->config()->readBoolEntry("showalt",true);
	showAltAction->setChecked(Area::showAlt);
	
	// if the image preview size changed update all images
	if (maxAreaPreviewHeight!=newHeight) {
		maxAreaPreviewHeight=newHeight;
	}
	
	updateAllAreas();	
	drawZone->viewport()->repaint();
	
}

bool KImageMapEditor::queryClose()
{
	if (modified) {
		int response=KMessageBox::warningYesNoCancel(this,
      i18n("The map was modified. Do you want to save it?"),
      i18n("Save modified map ?"));
    if (response==KMessageBox::Yes)
    	slotSaveFile();		
    else
    if (response==KMessageBox::Cancel)
    	return false;
	}
	
	saveProperties(kapp->config());
	if (_stdout) {
		cout << getHTMLImageMap() << endl;
	}
	return true;
}

void KImageMapEditor::slotEditKeys() {
  KKeyDialog::configureKeys(actionCollection(), xmlFile());
}

void KImageMapEditor::slotEditToolbars()
{
  KEditToolbar *dlg = new KEditToolbar(actionCollection());

  if (dlg->exec())
    createGUI("kimagemapeditorui.rc");

  delete dlg;
}

void KImageMapEditor::slotToggleToolbar()
{
	if (toolBar()->isVisible())
		toolBar()->hide();
	else
		toolBar()->show();
}

void KImageMapEditor::slotToggleStatusbar()
{
	if (statusBar()->isVisible())
		statusBar()->hide();
	else
		statusBar()->show();
}

void KImageMapEditor::slotBackOne()
{
	if (currentSelected->isEmpty())
		return;
		
	AreaList list = currentSelected->getAreaList();

		
	Area *a = 0L;
	// move every selected Area one step lower
	for (int i=areas->count()-2; i > -1; i--)
	{
	  if (list.find( areas->at(i) ) > -1 )
	  {
		  a = areas->at(i);
  		areas->remove(a);
	  	areas->insert((uint)i+1,a);
   		a->listViewItem()->moveItem( areas->at(i)->listViewItem() );	
	  }
	}
	// to update the up and down buttons
	updateUpDownBtn();			
	
}

void KImageMapEditor::slotForwardOne()
{
	if (currentSelected->isEmpty())
		return;
		
	AreaList list = currentSelected->getAreaList();
		
	Area *a = 0L;
	// move every selected Area one step higher
	for (int i=1; i < (int)areas->count(); i++)
	{
	  if (list.find( areas->at(i) ) > -1 )
	  {
		  a = areas->at(i);
  		areas->remove(a);
	  	areas->insert((uint)i-1,a);
   		areas->at(i)->listViewItem()->moveItem( a->listViewItem() );	
	  }
	}
	// to update the up and down buttons
	updateUpDownBtn();
}

void KImageMapEditor::slotToBack()
{
	if (currentSelected->isEmpty())
		return;
		
	while (backOneAction->isEnabled())
	  slotBackOne();
}

void KImageMapEditor::slotToFront()
{
	if (currentSelected->isEmpty())
		return;
		
	while (forwardOneAction->isEnabled())
	  slotForwardOne();
}


void KImageMapEditor::slotMoveUp()
{
	QRect r=selected()->rect();
  selected()->setMoving(true);	
	selected()->moveBy(0,-1);
	
	commandHistory()->addCommand(
		new MoveCommand( this, selected(), r.topLeft() ) ,true );
  selected()->setMoving(false);	
	slotAreaChanged(selected());
	slotUpdateSelectionCoords();
}

void KImageMapEditor::slotMoveDown()
{
	QRect r=selected()->rect();
  selected()->setMoving(true);	
	selected()->moveBy(0,1);
	
	commandHistory()->addCommand(
		new MoveCommand( this, selected(), r.topLeft() ) ,true );
  selected()->setMoving(false);	
	slotAreaChanged(selected());
	slotUpdateSelectionCoords();
}

void KImageMapEditor::slotMoveLeft()
{
	QRect r=selected()->rect();
  selected()->setMoving(true);	
	selected()->moveBy(-1,0);
	
	commandHistory()->addCommand(
		new MoveCommand( this, selected(), r.topLeft() ) ,true );
  selected()->setMoving(false);	
	slotAreaChanged(selected());
	slotUpdateSelectionCoords();
}

void KImageMapEditor::slotMoveRight()
{
	QRect r=selected()->rect();
  selected()->setMoving(true);	
 	selected()->moveBy(1,0);
	
	commandHistory()->addCommand(
		new MoveCommand( this, selected(), r.topLeft() ) ,true );
  selected()->setMoving(false);	
	slotAreaChanged(selected());
	slotUpdateSelectionCoords();
}

void KImageMapEditor::slotCancelDrawing()
{
	drawZone->cancelDrawing();
}

void KImageMapEditor::slotIncreaseHeight()
{
	Area *oldArea=selected()->clone();
	
	QRect r = selected()->rect();
	r.setHeight( r.height()+1 );
	r.moveBy(0,-1);
	
	selected()->setRect(r);
	
	commandHistory()->addCommand(
		new ResizeCommand( this, selected(), oldArea ) ,true );
	slotAreaChanged(selected());
	slotUpdateSelectionCoords();
}

void KImageMapEditor::slotDecreaseHeight()
{
	Area *oldArea=selected()->clone();
	
	QRect r = selected()->rect();
	r.setHeight( r.height()-1 );
	r.moveBy(0,1);
	
	selected()->setRect(r);
	
	commandHistory()->addCommand(
		new ResizeCommand( this, selected(), oldArea ) ,true );
	slotAreaChanged(selected());
	slotUpdateSelectionCoords();
}

void KImageMapEditor::slotIncreaseWidth()
{
	Area *oldArea=selected()->clone();
	
	QRect r = selected()->rect();
	r.setWidth( r.width()+1 );
	
	selected()->setRect(r);
	
	commandHistory()->addCommand(
		new ResizeCommand( this, selected(), oldArea ) ,true );
	slotAreaChanged(selected());
	slotUpdateSelectionCoords();
}

void KImageMapEditor::slotDecreaseWidth()
{
	Area *oldArea=selected()->clone();
	
	QRect r = selected()->rect();
	r.setWidth( r.width()-1 );
	
	selected()->setRect(r);
	
	commandHistory()->addCommand(
		new ResizeCommand( this, selected(), oldArea ) ,true );
	slotAreaChanged(selected());
	slotUpdateSelectionCoords();
}

void KImageMapEditor::slotHightlightAreas()
{
  bool b = highlightAreasAction->isChecked();

  highlightAreasAction->setChecked(b);
  Area::highlightArea = b;
  updateAllAreas();
  drawZone->viewport()->repaint();
}

void KImageMapEditor::slotShowAltTag()
{
  bool b = showAltAction->isChecked();
  showAltAction->setChecked(b);
  Area::showAlt = b;
  drawZone->viewport()->repaint();
}




