/*
  Copyright Mission Critical Linux, 2000

  Kimberlite is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2, or (at your option) any
  later version.

  Kimberlite is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Kimberlite; see the file COPYING.  If not, write to the
  Free Software Foundation, Inc.,  675 Mass Ave, Cambridge, 
  MA 02139, USA.
*/

/*
 *
 * Author: Brian Stevens <stevens@missioncriticallinux.com>
 * description: library for retrieving communication status
 * 	and config information
 */

static const char *version __attribute__ ((unused)) = "$Revision: 1.10 $";

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <unistd.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/param.h>
#include <errno.h>
#include <netdb.h>
#include <sys/socket.h>
#include <parseconf.h>
#include <clusterdefs.h>
#include <clucfg.h>
#include <clucfg_if.h>
#include <msgsvc.h>

#define CFG_CLU_NAME	"cluster%name"

#ifdef DEBUG
#define Dprintf(fmt,args...) printf(fmt,##args)
#else
#define Dprintf(fmt,args...)
#endif

/*
 *
 * get_clu_cfg:
 *
 * Description:
 *
 *	Return a description of the cluster configuration.
 *	NOTE: the caller is required to free() the returned data structure!
 *
 * Usage:
 *	cfg = get_clu_cfg(file);
 *
 *		"file" is the path of the cluster configuration file. If NULL,
 * 		the system default file as defined by the system constant
 *		CLU_CONFIG_FILE will be used.
 *		
 * Return values:
 * 
 *	Failure:
 * 		NULL - Failure obtaining cluster information
 * 		errno will be set to one of the following:
 *			ENOMEM - unable to allocate memory
 *			ENOENT - error processing configuration file
 *			EFAULT - unable to find the local node's 
 *				 member description in the config file
 *	
 *	Success:
 *		Pointer to a data area of type CluCfg which has been
 * 		malloc'd.  It is the caller's responsibility to free.
 *	Error: return of NULL and errno set accordingly.
 */

CluCfg *
get_clu_cfg(
    char		*cfg_file)
{
    char		*p;
    int			n, c;
    ChanCfg		*chans;
    char		qstr[80];
    CluCfg		*cinfo = NULL;
    struct ifreq	ifr;

    /*
     * Allow the user to override the default config file
     */

    if (cfg_file == NULL)
	cfg_file = CLU_CONFIG_FILE;

    /*
     * Allocate the cluster info structure if this is the first time
     * we have been called.
     */

    if (cinfo == NULL) {

        cinfo = (CluCfg *) malloc(sizeof(CluCfg) +
		MAX_NODES * sizeof(NodeCfg) +
		MAX_NODES * MAX_CHANNELS * sizeof(ChanCfg));
        if (cinfo == NULL) {
	    errno = ENOMEM;
	    return(NULL);
        }

        cinfo->nodes = (NodeCfg *) &cinfo[1];

	chans = (ChanCfg *) &cinfo->nodes[MAX_NODES];
        for (n = 0; n < MAX_NODES; n++)
            cinfo->nodes[n].chans = &chans[n * MAX_CHANNELS];
    }

    strcpy(cinfo->name, "");
    cinfo->num_nodes = 0;
    cinfo->num_chans = 0;
    cinfo->lid = -1;

    /*
     * Scan in the name of the cluster and its members
     */

    if (CFG_Get((char *) CFG_CLU_NAME, NULL, &p) == CFG_OK) {
        strcpy(cinfo->name, p);
    }

    for (n = 0; n < MAX_NODES; n++) {

	/* name */
	sprintf(qstr, "members%cmember%d%cname", CLU_CONFIG_SEPARATOR, n,
		CLU_CONFIG_SEPARATOR);
        if (CFG_Get((char *) qstr, NULL, &p) != CFG_OK) {
	    break;
        }
        strcpy(cinfo->nodes[n].name, p);
	/* quorumPartitionPrimary */
	sprintf(qstr, "members%cmember%d%cquorumPartitionPrimary", 
		CLU_CONFIG_SEPARATOR, n, CLU_CONFIG_SEPARATOR);
        if (CFG_Get((char *) qstr, NULL, &p) != CFG_OK) {
	    break;
        }
        strcpy(cinfo->nodes[n].quorumPartitionPrimary, p);
	/* quorumPartitionShadow */
	sprintf(qstr, "members%cmember%d%cquorumPartitionShadow", 
		CLU_CONFIG_SEPARATOR, n, CLU_CONFIG_SEPARATOR);
        if (CFG_Get((char *) qstr, NULL, &p) != CFG_OK) {
	    break;
        }
        strcpy(cinfo->nodes[n].quorumPartitionShadow, p);
	/* powerSerialPort */
	sprintf(qstr, "members%cmember%d%cpowerSerialPort", 
		CLU_CONFIG_SEPARATOR, n, CLU_CONFIG_SEPARATOR);
        if (CFG_Get((char *) qstr, NULL, &p) != CFG_OK) {
	    break;
        }
        strcpy(cinfo->nodes[n].powerSerialPort, p);
	/* powerSwitchType */
	sprintf(qstr, "members%cmember%d%cpowerSwitchType", 
		CLU_CONFIG_SEPARATOR, n, CLU_CONFIG_SEPARATOR);
        if (CFG_Get((char *) qstr, NULL, &p) != CFG_OK) {
	    break;
        }
        strcpy(cinfo->nodes[n].powerSwitchType, p);

        /*
         * Determine who we are as we share a common config file with our
         * other cluster member. The cluster name must be associated with
         * an address currently bound to one of the local interfaces.
         */

	if (if_lookup(cinfo->nodes[n].name, &ifr) == 0) 
    	    cinfo->lid = n;

	/*
	 * XXX - this routine was recursively being called causing the number
	 * of nodes to be twice the actual number; resulting in stack 
	 * corruption by offetting into the cinfo struct past the end.
	 */
	if (cinfo->num_nodes < MAX_NODES) {
            cinfo->num_nodes++;
	}
    }

    /*
     * If one of the config file members did not map to us, we are done.
     */

    if (cinfo->lid < 0) {
	errno = EFAULT;
	free(cinfo);
	return(NULL);
    }

    /*
     * Scan in the channel configurations
     */

    for (c = 0; (cinfo->num_chans < MAX_CHANNELS ) && (c < MAX_CHANNELS); c++) {

	for (n = 0; n < cinfo->num_nodes; n++) {
	    sprintf(qstr, "members%cmember%d%cchan%d%ctype",
		    CLU_CONFIG_SEPARATOR, n, CLU_CONFIG_SEPARATOR,
		    c, CLU_CONFIG_SEPARATOR);
	    if (CFG_Get((char *) qstr, (char *) NULL, &p) != CFG_OK) {
		break;
	    }
	    
	    if (!strcmp(p, "net")) {
		/*
		 * We've got ourselves a network interface.  Yee-hah.
		 */

		sprintf(qstr, "members%cmember%d%cchan%d%cname",
		    CLU_CONFIG_SEPARATOR, n, CLU_CONFIG_SEPARATOR,
		    c, CLU_CONFIG_SEPARATOR);
		if (CFG_Get((char *) qstr, (char *) NULL, &p) != CFG_OK) {
		    break;
		}
		/*
		 * If the channel is local, lookup the interface 
		 * name and address.
		 */
		if (cinfo->lid == n) {
		    if (if_lookup(p, &ifr)) 
			break;
		    strcpy(cinfo->nodes[n].chans[cinfo->num_chans].dev.net.if_name,
			   ifr.ifr_name);
		} else {
		    strcpy(cinfo->nodes[n].chans[cinfo->num_chans].dev.net.if_name,
			   "unknown");
		}

		strcpy(cinfo->nodes[n].chans[cinfo->num_chans].dev.net.name,p);
		cinfo->nodes[n].chans[cinfo->num_chans].type = CHAN_NET;

	    } else if (!strcmp(p, "serial")) {
		/*
		 * How cute, a little serial connection.  Awwwh.
		 */
		sprintf(qstr, "members%cmember%d%cchan%d%cdevice",
		    CLU_CONFIG_SEPARATOR, n, CLU_CONFIG_SEPARATOR,
		    c, CLU_CONFIG_SEPARATOR);
		if (CFG_Get((char *) qstr, (char *) NULL, &p) != CFG_OK) {
		    break;
		}

		strcpy(cinfo->nodes[n].chans[cinfo->num_chans].dev.serial.name,
		       p);
		cinfo->nodes[n].chans[cinfo->num_chans].type = CHAN_SERIAL;

	    } else {
		/*
		 * WHAT?  !@#$~
		 */
		Dprintf("Unknown channel type %s\n", p);
	    }

	}

	/*
	 * The following is a bit tricky.  If we went through the above
	 * loop for each node in the cluster, it means we successfully
	 * queried the database for a channel, and filled in our data
	 * structures as well.  Thus, we can bump the number of channels.
	 */
	if (n == cinfo->num_nodes)
	    cinfo->num_chans++;
	
	/*
	 * If we had an inconsistent channel definition where some
	 * nodes had a channel configured at this index while others
	 * did not, we want to remove the definition for all nodes.
	 */
	if (n && (n < cinfo->num_nodes)) {

	    while (n--) {
	        strcpy(cinfo->nodes[n].chans[cinfo->num_chans].dev.net.name, "");
	        cinfo->nodes[n].chans[cinfo->num_chans].type = 0;
	    }

	}

    }

    return(cinfo);
}


/*
 * API -- Exported Administration Routines
 * NOTE: the caller is required to free() the returned data structure!
 */
CluCfg *
cluGetConfig(void)
{
	return get_clu_cfg(CLU_CONFIG_FILE);
}


int
cluGetLocalNodeId(void)
{
	CluCfg *cfg;
	static int myID = -1; // Cache value for performance gain

	if (myID == -1) {
		cfg = get_clu_cfg(CLU_CONFIG_FILE);
		if (cfg == NULL)
			return -1;

		myID = cfg->lid;
		free(cfg);
	}
	return (myID);
}

#define MAXTOKENLEN 255
int
cluAddChan(ChanCfg *chan[MAX_NODES])
{
	CluCfg     *cfg;
	CFG_status  ret;
	char        token[MAXTOKENLEN];
	int         member;
	char        separator=CLU_CONFIG_SEPARATOR;


	cfg = get_clu_cfg(CLU_CONFIG_FILE);
	if (cfg == NULL)
		return -1;

	memset(token, 0, MAXTOKENLEN);
	/*
	 * Here we add the channel to both nodes.
	 */
	for (member = 0; member < cfg->num_nodes; member++) {
		/*
		 * Setup the new entry in the database.
		 */
		snprintf(token, MAXTOKENLEN, "members%cmember%d%cchan%d%ctype",
			 separator, member, separator, cfg->num_chans, 
			 separator);

		if (chan[member]->type == CHAN_NET) {
			ret = CFG_Set(token, "net");
			snprintf(token, MAXTOKENLEN, "members%cmember%d%cchan"
				 "%d%cname", separator, member, separator,
				 cfg->num_chans, separator);
			ret = CFG_Set(token, chan[member]->dev.net.name);
		} else if (chan[member]->type == CHAN_SERIAL) {
			ret = CFG_Set(token, "serial");
			snprintf(token, MAXTOKENLEN, "members%cmember%d%cchan"
				 "%d%cdevice", separator, member, separator,
				 cfg->num_chans, separator);
			ret = CFG_Set(token, chan[member]->dev.serial.name);
		} else {
			/*
			 * Invalid input.
			 */
			free(cfg);
			return -1;
		}
		/*
		 * Now that the on-disk copy is successfully written, we
		 * can update our in-memory structure.
		 */
		memcpy(&cfg->nodes[member].chans[cfg->num_chans],
		       chan[member], sizeof(ChanCfg));
	}
	cfg->num_chans++;

	free(cfg);
	return 0;
}


int
cluDelChan(int chan_id)
{
	CluCfg     *cfg;
	CFG_status  status;
	char        token[MAXTOKENLEN];
	int         member, c;
	char        separator=CLU_CONFIG_SEPARATOR;
	char        tmp[255];

	cfg = get_clu_cfg(CLU_CONFIG_FILE);
	if (cfg == NULL)
		return -1;

	memset(token, 0, MAXTOKENLEN);

	for (member = 0; member < cfg->num_nodes; member++) {
		for (c = chan_id; c < cfg->num_chans-1; c++) {
			memcpy(&cfg->nodes[member].chans[c],
			      &cfg->nodes[member].chans[c+1], sizeof(ChanCfg));
			cfg->nodes[member].chans[c].id--;
		}
		memset(&cfg->nodes[member].chans[cfg->num_chans-1], 0, 
		       sizeof(ChanCfg));
	}
	cfg->num_chans--;

	for (member = 0; member < cfg->num_nodes; member++) {
		for (c = 0; c < cfg->num_chans; c++) {

			/*
			 * First remove the channel definition from the
			 * configuration file.  This way we make sure
			 * not to leave any bogus entries lying around.
			 */
			snprintf(token, MAXTOKENLEN, 
				 "members%cmember%d%cchan%d*", separator, 
				 member, separator, c);
			CFG_RemoveMatch(token);

			/*
			 * Now start adding our entry from the in-memory copy
			 */
			snprintf(token, MAXTOKENLEN,
				 "members%cmember%d%cchan%d%ctype", separator,
				 member, separator, c, separator);

			snprintf(tmp, 255, "%s", cfg->nodes[member].chans[c].type == CHAN_NET ? "net" : "serial");
			Dprintf("Setting token %s = %s\n",token,tmp);
			CFG_Set(token, tmp);




			if (cfg->nodes[member].chans[c].type == CHAN_SERIAL) {
			    snprintf(token, MAXTOKENLEN, 
				     "members%cmember%d%cchan%d%cdevice",
				     separator, member, separator, c, 
				     separator);

			    Dprintf("Setting token %s = %s\n",token, 
				  cfg->nodes[member].chans[c].dev.serial.name);
			    CFG_Set(token, cfg->nodes[member].chans[c].dev.serial.name);

			}
			else if (cfg->nodes[member].chans[c].type == CHAN_NET){
			    snprintf(token, MAXTOKENLEN, 
				     "members%cmember%d%cchan%d%cname",
				     separator, member, separator, c, 
				     separator);

			    Dprintf("Setting token %s = %s\n", token,
				   cfg->nodes[member].chans[c].dev.net.name);
			    CFG_Set(token, cfg->nodes[member].chans[c].dev.net.name);

			}
			else {
				/*
				 * Error.
				 */
				Dprintf("Invalid channel type %d\n",
					cfg->nodes[member].chans[c].type);
			}

		}

	}


	/*
	 * Since we have shuffled everyone down and updated the database,
	 * we now need to remove the last channel entry in the db.
	 */
	for (member = 0; member < cfg->num_nodes; member++) {
		snprintf(token, MAXTOKENLEN, "members%cmember%d%cchan%d*",
			 separator, member, separator, cfg->num_chans);
		(void)CFG_RemoveMatch(token);
	}

	status = CFG_Write();
	if (status != CFG_OK) {
		Dprintf("error %d in CFG_Write.\n", status);
	}

	free(cfg);
	return 0;
}

ChanState *
cluGetChanState()
{
    int			c;
    static ChanState	*chan_state = NULL;
    msg_handle_t	con;
    int			hb_msg;
    int                 auth=0;

    /*
     * Allocate the channel status structure if this is the first time
     * we have been called.
     */

    if (chan_state == NULL) {

        chan_state = (ChanState *) malloc(MAX_CHANNELS * sizeof(ChanState));
        if (chan_state == NULL) {
	    errno = ENOMEM;
	    return(NULL);
        }
    }

    /*
     * Get the status of the channels from heartbeat
     */

    con = msg_open(PROCID_HEARTBEAT, cluGetLocalNodeId());
    if (con < 0) {
        Dprintf("Unable to connect to heartbeat mgr.\n");
	return(NULL);
    } else {
    
        hb_msg = HB_QUERY_NETUP;
        if (msg_send(con, &hb_msg, sizeof(hb_msg)) < 0) {
            Dprintf("Unable to send to heartbeat mgr\n");
	    msg_close(con);
	    return(NULL);
        }
        if (msg_receive(con,&hb_msg, sizeof(hb_msg),&auth) != sizeof(hb_msg)) {
            Dprintf("Bad reply from heartbeat mgr\n");
	    msg_close(con);
	    return(NULL);
        }

        for (c = 0; c < MAX_CHANNELS; c++) {
	    chan_state[c].id = c;
	    chan_state[c].state = (1 << c) & hb_msg;
	}

	msg_close(con);
    }

    return(chan_state);
}
/*
 * Local variables:
 *  c-basic-offset: 8
 *  c-indent-level: 8
 *  tab-width: 8
 * End:
 */

