/*
  Copyright Mission Critical Linux, 2000

  Kimberlite is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2, or (at your option) any
  later version.

  Kimberlite is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Kimberlite; see the file COPYING.  If not, write to the
  Free Software Foundation, Inc.,  675 Mass Ave, Cambridge, 
  MA 02139, USA.

 * Author: Gregory P. Myrdal <Myrdal@MissionCriticalLinux.Com>
 *
 * svcmgr_engine.c
 *
 *
 * This module contains functions that deal with node state changes
 * and what the local node should do with services with respect to
 * these changes.  In general this means:
 *	
 *	o Should the local system start the service if it is not running?
 *	o Should the local system wait to start the service?
 *	o Should the local system stop the service?
 *	o Should the local system wait to stop the service.
 */

/*
 * Version string that is filled in by CVS
 */
static const char *version __attribute__ ((unused)) = "$Id: svcmgr_engine.c,v 1.12 2000/08/23 15:59:52 myrdal Exp $";

/*
 * System includes
 */
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <sys/syslog.h>

/*
 * Cluster includes
 */
#include <clusterdefs.h>
#include <msgsvc.h>
#include <parseconf.h>
#include <svcmgr.h>
#include <logger.h>
#include "svcmgr_proto.h"

/*
 * Internal functions
 */
static int nodeDownWhileSvcStopped(int local, int nodeID,
                   ServiceBlock *svcStatusPtr, char *svcName, int *svcAction);
static int nodeDownWhileSvcStarting(int local, int nodeID,
                   ServiceBlock *svcStatusPtr, char *svcName, int *svcAction);
static int nodeDownWhileSvcRunning(int local, int nodeID,
                   ServiceBlock *svcStatusPtr, char *svcName, int *svcAction);
static int nodeDownWhileSvcStopping(int local, int nodeID,
                   ServiceBlock *svcStatusPtr, char *svcName, int *svcAction) ;
static int nodeUpWhileSvcStopped(int local, int nodeID,
                   ServiceBlock *svcStatusPtr, char *svcName, int *svcAction);
static int nodeUpWhileSvcStarting(int local, int nodeID,
                   ServiceBlock *svcStatusPtr, char *svcName, int *svcAction);
static int nodeUpWhileSvcRunning(int local, int nodeID,
                   ServiceBlock *svcStatusPtr, char *svcName, int *svcAction);
static int nodeUpWhileSvcStopping(int local, int nodeID,
                   ServiceBlock *svcStatusPtr, char *svcName, int *svcAction);
static int arbitrateServiceStart(ServiceBlock *svcStatusPtr, int *svcAction);
static int giveUpService(ServiceBlock *svcStatusPtr,int nodeID,int *svcAction);
static int takeOverService(ServiceBlock *svcStatusPtr, int *svcAction);
static int relocateOnNodeBoot(int svcID, int nodeID);
static int selectNodeToStartService(int svcID, char *svcName);

/*
 * Externally defined variables
 */
extern int myNodeID;
extern char *myNodeName;
extern char nodeStates[];

/*
 * Externally defined functions
 */
extern int svcmgrGetNodeState(int nodeID); 
extern void clu_lock(void);
extern void clu_un_lock(void);

/*
 * nodeChange()
 *
 * We received a node change event from the node monitor.  Determine
 * which type of node change it is and pass it onto the appropriate
 * node state change routine.
 *
 * NODE_DOWN:
 *
 * We received a node down event.  We first determine if its us
 * that is going down or someone else.
 *
 * If we are going down we should stop all of our services.
 *
 * If its someone else that is going down we parse through the list of
 * services to determine if we should start any services that the down 
 * node owned.  We will also double check to make sure that all other 
 * services are running and if we should start them.
 *
 *
 * NODE_UP:
 *
 * We received a node up event.  We first need to determine if its us that
 * is comming up or someone else.
 *
 * If its us that is comming up we need to determine if we should start any
 * services.
 *
 * If its someone else that is comming up we need to determine if we should
 * stop a service that might want to run on the newly available cluster
 * node.
 *
 * In either case we will also double check to make sure that all other
 * services are running and if we should start them.
 *
 * Note: This function indirectly takes out a lock so it may not be
 * called if the caller locks the service status information.
 */
int
nodeChange(int nodeID, int nodeStatus)
{
	int svcAction;
	int svcID=SERVICE_ID_NONE;
	int localTransition;
	char *svcName;
	char *nodeName;
	ServiceBlock svcStatus;
	char errMsg[MAX_ERROR_BUFLEN];
	char *svcOwnerName=(char *)NULL;

	getNodeName(nodeID, &nodeName);

	/*
	 * Make sure we understand the node state change
	 */
	switch (nodeStatus)
	  {
	  case NODE_DOWN:
	    nodeStates[nodeID] = nodeStatus;
	    clulog(LOG_NOTICE,
"Service Manager received a NODE_DOWN event for %s\n", nodeName);
	    break;
	  case NODE_UP:
	    nodeStates[nodeID] = nodeStatus;
	    clulog(LOG_NOTICE,
"Service Manager received a NODE_UP event for %s\n", nodeName);
	    break;
	  default:
	    clulog(LOG_ERR,
	           "Service Manager received an unknown node state change %d\n",
	            nodeStatus);
	    return(FAIL);
	  }

	/*
	 * Until we hear the state of all services from the quorumd we cannot
	 * change the state of any service.
	 */

	clulog(LOG_INFO, "Evaluating service states\n");

	/*
	 * Determine if we are going down, or if someone else went down.
	 */
	if (nodeID == myNodeID)
	    localTransition=YES;
	else
	    localTransition=NO;

	/*
	 * Loop through all of the services in the database to determine
	 * if there is anything to do with them (i.e. start or stop them).
	 *
	 * The service database contains a "Services" entry which defines
	 * all of the service entries.  We loop through this entry to find
	 * all service names.
	 */
	svcID=MIN_SERVICE-1;
	while (svcID < (MAX_SERVICES-1))
	  {
	    svcID++;

	    if (serviceExists(svcID) != YES)
	        continue;		// go to the next service ID

	    getSvcName(svcID, &svcName);

	    /*
	     * Make sure we got a valid value
	     */
	    if (svcName == (char *)NULL)
	      {
	        clulog(LOG_ERR, "Cannot get service name from database\n");
	        return(FAIL);
	      }

	    /*
	     * Lock the service information and get the current service
	     * status.
	     *
	     * Note: we do not use lockAndReqServiceStatus() here because
	     * we do not want to release the lock until we actually
	     * make the decision on what we are going to do.  I.e. we don't
	     * want someone else trying to change the service state at the
	     * same time that we are.
	     */
	    clu_lock();
	    if (reqServiceStatus(svcID, &svcStatus) != SUCCESS)
	      {
	        clulog(LOG_ERR,
"Cannnot get status for service %s\n", svcName);
		clu_un_lock();
	        continue;		// continue and try next service
	      }

	    clulog(LOG_DEBUG, "Service %s is in %s state\n", 
	            svcName, serviceStateStrings[svcStatus.state]);

	    /*
	     * If the service is defined to be disabled in the database
	     * then do not do anything with it.  This check is put in 
	     * here in the case that there is a mismatch between the
	     * database and the service information.  If for some reason
	     * the service is forced into the SVC_DISABLED state in the
	     * config file do not do anything with it.
	     */
	    if (isServiceDisabled(svcID) == YES)
	      {
	        clulog(LOG_INFO,
"Service %s is disabled in the cluster database, cannot change its state\n",
	                svcName);
		clu_un_lock();
	        /*
	         * If we find a service that is disabled, but is in an active
	         * state owned by a cluster member send a message.
	         */
	        if (svcStatus.state != SVC_DISABLED)
	          {
	            getNodeName(svcStatus.owner, &svcOwnerName);
	            clulog(LOG_ALERT,
"Service %s is disabled in the cluster database, but is currently %s owned by %s\n",
	                svcName, serviceStateStrings[svcStatus.state], 
	                svcOwnerName);
	          }
	        continue;
	      }

	    switch (svcStatus.state)
	      {
	      case SVC_STOPPED:
	        switch (nodeStatus)
	          {
	          case NODE_DOWN:
	            if ((nodeDownWhileSvcStopped(localTransition, nodeID,
	                 &svcStatus, svcName, &svcAction)) != SUCCESS)
	              {
		        clu_un_lock();
	                continue;	// continue and try next service
	              }
	            break;

	          case NODE_UP:
	            if ((nodeUpWhileSvcStopped(localTransition, nodeID,
	                 &svcStatus, svcName, &svcAction)) != SUCCESS)
	              {
		        clu_un_lock();
	                continue;	// continue and try next service
	              }
	            break;
	          }
	        break;

	      case SVC_STARTING:
	        switch (nodeStatus)
	          {
	          case NODE_DOWN:
	            if (nodeDownWhileSvcStarting(localTransition, nodeID,
	                 &svcStatus, svcName, &svcAction) != SUCCESS)
	              {
		        clu_un_lock();
	                continue;	// continue and try next service
	              }
	            break;

	          case NODE_UP:
	            if (nodeUpWhileSvcStarting(localTransition, nodeID,
	                 &svcStatus, svcName, &svcAction) != SUCCESS)
	              {
		        clu_un_lock();
	                continue;	// continue and try next service
	              }
	            break;
	          }
	        break;

	      case SVC_RUNNING:
	        switch (nodeStatus)
	          {
	          case NODE_DOWN:
	            if (nodeDownWhileSvcRunning(localTransition, nodeID,
	                 &svcStatus, svcName, &svcAction) != SUCCESS)
	              {
		        clu_un_lock();
	                continue;	// continue and try next service
	              }
	            break;

	          case NODE_UP:
	            if (nodeUpWhileSvcRunning(localTransition, nodeID,
	                 &svcStatus, svcName, &svcAction) != SUCCESS)
	              {
		        clu_un_lock();
	                continue;	// continue and try next service
	              }
	            break;
	          }
	        break;

	      case SVC_STOPPING:
	        switch (nodeStatus)
	          {
	          case NODE_DOWN:
	            if (nodeDownWhileSvcStopping(localTransition, nodeID,
	                 &svcStatus, svcName, &svcAction) != SUCCESS)
	              {
		        clu_un_lock();
	                continue;	// continue and try next service
	              }
	            break;

	          case NODE_UP:
	            if (nodeUpWhileSvcStopping(localTransition, nodeID,
	                 &svcStatus, svcName, &svcAction) != SUCCESS)
	              {
		        clu_un_lock();
	                continue;	// continue and try next service
	              }
	            break;
	          }
	        break;

	      case SVC_ERROR:		// Do nothing in these states
	      case SVC_DISABLED:
	      case SVC_DISABLING:
	        clulog(LOG_INFO,
                        "Service %s is %s, cannot change its state\n",
	                svcName, serviceStateStrings[svcStatus.state]);
		clu_un_lock();
	        continue;		// continue to next service

	      default:			// Error, unknown state
	        clulog(LOG_ERR,
	               "Unknown service state %d for service %s\n",
	               svcStatus.state, svcName);
		clu_un_lock();
	        continue;		// continue and try next service
	        break;
	      } // end of switch(svcStatus.state)

	    /*
	     * Based on our query above determine what we should do with the
	     * service.  If there is an action to perform, set the service into
	     * the correct state and perform that action (i.e. start or stop of
	     * a service).  Once we have changed the state of the service we
	     * release the service information lock so others can get at the
	     * service data.
	     */
	    switch (svcAction)
	      {
	      case SVC_START:		// Start the service
	        svcStatus.state = SVC_STARTING;
	        svcStatus.owner = myNodeID;
	        if (reqServiceStatusChange(&svcStatus) != SUCCESS)
	          {
	            clu_un_lock();// let someone else into the svc info
	            return(FAIL);
	          }
	        clu_un_lock();	// let someone else into the svc info

	        if (reqStartService(svcID, NO) != SUCCESS)
	          {
	            clulog(LOG_ERR, "Cannot start service %s\n", svcName);
	          }
	        break;

	      case SVC_WAITANDSTART:	// Wait and start the service
	        clu_un_lock();	// let someone else into the svc info
	        if (reqWaitAndStartService(svcID) != SUCCESS)
	          {
	            clulog(LOG_ERR, "Cannot wait and start service %s\n",
	                    svcName);
	          }
	        break;

	      case SVC_STOP:		// Stop the service
	        svcStatus.state = SVC_STOPPING;
	        svcStatus.owner = myNodeID;
	        if (reqServiceStatusChange(&svcStatus) != SUCCESS)
	          {
	            clu_un_lock();// let someone else into the svc info
	            return(FAIL);
	          }
	        clu_un_lock();	// let someone else into the svc info

	        if (reqStopService(svcID, NO) != SUCCESS)
	          {
	            clulog(LOG_ERR, "Cannot stop service %s\n", svcName);
	          }
	        break;

	      case SVC_WAITANDSTOP:	// Wait and stop the service
	        clu_un_lock();
	        if (reqWaitAndStopService(svcID) != SUCCESS)
	          {
	            clulog(LOG_ERR, "Cannot wait and stop service %s\n",
	                    svcName);
	          }
	        break;

	      case SVC_DISABLE:		// we should really never get here
	        svcStatus.state = SVC_DISABLING;
	        svcStatus.owner = myNodeID;
	        if (reqServiceStatusChange(&svcStatus) != SUCCESS)
	          {
	            clu_un_lock();// let someone else into the svc info
	            return(FAIL);
	          }
	        clu_un_lock();	// let someone else into the svc info

	        if (reqDisableService(svcID, errMsg) != SUCCESS)
	          {
	            clulog(LOG_ERR, "Cannot disable service %s\n",
	                    svcName);
	          }
	        break;

	      case SVC_ERROR:		// Uh oh, something bad happened
	        svcStatus.state = SVC_ERROR;
	        svcStatus.owner = myNodeID;

	        if (reqServiceStatusChange(&svcStatus) != SUCCESS)
	          {
	            clu_un_lock();// let someone else into the svc info
	            return(FAIL);
	          }

	        clu_un_lock();	// let someone else into the svc info
	        break;

	      case SVC_DONOTHING:
	        clu_un_lock();	// let someone else into the svc info
	        break;

	      default:
	        clulog(LOG_ERR, "unknown service action: %d\n", svcAction);
	        clu_un_lock();	// let someone else into the svc info
	        break;
	      }	// end of switch(svcAction)
	  } // end of while (svcID < MAX_SERVICES)

	/*
	 * If we are going down, run the checks in svcmgrExit().
	 */
	if ((nodeID == myNodeID) && (nodeStatus == NODE_DOWN))
	  {
	    svcmgrExit(0);
	  }

	return(SUCCESS);
}

/*****************************************************************************
 * NODE_DOWN routines
 *****************************************************************************/ 

/*
 * nodeDownWhileSvcStopped()
 *
 * We received a node NODE_DOWN and a service is in the SVC_STOPPED state.
 */
static int
nodeDownWhileSvcStopped(int local, int nodeID, ServiceBlock *svcStatusPtr,
	                char *svcName, int *svcAction)
{
	*svcAction=SVC_DONOTHING;	// default action

	nodeID=nodeID;			// quiet compiler warnings

	/*
	 * If we are going down we do not want to start any services,
	 * leave the service in the stopped state.
	 */
	if (local == YES)
	  {
	    clulog(LOG_DEBUG, "We are going down, leaving %s in %s state\n",
	            svcName, serviceStateStrings[svcStatusPtr->state]);
	    return(SUCCESS);		// do nothing
	  }
	else
	  {
	    /*
	     * If another node went down we need to arbitrate with any other
	     * nodes to see if we should start the service.  The called
	     * arbitrate function will fill in 'svcAction'.
	     */
	    if (arbitrateServiceStart(svcStatusPtr, svcAction) != SUCCESS)
	      {
	        clulog(LOG_ERR, 
"Cannot arbitrate start of service %s, not changing its state\n", svcName);
	        *svcAction=SVC_DONOTHING;	// reset just in case
	        return(FAIL);
	      }
	  }
	return(SUCCESS);
}

/*
 * nodeDownWhileSvcStarting()
 *
 * We received a node NODE_DOWN and a service is in the SVC_STARTING state.
 *
 * Note: This function indirectly takes out a lock so it may not be
 * called if the caller locks the service status information.
 */
static int
nodeDownWhileSvcStarting(int local, int nodeID, ServiceBlock *svcStatusPtr,
	                char *svcName, int *svcAction)
{
	char *svcOwnerName=(char *)NULL;	// default action

	nodeID=nodeID;				// quiet compiler warnings

	*svcAction=SVC_DONOTHING;

	/*
	 * If we are going down see if we are starting the service.  If
	 * we are we need to attempt to stop it.  Since its in the middle
	 * of starting the best we can do is to wait and stop it after it
	 * is running.  There is a good chance we may go down before all
	 * this happens, so this is a "best try" effort.
	 */
	if (local == YES)
	  {
	    if (svcStatusPtr->owner == myNodeID)
	      {
	        clulog(LOG_INFO, "We are %s service %s, waiting to stop it\n",
	                serviceStateStrings[svcStatusPtr->state], svcName);
	        *svcAction=SVC_WAITANDSTOP;
	      }
	    else
	      {
	        getNodeName(svcStatusPtr->owner, &svcOwnerName);
	        clulog(LOG_DEBUG, "Node %s is %s service %s\n",
	                svcOwnerName, 
	                serviceStateStrings[svcStatusPtr->state], svcName);
	      }
	    return(SUCCESS);
	  }
	else
	  {
	    /*
	     * If another node went down we abitrate to see if that node
	     * was the owner of the service and we should start it.
	     */
	    if (arbitrateServiceStart(svcStatusPtr, svcAction) != SUCCESS)
	      {
	        clulog(LOG_ERR, 
"Cannot arbitrate start of service %s, not changing its state\n",
	                svcName);
	        *svcAction=SVC_DONOTHING;	// reset just in case
	        return(FAIL);
	      }
	  }
	return(SUCCESS);
}

/*
 * nodeDownWhileSvcRunning()
 *
 * We received a node NODE_DOWN and a service is in the SVC_RUNNING state.
 *
 * Note: This function indirectly takes out a lock so it may not be
 * called if the caller locks the service status information.
 */
static int
nodeDownWhileSvcRunning(int local, int nodeID, ServiceBlock *svcStatusPtr,
	                char *svcName, int *svcAction)
{
	char *svcOwnerName=(char *)NULL;

	*svcAction=SVC_DONOTHING;	// default action

	/*
	 * If we are going down and a service is running we need to 
	 * determine if we are running the service.  If we are we need to
	 * stop it.  If not, we do not do anything, someone else is running
	 * the service.
	 */
	if (local == YES)
	  {
	    if (svcStatusPtr->owner == myNodeID)
	      {
	        clulog(LOG_INFO, "We are %s service %s, stop it\n",
	                serviceStateStrings[svcStatusPtr->state], svcName);
	        *svcAction=SVC_STOP;
	      }
	    else
	      {
	        getNodeName(svcStatusPtr->owner, &svcOwnerName);
	        clulog(LOG_INFO, "Node %s is %s service %s\n",
	                svcOwnerName, serviceStateStrings[svcStatusPtr->state],
	                svcName);
	      }
	    return(SUCCESS);
	  }
	else
	  {
	    /*
	     * If another node went down we need to determine if the down
	     * node was running the service.  If it is, we will arbitrate to
	     * see if we should start the service.
	     */
	    if (svcStatusPtr->owner == nodeID)
	      {
	        if (arbitrateServiceStart(svcStatusPtr, svcAction) != SUCCESS)
	          {
	             clulog(LOG_ERR, 
"Cannot arbitrate start of service %s, not changing its state\n", svcName);
	            *svcAction=SVC_DONOTHING;	// reset just in case
	            return(FAIL);
	          }
	      }
	    else
	      {
	        getNodeName(svcStatusPtr->owner, &svcOwnerName);
	        clulog(LOG_INFO, "Node %s is %s service %s\n",
	                svcOwnerName, serviceStateStrings[svcStatusPtr->state],
	                svcName);
	      }
	  }
	return(SUCCESS);
}

/*
 * nodeDownWhileSvcStopping()
 *
 * We received a node NODE_DOWN and a service is in the SVC_STOPPING state.
 *
 * Note: This function indirectly takes out a lock so it may not be
 * called if the caller locks the service status information.
 */
static int
nodeDownWhileSvcStopping(int local, int nodeID, ServiceBlock *svcStatusPtr,
	                char *svcName, int *svcAction)
{
	nodeID=nodeID;			// quiet compiler warnings

	*svcAction=SVC_DONOTHING;	// default action

	/*
	 * If we are going down we do not want to start any services,
	 * let the service continue to stop ... if someone else is
	 * up they will start the service when they get our node down
	 * event.
	 */
	if (local == YES)
	  {
	    clulog(LOG_INFO, 
"We are going down and service %s is %s, let it continue\n",
	            svcName, serviceStateStrings[svcStatusPtr->state]);
	    return(SUCCESS);
	  }
	else
	  {
	    /*
	     * If another node went down we abitrate to see if that node
	     * was the owner of the service and we should start it.
	     */
	    if (arbitrateServiceStart(svcStatusPtr, svcAction) != SUCCESS)
	      {
	        clulog(LOG_ERR, 
"Cannot arbitrate start of service %s, not changing its state\n", svcName);
	        *svcAction=SVC_DONOTHING;	// reset just in case
	        return(FAIL);
	      }
	  }
	return(SUCCESS);
}

/*****************************************************************************
 * NODE_UP routines
 *****************************************************************************/ 

/*
 * nodeUpWhileSvcStopped()
 *
 * We received a node NODE_UP and a service is in the SVC_STOPPED state.
 *
 * Note: This function indirectly takes out a lock so it may not be
 * called if the caller locks the service status information.
 */
static int
nodeUpWhileSvcStopped(int local, int nodeID, ServiceBlock *svcStatusPtr,
	                char *svcName, int *svcAction)
{

	*svcAction=SVC_DONOTHING;	// default action

	nodeID=nodeID;			// quiet compiler warnings
	local=local;

	/*
	 * If we are comming up OR another node is comming up arbitrate to
	 * determine who should start the service.
	 */
	if (arbitrateServiceStart(svcStatusPtr, svcAction) != SUCCESS)
	  {
	    clulog(LOG_ERR, 
"Cannot arbitrate start of service %s, not changing its state\n", svcName);
	    *svcAction=SVC_DONOTHING;	// reset just in case
	    return(FAIL);
	  }

	return(SUCCESS);
}

/*
 * nodeUpWhileSvcStarting()
 *
 * We received a node NODE_UP and a service is in the SVC_STARTING state.
 *
 * Note: This function indirectly takes out a lock so it may not be
 * called if the caller locks the service status information.
 */
static int
nodeUpWhileSvcStarting(int local, int nodeID, ServiceBlock *svcStatusPtr,
	                char *svcName, int *svcAction)
{
	*svcAction=SVC_DONOTHING;	// default action

	/*
	 * If we are comming up and we own the service we should never see
	 * our service in this state.  When we boot up all services that we
	 * own should have been placed into owned by NODE_ID_NONE and a state
	 * of SVC_STOPPED. 
	 */
	if (local == YES)
	  {
	    if (svcStatusPtr->owner == myNodeID)
	      {
	        clulog(LOG_ALERT, 
"During local boot found service %s owned by us in the %s state!\n",
	               svcName, serviceStateStrings[svcStatusPtr->state]);
	        return(FAIL);		// do nothing with the service
	      }
	    else
	      {
	        /*
	         * If the service relocates on a preferred node boot and
	         * we are the preferred node, we take over the service.
	         * If the service is in an active state we must wait before
	         * starting it.  svcAction is set to the action we should
	         * take for the service.
	         */
	        if (takeOverService(svcStatusPtr, svcAction) != SUCCESS)
	            return(FAIL);
	      }
	  }
	else
	  {
	    /*
	     * If the node comming up is the preferred node and we own
	     * the service, we need to determine if we should give it up.
	     * If we are to give it up the node booting is waiting for us
	     * to stop it.
	     */
	    if (giveUpService(svcStatusPtr, nodeID, svcAction) != SUCCESS)
	        return(FAIL);
	  }

	return(SUCCESS);
}

/*
 * nodeUpWhileSvcRunning()
 *
 * We received a node NODE_UP and a service is in the SVC_RUNNING state.
 *
 * Note: This function indirectly takes out a lock so it may not be
 * called if the caller locks the service status information.
 */
static int
nodeUpWhileSvcRunning(int local, int nodeID, ServiceBlock *svcStatusPtr,
	                char *svcName, int *svcAction)
{
	*svcAction=SVC_DONOTHING;		// default action

	/*
	 * If we are comming up and we own the service we should never see
	 * our service in this state.  When we boot up all services that we
	 * own should have been placed into owned by NODE_ID_NONE and a state
	 * of SVC_STOPPED. 
	 */
	if (local == YES)
	  {
	    if (svcStatusPtr->owner == myNodeID)
	      {
	        clulog(LOG_ALERT, 
"During local boot found service %s owned by us in the %s state!\n",
	               svcName, serviceStateStrings[svcStatusPtr->state]);
	        return(FAIL);		// do nothing with the service
	      }
	    else
	      {
	        /*
	         * If the service relocates on a preferred node boot and
	         * we are the preferred node, we take over the service.
	         * If the service is in an active state we must wait before
	         * starting it.  svcAction is set to the action we should
	         * take for the service.
	         */
	        if (takeOverService(svcStatusPtr, svcAction) != SUCCESS)
	            return(FAIL);
	      }
	  }
	else
	  {
	    /*
	     * If the node comming up is the preferred node and we own
	     * the service, we need to determine if we should give it up.
	     * If we are to give it up the node booting is waiting for us
	     * to stop it.
	     */
	    if (giveUpService(svcStatusPtr, nodeID, svcAction) != SUCCESS)
	        return(FAIL);
	  }

	return(SUCCESS);
}

/*
 * nodeUpWhileSvcStopping()
 *
 * We received a node NODE_UP and a service is in the SVC_STOPPING state.
 *
 * Note: This function indirectly takes out a lock so it may not be
 * called if the caller locks the service status information.
 */
static int
nodeUpWhileSvcStopping(int local, int nodeID, ServiceBlock *svcStatusPtr,
	                char *svcName, int *svcAction)
{
	char *svcOwnerName=(char *)NULL;

	nodeID=nodeID;				// quiet compiler warnings

	*svcAction=SVC_DONOTHING;		// default action

	if (local == YES)
	  {
	    /*
	     * If we are comming up and we own the service we should never see
	     * our service in this state.  When we boot up all services that we
	     * own should have been placed into owned by NODE_ID_NONE and a 
	     * state of SVC_STOPPED. 
	     */
	    if (svcStatusPtr->owner == myNodeID)
	      {
	        clulog(LOG_ALERT,
"During local boot found service %s owned by us in the %s state!\n",
	               svcName, serviceStateStrings[svcStatusPtr->state]);
	        return(FAIL);		// do nothing with the service
	      }
	    else
	      {
	        /*
	         * If another node is stopping the service we need to
	         * arbitrate with any other nodes to see if we should start
                 * the service.  The called arbitrate function will fill in 
	         * 'svcAction'.
	         */
	        if (arbitrateServiceStart(svcStatusPtr, svcAction) != SUCCESS)
	          {
	            clulog(LOG_ERR,
"Cannot arbitrate start of service %s, not changing its state\n", svcName);
	            *svcAction=SVC_DONOTHING;	// reset just in case
	            return(FAIL);
	          }
	      }
	    return(SUCCESS);
	  }
	else
	  {
	    /*
	     * If another node came up and a service is stopping, just
	     * let it stop.
	     */
	    getNodeName(svcStatusPtr->owner, &svcOwnerName);
	    clulog(LOG_INFO, "Node %s is %s service %s\n",
	            svcOwnerName, serviceStateStrings[svcStatusPtr->state],
	            svcName);
	  }
	return(SUCCESS);
}
 
/*****************************************************************************
 * Supporting functions
 *****************************************************************************/ 

/*
 * arbitrateServiceStart()
 *
 * Determine if we are a candidate to start a service.  This function is
 * called when a service requires a start.  First priority for the service
 * is to have a preferred node start the service.  After that its first
 * come first serve.  More than one nodes might run this routine and attempt
 * start a service.  We rely on service information locking to only allow
 * one cluster node to start the service.
 *
 * Note: This function indirectly takes out a lock so it may not be
 * called if the caller locks the service status information.
 */
static int
arbitrateServiceStart(ServiceBlock *svcStatusPtr, int *svcAction)
{
	int svcOwnerState;
	char *svcOwnerName=(char *)NULL;
	int startNode;
	char *svcName=(char *)NULL;

	*svcAction=SVC_DONOTHING;

	getSvcName(svcStatusPtr->id, &svcName);
	
	/*
	 * If the service is in SVC_DISABLED state then the admin has
	 * requested us not to start this service automatically.
	 *
	 * If the service is in the SVC_ERROR state we can not start as we
	 * do not know why it failed.
	 */
	if ((svcStatusPtr->state == SVC_DISABLED) ||
	    (svcStatusPtr->state == SVC_DISABLING) ||
	    (svcStatusPtr->state == SVC_ERROR))
	  {
	    clulog(LOG_INFO, "Service %s is %s, not starting\n",
	           svcName, serviceStateStrings[svcStatusPtr->state]);
	    return(SUCCESS);		// default svcAction, SVC_DONOTHING
	  }

	
	if ((svcOwnerState = svcmgrGetNodeState(svcStatusPtr->owner)) == FAIL)
	  {
	    clulog(LOG_ERR, 
"Cannot get service owner state for service %s, not arbitrate service start\n",
	           svcName);
	    return(FAIL);
	  }

	/*
	 * If the service is either stopped, nobody owns the service, or the
	 * node that owns it is down, someone should start it.  If the
	 * preferred node is up let it start the service (which could be us).
	 * If the the preferred node is not up or the service does not have a
	 * preferred node then we should attempt to start it.  This may
	 * allow for more than one nodes to trying to start the service,
	 * but this is ok, we rely on service information locking such
	 * that the first node that tries to start the service will win.
	 */
	if ((svcStatusPtr->state == SVC_STOPPED) ||
	    (svcStatusPtr->owner == NODE_ID_NONE) ||
	    (svcOwnerState == NODE_DOWN))
	  {
	    if (svcStatusPtr->owner == NODE_ID_NONE)
	        clulog(LOG_INFO, 
"There is no owner for service %s, service needs to be started\n", svcName);

	    if (svcOwnerState == NODE_DOWN)
	        clulog(LOG_INFO, 
"The owner of service %s is down, service needs to be started\n", svcName);

	    if ((startNode=selectNodeToStartService(svcStatusPtr->id, svcName))
	                  == FAIL)
	      {
	        clulog(LOG_ERR, "Cannot arbitrate service start for %s\n", 
	               svcName);
	        return(FAIL);		// default svcAction, SVC_DONOTHING
	      }

	    if (startNode == myNodeID)
	      {
	        *svcAction=SVC_START;
	        return(SUCCESS);
	      }
	    else
	      {
	        return(SUCCESS);	// default svcAction, SVC_DONOTHING
	      }
	  }

	/*
	 * The service is SVC_RUNNING and its owner is up, so don't start the
	 * service.  If the service is SVC_STARTING, let it start even
	 * if we are the preferred node.
	 */
	if ((svcStatusPtr->state == SVC_RUNNING) ||
	    (svcStatusPtr->state == SVC_STARTING))
	  {
	    getNodeName(svcStatusPtr->owner, &svcOwnerName);
	    clulog(LOG_INFO, "Node %s is %s service %s, not starting\n",
	            svcOwnerName, serviceStateStrings[svcStatusPtr->state],
	            svcName);
	    return(SUCCESS);		// default svcAction, SVC_DONOTHING
	  }

	/*
	 * If the service is SVC_STOPPING determine if we are the correct node
	 * to wait and start the service.
	 */
	if (svcStatusPtr->state == SVC_STOPPING)
	  {
	    clulog(LOG_INFO, 
"Service %s is %s, someone needs to wait and start\n",
	            svcName, serviceStateStrings[svcStatusPtr->state]);

	    if ((startNode=selectNodeToStartService(svcStatusPtr->id, svcName))
	                  == FAIL)
	      {
	        clulog(LOG_ERR, "Cannot arbitrate service start for %s\n", 
	               svcName);
	        return(FAIL);	
	      }

	    if (startNode == myNodeID)	// We will start the service
	      {
	        *svcAction=SVC_WAITANDSTART;
	        return(SUCCESS);
	      }
	    else			// Another node will start the service
	      {
	        return(SUCCESS);	// default svcAction, SVC_DONOTHING
	      }
	  }

	clulog(LOG_ERR, "Cannot arbitrate service start for %s\n", svcName);
	return(FAIL);		// should never get here
}
 
/*
 * takeOverService()
 *
 * Determine if the local node should take over the service because
 * it is the preferred node.
 *
 * Note: This function indirectly takes out a lock so it may not be
 * called if the caller locks the service status information.
 */
static int
takeOverService(ServiceBlock *svcStatusPtr, int *svcAction)
{
	char *svcOwnerName=(char *)NULL;
	char *svcName=(char *)NULL;

	*svcAction=SVC_DONOTHING;

	getSvcName(svcStatusPtr->id, &svcName);

	/*
	 * Determine if the service relocates on boot of the local node.
	 */
	switch (relocateOnNodeBoot(svcStatusPtr->id, myNodeID))
	  {
	    case NO:	return(SUCCESS);
	    case YES:	break;
	    case FAIL:
	    default:	return(FAIL);
	  }

	/*
	 * Check to make sure we do not own the service.  We should
	 * generally never be here because as part of init we stop all
	 * services removing the owner for any service we had owned.
	 */
	if (svcStatusPtr->owner == myNodeID)
	  {
	    clulog(LOG_INFO, 
"Cannot take over service %s, already owned by me\n", svcName);
	    return(SUCCESS);
	  }

	getNodeName(svcStatusPtr->owner, &svcOwnerName);

	/*
	 * If we are the preferred node and the service relocates on
	 * preferred member boot either start the service or wait to
	 * start it.
	 */
	switch (svcStatusPtr->state)
	  {
	    case SVC_STOPPING:		// wait and start the service
	    case SVC_STARTING:
	    case SVC_RUNNING:
	      *svcAction = SVC_WAITANDSTART;
	      clulog(LOG_INFO,
"Taking over service %s, waiting for %s to stop it before starting\n", 
	             svcOwnerName, svcName);
	      break;

	    case SVC_STOPPED:		// start the service
	      *svcAction = SVC_START;
	      clulog(LOG_INFO, "Taking over service %s\n", svcName);
	      break;

	    case SVC_ERROR:		// do nothing
	    case SVC_DISABLED:
	    case SVC_DISABLING:
	      clulog(LOG_DEBUG, "Cannot take over service %s in %s state\n",
	             svcName, serviceStateStrings[svcStatusPtr->state]);
	      return(SUCCESS);

	    default:
	      clulog(LOG_ERR, "Unknown service state %d\n", 
	             svcStatusPtr->state);
	  }

	if (*svcAction == SVC_DONOTHING)
	  {
	    clulog(LOG_DEBUG, 
"Node %s is %s service %s and we are not the preferred node\n",
	           svcOwnerName, 
	           serviceStateStrings[svcStatusPtr->state], svcName);
	  }

	return(SUCCESS);
}
 
/*
 * giveUpService()
 *
 * Determine if the local node should give up the service because the
 * preferred node is available.
 *
 * Note: This function indirectly takes out a lock so it may not be
 * called if the caller locks the service status information.
 */
static int
giveUpService(ServiceBlock *svcStatusPtr, int nodeID, int *svcAction)
{
	char *svcOwnerName=(char *)NULL;
	char *svcName=(char *)NULL;
	char *nodeName=(char *)NULL;

	*svcAction=SVC_DONOTHING;

	getSvcName(svcStatusPtr->id, &svcName);
        getNodeName(nodeID, &nodeName);

	/*
	 * Determine if the service relocates on boot of the local node.
	 */
	switch (relocateOnNodeBoot(svcStatusPtr->id, nodeID))
	  {
	    case NO:	return(SUCCESS);
	    case YES:	break;
	    case FAIL:
	    default:	return(FAIL);
	  }

	/*
	 * Check to make sure we own the service.  If we do not
	 * own it, we can not stop it.
	 */
	if (svcStatusPtr->owner != myNodeID)
	    return(SUCCESS);

	/*
	 * If we are the preferred node and the service relocates on
	 * preferred member boot either start the service or wait to
	 * start it.
	 */
	switch (svcStatusPtr->state)
	  {
	    case SVC_STARTING:
	      *svcAction = SVC_WAITANDSTOP;
	      clulog(LOG_INFO,
"Giving up service %s for preferred node %s, waiting before stopping it\n",
	             svcName, nodeName);
	      break;

	    case SVC_RUNNING:
	      *svcAction = SVC_STOP;
	      clulog(LOG_INFO, 
"Giving up service %s for preferred node %s\n", svcName, nodeName);
	      break;

	    case SVC_STOPPED:
	    case SVC_STOPPING:
	    case SVC_DISABLING:
	    case SVC_DISABLED:
	    case SVC_ERROR:
	      clulog(LOG_DEBUG, "Cannot give up service %s in %s state\n",
	             svcName, serviceStateStrings[svcStatusPtr->state]);
	      return(SUCCESS);

	    default:
	      clulog(LOG_ERR, "Unknown service state %d\n", 
	             svcStatusPtr->state);
	  }

	if (*svcAction == SVC_DONOTHING)
	  {
	    clulog(LOG_DEBUG,
"Not giving up service %s, node %s is %s it and it does not relocate on boot of %s\n",
	           svcName, svcOwnerName,
	           serviceStateStrings[svcStatusPtr->state], myNodeName);
	  }

	return(SUCCESS);
}

/*
 * relocateOnNodeBoot()
 *
 * Given a service ID and a node ID determine if:
 *
 *	o The service relocates on preferred node boot
 *	o The node 'nodeID' is the preferred node
 *	o The node 'nodeID' is in the NODE_UP state
 *
 * If true return YES, otherwise return NO.
 */
static int
relocateOnNodeBoot(int svcID, int nodeID)
{
	int preferredNodeID;
	char *svcName;
	char *preferredNodeName;

	getSvcName(svcID, &svcName);

	/*
	 * Determine if this service relocates due to a preferred node boot.
	 */
	if (doesServiceRelocateOnPreferredNodeBoot(svcID) == YES)
	  {
	    /*
	     * Get the service preferred node ID and that nodes current state
	     */
	    switch (getSvcPreferredNodeID(svcID, &preferredNodeID))
	      {
	        case FAIL:
	          clulog(LOG_ERR, 
"Cannot determine if service %s relocates on a preferred node boot\n", svcName);
	          return(FAIL);

	        case NOT_FOUND:
	          clulog(LOG_WARNING, 
"Service %s does not have a preferred node defined! However, is suppose to relocate on preferred node boot\n", svcName);
	          return(NO);
	      }

            getNodeName(preferredNodeID, &preferredNodeName);
	    clulog(LOG_DEBUG,
"Service %s relocates on boot of node %s\n", svcName, preferredNodeName);

	    /*
	     * Determine if the preferred node is up and matches the given
	     * nodeID.
	     */
	    if ((preferredNodeID != NODE_ID_NONE) &&	// node is defined
	        (preferredNodeID == nodeID))		// node matches
	      {
	        return(YES);
	      }

	    return(NO);
	  }

	    clulog(LOG_DEBUG,
"Service %s is not configured to relocate on preferred member boot\n", svcName);

	return(NO);
}

/*
 * selectNodeToStartService
 *
 * Given a service ID, determine who should try to start the service.  We
 * are only called if it is already determined that the service should start,
 * now its just a matter of who.  We either return the node ID, NODE_ID_NONE,
 * or FAIL.
 *
 * Note: in a cluster larger than two nodes we could have more than
 * one node trying to start the service.  We rely on service locking
 * to avoid this situation.
 */
static int
selectNodeToStartService(int svcID, char *svcName)
{
	int preferredNodeID;
	int preferredNodeState;
	char *preferredNodeName=(char *)NULL;

	clulog(LOG_INFO, "Selecting node to start service %s\n", svcName);

	/*
	 * Determine if there is a preferred node assigned to this service
	 */
	if (getSvcPreferredNodeID(svcID, &preferredNodeID) == FAIL)
	  {
	    clulog(LOG_ERR, 
"Cannot select node to start service %s\n", svcName);
	    return(FAIL);
	  }


	if (preferredNodeID == myNodeID) // We are the preferred node, start it
	  {
	    clulog(LOG_INFO, 
"I will start service %s, I am the preferred node\n",
	           svcName);
	    return(myNodeID);
	  }

	if (preferredNodeID == NODE_ID_NONE) // No preferred node, start it
	  {
	    clulog(LOG_INFO, 
"There is no preferred node set for service %s, I will start\n", svcName);
	    return(myNodeID);
	  }

	if (getNodeName(preferredNodeID, &preferredNodeName) != SUCCESS)
	  {
	    clulog(LOG_ERR, 
"Cannot select node to start service %s\n", svcName);
	    return(FAIL);
	  }

	/*
	 * The preferred node should start the service, but it can't if its
	 * not up.  Make sure the preferred node is up, if not we should try
	 * to start the service.
	 */
	if ((preferredNodeState=svcmgrGetNodeState(preferredNodeID)) == FAIL)
	  {
	    clulog(LOG_ERR, 
"Cannot select node to start service %s\n", svcName);
	    return(FAIL);
	  }

	switch (preferredNodeState)
	  {
	  case NODE_DOWN:		// Preferred node down, start service
	    clulog(LOG_INFO, 
"The preferred node (%s) for service %s is down, I will start\n",
	           preferredNodeName, svcName);
	    return(myNodeID);
	    break;

	  case NODE_UP:			// Preferred node up, do not start
	    clulog(LOG_INFO, 
"Expecting preferred node %s to start service %s\n",
	           preferredNodeName, svcName);
	    return(preferredNodeID);
	    break;

	  case NODE_UNINITIALIZED:	// Node state is uninitialized 
	    clulog(LOG_INFO, 
"Cannot select node to start service %s; do not know the state of preferred node %s yet\n", svcName, preferredNodeName);
	    return(NODE_ID_NONE);
	    break;

	  default:			// Unknown node state, do not start
	    clulog(LOG_ERR, "Unknown node state %d from svcmgrGetNodeState()\n",
	            preferredNodeState);
	    return(NODE_ID_NONE);
	    break;
	  }

	return(NODE_ID_NONE);		// Should never get here
}

