

#  Copyright Mission Critical Linux, 2000

#  Kimberlite is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License as published by the
#  Free Software Foundation; either version 2, or (at your option) any
#  later version.

#  Kimberlite is distributed in the hope that it will be useful, but
#  WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#  General Public License for more details.

#  You should have received a copy of the GNU General Public License
#  along with Kimberlite; see the file COPYING.  If not, write to the
#  Free Software Foundation, Inc.,  675 Mass Ave, Cambridge, 
#  MA 02139, USA.
#
# $Revision: 1.3 $
#
# Author: Gregory P. Myrdal <Myrdal@MissionCriticalLinux.Com>
#
#------------------------------------------------------------

#
# For proper lock operation all locks must be cleared at cluster
# start time.  The only way we will know which locks might be
# around is if they are registered here.  If they are not they
# will not be initialized.
#
typeset exportfs_lockfile=/tmp/exportfs
typeset rmtab_lockfile=/tmp/rmtab
typeset ifconfig_lockfile=/tmp/ifconfig

#
# NOTE: edit this list with all files to be locked during cluster
#       operation!
#
typeset lockfile_list="\
	$exportfs_lockfile \
	$ifconfig_lockfile \
"

#
# Lock variables
#
typeset lock_ext=.clu_lk		# lockfile extension
typeset -r lock_ext
typeset lockfiles_inuse=		# lockfiles outstanding

#
# For proper initialization all lockfiles should be registered with
# this function for removal at cluster start time.  This is done
# in the global variable 'lockfile_list'.
#
lockInit()
{
	typeset file

	for file in $lockfile_list
	do
		logAndPrint $LOG_DEBUG "Initializing lock for $file"
		rm -f $file$lock_ext
	done
}

#
# Register a lockfile that is in use such that on error exit it is 
# removed.
#
registerLockFile()
{
	#logAndPrint $LOG_DEBUG "Registering lockfile $lockfile"
	lockfiles_inuse="$lockfiles_inuse $1$lock_ext"
}

#
# Un-register a lockfile as it is no longer in use.
#
unregisterLockFile()
{
	#logAndPrint $LOG_DEBUG "Unregistering lockfile $lockfile"
	lockfiles_inuse="$(print $lockfiles_inuse | sed 's/ $1$lock_ext//')"
}

#
# If we are exiting clean up all locks we hold.
#
# Note: this should only be used on exit as it removes outstanding locks.
#
cleanUpLockfiles()
{
	for lockfile in $lockfiles_inuse
	do
		logAndPrint $LOG_DEBUG "Removing lockfile $lockfile on exit"
		rm -f $lockfile
	done
}

#
# Block for the lock on a given file.
#
lockFile()
{
	typeset file=$1
	typeset lockfile=$file$lock_ext
	typeset first_pass=1

	if [ -z "$1" ]; then
		logAndPrint $LOG_ERR "Usage: lockFile file"
		return $FAIL
	fi

	set -C noclobber	# needed for atomic file creation

	while :
	do
		if [ -f $lockfile ]; then
			if [ -n "$first_pass" ]; then
				logAndPrint $LOG_INFO \
					"Waiting for lock on $file"
				first_pass=
			fi
			usleep 1000
		else
			#logAndPrint $LOG_DEBUG "Acquiring lock on $file"
			echo "$(date) $USER $(basename $0) $$" > $lockfile
			if [ $? -ne 0 ]; then
				logAndPrint $LOG_DEBUG \
					"Lost lock on $file, retrying."
				first_pass=1
				continue
			else
				registerLockFile $file
				break
			fi
		fi
	done

	set +C noclobber
}

#
# Release the lock on a given file.
#
unlockFile()
{
	typeset file=$1
	typeset lockfile=$file$lock_ext

	#logAndPrint $LOG_DEBUG "Releasing lock on $file"
	rm -f $lockfile
	unregisterLockFile $file
}
