/*
*  oss.c
*
*     Copyright (C) Charles 'Buck' Krasic - January 2001
*     fragment handling addition by Tomoaki Hayasaka <hayasakas@postman.riken.go.jp>
*
*  This file is part of libdv, a free DV (IEC 61834/SMPTE 314M)
*  codec.
*
*  libdv is free software; you can redistribute it and/or modify it
*  under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2, or (at your
*  option) any later version.
*   
*  libdv is distributed in the hope that it will be useful, but
*  WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*  General Public License for more details.
*   
*  You should have received a copy of the GNU General Public License
*  along with GNU Make; see the file COPYING.  If not, write to
*  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
*
*  The libdv homepage is http://libdv.sourceforge.net/.  
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <libdv/dv_types.h>

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/soundcard.h>

#include "oss.h"

extern gchar *Preferences_getAudioDevice( void );

//static const gchar *default_device = "/dev/dsp";

dv_oss_t *
dv_oss_new( void )
{
	dv_oss_t * result;

	result = ( dv_oss_t * ) calloc( 1, sizeof( dv_oss_t ) );
	if ( result != NULL )
	{
		memset( result, 0, sizeof( dv_oss_t ) );
		result->fd = -1;
	}

	return result;
}

/* Very simplistic for sound output using the OSS API - returns 0 if unable to open or the
 * sampling rate used if successful. It is assumed that all subsequent play requests will
 * resample top this rate.
 */
gint
dv_oss_init( dv_audio_t *audio, dv_oss_t *oss )
{
	gint format = AFMT_S16_NE, rate_request, channels_request, fragment_request;
	gchar *device;
	gint used = audio->frequency;

	channels_request = audio->num_channels;
	rate_request = audio->frequency;
	fragment_request = 0x0004000B;
	oss->buffer = NULL;

	device = Preferences_getAudioDevice();

	if ( device == NULL || !strcmp( device, "" ) || channels_request == 0 || rate_request == 0 )
		return FALSE;

	/* open audio device
	   Open nonblocking first, to test if available. */
	oss->fd = open( device, O_WRONLY | O_NONBLOCK | O_NDELAY );
	if ( oss->fd != -1 )
	{
		close( oss->fd );
		oss->fd = open( device, O_WRONLY );
	}

	if ( oss->fd != -1 )
	{
		int error = 0;
		if ( ioctl( oss->fd, SNDCTL_DSP_SETFRAGMENT, &fragment_request ) == -1 )
		{
			perror( "SNDCTL_DSP_SETFMT" );
			error = 1;
		}
		if ( error == 0 && ioctl( oss->fd, SOUND_PCM_WRITE_BITS, &format ) == -1 )
		{
			perror( "SNDCTL_DSP_SETFMT" );
			error = 2;
		}
		if ( error == 0 && format != AFMT_S16_NE )
		{
			fprintf( stderr, "soundcard doesn't support format\n" );
			error = 3;
		}
		if ( error == 0 && ioctl( oss->fd, SOUND_PCM_WRITE_CHANNELS, &channels_request ) )
		{
			fprintf( stderr, "soundcard doesn't support %d channels\n", audio->num_channels );
			error = 4;
		}
		if ( error == 0 && channels_request != audio->num_channels )
		{
			fprintf( stderr, "soundcard doesn't support %d channels\n", audio->num_channels );
			error = 5;
		}
		if ( error == 0 && ioctl( oss->fd, SOUND_PCM_WRITE_RATE, &rate_request ) == -1 )
		{
			perror( "SNDCTL_DSP_SPEED" );
			error = 6;
		}
		if ( error == 0 && rate_request != audio->frequency )
		{
			perror( "SNDCTL_DSP_SPEED" );
			error = 7;
		}

		if ( error == 6 || error == 7 )
		{
			rate_request = 44100;
			used = rate_request;
			printf( ">>> audio at %d failed - forcing resample at %dhz\n", audio->frequency, rate_request );
			if ( ioctl( oss->fd, SOUND_PCM_WRITE_RATE, &rate_request ) == -1 )
			{
				perror( "SNDCTL_DSP_SPEED 44.1" );
				used = 0;
				error = 6;
			}
			else
			{
				error = 0;
			}
			if ( error == 0 && rate_request != 44100 )
			{
				perror( "SNDCTL_DSP_SPEED" );
				error = 7;
				used = 0;
			}
		}

		if ( error == 0 )
		{
			oss->buffer = malloc( DV_AUDIO_MAX_SAMPLES * audio->num_channels * sizeof( gint16 ) );
			if ( oss->buffer == NULL )
			{
				fprintf( stderr, "Unable to allocate memory for the buffer\n" );
				error = 8;
			}
		}

		if ( error != 0 )
		{
			close( oss->fd );
			oss->fd = -1;
			used = 0;
		}
	}
	else
	{
		fprintf( stderr, "Unable to open audio device %s\n", device );
		used = 0;
	}

	return used;
} /* dv_oss_init */

gboolean
dv_oss_player( dv_oss_t *oss, void *data, int size )
{
	return write( oss->fd, data, size ) == size;
}

gboolean
dv_oss_play( dv_audio_t *audio, dv_oss_t *oss, gint16 **out )
{
	gint ch, i, j = 0, total, written = 0, result;

	/* Interleave the audio into a single buffer */
	for ( i = 0; i < audio->samples_this_frame; i++ )
	{
		for ( ch = 0; ch < audio->num_channels; ch++ )
		{
			oss->buffer[ j++ ] = out[ ch ][ i ];
		} /* for */
	} /* for */

	/* Send the audio to the device */
	total = audio->samples_this_frame * audio->num_channels * sizeof( gint16 );
	do
	{
		result = write( oss->fd, oss->buffer, total );
		if ( result != total )
			goto write_error;
		written += result;
	}
	while ( total > written );
	if ( !oss->arg_audio_file )
	{
		if ( ioctl( oss->fd, SNDCTL_DSP_POST, NULL ) == -1 )
			goto post_ioctl;
	} /* if */

	return ( TRUE );

write_error:
	perror( "write" );
	return ( FALSE );

post_ioctl:
	perror( "SNDCTL_DSP_POST" );
	return ( FALSE );

} /* dv_oss_play */

void
dv_oss_close( dv_oss_t *oss )
{
	if ( oss != NULL )
	{
		if ( oss->fd != -1 )
		{
			close( oss->fd );
			oss->fd = -1;
		} /* if */
		if ( oss->buffer )
		{
			free( oss->buffer );
			oss->buffer = NULL;
		} /* if */
		free( oss );
	}
} /* dv_oss_close */
