/**
 * Nomad Jukebox KIO::Slave
 * @copyright Copyright 2004 Shaun Jackman
 * @author Shaun Jackman <sjackman@debian.org>
 */
static const char* rcsid __attribute__((unused)) =
"$Id: njb.cpp,v 1.16 2004/04/06 21:19:19 tsirc Exp $";


// kionjb
#include "mp3.h"
#include "njb.h"
#include "playlist.h"
#include "track.h"
#include "config.h"

// libnjb
#include <libnjb.h>

// libid3
#include <id3/tag.h>

// sqlite
#include <sqlite.h>

// kde
#include <kdebug.h>
#include <kinstance.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <ktempfile.h>
#include <kurl.h>

// qt
#include <qcstring.h>
#include <qdir.h>
#include <qfileinfo.h>
#include <qstring.h>
#include <qtextstream.h>

// posix
#include <stdint.h>
#include <sys/stat.h>
#include <time.h>
#include <unistd.h>


using namespace KIO;
extern "C" {

int
kdemain( int argc, char **argv)
{
	KInstance instance( "kio_njb");

	kdDebug( 7182) << "*** Starting kio_njb " << endl;

	if( argc != 4) {
		kdDebug( 7182)
			<< "Usage: kio_njb protocol domain-socket1 domain-socket2"
			<< endl;
		exit( -1);
	}

	kio_njbProtocol slave( argv[2], argv[3]);
	slave.dispatchLoop();

	kdDebug( 7182) << "*** kio_njb Done" << endl;
	return 0;
}

} // extern "C"


/* ------------------------------------------------------------------------ */
kio_njbProtocol::kio_njbProtocol(
		const QCString& pool_socket, const QCString& app_socket)
	: SlaveBase( "kio_njb", pool_socket, app_socket)
{
	kdDebug( 7182) << "constructor: pid=" << getpid() << endl;
	m_njb = NULL;
	m_captured = false;
	m_libcount = 0;
	cacheOpen();
}


/* ------------------------------------------------------------------------ */
kio_njbProtocol::~kio_njbProtocol( void)
{
	kdDebug( 7182) << "deconstructor: pid=" << getpid() << endl;
	disconnect();
	cacheClose();
}


/* ------------------------------------------------------------------------ */
void
kio_njbProtocol::err( int status, const QString& msg)
{
	QString fullMsg( msg);
	if( status == ERR_COULD_NOT_CONNECT)
		fullMsg = "Nomad Jukebox";
	if( !m_errMsg.isEmpty())
		fullMsg += '\n' + m_errMsg;
	error( status, fullMsg);
	m_errMsg = "";
}


/* ------------------------------------------------------------------------ */
int
kio_njbProtocol::open( void)
{
	if( m_njb)
		return 0;

	njb_t njbs[ NJB_MAX_DEVICES];
	int n;

	if( NJB_Discover( njbs, 0, &n) == -1 || n == 0) {
		kdDebug( 7182) << "connect: no NJBs found\n";
		return -1;
	}

	m_njb = new njb_t;
	*m_njb = njbs[ 0];

	if( NJB_Open( m_njb) == -1) {
		kdDebug( 7182) << "connect: couldn't open\n";
		delete m_njb;
		m_njb = NULL;
		return -1;
	}

	return 0;
}


/* ------------------------------------------------------------------------ */
int
kio_njbProtocol::capture( void)
{
	if( m_captured)
		return 0;
	if( open())
		return -1;
	if( NJB_Capture( m_njb) == -1) {
		kdDebug( 7182) << "connect: couldn't capture\n";
		return -1;
	}
	m_captured = true;
	return 0;
}


/* ------------------------------------------------------------------------ */
int
kio_njbProtocol::connect( void)
{
	if( m_captured)
		return 0;
	kdDebug( 7182) << "connect: pid=" << getpid() << endl;
	infoMessage( i18n( "Connecting to <b>%1</b>...").arg( "Nomad JukeBox"));
	for( unsigned tries = 0; tries < 3; tries++) {
		if( tries) {
			sleep( 2);
			infoMessage( i18n( "Retrying..."));
		}

		if( capture())
			continue;

		infoMessage( i18n( "Connected to <b>%1</b>").arg( "Nomad JukeBox"));
		kdDebug( 7182) << "connect: Connected" << endl;
		return 0;
	}
	return ERR_COULD_NOT_CONNECT;
}


/* ------------------------------------------------------------------------ */
int
kio_njbProtocol::disconnect( void)
{
	kdDebug( 7182) << __func__ << ": pid=" << getpid() << endl;
	if( m_captured) {
		NJB_Release( m_njb);
		m_captured = false;
	}
	if( m_njb) {
		NJB_Close( m_njb);
		delete m_njb;
		m_njb = NULL;
		kdDebug( 7182) << "disconnect: Disconnected" << endl;
	}
	return 0;
}


/* ------------------------------------------------------------------------ */
void
kio_njbProtocol::closeConnection( void)
{
	kdDebug( 7182) << "closeConnection" << endl;
	disconnect();
}


/* ------------------------------------------------------------------------ */
void
kio_njbProtocol::dataQString( const QString& string)
{
	QByteArray a( (QCString)string);
	a.resize( a.size() - 1);
	data( a);
}


/* ------------------------------------------------------------------------ */
int
kio_njbProtocol::getAlbum( const KURL& url)
{
	if( url.path().right( 4) == ".mp3")
		return 0;
	if( !url.path().startsWith( "/artists/") &&
			!url.path().startsWith( "/albums/"))
		return 0;

	int status = cacheLibrary();
	if( status)
		return status;

	char** result;
	int nrow, ncolumn;
	char* errmsg;
	sqlite_get_table_printf( m_db,
			"SELECT filename FROM tracks WHERE album = '%q' "
			"ORDER BY tracknum",
			&result, &nrow, &ncolumn, &errmsg,
			url.fileName().latin1());
	if( errmsg) {
		warning( errmsg);
		free( errmsg);
		return -1;
	}

	totalSize( nrow);
	for( int i = 1; i <= nrow; i++)
		dataQString( result[ i] + QString( "\n"));
	sqlite_free_table( result);

	return -1;
}


/* ------------------------------------------------------------------------ */
int
kio_njbProtocol::getEtc( const KURL& url)
{
	if( url.directory() != "/etc")
		return 0;

	int status;
	if( url.fileName() == "counter") {
		if( (status = connect()))
			return status;
		dataQString( QString::number( (long unsigned)m_njb->libcount) + "\n");
	} else
	if( url.fileName() == "diskfree") {
		if( (status = connect()))
			return status;
		u_int64_t total64, free64;
		int status = NJB_Get_Disk_Usage( m_njb, &total64, &free64);
		if( status) {
			kdDebug( 7182) << "getEtc: NJB_Get_Disk_Usage failed\n";
			return ERR_COULD_NOT_READ;
		}
		mimeType( "text/plain");
		unsigned long total = total64 / 1024;
		unsigned long free = free64 / 1024;
		unsigned long used = total - free;
		dataQString( "1k-blocks      Used Available Use%\n");
		QString df;
		df.sprintf( "%9lu %9lu %9lu %3lu%%\n",
				total, used, free, 100*used/total);
		dataQString( df);
	} else
	if( url.fileName() == "eax") {
		// xxx fixme unimplemented
		mimeType( "text/plain");
		dataQString( "eax info (unimplemented)\n");
	} else
	if( url.fileName() == "firmware") {
		if( (status = connect()))
			return status;
		njbid_t* njbid = NJB_Ping(m_njb);
		if(!njbid) {
			kdDebug( 7182) << "getEtc: NJB_Ping failed\n";
			return ERR_COULD_NOT_READ;
		}
		mimeType( "text/plain");
		dataQString( i18n( "%1 %2.%3.%4\n").arg( njbid->productName)
				.arg( njbid->fwMajor).arg( njbid->fwMinor).arg( njbid->fwRel));
		free( njbid);
	} else
	if( url.fileName() == "owner") {
		if( (status = connect()))
			return status;
		char* owner = NJB_Get_Owner_String( m_njb);
		if( !owner) {
			kdDebug( 7182) << "getEtc: NJB_Get_Owner_String failed\n";
			return ERR_COULD_NOT_READ;
		}
		mimeType( "text/plain");
		dataQString( (QString)owner + "\n");
		free( owner);
	} else
	if( url.fileName() == "serial") {
		if( (status = connect()))
			return status;
		mimeType( "text/plain");
		dataQString( (QString)m_njb->idstring + "\n");
	} else
	if( url.fileName() == "version") {
		mimeType( "text/plain");
		dataQString( (QString)"kionjb-" + VERSION);
	} else
		return ERR_DOES_NOT_EXIST;
	return -1;
}


/* ------------------------------------------------------------------------ */
int
kio_njbProtocol::getPlaylist( const KURL& url)
{
	if( url.directory() != "/playlists")
		return 0;

	int status = cacheLibrary();
	if( status)
		return status;

	char** result;
	int nrow, ncolumn;
	char* errmsg;
	sqlite_get_table_printf( m_db,
			"SELECT track FROM playlisttracks "
			"WHERE playlist='%q' ORDER BY number",
			&result, &nrow, &ncolumn, &errmsg,
			url.fileName().latin1());
	if( errmsg) {
		warning( errmsg);
		free( errmsg);
		return -1;
	}

	totalSize( nrow);
	for( int i = 1; i <= nrow; i++)
		dataQString( result[ i] + QString( "\n"));
	sqlite_free_table( result);

	return -1;
}


/* ------------------------------------------------------------------------ */
static kio_njbProtocol* getTrackProgress_njb;
static time_t getTrackProgress_start;


/* ------------------------------------------------------------------------ */
static int
getTrackProgress( u_int64_t sent, u_int64_t /*total*/,
		const char* buf, unsigned len, void* data)
{
	(void)data; // xxx fixme instance pointer
	getTrackProgress_njb->processedSize( sent);
	QByteArray block;
	block.setRawData( buf, len);
	getTrackProgress_njb->data( block);
	block.resetRawData( buf, len);
	time_t t_last = time( 0L);
	if( t_last - getTrackProgress_start)
		getTrackProgress_njb->speed(
				sent / (t_last - getTrackProgress_start));
	else
		getTrackProgress_njb->speed( 0);
	return 0;
}


/* ------------------------------------------------------------------------ */
int
kio_njbProtocol::getTrack( const KURL& url)
{
	if( url.directory() != "/all")
		return 0;

	if( getTrackProgress_njb) {
		// njb is already in the middle of a transfer
		return ERR_COULD_NOT_READ;
	}

	int status = cacheLibrary();
	if( status)
		return status;
	Track track;
	if( !trackByFilename( track, url.fileName()))
		return ERR_DOES_NOT_EXIST;
	totalSize( track.size);

	status = connect();
	if( status)
		return status;

	mimeType( "audio/x-mp3");
	getTrackProgress_njb = this;
	getTrackProgress_start = time( 0);
	status = NJB_Get_Track( m_njb, track.id, track.size,
			NULL, getTrackProgress, this);
	getTrackProgress_njb = NULL;
	if( status == -1) {
		kdDebug( 7182) << "getTrack: NJB_Get_Track failed\n";
		njb_error_dump( stderr);
		return ERR_COULD_NOT_READ;
	}
	return -1;
}


/* ------------------------------------------------------------------------ */
void
kio_njbProtocol::get( const KURL& url )
{
	kdDebug( 7182) << "get: " << url.prettyURL() << endl;

	int status = 0;
	if( !status) status = getAlbum( url);
	if( !status) status = getEtc( url);
	if( !status) status = getPlaylist( url);
	if( !status) status = getTrack( url);
	if( status < 0) {
		data( QByteArray());
		finished();
	} else
		err( status ? status : ERR_DOES_NOT_EXIST, url.fileName());
	disconnect();
}


/* ------------------------------------------------------------------------ */
int
kio_njbProtocol::delEtc( const KURL& url)
{
	if( url.directory() != "/etc")
		return 0;
	if( url.fileName() == "owner") {
		int status = connect();
		if( status)
			return status;
		status = NJB_Set_Owner_String( m_njb, "");
		if( status) {
			kdDebug( 7182) << "delEtc: NJB_Set_Owner_String failed\n";
			return ERR_CANNOT_DELETE;
		}
		return -1;
	}
	return ERR_DOES_NOT_EXIST;
}


/* ----------------------------------------------------------------------- */
int
kio_njbProtocol::delPlaylist( const KURL& url)
{
	if( url.directory() != "/playlists")
		return 0;

	int status = cacheLibrary();
	if( status)
		return status;

	char** result;
	int nrow, ncolumn;
	char* errmsg;
	sqlite_get_table_printf( m_db,
			"SELECT id FROM playlists WHERE name='%q'",
			&result, &nrow, &ncolumn, &errmsg,
			url.fileName().latin1());
	if( errmsg) {
		warning( errmsg);
		free( errmsg);
		return -1;
	}
	if( nrow) {
		status = connect();
		if( status)
			return status;
		int id = atoi( result[ 1]);
		status = NJB_Delete_Playlist( m_njb, id);
		if( status) {
			kdDebug( 7182) << "delPlaylist: NJB_Delete_Playlist failed\n";
			return ERR_CANNOT_DELETE;
		}
		sqlite_exec_printf( m_db,
				"DELETE FROM playlists WHERE name='%q'",
				NULL, NULL, &errmsg,
				url.fileName().latin1());
		if( errmsg) {
			warning( errmsg);
			free( errmsg);
			return -1;
		}
		sqlite_exec_printf( m_db,
				"DELETE FROM playlisttracks WHERE playlist='%q'",
				NULL, NULL, &errmsg,
				url.fileName().latin1());
		if( errmsg) {
			warning( errmsg);
			free( errmsg);
			return -1;
		}
	} else
		warning( "Cannot find playlist");
	sqlite_free_table( result);

	return -1;
}


/* ------------------------------------------------------------------------ */
int
kio_njbProtocol::delTrack( const KURL& url)
{
	if( url.directory() != "/all" &&
			!url.path().startsWith( "/artists/"))
		return 0;

	int status = cacheLibrary();
	if( status)
		return status;
	Track track;
	if( !trackByFilename( track, url.fileName()))
		return ERR_DOES_NOT_EXIST;

	status = connect();
	if( status)
		return status;
	status = NJB_Delete_Track( m_njb, track.id);
	if( status) {
		kdDebug( 7182) << "delTrack: NJB_Delete_Track failed";
		return ERR_CANNOT_DELETE;
	}

	// remove from the cache
	cacheDel( track);

	return -1;
}


/* ------------------------------------------------------------------------ */
void
kio_njbProtocol::del( const KURL& url, bool /*isfile*/)
{
	kdDebug( 7182) << "del: " << url.prettyURL() << endl;

	int status = 0;
	if( !status) status = delEtc( url);
	if( !status) status = delPlaylist( url);
	if( !status) status = delTrack( url);
	kdDebug( 7182) << "del: status = " << status << endl;
	if( status < 0)
		finished();
	else
		err( status ? status : ERR_DOES_NOT_EXIST, url.fileName());
	disconnect();
}


/* ------------------------------------------------------------------------ */
int
kio_njbProtocol::putEtc( const KURL& url)
{
	if( url.path() != "/etc/owner")
		return 0;

	int status = connect();
	if( status)
		return status;
	kdDebug( 7182) << "put datareq" << endl;
	dataReq();
	kdDebug( 7182) << "put readdata" << endl;
	QCString owner( OWNER_STRING_LENGTH);
	int len = readData( owner);
	if( len < 0)
		return ERR_COULD_NOT_READ;
	owner.resize( len);
	kdDebug( 7182) << "put owner: " << owner << endl;
	status = NJB_Set_Owner_String( m_njb, owner);
	if( status) {
		kdDebug( 7182) << "putEtc: NJB_Set_Owner_String failed";
		return ERR_COULD_NOT_WRITE;
	}
	return -1;
}


/* ------------------------------------------------------------------------ */
int
kio_njbProtocol::putPlaylist( const KURL& url)
{
	if( url.directory() != "/playlists")
		return 0;
	kdDebug( 7182) << "putPlaylist: " << url.fileName() << endl;

	int status = cacheLibrary();
	if( status)
		return status;
	status = connect();
	if( status)
		return status;

	// read data in
	QByteArray slurp;
	dataReq();
	int len = readData( slurp);
	if( len < 0)
		return ERR_COULD_NOT_READ;
	kdDebug( 7182) << "putPlaylist: read " << len  << " bytes\n";
	QTextStream stream( slurp, IO_ReadOnly);

	Playlist playlist( this);
	status = playlist.load( url.fileName());
	if( status)
		return status;
	for(;;) {
		QString line = stream.readLine();
		if( line.isEmpty())
			break;
		status = playlist.add( line);
		if( status)
			return status;
	}
	status = playlist.update();
	if( status)
		return status;

	// update cache
	cachePlaylists();
	return -1;
}


/* ------------------------------------------------------------------------ */
static kio_njbProtocol* putTrackProgress_njb;
static time_t putTrackProgress_start;


/* ------------------------------------------------------------------------ */
static int
putTrackProgress( u_int64_t sent, u_int64_t total,
		const char* /*buf*/, unsigned /*len*/, void* data)
{
	(void)data; // xxx fixme instance pointer
	putTrackProgress_njb->infoMessage(
			i18n( "Sent %1%...").arg( (unsigned)(100 * sent / total)));
	return 0;
}


/* ------------------------------------------------------------------------ */
int
kio_njbProtocol::putTrack( const KURL& url, bool overwrite)
{
	if( url.directory() != "/all" &&
			!url.path().startsWith( "/artists/"))
		return 0;

	if( putTrackProgress_njb) {
		// njb is already in the middle of a transfer
		return ERR_COULD_NOT_CONNECT;
	}

	int status = cacheLibrary();
	if( status)
		return status;

	Track track;
	if( trackByFilename( track, url.fileName())) {
		if( overwrite)
			delTrack( url);
		else
			return ERR_FILE_ALREADY_EXIST;
	}

	status = connect();
	if( status)
		return status;

	// read in the file
	KTempFile tmpfile( "kionjb-", ".mp3");
	unsigned length = 0;
	for(;;) {
		QByteArray buf;
		dataReq();
		int len = readData( buf);
		if( len < 0)
			return ERR_COULD_NOT_READ;
		if( !len)
			break;
		tmpfile.dataStream()->writeRawBytes( buf.data(), len);
		length += len;
	}
	tmpfile.close();
	kdDebug( 7182) << "putTrack: read " << length << " bytes" << endl;
	QString tmpfilename = tmpfile.name();
	kdDebug( 7182) << "putTrack: wrote " << tmpfilename << endl;
	kdDebug( 7182) << "putTrack: renaming "
		<< tmpfilename << " to " << url.fileName() << endl;
	status = QDir().rename( tmpfilename, url.fileName());
	if( !status) {
		kdDebug( 7182) << "putTrack: error renaming local file\n";
		return ERR_COULD_NOT_WRITE;
	}

	// read the mp3 header
	unsigned duration = getDuration( url.path());
	if( !duration) {
		m_errMsg = i18n( "Not a valid mp3 file");
		return ERR_COULD_NOT_READ;
	}
	// read the id3 tags
	ID3_Tag tag;
	tag.Link( url.fileName(), ID3TT_ID3V1);
	track = Track( tag);
	// filename
	track.filename = url.fileName();
	// file size in bytes
	track.size = length;
	// duration in seconds
	track.duration = duration;

	// send the track
	kdDebug( 7182) << "putTrack: sending..." << endl;
	putTrackProgress_njb = this;
	kdDebug( 7182) << "putTrack: "
		<< track.title << " " << track.album << " "
		<< track.genre << " " << track.artist << endl;
	infoMessage( i18n( "Sending <b>%1</b> to the NJB\nPlease wait...").arg(
				url.fileName()));
	u_int32_t id;
	putTrackProgress_start = time( 0);
	status = NJB_Send_Track( m_njb, url.fileName(), track.codec,
		track.title, track.album, track.genre, track.artist,
		track.duration, track.tracknum, track.year, false,
		putTrackProgress, this, &id);
	if( status) {
		kdDebug( 7182) << "putTrack: NJB_Send_Track failed\n";
		njb_error_dump( stderr);
		putTrackProgress_njb = NULL;
		return ERR_COULD_NOT_WRITE;
	}
	putTrackProgress_njb = NULL;
	track.id = id;

	// cache the track
	cacheTrack( track);

	// cleanup
	QDir().remove( url.fileName());

	return -1;
}


/* ----------------------------------------------------------------------- */
void
kio_njbProtocol::put( const KURL& url,
		int /*permissions*/, bool overwrite, bool /*resume*/)
{
	kdDebug( 7182) << "put: " << url.prettyURL() << endl;
	int status = 0;
	if( !status) status = putEtc( url);
	if( !status) status = putPlaylist( url);
	if( !status) status = putTrack( url, overwrite);
	kdDebug( 7182) << "put: status = " << status << endl;
	if( status < 0)
		finished();
	else
		err( status ? status : ERR_COULD_NOT_WRITE, url.fileName());
	disconnect();
}


/* ------------------------------------------------------------------------ */
static int
copyTrackProgress( u_int64_t sent, u_int64_t /*total*/,
		const char* /*buf*/, unsigned /*len*/, void* data)
{
	(void)data; // xxx fixme instance pointer
	putTrackProgress_njb->processedSize( sent);
	unsigned secs = time( 0) - putTrackProgress_start;
	putTrackProgress_njb->speed( secs ? sent / secs : 0);
	return 0;
}


/* ----------------------------------------------------------------------- */
int
kio_njbProtocol::copyTrack( const KURL& src, const KURL& dst, bool overwrite)
{
	if( dst.directory() != "/all" &&
			!dst.path().startsWith( "/artists/"))
		return 0;

	if( putTrackProgress_njb) {
		// njb is already in the middle of a transfer
		return ERR_COULD_NOT_CONNECT;
	}

	int status = cacheLibrary();
	if( status)
		return status;

	Track track;
	if( trackByFilename( track, src.fileName())) {
		if( overwrite)
			delTrack( dst);
		else
			return ERR_FILE_ALREADY_EXIST;
	}

	status = connect();
	if( status)
		return status;

	// read the mp3 header
	unsigned duration = getDuration( src.path());
	if( !duration) {
		m_errMsg = i18n( "Not a valid mp3 file");
		return ERR_COULD_NOT_READ;
	}
	// read the id3 tags
	ID3_Tag tag;
	tag.Link( src.path(), ID3TT_ID3V1);
	track = Track( tag);
	track.size = QFileInfo( src.path()).size();
	track.duration = duration;
	track.filename = src.fileName();

	// send the track
	totalSize( track.size);
	kdDebug( 7182) << "copyTrack: sending..." << endl;
	putTrackProgress_njb = this;
	kdDebug( 7182) << "copyTrack: "
		<< track.title << " " << track.album << " "
		<< track.genre << " " << track.artist << endl;
	u_int32_t id;
	putTrackProgress_start = time( 0);
	status = NJB_Send_Track( m_njb, src.path(), track.codec,
		track.title, track.album, track.genre, track.artist,
		track.duration, track.tracknum, NULL, false,
		copyTrackProgress, this, &id);
	if( status) {
		kdDebug( 7182) << "copyTrack: NJB_Send_Track failed\n";
		njb_error_dump( stderr);
		putTrackProgress_njb = NULL;
		return ERR_COULD_NOT_WRITE;
	}
	putTrackProgress_njb = NULL;
	track.id = id;

	// cache the track
	cacheTrack( track);

	return -1;
}


/* ----------------------------------------------------------------------- */
void
kio_njbProtocol::copy( const KURL& src, const KURL& dst,
		int /*permissions*/, bool overwrite)
{
	kdDebug( 7182) << "copy: " << src.prettyURL() << " to "
		<< dst.prettyURL() << endl;
	int status = 0;
	if( src.protocol() != "file")
		status = ERR_UNSUPPORTED_ACTION;
	if( !status) status = copyTrack( src, dst, overwrite);
	kdDebug( 7182) << "copy: status = " << status << endl;
	if( status < 0)
		finished();
	else
		err( status ? status : ERR_UNSUPPORTED_ACTION, src.fileName());
	disconnect();
}


/* ----------------------------------------------------------------------- */
UDSEntry
kio_njbProtocol::createUDSEntry( const KURL& url)
{
#if 0 // very noisy
	kdDebug( 7182) << "createUDSEntry: " << url.prettyURL() << endl;
#endif
	UDSEntry entry;
	UDSAtom atom;

	int type;
	const char* mime = 0;
	int access;
	int size = 0;

	if( url.path() == "/") {
		type = S_IFDIR;
		access = 0555;
	} else
	if( url.path( -1) == "/albums") {
		type = S_IFDIR;
		access = 0555;
	} else
	if( url.path( -1) == "/all") {
		type = S_IFDIR;
		access = 0777;
	} else
	if( url.path( -1) == "/artists") {
		type = S_IFDIR;
		access = 0777;
	} else
	if( url.path( -1) == "/etc") {
		type = S_IFDIR;
		access = 0777;
	} else
	if( url.path( -1) == "/playlists") {
		type = S_IFDIR;
		access = 0777;
	} else
	if( url.directory() == "/albums") {
		// an album listing
		type = S_IFREG;
		mime = "text/plain";
		access = 0444;
	} else
	if( url.directory() == "/all") {
		// a track
		Track track;
		if( !trackByFilename( track, url.fileName())) {
			err( ERR_DOES_NOT_EXIST, url.fileName());
			return UDSEntry();
		}
		type = S_IFREG;
		mime = "audio/x-mp3";
		access = 0666;
		size = track.size;
	} else
	if( url.directory() == "/artists") {
		// an artist
		type = S_IFDIR;
		access = 0555;
	} else
	if( url.path().startsWith( "/artists/")) {
		if( url.fileName().right( 4) == ".mp3") {
			// a track;
			Track track;
			if( !trackByFilename( track, url.fileName())) {
				err( ERR_DOES_NOT_EXIST, url.fileName());
				return UDSEntry();
			}
			type = S_IFREG;
			mime = "audio/x-mp3";
			access = 0666;
			size = track.size;
		} else {
			// an album listing
			type = S_IFREG;
			mime = "text/plain";
			access = 0444;
		}
	} else
	if( url.directory() == "/etc") {
		type = S_IFREG;
		mime = "text/plain";
		if( url.fileName() == "owner")
			access = 0666;
		else
			access = 0444;
	} else
	if( url.directory() == "/playlists") {
		type = S_IFREG;
		mime = "text/plain";
		access = 0666;
	} else
	/* default */ {
		err( ERR_DOES_NOT_EXIST, url.fileName());
		return UDSEntry();
	}

	atom.m_uds = UDS_NAME;
	atom.m_str = url.fileName();
	entry.append( atom);

	atom.m_uds = UDS_FILE_TYPE;
	atom.m_long = type;
	entry.append( atom);

	if( mime) {
		atom.m_uds = UDS_MIME_TYPE;
		atom.m_str = mime;
		entry.append( atom);
	}

	atom.m_uds = UDS_ACCESS;
	atom.m_long = access;
	entry.append( atom);

	atom.m_uds = UDS_SIZE;
	atom.m_long = size;
	entry.append( atom);

	return entry;
}


/* ------------------------------------------------------------------------ */
UDSEntry
kio_njbProtocol::createUDSEntry( const KURL& url, const QString& filename)
{
#if 0 // very noisy
	kdDebug( 7182) << "createUDSEntry: "
		<< url.prettyURL() << " " << filename << endl;
#endif
	KURL file( url);
	file.addPath( filename);
	return createUDSEntry( file);
}


/* ------------------------------------------------------------------------ */
void
kio_njbProtocol::stat( const KURL& url)
{
	kdDebug( 7182) << "stat: " << url.prettyURL() << endl;
	statEntry( createUDSEntry( url));
	finished();
	disconnect();
}


/* ------------------------------------------------------------------------ */
int
kio_njbProtocol::listAlbums( const KURL& url)
{
	if( url.path( -1) != "/albums")
		return 0;
	kdDebug( 7182) << "listAlbums\n";

	int status = cacheLibrary();
	if( status)
		return status;

	char** result;
	int nrow, ncolumn;
	char* errmsg;
	sqlite_get_table( m_db,
			"SELECT DISTINCT album FROM tracks ORDER BY album",
			&result, &nrow, &ncolumn, &errmsg);
	if( errmsg) {
		warning( errmsg);
		free( errmsg);
		return -1;
	}

	totalSize( nrow);
	for( int i = 1; i <= nrow; i++)
		listEntry( createUDSEntry( url, result[ i]), false);
	sqlite_free_table( result);

	return -1;
}


/* ------------------------------------------------------------------------ */
int
kio_njbProtocol::listAll( const KURL& url)
{
	if( url.path( -1) != "/all")
		return 0;
	kdDebug( 7182) << "listAll\n";

	int status = cacheLibrary();
	if( status)
		return status;

	char** result;
	int nrow, ncolumn;
	char* errmsg;
	sqlite_get_table( m_db,
			"SELECT filename FROM tracks ORDER BY filename",
			&result, &nrow, &ncolumn, &errmsg);
	if( errmsg) {
		warning( errmsg);
		free( errmsg);
		return -1;
	}

	totalSize( nrow);
	for( int i = 1; i <= nrow; i++)
		listEntry( createUDSEntry( url, result[ i]), false);
	sqlite_free_table( result);
	return -1;
}


/* ------------------------------------------------------------------------ */
int
kio_njbProtocol::listArtists( const KURL& url)
{
	if( url.path( -1) != "/artists")
		return 0;
	kdDebug( 7182) << "listArtists\n";

	int status = cacheLibrary();
	if( status)
		return status;

	char** result;
	int nrow, ncolumn;
	char* errmsg;
	sqlite_get_table( m_db,
			"SELECT DISTINCT artist FROM tracks ORDER BY artist",
			&result, &nrow, &ncolumn, &errmsg);
	if( errmsg) {
		warning( errmsg);
		free( errmsg);
		return -1;
	}

	totalSize( nrow);
	for( int i = 1; i <= nrow; i++)
		listEntry( createUDSEntry( url, result[ i]), false);
	sqlite_free_table( result);

	return -1;
}


/* ------------------------------------------------------------------------ */
int
kio_njbProtocol::listArtist( const KURL& url)
{
	if( url.directory() != "/artists")
		return 0;
	kdDebug( 7182) << "listArist\n";

	int status = cacheLibrary();
	if( status)
		return status;

	char** result;
	int nrow, ncolumn;
	char* errmsg;
	sqlite_get_table_printf( m_db,
			"SELECT filename FROM tracks WHERE artist='%q' UNION "
			"SELECT DISTINCT album FROM tracks WHERE artist='%q' "
			"ORDER BY filename",
			&result, &nrow, &ncolumn, &errmsg,
			url.fileName().latin1(), url.fileName().latin1());
	if( errmsg) {
		warning( errmsg);
		free( errmsg);
		return -1;
	}

	totalSize( nrow);
	for( int i = 1; i <= nrow; i++)
		listEntry( createUDSEntry( url, result[ i]), false);
	sqlite_free_table( result);

	return -1;
}


/* ----------------------------------------------------------------------- */
int
kio_njbProtocol::listPlaylists( const KURL& url)
{
	if( url.path( -1) != "/playlists")
		return 0;
	kdDebug( 7182) << "listPlaylists\n";

	int status = cacheLibrary();
	if( status)
		return status;

	char** result;
	int nrow, ncolumn;
	char* errmsg;
	sqlite_get_table( m_db,
			"SELECT name FROM playlists ORDER BY name",
			&result, &nrow, &ncolumn, &errmsg);
	if( errmsg) {
		warning( errmsg);
		free( errmsg);
		return -1;
	}

	totalSize( nrow);
	for( int i = 1; i <= nrow; i++)
		listEntry( createUDSEntry( url, result[ i]), false);
	sqlite_free_table( result);
	return -1;

	return -1;
}


/* ----------------------------------------------------------------------- */
int
kio_njbProtocol::listEtc( const KURL& url)
{
	if( url.path( -1) != "/etc")
		return 0;
	kdDebug( 7182) << "listEtc\n";

	totalSize( 7);
	listEntry( createUDSEntry( url, "counter"), false);
	listEntry( createUDSEntry( url, "diskfree"), false);
	listEntry( createUDSEntry( url, "eax"), false);
	listEntry( createUDSEntry( url, "firmware"), false);
	listEntry( createUDSEntry( url, "owner"), false);
	listEntry( createUDSEntry( url, "serial"), false);
	listEntry( createUDSEntry( url, "version"), false);
	return -1;
}


/* ------------------------------------------------------------------------ */
int
kio_njbProtocol::listRoot( const KURL& url) {
	if( url.path() != "/")
		return 0;
	kdDebug( 7182) << "listRoot\n";

	totalSize( 5);
	listEntry( createUDSEntry( url, "albums/"), false);
	listEntry( createUDSEntry( url, "all/"), false);
	listEntry( createUDSEntry( url, "artists/"), false);
	listEntry( createUDSEntry( url, "etc/"), false);
	listEntry( createUDSEntry( url, "playlists/"), false);
	return -1;
}


/* ------------------------------------------------------------------------ */
void
kio_njbProtocol::listDir( const KURL& url)
{
	kdDebug( 7182) << "listDir: " << url.prettyURL() << endl;

	int status = 0;
	if( !status) status = listRoot( url);
	if( !status) status = listAlbums( url);
	if( !status) status = listAll( url);
	if( !status) status = listArtists( url);
	if( !status) status = listPlaylists( url);
	if( !status) status = listEtc( url);
	if( !status) status = listArtist( url);
	kdDebug( 7182) << "listDir: status = " << status << endl;
	if( status < 0) {
		listEntry( UDSEntry(), true);
		finished();
	} else
		err( status ? status : ERR_DOES_NOT_EXIST, url.fileName());
	disconnect();
}
