#include <string.h>
#include <time.h>

#include <glib-object.h>
#include <glib.h>

#include "../kptraininglog.h"
#include "../kipina-i18n.h"
#include "../kpsettings.h"
#include "../kputil.h"

#include "kpcalendarentryinfodialog.h"
#include "kpnewsplitworkoutdialog.h"
#include "kpnewworkoutdialog.h"
#include "kpnewcommentdialog.h"
#include "kpchartview.h"
#include "kpviewmodel.h"
#include "kpguiutils.h"
#include "kpchart.h"

/* Callbacks */
static void       kp_chart_view_class_init      (KPChartViewClass *klass);
static void       kp_chart_view_init            (KPChartView *cv);
static void       kp_chart_view_model_init      (KPViewModelIface *iface);
static void       kp_chart_view_finalize        (GObject *object);

/* Follow log changes.. */
static void       log_connect_signals           (KPTrainingLog *log,
                                                 KPChartView *cv);
static void       log_disconnect_signals        (KPTrainingLog *log,
                                                 KPChartView *cv);
static void       log_changed                   (KPTrainingLog *log,
                                                 KPChartView *cv);
static void       kp_chart_view_set_dmy         (KPViewModel *view,
                                                 guint d,
                                                 guint m,
                                                 guint y);
static void       kp_chart_view_get_dmy         (KPViewModel *view,
                                                 guint *d,
                                                 guint *m,
                                                 guint *y);
static void       kp_chart_view_set_log         (KPViewModel *view,
                                                 KPTrainingLog *log);
static void       kp_chart_view_unset_log       (KPViewModel *view);
static void       kp_chart_view_set_view_type   (KPViewModel *view,
                                                 KPViewModelType type);
static
KPViewModelType   kp_chart_view_get_view_type   (KPViewModel *view);
static void       kp_chart_view_update          (KPChartView *cv);


static void       param_combo_box_changed       (GtkComboBox *box,
                                                 KPChartView *cv);
static void       mode_combo_box_changed        (GtkComboBox *box,
                                                 KPChartView *cv);
static void       sport_combo_box_changed       (GtkComboBox *box,
                                                 KPChartView *cv);
static void       update_chart_data             (KPChartView *cv);
#define ITEMS_MAX     31
#define GROUPS_MAX    4

typedef struct KPChartViewPrivateData_
{
  KPViewModelType       type;
  KPTrainingLog        *log;

  GDate                *date_s;
  GDate                *date_e;
  GDate                *date;

  gdouble             **graph_data;

  KPTrainingLogDataMode mode;
  GString              *sport;
  GString              *param;
  
  GtkWidget            *sport_combo;
  GtkWidget            *chart;
} KPChartViewPrivateData;

#define KP_CHART_VIEW_PRIVATE_DATA(widget) (((KPChartViewPrivateData*) \
      (KP_CHART_VIEW (widget)->private_data)))

#define CONST_ENTRY_SIZE    16

static const gchar const_params[][CONST_ENTRY_SIZE] = 
{
  N_("Duration"),
  N_("Distance"),
  N_("Pace")
};

static const gchar internal_params[][CONST_ENTRY_SIZE] = 
{
  "duration",
  "distance",
  "pace" 
};

/*static const gchar const_modes[][CONST_ENTRY_SIZE] = 
{
  N_("Sum"),
  N_("Average"),
  N_("Maximum"),
  N_("Minimum")
};*/

static const gchar menu_texts[][CONST_ENTRY_SIZE] =
{
  N_("Bars   "),
  N_("Lines  "),
  N_("SumBars")
};

static gchar hint_text_weekdays[][CONST_ENTRY_SIZE] =
{
  N_("Mon"),
  N_("Tue"),
  N_("Wed"),
  N_("Thu"),
  N_("Fri"),
  N_("Sat"),
  N_("Sun"),
};

static gchar hint_text_months[][12] =
{
  N_("Jan"),
  N_("Feb"),
  N_("Mar"),
  N_("Apr"),
  N_("May"),
  N_("Jun"),
  N_("Jul"),
  N_("Aug"),
  N_("Sep"),
  N_("Oct"),
  N_("Nov"),
  N_("Dec"),
};


GType
kp_chart_view_get_type (void)
{
  static GType        kp_chart_view_type = 0;

  if (!kp_chart_view_type) {
    static const GTypeInfo kp_chart_view_info = {
      sizeof (KPChartViewClass),
      NULL,
      NULL,
      (GClassInitFunc) kp_chart_view_class_init,
      NULL,
      NULL,
      sizeof (KPChartView),
      0,
      (GInstanceInitFunc) kp_chart_view_init,
      NULL
    };
    static const GInterfaceInfo view_model_info = {
      (GInterfaceInitFunc) kp_chart_view_model_init,
      NULL,
      NULL
    };
    kp_chart_view_type = g_type_register_static (GTK_TYPE_VBOX,
                                                "KPChartView",
                                                &kp_chart_view_info, 0);
    g_type_add_interface_static (kp_chart_view_type,
                                 KP_TYPE_VIEW_MODEL,
                                &view_model_info);

  }
  return kp_chart_view_type;
}

static void
kp_chart_view_class_init (KPChartViewClass *klass)
{
  GObjectClass *object_class;

  object_class = G_OBJECT_CLASS (klass);
  object_class->finalize = kp_chart_view_finalize;
}

  

static void
kp_chart_view_init (KPChartView *cv)
{
  KPChartViewPrivateData *p_data;
  GtkWidget *expander;
  GtkWidget *mode_combo;
  GtkWidget *param_combo;
  GtkWidget *hbox;
  GSList *details;
  GSList *list;
  gchar *sport;
  gchar *param;
  guint i;
  
  cv->private_data = g_new (KPChartViewPrivateData, 1);
  p_data = KP_CHART_VIEW_PRIVATE_DATA (cv);
  p_data->date_s = g_date_new ();
  p_data->date_e = g_date_new ();
  p_data->date = g_date_new ();
  p_data->log = NULL;
  p_data->type = KP_VIEW_MODEL_TYPE_MONTH;
  p_data->mode = KP_TRAINING_LOG_DATA_SUM;

  p_data->graph_data = (gdouble **) malloc (GROUPS_MAX * sizeof (gdouble *));
  for (i=0; i < GROUPS_MAX; i++)
    p_data->graph_data[i] = (gdouble *) malloc (ITEMS_MAX * sizeof (gdouble));
  
  expander = gtk_expander_new ("Show filters");
  gtk_box_pack_start (GTK_BOX (cv), expander, FALSE, TRUE, 0);
  gtk_container_set_border_width (GTK_CONTAINER (expander), 0);
  gtk_widget_show (expander);

  hbox = gtk_hbox_new (FALSE, 12);
  gtk_container_set_border_width (GTK_CONTAINER (hbox), 12);
  gtk_container_add (GTK_CONTAINER (expander), hbox);
  gtk_widget_show (hbox);

  param_combo = gtk_combo_box_new_text ();

  for (i=0; i < (sizeof (const_params) / CONST_ENTRY_SIZE); i++)
    gtk_combo_box_append_text (GTK_COMBO_BOX (param_combo), _(const_params[i]));
    
  list = kp_settings_get_list ("preset_detail");
  details = list;

  while (list) {
    gtk_combo_box_append_text (GTK_COMBO_BOX (param_combo), list->data);
    list = list->next;
  }
  kp_settings_list_free (details);
  
  gtk_combo_box_set_active (GTK_COMBO_BOX (param_combo), 0);
  gtk_box_pack_start (GTK_BOX (hbox), param_combo, FALSE, TRUE, 0);
  gtk_widget_show (param_combo);

  g_signal_connect (G_OBJECT (param_combo), "changed",
                    G_CALLBACK (param_combo_box_changed), cv);

  mode_combo = gtk_combo_box_new_text ();
  gtk_combo_box_append_text (GTK_COMBO_BOX (mode_combo), _("Sum"));
  gtk_combo_box_append_text (GTK_COMBO_BOX (mode_combo), _("Average"));
  gtk_combo_box_append_text (GTK_COMBO_BOX (mode_combo), _("Maximum"));
  gtk_combo_box_append_text (GTK_COMBO_BOX (mode_combo), _("Minimum"));
  gtk_combo_box_set_active (GTK_COMBO_BOX (mode_combo), 0);
  gtk_box_pack_start (GTK_BOX (hbox), mode_combo, FALSE, TRUE, 0);
  gtk_widget_show (mode_combo);

  g_signal_connect (G_OBJECT (mode_combo), "changed",
                    G_CALLBACK (mode_combo_box_changed), cv);
 
  p_data->sport_combo = gtk_combo_box_new_text ();
  
  gtk_combo_box_set_active (GTK_COMBO_BOX (p_data->sport_combo), 0);
  gtk_box_pack_start (GTK_BOX (hbox), p_data->sport_combo, FALSE, TRUE, 0);
  gtk_widget_show (p_data->sport_combo);

  g_signal_connect (G_OBJECT (p_data->sport_combo), "changed",
                    G_CALLBACK (sport_combo_box_changed), cv);
  
  p_data->chart = kp_chart_new ();
  gtk_box_pack_start (GTK_BOX (cv), p_data->chart, TRUE, TRUE, 0);
  gtk_widget_show (p_data->chart);

  p_data->param = g_string_new (NULL);
  p_data->sport = g_string_new (NULL);

  sport = kp_get_combo_box_first_entry (GTK_COMBO_BOX (p_data->sport_combo));
  param = kp_get_combo_box_first_entry (GTK_COMBO_BOX (param_combo));

  g_string_assign (p_data->sport, (sport) ? sport : "");
  g_string_assign (p_data->param, (param) ? param : "");
}

static void
kp_chart_view_model_init (KPViewModelIface *iface)
{
  iface->get_dmy = kp_chart_view_get_dmy;
  iface->set_dmy = kp_chart_view_set_dmy;
  iface->set_log = kp_chart_view_set_log;
  iface->unset_log = kp_chart_view_unset_log;
  iface->set_view_type = kp_chart_view_set_view_type;
  iface->get_view_type = kp_chart_view_get_view_type;
}

static void
kp_chart_view_finalize (GObject *object)
{
  KPChartViewPrivateData *p_data;
  guint i;

  p_data = KP_CHART_VIEW_PRIVATE_DATA (object);

  g_date_free (p_data->date);

  if (p_data->log) {
    log_disconnect_signals (p_data->log, KP_CHART_VIEW (object));
    p_data->log = NULL;
  }
  
  for (i=0; i < GROUPS_MAX; i++)
    g_free (p_data->graph_data[i]);
  g_free (p_data->graph_data);

  KP_CHART_VIEW (object)->private_data = NULL;
  g_free (KP_CHART_VIEW (object)->private_data);  
}


GtkWidget *
kp_chart_view_new (void)
{
  KPChartViewPrivateData *p_data;
  GtkWidget *widget;

  widget = g_object_new (kp_chart_view_get_type (), NULL);

  p_data = KP_CHART_VIEW_PRIVATE_DATA (widget);
  
  gtk_widget_show_all (GTK_WIDGET (widget));
  
  return widget;
}

static void
kp_chart_view_set_dmy (KPViewModel *view, guint d, guint m, guint y)
{
  KPChartViewPrivateData *p_data;
  
  p_data = KP_CHART_VIEW_PRIVATE_DATA (view);

  kp_view_model_set_dmy (KP_VIEW_MODEL (p_data->chart), d, m, y);
  
  g_return_if_fail (g_date_valid_dmy (d, m, y));
  g_date_set_dmy (p_data->date, d, m, y);

  kp_chart_view_update (KP_CHART_VIEW (view));
  
  kp_debug ("Date set: %u.%u.%u", d, m, y);
}

static void
kp_chart_view_get_dmy (KPViewModel *view, guint *d, guint *m, guint *y)
{
  KPChartViewPrivateData *p_data;
  p_data = KP_CHART_VIEW_PRIVATE_DATA (view);

  g_return_if_fail (KP_IS_CHART_VIEW (view));
  g_return_if_fail (g_date_valid (p_data->date));
  
  if (d)
    *d = g_date_get_day (p_data->date);
  if (m)
    *m = g_date_get_month (p_data->date);
  if (y)
    *y = g_date_get_year (p_data->date);
}

static void
kp_chart_view_set_view_type (KPViewModel *view, KPViewModelType type)
{
  KPChartViewPrivateData *p_data;
  p_data = KP_CHART_VIEW_PRIVATE_DATA (view);
  p_data->type = type;

  kp_debug ("Setting view.");
  
  kp_view_model_set_view_type (KP_VIEW_MODEL (p_data->chart), type);
  kp_chart_view_update (KP_CHART_VIEW (view));
}


static KPViewModelType
kp_chart_view_get_view_type (KPViewModel *view)
{
  KPChartViewPrivateData *p_data;
  p_data = KP_CHART_VIEW_PRIVATE_DATA (view);
  
  return p_data->type;
}


static void
kp_chart_view_set_log (KPViewModel *model, KPTrainingLog *log)
{
  KPChartViewPrivateData *p_data;
  KPSportEntry *entry;
  GSList *list;

  kp_debug ("Setting log.");
 
  if (!log)
    return;
  
  p_data = KP_CHART_VIEW_PRIVATE_DATA (model);
  p_data->log = log;

  kp_view_model_set_log (KP_VIEW_MODEL (p_data->chart), log);

  list = kp_training_log_get_sports (log);

  while (list) {
    entry = (KPSportEntry *) list->data;
    gtk_combo_box_prepend_text (GTK_COMBO_BOX (p_data->sport_combo), entry->name);
    list = list->next;
  }
  gtk_combo_box_set_active (GTK_COMBO_BOX (p_data->sport_combo), 0);
  
  kp_chart_view_update (KP_CHART_VIEW (model));

  log_connect_signals (log, KP_CHART_VIEW (model));
}

static void
kp_chart_view_unset_log (KPViewModel *model)
{
  KPChartViewPrivateData *p_data;
  GtkTreeModel *listmodel;
  p_data = KP_CHART_VIEW_PRIVATE_DATA (model);

  log_disconnect_signals (p_data->log, KP_CHART_VIEW (model));
  p_data->log = NULL;

  listmodel = gtk_combo_box_get_model (GTK_COMBO_BOX (p_data->sport_combo));

  if (GTK_IS_LIST_STORE (listmodel)) 
    gtk_list_store_clear (GTK_LIST_STORE (listmodel));
}


static void
kp_chart_view_update_dates (KPChartView *view)
{
  KPChartViewPrivateData *p_data;

  p_data = KP_CHART_VIEW_PRIVATE_DATA (view);

  if (!p_data->date) 
    return;

  /* For all time, the log is needed to figure out which are the first
   * and the last date of the log */
  if (p_data->type == KP_VIEW_MODEL_TYPE_ALL_TIME
   && !KP_IS_TRAINING_LOG (p_data->log))
    g_return_if_reached ();
    
  kp_gui_get_dates_for_view_type (p_data->date, p_data->type, &p_data->date_s,
                                 &p_data->date_e, p_data->log);
}


static void
log_changed (KPTrainingLog *log, KPChartView *cv)
{
  kp_chart_view_update (cv);
}

static void
log_sport_list_changed (KPTrainingLog *log, gboolean added, const gchar *sport,
                        KPCalendarView *cv)
{
  KPChartViewPrivateData *p_data;
  GtkTreeModel *model;
  GtkTreePath *path;
  GtkTreeIter iter;
  gchar *text;
  guint n;
  
  p_data = KP_CHART_VIEW_PRIVATE_DATA (cv);

  kp_debug ("Sport list changed!");
  
  if (added) {
    gtk_combo_box_prepend_text (GTK_COMBO_BOX (p_data->sport_combo), sport);
    return;
  }
  /* If we come here, it means that the sport was removed */
  model = gtk_combo_box_get_model (GTK_COMBO_BOX (p_data->sport_combo));

  path = gtk_tree_path_new_first ();
  
  n = 0;
  while (gtk_tree_model_get_iter (model, &iter, path)) {
      
    gtk_tree_model_get (model, &iter, 0, &text, -1);
      
    if (strcmp (text, sport) == 0) {
      gtk_combo_box_remove_text (GTK_COMBO_BOX (p_data->sport_combo), n);
      break;
    }
      
    n++;
    gtk_tree_path_next (path);
  }
  gtk_tree_path_free (path);
}

static void
log_connect_signals (KPTrainingLog *log, KPChartView *cv)
{
  g_signal_connect (G_OBJECT (log), "changed",
                    G_CALLBACK (log_changed), cv);
  g_signal_connect (G_OBJECT (log), "sport-list-changed",
                    G_CALLBACK (log_sport_list_changed), cv);
}

static void
log_disconnect_signals (KPTrainingLog *log, KPChartView *cv)
{
  g_signal_handlers_disconnect_by_func (log, log_changed, cv);
}


static void
param_combo_box_changed (GtkComboBox *box, KPChartView *cv)
{
  KPChartViewPrivateData *p_data;
  GtkTreeModel *model;
  GtkTreeIter iter;
  const gchar *param;
  gchar *str;
  guint i;
  
  g_return_if_fail (KP_IS_CHART_VIEW (cv));
  p_data = KP_CHART_VIEW_PRIVATE_DATA (cv);

  if (gtk_combo_box_get_active_iter (box, &iter)) {
    model = gtk_combo_box_get_model (box);
  
    gtk_tree_model_get (model, &iter, 0, &str, -1);    
  } else
    return;

  param = (const gchar *) str;

  for (i=0; i < sizeof (const_params) / CONST_ENTRY_SIZE; i++)
    if (strcmp (_(const_params[i]), param) == 0)
      param = &internal_params[i][0];
  
  g_string_assign (p_data->param, param);
  kp_debug ("Assigning param: %s", param);

  if (strcmp (param, "pace") == 0
   || strcmp (param, "duration") == 0)
    kp_chart_set_data_unit (KP_CHART (p_data->chart), KP_CHART_UNIT_MS);
  else
    kp_chart_set_data_unit (KP_CHART (p_data->chart), KP_CHART_UNIT_RAW);

  update_chart_data (cv);
}


static void
mode_combo_box_changed (GtkComboBox *box, KPChartView *cv)
{
  KPChartViewPrivateData *p_data;
  KPTrainingLogDataMode mode;
  GtkTreeModel *model;
  GtkTreeIter iter;
  gchar *mode_str;
  
  g_return_if_fail (KP_IS_CHART_VIEW (cv));
  p_data = KP_CHART_VIEW_PRIVATE_DATA (cv);

  if (gtk_combo_box_get_active_iter (box, &iter)) {
    model = gtk_combo_box_get_model (box);
  
    gtk_tree_model_get (model, &iter, 0, &mode_str, -1);
  } else
    return;
  
  if (!mode_str)
    return;

  if (strcmp (mode_str, _("Average")) == 0)
    mode = KP_TRAINING_LOG_DATA_AVG;
  else if (strcmp (mode_str, _("Sum")) == 0)
    mode = KP_TRAINING_LOG_DATA_SUM;
  else if (strcmp (mode_str, _("Minimum")) == 0)
    mode = KP_TRAINING_LOG_DATA_MIN;
  else if (strcmp (mode_str, _("Maximum")) == 0)
    mode = KP_TRAINING_LOG_DATA_MAX;
  else
    g_return_if_reached ();

  kp_debug ("Setting mode: %d\n", mode);

  kp_chart_set_chart_mode (KP_CHART (p_data->chart), mode);
  p_data->mode = mode;
  
  kp_debug ("Got param: %s\n", mode_str);
  update_chart_data (cv);
}

static void
sport_combo_box_changed (GtkComboBox *box, KPChartView *cv)
{
  KPChartViewPrivateData *p_data;
  GtkTreeModel *model;
  GtkTreeIter iter;
  gchar *sport;
  
  g_return_if_fail (KP_IS_CHART_VIEW (cv));
  p_data = KP_CHART_VIEW_PRIVATE_DATA (cv);

  if (gtk_combo_box_get_active_iter (box, &iter)) {
    model = gtk_combo_box_get_model (box);
  
    gtk_tree_model_get (model, &iter, 0, &sport, -1);
  } else
    return;
  
  if (sport)
    g_string_assign (p_data->sport, sport);
  
  kp_debug ("Got sport %s\n", sport);
  update_chart_data (cv);
}

static void
update_chart_data_day (KPChartView *view, guint days, gdouble **data)
{
  KPChartViewPrivateData *p_data;
  gchar *format;
  struct tm tm;
  gchar buf[32];
  guint n_items;
  guint i;
  
  g_return_if_fail (KP_IS_CHART_VIEW (view));
  p_data = KP_CHART_VIEW_PRIVATE_DATA (view);
 
  if (g_date_valid (p_data->date)) {
    g_date_to_struct_tm (p_data->date, &tm);
    format = "%x"; /* Hack to prevent a compiler warning */
    strftime (buf, sizeof(buf)-1, format, &tm);
    kp_chart_set_item_title (KP_CHART (p_data->chart), 0, buf);
  }
  n_items = kp_chart_get_n_items (KP_CHART (p_data->chart));
  for (i=0; i < n_items; i++)
    p_data->graph_data[0][i] = data[0][i];
}


static void
update_chart_data_week (KPChartView *view, guint days, gdouble **data)
{
  KPChartViewPrivateData *p_data;
  guint n_items;
  guint i;
  
  g_return_if_fail (KP_IS_CHART_VIEW (view));
  p_data = KP_CHART_VIEW_PRIVATE_DATA (view);
  
  for (i=0; i < 7; i++) {
    kp_chart_set_item_title (KP_CHART (p_data->chart), i, &hint_text_weekdays[i][0]);
  }

  n_items = kp_chart_get_n_items (KP_CHART (p_data->chart));
  
  for (i=0; i < n_items; i++)
    p_data->graph_data[0][i] = data[0][i];
}


static void
update_chart_data_month (KPChartView *view, guint days, gdouble **data)
{
  KPChartViewPrivateData *p_data;
  guint n_items;
  guint i;
  
  g_return_if_fail (KP_IS_CHART_VIEW (view));
  p_data = KP_CHART_VIEW_PRIVATE_DATA (view);
  
  n_items = kp_chart_get_n_items (KP_CHART (p_data->chart));
  
  for (i=0; i < n_items; i++) {
    p_data->graph_data[0][i] = data[0][i];
   
    kp_chart_set_item_title_int (KP_CHART (p_data->chart), i, i+1);
  }
}


static void
update_chart_data_year (KPChartView *view, guint days, gdouble **data)
{
  KPChartViewPrivateData *p_data;
  gboolean leap;
  guint m;
  guint i;
  gdouble *src;
  gdouble *loc;

  /* For avg */
  guint *avg_n;
  gdouble *avg_sum;
  
  g_return_if_fail (KP_IS_CHART_VIEW (view));
  p_data = KP_CHART_VIEW_PRIVATE_DATA (view);

  /*memset (&p_data->graph_data[0][0], 0, days-1 * sizeof (gdouble));*/
  
  leap = kp_leap (g_date_get_year (p_data->date));

  if (p_data->mode == KP_TRAINING_LOG_DATA_AVG) {
    avg_n = g_malloc0 (days * sizeof (*avg_n));
    avg_sum = g_malloc0 (days * sizeof (*avg_sum));
  } else {
    /* Prevent a compiler warning */
    avg_n = NULL;
    avg_sum = NULL;
  }
 
  for (i=1; i <= (365 + (guint)leap); i++) {
    g_return_if_fail (i <= days);
    m = kp_month_num_by_year_day_number (i, leap);

    loc = &p_data->graph_data[0][m-1];
    src = &data[0][i-1];
 
    switch (p_data->mode)
    {
      case KP_TRAINING_LOG_DATA_SUM:
        *loc += *src;
        break;

      case KP_TRAINING_LOG_DATA_MIN:
        if (*src) {
          if (*loc) {
            if (*src < *loc)
              *loc = *src;
          } else
            *loc = *src;
        }
        break;

      case KP_TRAINING_LOG_DATA_MAX:
        if (*src > *loc)
          *loc = *src;
        break;

      case KP_TRAINING_LOG_DATA_AVG:
        if (*src) {
          avg_n[m-1]++;
          avg_sum[m-1] += *src;
          *loc = ((gdouble) avg_sum[m-1] / (gdouble) avg_n[m-1]);
        }
        break;

      default:
        kp_debug ("Not implemented");
    }
  }
  
  if (p_data->mode == KP_TRAINING_LOG_DATA_AVG) {
    g_free (avg_n);
    g_free (avg_sum);
  }
  
  for (i=0; i < 12; i++)
    kp_chart_set_item_title (KP_CHART (p_data->chart), i, &hint_text_months[i][0]);
}


static void
update_chart_data_all_time (KPChartView *view, guint days, gdouble **data)
{
  KPChartViewPrivateData *p_data;
  gdouble *avg_sum;
  guint *avg_n;
  gdouble *loc, *src;
  GDate *date;
  guint prev_y;
  guint y;
  guint i;
 
  g_return_if_fail (KP_IS_CHART_VIEW (view));
  p_data = KP_CHART_VIEW_PRIVATE_DATA (view);
 
  date = g_date_new_julian (g_date_get_julian (p_data->date_s));
  g_return_if_fail (g_date_valid (date));

  /* There are not any years (not any entries) in the log,
   * and we don't want to crash */
  if (!p_data->log->year_list)
    return;

  if (days == 0)
    return;

  prev_y = g_date_get_year (date);
  y = 0;

  if (p_data->mode == KP_TRAINING_LOG_DATA_AVG) {
    avg_n = g_malloc0 (days * sizeof (*avg_n));
    avg_sum = g_malloc0 (days * sizeof (*avg_sum));
  } else {
    /* Prevent a compiler warning */
    avg_n = NULL;
    avg_sum = NULL;
  }

  for (i=0; i < days; i++) {

    loc = &p_data->graph_data[0][y];
    src = &data[0][i];
 
    switch (p_data->mode)
    {
      case KP_TRAINING_LOG_DATA_SUM:
        *loc += *src;
        break;

      case KP_TRAINING_LOG_DATA_MIN:
        if (*src) {
          if (*loc) {
            if (*src < *loc)
              *loc = *src;
          } else
            *loc = *src;
        }
        break;

      case KP_TRAINING_LOG_DATA_MAX:
        if (*src > *loc)
          *loc = *src;
        break;

      case KP_TRAINING_LOG_DATA_AVG:
        if (*src) {
          avg_n[y]++;
          avg_sum[y] += *src;
          *loc = ((gdouble) avg_sum[y] / (gdouble) avg_n[y]);
        }
        break;

      default:
        g_return_if_reached ();
    }
    g_date_add_days (date, 1);

    if (g_date_get_year (date) > prev_y) {
      kp_chart_set_item_title_int (KP_CHART (p_data->chart), y, prev_y);
      prev_y = g_date_get_year (date);
      y++;
    }

    kp_chart_set_item_title_int (KP_CHART (p_data->chart), y, prev_y);
  }
}
  

static void
update_chart_data (KPChartView *view)
{
  KPChartViewPrivateData *p_data;
  gdouble **data;
  gchar *sport;
  guint n_groups;
  guint days;
  guint n_items;
  guint i;

  n_items = 0;
  data = NULL;
  
  g_return_if_fail (KP_IS_CHART_VIEW (view));

  p_data = KP_CHART_VIEW_PRIVATE_DATA (view);

  if (!KP_IS_TRAINING_LOG (p_data->log))
    return;
  
  n_groups = kp_chart_get_n_groups (KP_CHART (p_data->chart));

  memset (&p_data->graph_data[0][0], 0, sizeof (gdouble)*ITEMS_MAX);
  
  for (i=0; i < n_groups; i++) {

    sport = (p_data->sport->len > 0) ? p_data->sport->str : NULL;

    data = kp_training_log_get_workout_params_between (p_data->log,
                                                       p_data->mode,
                                                       sport,
                                                       p_data->param->str,
                                                       p_data->date_s,
                                                       p_data->date_e,
                                                      &days);    
    switch (p_data->type)
    {
      case KP_VIEW_MODEL_TYPE_YEAR:
        update_chart_data_year (view, days, data);
        n_items = 12;
        break;

      case KP_VIEW_MODEL_TYPE_ALL_TIME:
        update_chart_data_all_time (view, days, data);
        n_items = kp_training_log_get_n_years (p_data->log);
        break;

      case KP_VIEW_MODEL_TYPE_WEEK:
        update_chart_data_week (view, days, data);
        n_items = 7;
        break;
     
      case KP_VIEW_MODEL_TYPE_MONTH:
        update_chart_data_month (view, days, data);
        n_items = kp_get_month_len (g_date_get_month (p_data->date),
                           kp_leap (g_date_get_year (p_data->date)));
        break;

      case KP_VIEW_MODEL_TYPE_DAY:
        update_chart_data_day (view, days, data);
        n_items = 1;
        break;

      default:
        g_return_if_reached ();
    }
  }
  if (data) {
    kp_chart_set_n_items (KP_CHART (p_data->chart), n_items);
    kp_chart_set (KP_CHART (p_data->chart), p_data->graph_data); 
  }
}

static void
kp_chart_view_update (KPChartView *cv)
{
  KPChartViewPrivateData *p_data;
  
  g_return_if_fail (KP_IS_CHART_VIEW (cv));
  p_data = KP_CHART_VIEW_PRIVATE_DATA (cv);

  kp_chart_view_update_dates (cv);
  update_chart_data (cv);
}


