#include <string.h>
#include <math.h>

#include <glib-object.h>
#include <glib.h>

#include "kpviewmodel.h"
#include "kpstatsview.h"
#include "kpguiutils.h"

#include "../kpworkoutmodel.h"
#include "../kipina-i18n.h"

static void            kp_stats_view_class_init        (GObjectClass *klass,
                                                        gpointer data);
static void            kp_stats_view_model_init        (KPViewModelIface *iface);
static void            kp_stats_view_instance_init     (GObject *object,
                                                        gpointer data);
static void            kp_stats_view_instance_finalize (GObject *object);
static void            kp_stats_view_update_workouts_and_sports
                                                       (KPStatsView *view);
static void            kp_stats_update                 (KPStatsView *model);
static void            kp_stats_update_dates           (KPStatsView *view);
static void            kp_stats_set_statistics         (KPStatsView *model);
static void            kp_stats_set_distance           (KPStatsView *model);
static void            kp_stats_set_time               (KPStatsView *model);
static void            kp_stats_view_activate          (KPViewModel *model);
static void            kp_stats_view_deactivate        (KPViewModel *model);
static void            kp_stats_view_set_dmy           (KPViewModel *view,
                                                        guint d,
                                                        guint m,
                                                        guint y);
static void            kp_stats_view_get_dmy           (KPViewModel *view,
                                                        guint *d,
                                                        guint *m,
                                                        guint *y);
static void            kp_stats_view_set_log           (KPViewModel *view,
                                                        KPTrainingLog *log);
static void            kp_stats_view_unset_log         (KPViewModel *view);
static void            kp_stats_view_set_view_type     (KPViewModel *view,
                                                        KPViewModelType type);
static KPViewModelType kp_stats_view_get_view_type     (KPViewModel *view);
static gchar          *kp_stats_view_get_icon_name     (KPViewModel *view);

static void            kp_stats_set_field              (KPStatsView *model,
                                                        guint field,
                                                        const gchar *text,
                                                        gboolean use_markup);
static void            kp_stats_set_field_uint         (KPStatsView *model,
                                                        guint field,
                                                        guint num);
static void            kp_stats_set_field_double       (KPStatsView *model,
                                                        guint field,
                                                        gdouble value);
static void            kp_stats_set_field_percent      (KPStatsView *model,
                                                        guint field,
                                                        gdouble value);
static void            kp_stats_set_field_time         (KPStatsView *model,
                                                        guint field,
                                                        guint msec);
static void            sport_menu_selection_changed    (GtkOptionMenu *men,
                                                        KPStatsView *view);

static gboolean        kp_stats_filter                 (KPStatsView *view,
                                                        KPCalendarEntry *entry);
static void            kp_stats_set_filter             (KPStatsView *view,
                                                        const gchar *sport);

enum {
  LABEL_DAYS      = 0,
  LABEL_WORKOUTS,
  LABEL_WORKOUTS_PER_DAY,
  LABEL_ALL_TIME_RANK,
  LABEL_DISTANCE_MIN,
  LABEL_DISTANCE_AVG,
  LABEL_DISTANCE_PER_DAY,
  LABEL_DISTANCE_MAX,
  LABEL_DISTANCE_TOTAL,
  LABEL_DISTANCE_PERCENT,
  LABEL_TIME_MIN,
  LABEL_TIME_AVG,
  LABEL_TIME_PER_DAY,
  LABEL_TIME_MAX,
  LABEL_TIME_TOTAL,
  LABEL_TIME_PERCENT,
  LABEL_N
};

struct LabelData
{
  gint  n;
  gchar *name;
}
labels_data[] = {
  { LABEL_DAYS,             "l_days"             },
  { LABEL_WORKOUTS,         "l_workouts"         },
  { LABEL_WORKOUTS_PER_DAY, "l_workouts_per_day" },
  { LABEL_ALL_TIME_RANK,    "l_rank"             },
  { LABEL_DISTANCE_MIN,     "l_distance_min"     },
  { LABEL_DISTANCE_AVG,     "l_distance_avg"     },
  { LABEL_DISTANCE_PER_DAY, "l_distance_per_day" },
  { LABEL_DISTANCE_MAX,     "l_distance_max"     }, 
  { LABEL_DISTANCE_TOTAL,   "l_distance_total"   },
  { LABEL_DISTANCE_PERCENT, "l_distance_percent" },
  { LABEL_TIME_MIN,         "l_time_min"         },
  { LABEL_TIME_AVG,         "l_time_avg"         },
  { LABEL_TIME_PER_DAY,     "l_time_per_day"     },
  { LABEL_TIME_MAX,         "l_time_max"         },
  { LABEL_TIME_TOTAL,       "l_time_total"       },
  { LABEL_TIME_PERCENT,     "l_time_percent"     },
  { LABEL_N,                 NULL                }
};


#define FILTER_ALL_TEXT _("All Sports")

typedef struct KPStatsViewPrivateData_
{
  KPViewModelType   type;
  GDate            *date;

  GDate            *date_s;
  GDate            *date_e;

  KPTrainingLog    *log;

  GList            *entries;

  GString          *filter;

  GtkWidget        *sports_menu;
  GtkLabel         *labels[LABEL_N];
} KPStatsViewPrivateData;

#define KP_STATS_VIEW_PRIVATE_DATA(widget) ((KPStatsViewPrivateData*) \
        (KP_STATS_VIEW (widget)->private_data))

enum {
  COLUMN_LABEL,
  COLUMN_VALUE,
  COLUMN_N,
};

GType
kp_stats_view_get_type ()
{
  static GType kp_stats_view_type = 0;

  if (!kp_stats_view_type) {
    static const GTypeInfo kp_stats_view_info = {
      sizeof (KPStatsViewClass),
      (GBaseInitFunc) NULL,
      (GBaseFinalizeFunc) NULL,
      (GClassInitFunc) kp_stats_view_class_init,
      (GClassFinalizeFunc) NULL,
      NULL,
      sizeof (KPStatsView),
      0,
      (GInstanceInitFunc) kp_stats_view_instance_init,
      NULL
    };
    static const GInterfaceInfo view_model_info = {
     (GInterfaceInitFunc) kp_stats_view_model_init,
      NULL,
      NULL
    };

    kp_stats_view_type = g_type_register_static (GTK_TYPE_VBOX,
                                                "KPStatsView",
                                                &kp_stats_view_info,
                                                 0);
    g_type_add_interface_static (kp_stats_view_type,
                                 KP_TYPE_VIEW_MODEL,
                                &view_model_info);
  }
  return kp_stats_view_type;
}

static void
kp_stats_view_class_init (GObjectClass * klass,
                          gpointer data)
{
  GObjectClass *object_class;

  object_class = G_OBJECT_CLASS (klass);
  object_class->finalize = kp_stats_view_instance_finalize;
}

static void
kp_stats_view_model_init (KPViewModelIface *iface)
{
  iface->get_dmy = kp_stats_view_get_dmy;
  iface->set_dmy = kp_stats_view_set_dmy;
  iface->set_log = kp_stats_view_set_log;
  iface->unset_log = kp_stats_view_unset_log;
  iface->set_view_type = kp_stats_view_set_view_type;
  iface->get_view_type = kp_stats_view_get_view_type;
  iface->get_icon_name = kp_stats_view_get_icon_name;
  iface->activate = kp_stats_view_activate;
  iface->deactivate = kp_stats_view_deactivate;
}

static void
kp_stats_view_instance_init (GObject *object, gpointer data)
{
  KPStatsViewPrivateData *p_data;
  KPStatsView *view;
  GtkWidget *box;
  GladeXML *xml;
  guint i;
 
  view = KP_STATS_VIEW (object);
  view->private_data = g_new0 (KPStatsViewPrivateData, 1);
  p_data = KP_STATS_VIEW_PRIVATE_DATA (object);
  g_return_if_fail (p_data != NULL);
  
  p_data->date = g_date_new ();
  p_data->date_s = g_date_new ();
  p_data->date_e = g_date_new ();
  p_data->log = NULL;
  p_data->filter = g_string_new (NULL);

  xml = kp_gui_load ("statistics", "box");
  
  box = KP_W (xml, "box");
  p_data->sports_menu = KP_W (xml, "sports_menu");

  g_signal_connect (G_OBJECT (p_data->sports_menu), "changed",
                    G_CALLBACK (sport_menu_selection_changed), view);
 
  for (i=0; labels_data[i].name != NULL; i++)
    p_data->labels[i] = GTK_LABEL (KP_W (xml, labels_data[i].name));
 
  gtk_label_set_text (p_data->labels[LABEL_DAYS], "666");

  gtk_box_pack_start (GTK_BOX (view), GTK_WIDGET (box), TRUE, TRUE, 0);
  gtk_widget_show (GTK_WIDGET (view));

  g_object_unref (xml);

  kp_stats_update (KP_STATS_VIEW (view));
}


static void
kp_stats_view_instance_finalize (GObject *object)
{
  KPStatsViewPrivateData *p_data;
  KPStatsView *view;
  GObjectClass *parent_class;

  view = KP_STATS_VIEW (object);
  p_data = KP_STATS_VIEW_PRIVATE_DATA (object);

  if (!p_data)
    return;
  
  g_string_free (p_data->filter, TRUE);
  g_date_free (p_data->date);
  g_free (view->private_data);
  view->private_data = NULL;
  
  parent_class = g_type_class_peek_parent (G_OBJECT_GET_CLASS (object));
  parent_class->finalize (object);
}


/**
 * kp_stats_view_new:
 * 
 * Create a new instance of #KPStatsView.
 *
 * Returns: A #KPStatsView 
 */
KPStatsView *
kp_stats_view_new (void)
{
  return g_object_new (kp_stats_view_get_type (), NULL);
}

 
static void 
kp_stats_set_statistics (KPStatsView *model)
{
  KPStatsViewPrivateData *p_data;
  GList *list;
  guint n_days;
  guint n;

  p_data = KP_STATS_VIEW_PRIVATE_DATA (model);
  list = p_data->entries;

  n=0;
  while (list) {
    if (kp_stats_filter (model, KP_CALENDAR_ENTRY (list->data)))
      n++;
    list = list->next;
  }
  n_days = g_date_days_between (p_data->date_s, p_data->date_e) + 1;
  kp_stats_set_field_uint (model, LABEL_WORKOUTS, n);
  kp_stats_set_field_uint (model, LABEL_DAYS, n_days);
  kp_stats_set_field_double (model, LABEL_WORKOUTS_PER_DAY,
             (n_days > 0) ? (gdouble) n / (gdouble) n_days : n);
}

static void
kp_stats_set_distance (KPStatsView *model)
{
  KPStatsViewPrivateData *p_data;
  gdouble distance;
  gdouble min;
  gdouble max;
  gdouble sum;
  GList *list;
  guint n_days; /* how many days are there? */
  guint n_dist; /* how many workouts have distance field? */
  guint n;      /* how many workouts are there? */

  p_data = KP_STATS_VIEW_PRIVATE_DATA (model);
  list = p_data->entries;

  sum = min = max = 0.0;
  n_dist = n = 0;

  while (list) {
    if (kp_stats_filter (model, KP_CALENDAR_ENTRY (list->data))) {
      distance = kp_workout_model_get_distance (KP_WORKOUT_MODEL (list->data));
    
      if (n == 0) {
        min = distance;
        max = distance;
      }
      sum += distance;
      n_dist += (distance > 0.0);
      
      if (distance > 0 && distance < min) 
        min = distance;
      
      if (distance > max)
        max = distance;
        
      n++;
    }
    list = list->next;
  }
  n_days = g_date_days_between (p_data->date_s, p_data->date_e) + 1;
 
  kp_stats_set_field_double (model, LABEL_DISTANCE_TOTAL, sum);
  kp_stats_set_field_double (model, LABEL_DISTANCE_MAX, max);
  kp_stats_set_field_double (model, LABEL_DISTANCE_MIN, min);
  kp_stats_set_field_double (model, LABEL_DISTANCE_PER_DAY,
                             sum / (gdouble) n_days);
  kp_stats_set_field_double (model, LABEL_DISTANCE_AVG,
                            (n > 0) ? sum / (gdouble) n : 0.0);
  kp_stats_set_field_percent (model, LABEL_DISTANCE_PERCENT,
                            (n > 0) ? (gdouble) n_dist / (gdouble) n : 0.0);
}

static void
kp_stats_set_time (KPStatsView *model)
{
  KPStatsViewPrivateData *p_data;
  guint duration;
  guint min;
  guint max;
  guint sum;
  GList *list;
  guint n_time;
  guint n_days;
  guint n;

  p_data = KP_STATS_VIEW_PRIVATE_DATA (model);
  list = p_data->entries;

  sum = min = max = 0;
  n_time = 0;
  n=0;

  while (list) {
    if (kp_stats_filter (model, KP_CALENDAR_ENTRY (list->data))) {
      duration = kp_workout_model_get_duration (KP_WORKOUT_MODEL (list->data));
      kp_debug ("Got duration: %u", duration);
    
      if (n == 0) {
        min = duration;
        max = duration;
      }
      sum += duration;
      n_time += (duration > 0);
      
      if (duration > 0 && duration < min) 
        min = duration;
      
      if (duration > max)
        max = duration;
        
      n++;
    }
    list = list->next;
  }
  n_days = g_date_days_between (p_data->date_s, p_data->date_e) + 1;
 
  kp_stats_set_field_time (model, LABEL_TIME_TOTAL, sum);
  kp_stats_set_field_time (model, LABEL_TIME_MAX, max);
  kp_stats_set_field_time (model, LABEL_TIME_MIN, min);
  kp_stats_set_field_time (model, LABEL_TIME_PER_DAY, sum / (gdouble) n_days);
  kp_stats_set_field_time (model, LABEL_TIME_AVG,
                          (n > 0) ? floor ((gdouble) sum / (gdouble) n) : 0);
  kp_stats_set_field_percent (model, LABEL_TIME_PERCENT,
                          (n > 0) ? ((gdouble) n_time / (gdouble) n) : 0);

}


static void
kp_stats_view_update_workouts_and_sports (KPStatsView *view)
{
  KPStatsViewPrivateData *p_data;
  GtkWidget *menu, *mi;
  GList *sports;
  GList *items;
  GList *tmp;
  
  p_data = KP_STATS_VIEW_PRIVATE_DATA (view);

  if (!g_date_valid (p_data->date))
    return;

  g_return_if_fail (KP_IS_TRAINING_LOG (p_data->log));
 
  p_data->entries = kp_training_log_get_all_entries_between (p_data->log,
                                                             p_data->date_s,
                                                             p_data->date_e,
                                                            &sports);

  menu = gtk_option_menu_get_menu (GTK_OPTION_MENU (p_data->sports_menu));
  items = gtk_container_get_children (GTK_CONTAINER (menu));

  while (items) {
    gtk_container_remove (GTK_CONTAINER (menu),
                          GTK_WIDGET (items->data));
    tmp = items;
    items = items->next;

    g_list_free_1 (tmp);
  }
 
  sports = g_list_prepend (sports, g_strdup (FILTER_ALL_TEXT));
  
  while (sports) {
    mi = gtk_menu_item_new_with_label (sports->data);
    gtk_menu_shell_append (GTK_MENU_SHELL (menu), mi);

    tmp = sports;    
    sports = sports->next;
    
    g_free (tmp->data);
    g_list_free_1 (tmp);
  }
  gtk_widget_show_all (p_data->sports_menu);
}

static void
kp_stats_update_dates (KPStatsView *view)
{
  KPStatsViewPrivateData *p_data;
  
  p_data = KP_STATS_VIEW_PRIVATE_DATA (view);

  if (!p_data->date) {
    kp_debug ("Date not valid, return.\n");
    return;
  }

  /* The function needs the log to figure out which are the first
   * and the last date of the log */
  if (p_data->type == KP_VIEW_MODEL_TYPE_ALL_TIME
   && !KP_IS_TRAINING_LOG (p_data->log))
    g_return_if_reached ();
    
  kp_gui_get_dates_for_view_type (p_data->date, p_data->type, &p_data->date_s,
                                 &p_data->date_e, p_data->log);
}

static void
kp_stats_update (KPStatsView *view)
{
  KPStatsViewPrivateData *p_data;
  
  p_data = KP_STATS_VIEW_PRIVATE_DATA (view);

  if (!g_date_valid (p_data->date_s)) {
    return;
  }

  kp_debug ("Ready to update fields.");
  
  kp_stats_set_statistics (view);
  kp_stats_set_distance (view);
  kp_stats_set_time (view);
}

static void
kp_stats_view_set_dmy (KPViewModel *view, guint d, guint m, guint y)
{
  KPStatsViewPrivateData *p_data;
  
  p_data = KP_STATS_VIEW_PRIVATE_DATA (view);

  g_return_if_fail (g_date_valid_dmy (d, m, y));
  g_date_set_dmy (p_data->date, d, m, y);

  kp_stats_update_dates (KP_STATS_VIEW (view));
  
  if (!p_data->log)
    return;

  kp_stats_view_update_workouts_and_sports (KP_STATS_VIEW (view));
  kp_stats_update (KP_STATS_VIEW (view));
}

static void
kp_stats_view_get_dmy (KPViewModel *view, guint *d, guint *m, guint *y)
{
  KPStatsViewPrivateData *p_data;
  p_data = KP_STATS_VIEW_PRIVATE_DATA (view);

  g_return_if_fail (KP_IS_STATS_VIEW (view));
  g_return_if_fail (g_date_valid (p_data->date));
  
  if (d)
    *d = g_date_get_day (p_data->date);
  if (m)
    *m = g_date_get_month (p_data->date);
  if (y)
    *y = g_date_get_year (p_data->date);
}
  
static void
kp_stats_view_activate (KPViewModel *model)
{
  /* nothing to do */
}

static void
kp_stats_view_deactivate (KPViewModel *model)
{
  /* nothing to do */
}

static void
kp_stats_view_set_log (KPViewModel *view, KPTrainingLog *log)
{
  KPStatsViewPrivateData *p_data;

  g_return_if_fail (KP_IS_VIEW_MODEL (view));
  
  p_data = KP_STATS_VIEW_PRIVATE_DATA (view);
  p_data->log = log;

  kp_stats_view_update_workouts_and_sports (KP_STATS_VIEW (view));
  kp_stats_update (KP_STATS_VIEW (view));
}

static void
kp_stats_view_unset_log (KPViewModel *view)
{
  KPStatsViewPrivateData *p_data;

  g_return_if_fail (KP_IS_VIEW_MODEL (view));
  p_data = KP_STATS_VIEW_PRIVATE_DATA (view);
  if (!KP_IS_STATS_VIEW (p_data->log))
    return;
  else
    p_data->log = NULL;

  kp_stats_view_update_workouts_and_sports (KP_STATS_VIEW (view));
  kp_stats_update (KP_STATS_VIEW (view));
}

static void
kp_stats_view_set_view_type (KPViewModel *view, KPViewModelType type)
{
  KPStatsViewPrivateData *p_data;
  p_data = KP_STATS_VIEW_PRIVATE_DATA (view);
  p_data->type = type;

  kp_stats_update_dates (KP_STATS_VIEW (view));
  kp_stats_view_update_workouts_and_sports (KP_STATS_VIEW (view));
  kp_stats_update (KP_STATS_VIEW (view));
}


static KPViewModelType
kp_stats_view_get_view_type (KPViewModel *view)
{
  KPStatsViewPrivateData *p_data;
  p_data = KP_STATS_VIEW_PRIVATE_DATA (view);
  
  return p_data->type;
}

static gchar *
kp_stats_view_get_icon_name (KPViewModel *view)
{
  return g_strdup ("statistics.png");
}

static void
sport_menu_selection_changed (GtkOptionMenu *menu, KPStatsView *view)
{
  KPStatsViewPrivateData *p_data;
  gchar *text;

  p_data = KP_STATS_VIEW_PRIVATE_DATA (view);
 
  text = kp_gui_get_option_menu_active (menu);

  if (text) {
    if (strcmp (text, FILTER_ALL_TEXT) == 0)
      kp_stats_set_filter (view, NULL);
    else 
      kp_stats_set_filter (view, text);
    
    g_free (text);
  }
}


static gboolean
kp_stats_filter (KPStatsView *view, KPCalendarEntry *entry)
{
  KPStatsViewPrivateData *p_data;
  const gchar *sport;

  if (!KP_IS_WORKOUT (entry))
    return FALSE;
  
  p_data = KP_STATS_VIEW_PRIVATE_DATA (view);

  if (!p_data->filter->len)
    return TRUE;

  sport = kp_workout_get_sport (KP_WORKOUT (entry));
  
  if (strcmp (p_data->filter->str, sport) == 0)
    return TRUE;

  return FALSE;
}


static void
kp_stats_set_filter (KPStatsView *view, const gchar *sport)
{
  KPStatsViewPrivateData *p_data;

  p_data = KP_STATS_VIEW_PRIVATE_DATA (view);

  g_string_assign (p_data->filter, ((sport) ? sport : ""));

  kp_stats_view_update_workouts_and_sports (view);
  kp_stats_update (view);
}


static void
kp_stats_set_field (KPStatsView *model, guint field, const gchar *text,
                    gboolean use_markup)
{
  KPStatsViewPrivateData *p_data;

  g_return_if_fail (KP_IS_VIEW_MODEL (model));
  g_return_if_fail (field < LABEL_N);
  
  p_data = KP_STATS_VIEW_PRIVATE_DATA (model);
  if (use_markup)
    gtk_label_set_markup (p_data->labels[field], text);
  else
    gtk_label_set_text (p_data->labels[field], text);
}



static void
kp_stats_set_field_uint (KPStatsView *model, guint field, guint num)
{
  gchar buf[32];
  g_snprintf (buf, sizeof (buf)-1, "%u", num);
  kp_stats_set_field (model, field, buf, FALSE);
}


static void
kp_stats_set_field_double (KPStatsView *model, guint field, gdouble value)
{
  gchar buf[32];
  g_snprintf (buf, sizeof (buf)-1, "%.2f", value);
  kp_stats_set_field (model, field, buf, FALSE);
}


static void
kp_stats_set_field_time (KPStatsView *model, guint field, guint msec)
{
  guint h, m, s, t;
  gchar buf[32];

  h = (msec / (60 * 60 * 1000));
  m = ((msec - h * 60 * 60 * 1000) / (60 * 1000));
  s = (msec - h * 1000 * 60 * 60 - m * 60 * 1000) / 1000;
  t = (msec - h * 1000 * 60 * 60 - m * 60 * 1000 - s * 1000);

  kp_debug ("h: %u, m: %u, s: %u, t: %u", h, m, s, t);
  
  g_snprintf (buf, sizeof (buf)-1,"%.2u:%.2u:%.2u.%.3u", h, m, s, t);
  kp_stats_set_field (model, field, buf, FALSE);
}


static void
kp_stats_set_field_percent (KPStatsView *model, guint field, gdouble value)
{
  gchar buf[32];
  g_snprintf (buf, sizeof (buf)-1, "<small>%.1f%%</small>", value * 100.0);
  kp_stats_set_field (model, field, buf, TRUE);
}


