#include <glib.h>

#include "kphtmldiagram.h"
#include "kphtmlutil.h"
#include "../kputil.h"


/**
 * kp_html_diagram_new:
 * 
 * Allocate a new #KPHtmlDiagram structure.
 */
KPHtmlDiagram *
kp_html_diagram_new (guint groups, guint items)
{
  KPHtmlDiagram *diagram;
  
  diagram = g_new0 (KPHtmlDiagram, 1);
  diagram->max = 0.0;
  diagram->min = 0.0;
  diagram->groups = groups;
  diagram->items = items;
  diagram->data = NULL;
  diagram->x_title = g_string_new ("");
  diagram->y_title = g_string_new ("");
  diagram->title = g_string_new ("");
  
  diagram->x_hints = g_malloc0 (items * sizeof (gchar *));
  diagram->group_titles = g_malloc0 (groups * sizeof (gchar *));

  diagram->height = 150;

  return diagram;
}

static void
kp_html_diagram_write_hint_box (KPHtmlDiagram *diagram, FILE *fp)
{
  guint i;
  
  for (i=0; i < diagram->groups; i++) {
    fprintf (fp, "<img src=\"horiz_%u.png\" style=\"width:10px;height:10px;"
                 "border: 1px solid black\" alt=\"\" />&nbsp;%s<br/>\n",
                  i + 1, NOT_NULL (diagram->group_titles[i]));
  }
}


/**
 * kp_html_diagram_write:
 * @diagram: A #KPHtmlDiagram
 * @fp: A file handle to an opened file
 *
 * Write the HTML Code for the diagram to @fp.
 * Diagram data must be set via kp_html_diagram_set_data ().
 */
void
kp_html_diagram_write (KPHtmlDiagram *diagram, FILE *fp)
{
  guint i, j;

  g_return_if_fail (diagram != NULL);
  g_return_if_fail (fp != NULL);
  g_return_if_fail (diagram->data != NULL);

  /* Table start */
  fprintf (fp, "\n\n<!-- Diagram Table -->\n"
               "<table border=\"0\" class=\"table_stats\" style=\"width:100%\""
               " cellspacing=\"0\" cellpadding=\"0\">\n"
               "<thead>\n"
               "<tr>\n"
               "  <th class=\"th_theadleft\" colspan=\"%u\">%s</th>\n"
               "</tr>\n"
               "</thead>\n"
               "<tbody>\n"
               "<tr valign=\"bottom\">\n"
               "  <td class=\"diagram_axis_title_y\" colspan=\"%u\">%s</td>\n"
               "</tr>\n"
               "<tr>\n"
               "  <td class=\"diagram_hint\" style=\"height: %upx\">%.2f</td>\n",
               diagram->items * diagram->groups + 2, 
               diagram->title->str,
               diagram->items * diagram->groups + 2,
               diagram->y_title->str,
               diagram->height / 4,
               diagram->max);
 

  /* Diagram bars */ 
  for (j=0; j < diagram->items; j++) {
    for (i=0; i < diagram->groups; i++) {
      fprintf (fp, "<td rowspan=\"4\" class=\"diagram_bar\" style=\"height: %upx;\">"
                   "  <small>%.1f</small><br/>"
                   "  <img src=\"horiz_%u.png\" width=\"10\" height=\"%u\" "
                   "class=\"diagram_tower\" alt=\"\" />\n"
                   "</td>\n",
                   (guint) diagram->height + 15,
                   diagram->data[diagram->groups * j + i], 
                   i+1, 
                   (guint) (diagram->data[diagram->groups * j + i] /
                            diagram->max * (gdouble) diagram->height) - 2);
    }
  }

  /* Hint box on the right side of the diagram */
  fprintf (fp, "<td class=\"diagram_hint_table\" rowspan=\"4\">\n");
  kp_html_diagram_write_hint_box (diagram, fp);
  fprintf (fp, "</td>\n"
               "</tr>\n");

  /* Y Axis hints */
  for (i=75; i > 0; i -= 25) {
    fprintf (fp, "<tr>\n"
                 "  <td class=\"diagram_hint\" style=\"height: %upx\">%.2f</td>\n"
                 "</tr>\n",
                  diagram->height / 4,
                  diagram->max * (gdouble)i / 100.0);
  }
  
  /* Last Y Axis hint */
  fprintf (fp, "<tr>"
               "  <td style=\"vertical-align: top;\">%.2f</td>\n", diagram->min);

  /* X Axis hints */
  for (j=0; j < diagram->items; j++)
    fprintf (fp, "  <td colspan=\"%u\">%s</td>\n", 
             diagram->groups,
             NOT_NULL (diagram->x_hints[j]));
  
  /* X Axis title & Table footer */
  fprintf (fp, "</tr>\n"
               "<tr>\n"
               "  <td class=\"diagram_axis_title_x\" colspan=\"%u\">%s</td>"
               "</tr>\n"
               "</tbody>\n"
               "</table>\n",
               diagram->items * diagram->groups + 2,
               diagram->x_title->str);
}



/**
 * kp_html_diagram_set_data:
 * @diagram: A #KPHtmlDiagram
 * @data: Pointer to double
 * @groups: Num of datasets
 * @items: Num of items in a dataset
 * 
 * Set data for diagram.
 */
void
kp_html_diagram_set_data (KPHtmlDiagram *diagram, gdouble *data)
{
  guint i, j;
 
  g_return_if_fail (diagram != NULL);
 
  g_free (diagram->data); 
  diagram->data = g_new0 (gdouble, diagram->groups * diagram->items);
  diagram->max = 0.0;
  diagram->min = 0.0;
    
  for (j=0; j < diagram->items; j++)
    for (i=0; i < diagram->groups; i++) {
      diagram->data[diagram->groups * j + i] = data[diagram->groups * j + i];
      diagram->max = MAX (data[diagram->groups * j + i], diagram->max);
      diagram->min = MIN (data[diagram->groups * j + i], diagram->min);
    }

  kp_autoscale_axis (&diagram->min, &diagram->max);
}


void
kp_html_diagram_set_x_hint (KPHtmlDiagram *diagram, guint n, 
                            const gchar *title) 
{
  g_return_if_fail (diagram != NULL);
  g_return_if_fail (n < diagram->items);

  diagram->x_hints[n] = (title) ? g_strdup (title) : NULL;
}


void
kp_html_diagram_set_group_title (KPHtmlDiagram *diagram, guint n, 
                                 const gchar *title)
{
  g_return_if_fail (diagram != NULL);
  g_return_if_fail (n < diagram->groups);

  diagram->group_titles[n] = (title) ? g_strdup (title) : NULL;
}


void
kp_html_diagram_set_x_title (KPHtmlDiagram *diagram, const gchar *title)
{
  g_string_assign (diagram->x_title, title);
}


void
kp_html_diagram_set_y_title (KPHtmlDiagram *diagram, const gchar *title)
{
  g_string_assign (diagram->y_title, title);
}


void
kp_html_diagram_set_title (KPHtmlDiagram *diagram, const gchar *title)
{
  g_string_assign (diagram->title, title);
}


/**
 * kp_html_diagram_free:
 * @diagram: A #KPHtmlDiagram
 *
 * Free memory used by @diagram.
 */
void
kp_html_diagram_free (KPHtmlDiagram *diagram)
{
  guint i;
  g_string_free (diagram->x_title, TRUE);
  g_string_free (diagram->y_title, TRUE);
  g_string_free (diagram->title, TRUE);

  for (i=0; i < diagram->items; i++)
    g_free (diagram->x_hints[i]);
  for (i=0; i < diagram->groups; i++)
    g_free (diagram->group_titles[i]);
  
  g_free (diagram->data);
  diagram->data = NULL;
  g_free (diagram);
}

