#include "config.h"
 
#include <stdlib.h> 
#include <stdio.h>
#include <string.h>

#include "vm/classfil.h" 
#include "vm/jni.h" 

#include "vm/classfile_methods.h"
#include "vm/classloader_tuple.h"
#include "vm/interp_methods.h" 
#include "vm/cptuplelist.h"
#include "vm/newobject.h"
#include "vm/sys_linux_host/wrappers.h"
#include "lib/indigenous/java.lang/VMClass.h"
#include "lib/indigenous/java.lang/Class_Reflection.h"


extern tOBREF OOMExceptionObject;

extern tClassLoaderTuple* pstObjectType; // this can be globally used to get the type for object
extern tClassLoaderTuple* pstClassType; // this can be globally used to get the type for class

 
typedef struct primitiveStructure {
  tAType type;
  tClassLoaderTuple* pstType;
  jclass classObject;        /* the class object for "int", "byte" etc */
  jclass wrapperClassObject; /* the class object "java/lang/Integer", 
				"java/lang/Byte" etc */
} tPrimitiveStructure;

static tPrimitiveStructure primTypes[9];

static char* typeStrings[] = {
  "boolean", "char", "float", "double", "byte", "short", 
  "int", "long", "void"};
static tAType typeCodes[] = { 
  T_BOOLEAN, T_CHAR, T_FLOAT, T_DOUBLE, T_BYTE, T_SHORT, 
  T_INT, T_LONG, T_VOID 
};

//an even more elegant way is for those codes to be array indices but
//we share them with the types for the array elements, so we won't
//change them


jclass java_lang_VMClassLoader_getPrimitiveClassFromCode(JNIEnv* env,
							 tAType code)
{
  int i;
  for (i = 0; i < 9; i++) {
    if (primTypes[i].type == code) {
      return primTypes[i].classObject;
    }
  }
  return NULL;
}


jclass java_lang_VMClassLoader_getPrimitiveWrapperClassFromCode(JNIEnv* env, 
								tAType code)
{
  int i;
  for (i = 0; i < 9; i++) {
    if (primTypes[i].type == code) {
      return primTypes[i].wrapperClassObject;
    }
  }
  return NULL;
}


static int primitiveStringToIndex(const char* string)
{
  int i;
  for (i = 0; i < 9; i++) {
    if (strcmp(typeStrings[i], string) == 0) {
      return i;
    }
  }
  return -1;
}


static jclass primitiveCharToClass(const jchar type)
{
  int i;
  switch (type) {
  case 'Z': /* boolean */
    i = 0;
    break;
  case 'C': /* char */
    i = 1;
    break;
  case 'F': /* float */
    i = 2;
    break;
  case 'D': /* double */
    i = 3;
    break;
  case 'B': /* byte */
    i = 4;
    break;
  case 'S': /* short */
    i = 5;
    break;
  case 'I': /* int */
    i = 6;
    break;
  case 'J': /* long */
    i = 7;
    break;
  case 'V': /* void */
    i = 8;
    break;
  default:
    i = -1;
  }
  
  if (i >= 0) {
    return primTypes[i].classObject;
  }
  else {
    return NULL;
  }
}


/**
 * Create the java.lang.Class object for a primitive type and attach it
 * and the class loader ref to the prititive type's tuple.
 *
 * (This used to be a Kissme-specific JNI call, but I've changed this
 * to allow us to merge the Classpath and Kissme declarations of the
 * JNINativeInterface struct.  Besides, this file is the only place
 * that it is used now.  Steve C - 2002-11-21)
 **/
static jclass DefinePrimitiveClass (JNIEnv* env, jobject loader, 
				    tClassLoaderTuple* pstClass)
{
  tOBREF pstExOb;
  
  pstExOb = CLASSLOADER_TUPLE_MakeClassObject(env, pstClass);
  if (pstExOb != NULL) {
    (*env)->Throw(env, pstExOb);
    return NULL;
  }
  pstClass->classLoader = loader;
  (*env)->NewLocalRef(env, pstClass->classObject);
  return pstClass->classObject;
}


/*
 * Makes the actual tClass* structure for a primitive type, and also
 * creates a class object.  Stores the result in a special structure for
 * primitive types.  The prim->type field is already set when this
 * method is called.
 */
static int java_lang_VMClassLoader_makePrimitiveType(JNIEnv* env, char* name, 
						     tPrimitiveStructure* prim)
{
  tClassLoaderTuple* tuple = CLASSFILE_NewTuple(env, name, NULL);
  tClassLoaderTuple* temp = NULL;
  tClass* c = NULL;  
  jclass classObject = NULL; 
  
  assert(tuple);
  c = tuple->pstClass;

  // Fake up various fields that are usually initialised by loading
  // a class file.
  c->bInitialised = 1;
  c->u16AccessFlags = ACC_PRIMITIVE | ACC_PUBLIC | ACC_FINAL;
  c->u16ConstPoolCount = 4;  // XXX Shouldn't this be zero?
  c->u16MajorVersion = 45; 
  c->u16MinorVersion = 3; 
  c->pstSuperClass = CLASSFILE_FindOrLoad(env, "java/lang/Object",
					  pstObjectType);

  // Create the class object
  classObject = DefinePrimitiveClass(env, NULL, tuple); 

  // Record the tuple and the class object
  prim->classObject = classObject;
  prim->pstType = tuple;

  // Load the corresponding wrapper class.
  switch (prim->type) {
  case T_BOOLEAN:
    temp = CLASSFILE_FindOrLoad(env, "java/lang/Boolean", NULL);
    break;
  case T_CHAR:
    temp = CLASSFILE_FindOrLoad(env, "java/lang/Character", NULL);
    break;
  case T_FLOAT:
    temp = CLASSFILE_FindOrLoad(env, "java/lang/Float", NULL);
    break;
  case T_DOUBLE:
    temp = CLASSFILE_FindOrLoad(env, "java/lang/Double", NULL);
    break;
  case T_BYTE:
    temp = CLASSFILE_FindOrLoad(env, "java/lang/Byte", NULL);
    break;
  case T_SHORT:
    temp = CLASSFILE_FindOrLoad(env, "java/lang/Short", NULL);
    break;
  case T_INT:
    temp = CLASSFILE_FindOrLoad(env, "java/lang/Integer", NULL);
    break;
  case T_LONG:
    temp = CLASSFILE_FindOrLoad(env, "java/lang/Long", NULL);
    break;
  case T_VOID:
    temp = CLASSFILE_FindOrLoad(env, "java/lang/Void", NULL);
    break;
  default:
    panic("VMClassLoader.makePrimitiveType: bad type %d", prim->type);
    break;
  }
  if (temp) {
    prim->wrapperClassObject = temp->classObject;
    return 0;
  }
  else {
    return -1;
  }
}


/**
 * Return the location containing Class object for the i'th primitive type.
 * Used by the garbage collector.
 */
tOBREF* java_lang_VMClassLoader_getPrimitiveClassLoc(int index)
{
  return &(primTypes[index].classObject);
}

/**
 * Return the location containing wrapper Class object for the i'th
 * primitive type.  Used by the garbage collector.  
 */
tOBREF* java_lang_VMClassLoader_getPrimitiveWrapperClassLoc(int index)
{
  return &(primTypes[index].wrapperClassObject);
}


int java_lang_VMClassLoader_makePrimitiveTypes(JNIEnv* env)
{
  int i;
  for (i = 0; i < 9; i++) {
    primTypes[i].type = typeCodes[i];
    if (java_lang_VMClassLoader_makePrimitiveType(env, typeStrings[i], 
						  &primTypes[i]) != 0) {
      return -1;
    }
    assert(primTypes[i].wrapperClassObject);
  }
  return 0;
}



/* Looks up the primitive class definitions in the static array */
jclass java_lang_VMClassLoader_getPrimitiveClassFromAsciz(JNIEnv* env, 
							  const char* name)
{ 
  int failure;
  int index = primitiveStringToIndex(name);
  
  //just check that the primitive types have been created
  assert(primTypes[0].type == T_BOOLEAN);
  
  if (index != -1) {
    return primTypes[index].classObject;
  }
  (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/ClassNotFoundException", &failure)); 
  return NULL;
}


//java/lang/VMClassLoader.getPrimitiveClass(Ljava/lang/String;)Ljava/lang/Class;              

jclass java_lang_VMClassLoader_getPrimitiveClass(JNIEnv* env, jclass clazz, 
						 jchar type)
{ 
  //Note clazz is allowed to be NULL! 
  jclass ret;
  int failure;
  
  //just check that the primitive types have been created
  assert(primTypes[0].type == T_BOOLEAN);
  
  ret = primitiveCharToClass(type);
  if (ret == NULL) {
    (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/NoClassDefFoundError", &failure));
  }
  return ret; 
} 


/** Helper to resolve all references to other classes from this class.
 ** @param c the class to resolve.
 **/

/*	final static native void resolveClass(Class c); */


void java_lang_VMClassLoader_resolveClass(JNIEnv* env, jclass clazz, 
					  jclass cls)
{ 
  // This is a no-op.  References are resolved during the loadClass call.
  return;
}


jclass java_lang_VMClassLoader_loadClass(JNIEnv* env, jclass clazz, 
					 jobject name, jboolean resolve)
{
  tClassLoaderTuple* tuple;
  char *pszNameString;
  jobject pstExOb;
  int failure;
  int i;

  if (name == NULL) {
    (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/NullPointerException", &failure));
    return NULL;    
  }

  pszNameString = INTERP_AscizFromString(env, name);
  if (pszNameString == NULL) {
    (*env)->Throw(env, OOMExceptionObject);
    return NULL;
  }

  for (i = 0; pszNameString[i]; i++) {
    if (pszNameString[i] == '.') {
      pszNameString[i] = '/';
    }
  }

  tuple = CLASSFILE_FindOrLoad(env, pszNameString, NULL);
  sys_free(pszNameString);

  if (tuple == NULL) {
    return NULL;
  }

  pstExOb = CLASSLOADER_TUPLE_MakeClassObject(env, tuple);
  if (pstExOb) {
    (*env)->Throw(env, pstExOb);
    return NULL;
  }
  else {
    assert(tuple->classObject);
    return tuple->classObject;
  }
}

jobject java_lang_VMClassLoader_nativeGetResource(JNIEnv* env, jclass clazz, 
						  jobject name)
{
  tClassLoaderTuple* tuple;
  char *pszNameString;
  jobject pstExOb;
  int failure;
  int i;

  if (name == NULL) {
    (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/NullPointerException", &failure));
    return NULL;    
  }

  pszNameString = INTERP_AscizFromString(env, name);
  if (pszNameString == NULL) {
    (*env)->Throw(env, OOMExceptionObject);
    return NULL;
  }

}


jclass java_lang_VMClassLoader_defineClass(JNIEnv* env, jclass clazz, 
					   jobject classloader,
					   jobject namestring,
					   jarray data, jint offset,
					   jint len) 
{  
  tClass* c;
  tClassLoaderTuple* newtuple;
  jbyte* realData;
  jboolean aboolean;
  uint32 errorCode;
  tOBREF pstExOb;
  char* pszNameString;
  int i;
  int failure;

  if (data == NULL) {   
    (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/ClassFormatError", &failure));
    return NULL;
  }
  realData = (*env)->GetByteArrayElements(env, data, &aboolean);

  if (ADEREF(data)->i32Number <= offset) {
    (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/ClassFormatError", &failure));
    return NULL;
  }

  if (namestring == NULL) {
    (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/NullPointerException", &failure));
    return NULL;    
  }

  pszNameString = INTERP_AscizFromString(env, namestring);
  if (pszNameString == NULL) {
    (*env)->Throw(env, OOMExceptionObject);
    return NULL;
  }
  for (i = 0; pszNameString[i]; i++) {
    if (pszNameString[i] == '.') {
      pszNameString[i] = '/';
    }
  }

  newtuple = CLASSFILE_ClassCreate(env, realData + offset, pszNameString, 
				   classloader, &errorCode);
  sys_free(pszNameString);
  if (newtuple == NULL) {
    return NULL;
  }
  
  c = newtuple->pstClass;

  //Now do the housework stuff
  CLASSFILE_CompleteClass(env, newtuple);

  //XXX - should we do this now ???
  if (c->bInitialised == 0) {
    pstExOb = CLASSFILE_InitClass(env, newtuple);
    if (pstExOb) {
      //Construct a java.lang.ExceptionInInitializerError
      jobject newExOb;
      jclass exClass =
	(*env)->FindClass(env, "java/lang/ExceptionInInitializerError");
      
      jmethodID methodID = (*env)->GetMethodID(env, exClass, "<init>", 
					       "(Ljava/lang/Throwable;)V");
      if (methodID == NULL) {
	(*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/InternalError", "Could not find constructor method for java.lang.ExceptionInInitializerError"));
	return NULL;
      }
      
      newExOb = (*env)->NewObject(env, exClass, methodID, pstExOb);
      if (newExOb == NULL) { 	
	(*env)->Throw(env, OOMExceptionObject); 
	return NULL;
      }
      (*env)->Throw(env, newExOb);
      return NULL;
    }
  }
  
  assert(newtuple->classObject);
  return newtuple->classObject;
}

 
 
 
 
 
 
 
 
