/*
 * Copyright (C) 2001, John Leuner.
 *
 * This file is part of the kissme/teaseme project, which in turn is part of the JOS project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"
#include <string.h>

#include "vm/jni.h"
#include "vm/interp.h"
#include "vm/interp_methods.h"
#include "vm/classfile_methods.h"
#include "vm/classloader_tuple.h"
#include "vm/newobject.h"

#include "vm/garbage.h"
#include "lib/indigenous/java.lang/Class.h"
#include "lib/indigenous/java.lang/Class_Reflection.h"

#include "lib/indigenous/java.lang/VMSystem.h"


jint Java_java_lang_VMSystem_identityHashCode(JNIEnv* env,
					      jclass class,
					      jobject obj)
{
  return (jint) obj; /* just use pointer as hash code */
}


void java_lang_VMSystem_kprint(JNIEnv* env, jclass class, jstring str)
{
  char* pszString = INTERP_AscizFromString(env,str); 
  eprintf("%s\n", pszString);
  sys_free(pszString);
}


char** command_line_args = NULL;
int command_line_num_args = 0;

void java_lang_VMSystem_setArgsPointer(char** argv, int argc)
{
  command_line_args = argv;
  command_line_num_args = argc;
}

#include <unistd.h>


/*
 * @doc NATFUNC
 * @func
 * Implementation of: public static native void arraycopy(Object src,
 *                    int src_position, Object dst, int dst_position,
 *                    int length);
 *
 * @rdesc Returns one of:
 *
 * @flag 0 | Normal return
 * @flag -1 | Return with exception thrown
 *
 */

void Java_java_lang_VMSystem_arraycopy(JNIEnv* env, jclass thisclass,
				       jarray src, jint src_pos,
				       jarray dst, jint dst_pos,
				       jint length)
{
  tArray* pstA;
  tArray* pstB;
  int iSize;
  
  if ((src == NULL) || (dst == NULL)) {
    int failure;
    (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/NullPointerException", &failure));
    return;
  }

  pstA = ADEREF((tARREF) src);
  pstB = ADEREF((tARREF) dst);

  // Check if they are really arrays

  if (((pstA->i32Flags & GARBAGE_TYPEMASK) != GARBAGE_ARRAY) ||
      ((pstB->i32Flags & GARBAGE_TYPEMASK) != GARBAGE_ARRAY)) {
    int failure;
    (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/ArrayStoreException", &failure));
    return;
  }
  
  // Check their types are the same
  if (pstA->enType != pstB->enType) {
    if (!((pstA->enType == T_OBJECT) && (pstB->enType == T_ARRAY))) {
      int failure;
      (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/ArrayStoreException", &failure));
      return;
    }
  }

  if ((dst_pos < 0) || (src_pos < 0) || (dst_pos > pstB->i32Number) || 
      (src_pos > pstA->i32Number) || (length < 0)) {
    int failure;
    (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/ArrayIndexOutOfBoundsException", &failure));
    return;
  }
  
  if (((dst_pos + length) > pstB->i32Number) || ((dst_pos + length) < 0) ||
      ((src_pos + length) > pstA->i32Number) || ((src_pos + length) < 0)) {
    int failure;
    (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/ArrayIndexOutOfBoundsException", &failure));
    return;
  }

  switch (pstA->enType) {
  case T_BYTE:
  case T_BOOLEAN:
    iSize = 1;
    break;
  case T_CHAR:
  case T_SHORT:
    iSize = 2;
    break;
  case T_ARRAY:
  case T_OBJECT:
  case T_FLOAT:
  case T_INT:
    iSize = 4;
    break;
  case T_DOUBLE:
  case T_LONG:
    iSize = 8;
    break;
  default:
    panic0("Invalid array passed to arraycopy");
  }

  if (pstA != pstB) {
    if ((pstA->enType == T_ARRAY) || (pstA->enType == T_OBJECT)) {
      int i;
      tClassLoaderTuple* targetType;
      jclass classClass;
      jclass destArrayClass;
      jclass destArrayComponentClass;
      jmethodID componentMethod;
      
      // If these are references we should actually be checking the 
      // 'castability' of each item we copy 
      
      destArrayClass = pstB->pstType->classObject;
      classClass = (*env)->GetObjectClass(env, destArrayClass);
      
      componentMethod = (*env)->GetMethodID(env, classClass, 
					    "getComponentType", 
					    "()Ljava/lang/Class;");
      assert(componentMethod);
      
      destArrayComponentClass = (*env)->CallObjectMethod(env, destArrayClass,
							 componentMethod);
      if ((*env)->ExceptionOccurred(env) || 
	  (destArrayComponentClass == NULL)) {
	int failure;
	(*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/ArrayStoreException", &failure));
	return;		      
      }
      targetType = CLASS_GetClassStruct(env, destArrayComponentClass);
      
      assert(targetType);
      
      for (i = 0; i < length; i++) {
	tOBREF item = ((tOBREF*) pstA->pvElements)[src_pos + i];
	if (item != NULL) {
	  if (INTERP_CheckCast(env, targetType, DEREF(item)->pstType) == 1) {
	    ((tOBREF*)pstB->pvElements)[dst_pos + i] = item;
	  }
	  else {
	    int failure;
	    (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/ArrayStoreException", &failure));
	    return;		      
	  }
	}
	else {
	  ((tOBREF*)pstB->pvElements)[dst_pos + i] = NULL;
	}
      }
    }
    else {
      memcpy((byte*) pstB->pvElements + iSize * dst_pos, 
	     (byte*) pstA->pvElements + iSize * src_pos, length * iSize);
    }
  }
  else {
#ifndef KISSME_LINUX_USER
    panic0("unimplemented array copy case");
#else
    memmove((byte*) pstB->pvElements + iSize * dst_pos, 
	    (byte*) pstA->pvElements + iSize * src_pos, length * iSize);
#endif
  }
}


jboolean Java_java_lang_VMSystem_isWordsBigEndian(JNIEnv* env, jclass class)
{
#ifdef __powerpc__
  return JNI_TRUE;
#else
  return JNI_FALSE;
#endif
}


extern jint Java_java_lang_VMSystem_identityHashCode (JNIEnv *env, jclass,
						      jobject);
extern void Java_java_lang_VMSystem_setIn (JNIEnv *env, jclass, jobject);
extern void Java_java_lang_VMSystem_setOut (JNIEnv *env, jclass, jobject);
extern void Java_java_lang_VMSystem_setErr (JNIEnv *env, jclass, jobject);
extern jlong Java_java_lang_VMSystem_currentTimeMillis (JNIEnv *env, jclass);

