/*
 * Copyright (C) 2001, John Leuner.
 *
 * This file is part of the kissme project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"

#include "vm/classfil.h"
#include "vm/loading_errors.h"
#include "vm/cptuplelist.h"
#include "vm/interp.h"

#include "vm/jni.h"

#include "vm/classfile_methods.h"
#include "vm/interp_methods.h"
#include "vm/newobject.h"
#include "lib/indigenous/java.lang/Class.h"
#include "vm/sys_linux_host/kissme_classload.h"


#ifdef KISSME_LINUX_USER
#include <stdio.h>
#include <string.h>
#endif

// this can be globally used to get the type for class
extern tClassLoaderTuple* pstClassType; 
extern tClassLoaderTuple* pstVMClassType; 

tMutex* classTupleMutex;

static tList* fixupList;

static char* pszObjectString = "java/lang/Object";


tClassLoaderTuple* 
CLASSFILE_FindOrLoadWithError(JNIEnv* env, char* pszClassName, 
			      tClassLoaderTuple* loadingClass, 
			      uint32* errorCode)
{
  tClassLoaderTuple* pstClass;
  char* ptr;
  
#ifdef LOADING_VERBOSE
  traceLoading("CLASSFILE_FindOrLoadWithError(%s)", pszClassName);
#endif
  sys_mutex_lock(classTupleMutex);

  if (pszClassName[0] == '[') {
    /* if we are looking for an array class, we must use the same
       class loader as the ultimate component class. */
    int i;
    
    for (i = 1; pszClassName[i] == '['; i++) { /* */ }

    if (pszClassName[i++] == 'L') {
      /* the ultimate component type is a class/interface */
      int len = strlen(pszClassName + i);
      char* pszComponentClassName = sys_malloc(len + 1);
      strcpy(pszComponentClassName, pszClassName + i);
      if (pszComponentClassName[len - 1] == ';') {
	pszComponentClassName[len - 1] = '\000';
      }
      loadingClass = CLASSFILE_FindOrLoadWithError(env, pszComponentClassName,
						   loadingClass, errorCode);
      sys_free(pszComponentClassName);
    }
    else {
      /* the ultimate component type is a primitive type */
      loadingClass = NULL;
    }
  }
  else {
    /* convert all dots to slashes first */
    ptr = strchr(pszClassName, '.');
    while (ptr){
      ptr[0] = '/';
      ptr = strchr(ptr, '.');
    }
  }

  /* just return the pointer if it is already loaded */
  pstClass = CPTUPLELIST_Find(pszClassName, 
			      ((loadingClass == NULL) ? NULL : 
			       loadingClass->classLoader));
  if (pstClass != NULL) {
    sys_mutex_unlock(classTupleMutex);
    return pstClass;
  }
  
  sys_mutex_unlock(classTupleMutex);
  return CLASSFILE_LoadAndLink(env, pszClassName, errorCode, loadingClass); 
}


tClassLoaderTuple* CLASSFILE_FindOrLoad(JNIEnv* env, char* pszClassName, 
					tClassLoaderTuple* loadingClass)
{
  uint32 lostError = 0;
  tClassLoaderTuple* ret;
  ret = CLASSFILE_FindOrLoadWithError(env, pszClassName, loadingClass, 
				      &lostError);
  if (ret == NULL) {
    traceLoading("Class load failed for class %s from %p (%s) loader %p (%s)",
		 pszClassName, loadingClass, 
		 loadingClass ? loadingClass->uidName : "none",
		 loadingClass ? loadingClass->classLoader : NULL,
		 ((loadingClass && loadingClass->classLoader) ?
		  DEREF(loadingClass->classLoader)->pstType->uidName : "none"));
    if (lostError != 0) {
      traceLoading("Class loading error: %s", loadingErrorMessage(lostError));
    }
  }
  else if (lostError != 0) {
    traceLoading("lost classloader error %i, (loading class %s from %s): %s", 
		 (int) lostError, pszClassName, 
		 (loadingClass == NULL ? "null" : loadingClass->uidName),
		 loadingErrorMessage(lostError));
  }

  return ret;
}


tClassLoaderTuple* getClassFromByteArray(JNIEnv* env, byte* pbData, 
					 char* pszFileName, uint32* errorCode,
					 tMutex* classfilMutex)
{
  tClassLoaderTuple* pstClass;
  
  sys_mutex_lock(classfilMutex);
  
  // Note: using boot class loader
  pstClass = CLASSFILE_ClassCreate(env, pbData, pszFileName, NULL, errorCode); 
  if (pstClass == NULL) {
    // XXX - should throw an exception ...
    eprintf("Error occurred creating class from array %s\n", pszFileName);
  }
  sys_mutex_unlock(classfilMutex);
  return pstClass;
}


void CLASSFILE_CompleteClass(JNIEnv* env, tClassLoaderTuple* pstClass) 
{
  tClassLoaderTuple* pstTempClass;
  int failure;

  traceLoading("CLASSFILE_CompleteClass(%s)", pstClass->uidName);
  if (pstClass->pstClass == 0) {
    (*env)->Throw(env, INTERP_NewObjectFromName(env, 
						"java/lang/ClassFormatError",
						&failure));
    return;
  }

  pstTempClass = CPTUPLELIST_Find(pstClass->uidName, pstClass->classLoader);
  if (pstTempClass == NULL) {
     panic("Class %s %p (%p) missing from CPLIST",
	    pstClass->uidName, pstClass, pstClass->classLoader);
  }

  // Check that we haven't accidentally loaded the class a second time.
  // (This might trap VM bootstrap problems a bit earlier.)
  if (pstTempClass == pstClass) {
    traceLoading("Class tuple %s %p (%p) already in the CPLIST",
		 pstClass->uidName, pstClass, pstClass->classLoader);
  }
  else {
    panic("Class %s %p (%p) loaded twice",
	  pstClass->uidName, pstClass, pstClass->classLoader);
  }

  if (pstClass->classObject == NULL) {
    if (pstClassType) {
      if (CLASSLOADER_TUPLE_MakeClassObject(env, pstClass) != NULL) {
	panic("CLASSLOADER_TUPLE_MakeClassObject failed");
      }
    }
    else {
      LLIST_AddAtHead(fixupList, pstClass);
    }
  }

  if (pstClass->pi32StatVars == NULL) {
    if (CLASSLOADER_TUPLE_InitializeConstants(pstClass) != 0) {
	//XXX fix me, throw some exception here
      panic("CLASSLOADER_TUPLE_InitializeConstants failed");
    }
  }
  if (CLASSLOADER_TUPLE_ResolveInnerClasses(env, pstClass) != 0) {
    panic("CLASSLOADER_TUPLE_ResolveInnerClasses failed");
  }
}


tClassLoaderTuple* CLASSFILE_LoadAndLink(JNIEnv* env, char* pszClassName,
					 uint32* errorCode, 
					 tClassLoaderTuple* loadingClass)
{
  tClassLoaderTuple* pstClass;
  byte* pbData;
  tClassLoaderTuple* pstTempClass;
  char* pszClassFileName;
  tOBREF pstExOb;
  
  /* handle the array case separately */
  if (pszClassName[0] == '[') {
    return CLASSFILE_LoadAndLinkArray(env, pszClassName,
				      errorCode, loadingClass);
  }

  traceLoading("CLASSFILE_LoadAndLink(%s, %s)", 
	       pszClassName, loadingClass ? loadingClass->uidName : "NULL");

  pszClassFileName = pszClassName; 
  
  if (loadingClass != NULL) {
    if (loadingClass->classLoader != NULL) {
      /* Try use the class loader object to load this class */
      jclass clazz;
      jmethodID methodID;
      jclass retclazz;
      char* pszCopyString;
      int i;
      
      pszCopyString = sys_malloc(strlen(pszClassFileName) + 1);
      if (pszCopyString == NULL) {
	return NULL;
      }
      strcpy(pszCopyString, pszClassFileName);
      for (i = 0; pszCopyString[i] != 0; i++) {
	if (pszCopyString[i] == '/') {
	  pszCopyString[i] = '.';
	}
      }
      traceLoading("CLASSFILE_LoadAndLink calling %s.loadClass(%s)",
		   DEREF(loadingClass->classLoader)->pstType->uidName,
		   pszCopyString);
      clazz = (*env)->GetObjectClass(env, loadingClass->classLoader);
      assert(clazz);
      methodID = (*env)->GetMethodID(env, clazz, "loadClass", 
				     "(Ljava/lang/String;Z)Ljava/lang/Class;");
      assert(methodID);
      
      retclazz = 
	(*env)->CallObjectMethod(env, loadingClass->classLoader,
				 methodID, 
				 INTERP_NewStringFromAsciz(env, pszCopyString),
				 JNI_FALSE);
      traceLoading("CLASSFILE_LoadAndLink %s.loadClass(%s) completed",
		   DEREF(loadingClass->classLoader)->pstType->uidName,
		   pszCopyString);
      sys_free(pszCopyString);
      if ((*env)->ExceptionOccurred(env) == NULL && retclazz != NULL) {
	return CLASS_GetClassStruct(env, retclazz);
      }
      else {
	return NULL;
      }
    }
  }

  pbData = getClassData(env, pszClassFileName, errorCode);
  if (pbData == NULL) {
    return NULL;
  }

  pstClass = CLASSFILE_ClassCreate(env, pbData, pszClassName,
				   NULL, errorCode);

  if (pstClass == NULL) {
    return NULL;
  }

  CLASSFILE_CompleteClass(env, pstClass);
  return CPTUPLELIST_Find(pstClass->uidName, pstClass->classLoader);  
}



tClassLoaderTuple* CLASSFILE_LoadAndLinkArray(JNIEnv* env, char* pszClassName,
					      uint32* errorCode, 
					      tClassLoaderTuple* loadingClass)
{
  tClassLoaderTuple* pstClass;
  byte* pbData;
  tClassLoaderTuple* pstTempClass;
  char* pszClassFileName;

  tClassLoaderTuple* baseComponentType = NULL; // base type for object arrays
  int isPrimArray;  // says whether this is an array of primitives or not

  traceLoading("CLASSFILE_LoadAndLinkArray(%s)", pszClassName);

  assert(pszClassName[0] == '[');

  {
    tClassLoaderTuple* findArray;
    /* just use Object class for arrays */
    int biggest = strlen("java/lang/Object");
    int actual = ((strlen(pszClassName) > biggest) ? 
		  strlen(pszClassName) : biggest);
    int strsize =  actual + 10;
    
    findArray = CPTUPLELIST_Find(pszClassName, 
				 ((loadingClass == NULL) ? NULL : 
				  loadingClass->classLoader));
    if (findArray != NULL) {
      return findArray;
    }
    
    pszClassFileName = (char*) sys_malloc(strsize);
    
    /* check we can load the component type; i.e. it's an array of objects */
    strcpy(pszClassFileName, pszClassName);
    if (strstr(pszClassFileName, "L") != NULL) {
      isPrimArray = 0;
      assert(pszClassFileName[strlen(pszClassFileName) - 1] == ';');
      pszClassFileName[strlen(pszClassFileName) - 1] = '\0';
      baseComponentType = 
	CLASSFILE_FindOrLoad(env, strstr(pszClassFileName, "L") + 1, 
			     loadingClass);
      if (baseComponentType == NULL) {
	// XXX - we can't find the component type: should return an error code
	return NULL;
      }
      else {
	findArray = CPTUPLELIST_Find(pszClassName, 
				     baseComponentType->classLoader);
	if (findArray != NULL) {
	  return findArray;
	}
      }
    }
    else {
      isPrimArray = 1;
    }
    assert(pszClassFileName);
    strcpy(pszClassFileName, pszObjectString);
  }

  pstClass = 
    CLASSFILE_ArrayClassCreate(env, pszClassName, 
			       isPrimArray ? NULL : baseComponentType);
  
  CLASSFILE_CompleteClass(env, pstClass);
  return CPTUPLELIST_Find(pstClass->uidName, pstClass->classLoader);  
}


/**
 * Make the class object for a class and attach it to the class tuple.
 * The result is NULL on success, and an exception object otherwise.
 */
jobject CLASSLOADER_TUPLE_MakeClassObject(JNIEnv* env, 
					  tClassLoaderTuple* pstClass)
{
  int failure;
  jobject pstClassObject, pstVMClassObject;

  if (pstClass->classObject != NULL) {
    // Already created
    return NULL;
  }

  // Find the tuples for java.lang.Class & java.lang.VMClass
  if (pstClassType == NULL) {
    pstClassType = CLASSFILE_FindOrLoad(env, "java/lang/Class", NULL);
    pstVMClassType = CLASSFILE_FindOrLoad(env, "java/lang/VMClass", NULL);
  }

  // XXX must use system class?
  pstVMClassObject = INTERP_NewObject(env, pstVMClassType, &failure);

  if (failure) {
    // This should be an exception ...
    return pstVMClassObject;
  }

  pstClassObject = INTERP_NewObject(env, pstClassType, &failure);
  
  if (failure) {
    // This should be an exception ...
    return pstClassObject;
  }

  /* set _classStruct field in new VMClass object */
  CLASS_SetVMClassStruct(pstVMClassObject, pstClass);

  /* set vmClass field in new Class object */
  CLASS_SetVMClass(pstClassObject, pstVMClassObject);

  /* set class struct to point to Class & VMClass objects */
  pstClass->classObject = pstClassObject;
  pstClass->vmClassObject = pstVMClassObject;
  return NULL;
}


/*
 * Get the class object associated with a class tuple, creating it
 * if necessary.
 */
jclass CLASSLOADER_TUPLE_GetClassObject(JNIEnv* env, 
					tClassLoaderTuple* pstClass)
{
  if (pstClass->classObject == NULL) {
    CLASSLOADER_TUPLE_MakeClassObject(env, pstClass);
  }
  return pstClass->classObject;
}


void CLASSLOADER_TUPLE_Init()
{
  fixupList = LLIST_Create_List();
}


jobject CLASSLOADER_TUPLE_FixupClassObjects(JNIEnv* env)
{
  tListIterator* iter = LLIST_FindFirst(fixupList);
  while (iter != NULL) {
    tClassLoaderTuple* pstClass = (tClassLoaderTuple*) iter->entry->payload;
    jobject pstExOb = CLASSLOADER_TUPLE_MakeClassObject(env, pstClass);
					
    if (pstExOb) {
      return pstExOb;
    }
    iter = LLIST_FindNext(fixupList, iter);
  }
  return 0;
}


static tClassLoaderTuple* doResolve(JNIEnv* env, tClassLoaderTuple* tuple,
				    uint16 u16InfoIndex)
{
  tClass *pstClass = tuple->pstClass;
  tClassLoaderTuple* res;
  char *className;

  // According to the JVM spec, inner and outer classes are
  // denoted by a ClassInfo entry in the constant pool.  However
  // some compilers (e.g. some versions of Jikes) represent the
  // classes (names) as a Utf8 string.
  
  if (CONSTTAG(pstClass, u16InfoIndex) == CONSTANT_Utf8) {
    className = CONSTGET_UidAsciz(pstClass, u16InfoIndex);
    res = CLASSFILE_FindOrLoad(env, className, tuple);
  }
  else {
    assert(CONSTTAG(pstClass, u16InfoIndex) == CONSTANT_Class);
    res = CONSTGET_Class(env, tuple, u16InfoIndex);
  }
  // XXX - should throw an exception if FindOrLoad/ResolveClass fails.
  assert(res);
  return res;
}


/**
 * Resolve the classes mentioned in a class's "inner classes" array.
 */
int CLASSLOADER_TUPLE_ResolveInnerClasses(JNIEnv* env, 
					  tClassLoaderTuple* tuple)
{
  int i;
  tClass *pstClass = tuple->pstClass;
  
  for (i = 0; i < pstClass->u16InnerClassesCount; i++) {
    uint16 u16InfoIndex;
    
    // Resolve the inner class reference
    u16InfoIndex = pstClass->pstInnerClasses[i].u16InnerClassInfo;
    pstClass->pstInnerClasses[i].pstInnerClass = doResolve(env, tuple, 
							   u16InfoIndex);
    if (pstClass->pstInnerClasses[i].pstInnerClass == tuple) {
      pstClass->bIsInnerClass = 1;
    }
    
    // Resolve the outer class reference
    u16InfoIndex = pstClass->pstInnerClasses[i].u16OuterClassInfo;
    if (u16InfoIndex) {
      pstClass->pstInnerClasses[i].pstOuterClass = doResolve(env, tuple, 
							     u16InfoIndex);
    }
  }
  return 0;
}


int CLASSLOADER_TUPLE_InitializeConstants(tClassLoaderTuple* tuple)
{
  int numStatics;
  int i;
  int var_index;
  int field_index;
  
  numStatics = tuple->pstClass->u16StatCount;
  
  if (tuple->pstClass->u16StatSize == 0) {
    tuple->pi32StatVars = NULL;
  }
  else {
    tuple->pi32StatVars = 
      sys_malloc(sizeof(int32) * tuple->pstClass->u16StatSize);
    if (tuple->pi32StatVars == NULL) {
      return -2;
    }
    
    memset(tuple->pi32StatVars, 0, 
	   sizeof(int32) * tuple->pstClass->u16StatSize);
    
    /* We must take into account static variables in the parent class(es)*/
    for (i = 0, var_index = 0; i < numStatics; i++) {
      byte tagtype;
      
      // position in constant pool 
      field_index = tuple->pstClass->pstStatOffsets[i].u16ConstValueIndex; 
      if (field_index != 0) {
	assert(var_index >= 0 && var_index < tuple->pstClass->u16StatSize);

	switch (tagtype = CONSTTAG(tuple->pstClass, field_index)) {
        case CONSTANT_Integer:
	  tuple->pi32StatVars[var_index] = 
	    CONSTGET_Int(tuple->pstClass, field_index);
          break;
	  
        case CONSTANT_Float:
          *((float *) &(tuple->pi32StatVars[var_index])) = 
	    CONSTGET_Float(tuple->pstClass, field_index);
          break;
	  
        case CONSTANT_Long:
          tuple->pi32StatVars[var_index] = 
	    CONSTGET_Int(tuple->pstClass, field_index);
          tuple->pi32StatVars[var_index + 1] = 
	    CONSTGET_Int(tuple->pstClass, field_index + 1);
	  //XXX check for endianness issues here
	  var_index++;
          break;
        
        case CONSTANT_Double:
          tuple->pi32StatVars[var_index] = 
	    CONSTGET_Double_High(tuple->pstClass, field_index);
          tuple->pi32StatVars[var_index + 1] = 
	    CONSTGET_Double_Low(tuple->pstClass, field_index);
	  var_index++;
          break;
	  
	case CONSTANT_Fieldref:
	  panic("Unexpected CONSTANT_Fieldref found in class %s", 
		tuple->uidName);
	  break;
	  
	case CONSTANT_String:
	  tuple->pi32StatVars[var_index] = 
	    CONSTGET_String_Index(tuple->pstClass, field_index);
	  break;
	
	default:
	  panic("Unknown constant value %i encountered in class %s\n", 
		(int) tagtype, tuple->uidName);
	}
      }
      var_index++;
    }
  }
  return 0;
}













