/*
 * Copyright (C) 2002, Stephen Crawley
 *
 * This file is part of the kissme project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef __DIAG_H__
#define __DIAG_H__

#include <stdio.h>

/**
 * The DIAG module contains routines and methods for VM assertion
 * checking, reporting fatal errors and console printing.  
 * 
 * It also provides toggleable trace printing based on the notion of
 * trace 'channels' identified by small integers.
 **/


/*
 * Basic 'console' output
 */

#ifdef KISSME_LINUX_USER
/* eprinf(format, args...) */
#define eprintf(...) fprintf(stderr, __VA_ARGS__)
#define eprint(msg) fputs(msg, stderr)
#else
#define eprint(msg) eprintf(msg)
#endif


/**
 * MACRO 'functions' for assertion checking and error messages.
 */
#ifdef ASSERTIONS
#define assert(condition) \
  if (!(condition)) { \
    DIAG_AssertFail( __FILE__, __LINE__, #condition ); \
  }
#else
#define assert(condition) /* ignored */
#endif

#define panic0(msg) \
  DIAG_Panic0(msg)
    
/* panic(format, args...) */
#define panic(...) \
  DIAG_Panic(__VA_ARGS__)
    
#define fatalError0(rc, msg) \
  DIAG_FatalError0(rc, msg)

/* fatalError(rc, format, args...) */
#define fatalError(...) \
  DIAG_FatalError(__VA_ARGS__)

#define nonfatalError0(msg) \
  DIAG_Error0(msg)

/* nonfatalError(format, args...) */
#define nonfatalError(...) \
  DIAG_Error(__VA_ARGS__)

/**
 * MACRO 'functions' for trace printing
 */


#ifdef TRACING
#define trace(...) \
  DIAG_Log(DEFAULT_DIAG_CHANNEL, __VA_ARGS__)
#define trace0(msg) \
  DIAG_Log0(DEFAULT_DIAG_CHANNEL, msg)
#else
#define trace(...) /* ignore */
#define trace0(msg) /* ignore */
#endif

#ifdef TRACING
#define traceLoading(...) \
  DIAG_Log(LOADER_DIAG_CHANNEL, __VA_ARGS__)
#define traceLoading0(msg) \
  DIAG_Log0(LOADER_DIAG_CHANNEL, msg)
#else
#define traceLoading(...) /* ignore */
#define traceLoading0(msg) /* ignore */
#endif

#ifdef TRACING
#define traceInit(...) \
  DIAG_Log(INIT_DIAG_CHANNEL, __VA_ARGS__)
#define traceInit0(msg) \
  DIAG_Log0(INIT_DIAG_CHANNEL, msg)
#else
#define traceInit(...) /* ignore */
#define traceInit0(msg) /* ignore */
#endif

#ifdef TRACING
#define traceCall(...) \
  DIAG_Log(CALL_DIAG_CHANNEL, __VA_ARGS__)
#define traceCall0(msg) \
  DIAG_Log0(CALL_DIAG_CHANNEL, msg)
#else
#define traceCall(...) /* ignore */
#define traceCall0(msg) /* ignore */
#endif

#ifdef TRACING
#define traceJNI(...) \
  DIAG_Log(JNI_DIAG_CHANNEL, __VA_ARGS__)
#define traceJNI0(msg) \
  DIAG_Log0(JNI_DIAG_CHANNEL, msg)
#else
#define traceJNI(...) /* ignore */
#define traceJNI0(msg) /* ignore */
#endif

#ifdef TRACING
#define traceOpcodes(...) \
  DIAG_Log(OPCODE_DIAG_CHANNEL, __VA_ARGS__)
#define traceOpcodes0(msg) \
  DIAG_Log0(OPCODE_DIAG_CHANNEL, msg)
#else
#define traceOpcodes(...) /* ignore */
#define traceOpcodes0(msg) /* ignore */
#endif

#ifdef TRACING
#define traceOpstacks(...) \
  DIAG_Log(OPSTACK_DIAG_CHANNEL, __VA_ARGS__)
#define traceOpstacks0(msg) \
  DIAG_Log0(OPSTACK_DIAG_CHANNEL, msg)
#else
#define traceOpstacks(...) /* ignore */
#define traceOpstacks0(msg) /* ignore */
#endif

#ifdef TRACING
#define traceFrames(...) \
  DIAG_Log(FRAME_DIAG_CHANNEL, __VA_ARGS__)
#define traceFrames0(msg) \
  DIAG_Log0(FRAME_DIAG_CHANNEL, msg)
#else
#define traceFrames(...) /* ignore */
#define traceFrames0(msg) /* ignore */
#endif

#ifdef TRACING
#define traceExceptions(...) \
  DIAG_Log(EXCEPTION_DIAG_CHANNEL, __VA_ARGS__)
#define traceExceptions0(msg) \
  DIAG_Log0(EXCEPTION_DIAG_CHANNEL, msg)
#else
#define traceExceptions(...) /* ignore */
#define traceExceptions0(msg) /* ignore */
#endif

#ifdef TRACING
#define traceAlloc(...) \
  DIAG_Log(ALLOC_DIAG_CHANNEL, __VA_ARGS__)
#define traceAlloc0(msg) \
  DIAG_Log0(ALLOC_DIAG_CHANNEL, msg)
#else
#define traceAlloc(...) /* ignore */
#define traceAlloc0(msg) /* ignore */
#endif

#ifdef TRACING
#define traceGC(...) \
  DIAG_Log(GC_DIAG_CHANNEL, __VA_ARGS__)
#define traceGC0(msg) \
  DIAG_Log0(GC_DIAG_CHANNEL, msg)
#else
#define traceGC(...) /* ignore */
#define traceGC0(msg) /* ignore */
#endif

#ifdef TRACING
#define traceGCThread(...) \
  DIAG_Log(GCTHREAD_DIAG_CHANNEL, __VA_ARGS__)
#define traceGCThread0(msg) \
  DIAG_Log0(GCTHREAD_DIAG_CHANNEL, msg)
#else
#define traceGCThread(...) /* ignore */
#define traceGCThread0(msg) /* ignore */
#endif

#ifdef TRACING
#define traceThreads(...) \
  DIAG_Log(THREAD_DIAG_CHANNEL, __VA_ARGS__)
#define traceThreads0(msg) \
  DIAG_Log0(THREAD_DIAG_CHANNEL, msg)
#else
#define traceThreads(...) /* ignore */
#define traceThreads0(msg) /* ignore */
#endif

#ifdef TRACING
#define traceLock(...) \
  DIAG_Log(LOCK_DIAG_CHANNEL, __VA_ARGS__)
#define traceLock0(msg) \
  DIAG_Log0(LOCK_DIAG_CHANNEL, msg)
#else
#define traceLock(...) /* ignore */
#define traceLock0(msg) /* ignore */
#endif

#ifdef TRACING
#define traceStartup(...) \
  DIAG_Log(STARTUP_DIAG_CHANNEL, __VA_ARGS__)
#define traceStartup0(msg) \
  DIAG_Log0(STARTUP_DIAG_CHANNEL, msg)
#else
#define traceStartup(...) /* ignore */
#define traceStartup0(msg) /* ignore */
#endif

#ifdef TRACING
#define tracePersist(...) \
  DIAG_Log(PERSIST_DIAG_CHANNEL, __VA_ARGS__)
#define tracePersist0(msg) \
  DIAG_Log0(PERSIST_DIAG_CHANNEL, msg)
#else
#define tracePersist(...) /* ignore */
#define tracePersist0(msg) /* ignore */
#endif


/*
 * Global 'static' diagnostic channel numbers.
 */
#define DEFAULT_DIAG_CHANNEL     0
#define LOADER_DIAG_CHANNEL      1
#define CALL_DIAG_CHANNEL        2
#define JNI_DIAG_CHANNEL         3
#define OPCODE_DIAG_CHANNEL      4
#define OPSTACK_DIAG_CHANNEL     5
#define FRAME_DIAG_CHANNEL       6
#define EXCEPTION_DIAG_CHANNEL   7
#define ALLOC_DIAG_CHANNEL       8
#define GC_DIAG_CHANNEL          9
#define GCTHREAD_DIAG_CHANNEL   10
#define THREAD_DIAG_CHANNEL     11
#define LOCK_DIAG_CHANNEL       12
#define STARTUP_DIAG_CHANNEL    13
#define PERSIST_DIAG_CHANNEL    14
#define INIT_DIAG_CHANNEL       15


/*
 * Bounds on the number of logging channels.
 */
#define NOS_STATIC_DIAG_CHANNELS  16
#define NOS_DYNAMIC_DIAG_CHANNELS 10

#define NOS_DIAG_CHANNELS \
  (NOS_STATIC_DIAG_CHANNELS + NOS_DYNAMIC_DIAG_CHANNELS)
#define LAST_DIAG_CHANNEL (NOS_DIAG_CHANNELS - 1)


/*
 * DIAG routine signatures.
 */


/* Exit with an assertion failure message */
void DIAG_AssertFail(char *file, unsigned int line, char *msg);

/* Exit by printing a message and calling BailOut. */
void DIAG_Panic0(char* message);
void DIAG_Panic(char* format, ...);

/*
 * Exit by printing an error message and then calling 
 * shutdown_machine(NULL, 1, rc)
 */
int DIAG_FatalError0(int rc, char* message);
int DIAG_FatalError(int rc, char* format, ...);

/* Print an error message and continue */
void DIAG_Error0(char* message);
void DIAG_Error(char* format, ...);

/* Terminate immediately, generating a core dump (or whatever) */
void DIAG_BailOut();

/* Send a message to a logging channel */
void DIAG_Log0(int channel, char* message);
void DIAG_Log(int channel, char* format, ...);

/* 
 * Define a logging channel with a known channel number.  The 'name'
 * is prefixed to log messages, and the 'description' is displayed
 * by DIAG_ListChannels.
 */ 
void DIAG_DefineStaticChannel(int channel, char *name, char *description);

/*
 * Same as DIAG_DefineStaticChannel, except that the channel number is
 * allocated on the fly.
 */
int DIAG_DefineDynamicChannel(char *name, char *description);

/*
 * Enable or disable a channel.  The result is the previous state.
 */
int DIAG_SetChannelState(int channel, int enabled);

/*
 * Get a channel's current state.
 */
int DIAG_ChannelState(int channel);

/*
 * Set the FILE associated with a channel.
 */
void DIAG_SetChannelFile(int channel, FILE *file);

/*
 * Flush all trace print channels
 */
void DIAG_FlushChannels();

/*
 * List the channel names, descriptions and their states.
 */
void DIAG_ListChannels();

/*
 * Map a channel name to a channel number.
 */
int DIAG_LookupChannel(char *name);

/*
 * Initialise logging channels.
 */
void DIAG_InitChannels();


#endif

