/* Manages the allocation and setting of handles for the allocator */

#include "config.h"

#include "vm/garbage/public_methods.h"

/* 
 * Set up the block to contain a whole bunch of handles pointing to
 * each other, ending in a handle pointing to NULL
 */
int HANDLES_Init(int32* pi32HandleBlock, int32 i32NumHandles)
{
  int32* iter;
  
  // The last handle points to NULL
  pi32HandleBlock[i32NumHandles - 1] = (int32) NULL;
  
  // Then we make each handle point to the following handle
  for(iter = pi32HandleBlock + i32NumHandles - 2; 
      iter >= pi32HandleBlock; 
      iter--) {
    *iter = (int32) (iter + 1);
  }

  return 0;
}


tOBREF HANDLES_FirstHandle(int32* pi32HandleBlock)
{
  return (tOBREF) pi32HandleBlock;
}


tOBREF HANDLES_getFreeHandle(tAllocatorHeap* heap)
{
  tOBREF ret;
  
  if (heap->hNextFreeHandle == NULL) {
    return NULL;
  }
  
  ret = heap->hNextFreeHandle;
  heap->hNextFreeHandle = (tOBREF) *ret;
  *ret = (int) NULL; //Just for safety
  return ret;
}


int32 HANDLES_nosFreeHandles(tAllocatorHeap* heap) 
{
  int32 count = 0;

  tOBREF free = heap->hNextFreeHandle;
  while (free) {
    count++;
    assert(HANDLES_InHandleBlock(heap, free));
    assert(*((tOBREF *) free) != free);
    assert(count <= heap->i32HandleBlockSize);
    free = *((tOBREF *) free);
  }
  return count;
}


int inline HANDLES_InHandleBlock(tAllocatorHeap* heap, tOBREF h)
{
  if ((((int32*) h) >= heap->pi32HandleBlock) && 
      ((((int32*) h) < (heap->pi32HandleBlock + heap->i32HandleBlockSize)))) {
    return 1;
  }
  else {
    return 0;
  }
}


void HANDLES_ReleaseHandle(tAllocatorHeap* heap, tOBREF h)
{
  /* The free handle list does not have to be in memory order
     so we can just add this one to the beginning of the list 
     quickly regardless of where it is in memory */
#ifdef DEBUG
  tOBREF free;
  int32 count = 0;

  assert(HANDLES_InHandleBlock(heap, h));
#endif

  *((tOBREF*) h) = heap->hNextFreeHandle;
  heap->hNextFreeHandle = h;
}
