/*
 * Copyright (C) 2001, John Leuner.
 *
 * This file is part of the kissme/teaseme project, which in turn is part of the JOS project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include "config.h"

/* 
 * Hash table support. Originally from Tcl. 
 */ 
 
#include "vm/wrappers.h"

#ifdef KISSME_LINUX_USER 
#include <string.h> 
#include <stdio.h>
#include <stdlib.h> 
#endif 

#include "vm/hash.h" 
 
#include "vm/global.h" 
 
/* 
 * When there are this many entries per bucket, on average, rebuild 
 * the hash table to make it larger. 
 */ 
 
#define REBUILD_MULTIPLIER	3 
 
/* 
 * The following macro takes a preliminary integer hash value and 
 * produces an index into a hash tables bucket list.  The idea is 
 * to make it so that preliminary values that are arbitrarily similar 
 * will end up in different papstBuckets.  The hash function was taken 
 * from a random-number generator. 
 */ 
 
#define RANDOM_INDEX(pstTable, i32) \
  (((((long) (i32))*1103515245) >> (pstTable)->i32DownShift) & \
   (pstTable)->i32Mask) 
 
/* 
 * Procedure prototypes for static procedures in this file: 
 */ 
 
static HASH_tstHashEntry* ArrayFind(HASH_tstHashTable *pstTable, 
				    char *pchKey); 
static HASH_tstHashEntry* ArrayCreate(HASH_tstHashTable *pstTable, 
				      char *pchKey, int32 *pi32New); 
static HASH_tstHashEntry* BogusFind(HASH_tstHashTable *pstTable, 
				    char *pchKey); 
static HASH_tstHashEntry* BogusCreate(HASH_tstHashTable *pstTable, 
				      char *pchKey, int32 *pi32New); 
static uint32 HashString(char *psz); 
static void RebuildTable(HASH_tstHashTable *pstTable); 
static HASH_tstHashEntry* StringFind(HASH_tstHashTable *pstTable, 
				     char *pchKey); 
static HASH_tstHashEntry* StringCreate(HASH_tstHashTable *pstTable, 
				       char *pchKey, int32 *pi32New); 
static HASH_tstHashEntry* OneWordFind(HASH_tstHashTable *pstTable, 
				      char *pchKey); 
static HASH_tstHashEntry* OneWordCreate(HASH_tstHashTable *pstTable,
					char *pchKey, int32 *pi32New); 
void  HASH_IntegrityCheck(HASH_tstHashTable* pstTable); 

 
/* 
 * @doc FUNC 
 * @func 
 *	Given storage for a hash table, set up the fields to prepare 
 *	the hash table for use. 
 *	TablePtr is now ready to be passed to HASH_FindHashEntry and 
 *	HASH_CreateHashEntry. 
 */ 
 
void HASH_InitHashTable(HASH_tstHashTable *pstTable,	
			/* @parm Pointer to table record, which is supplied
			   by the caller. */ 
			int32 i32KeyType
			/* @parm Type of keys to use in table: 
			   HASH_STRING_KEYS, HASH_ONE_WORD_KEYS, 
			   or an integer >= 2. */) 
{ 
  pstTable->papstBuckets = pstTable->apstStaticBuckets; 
  pstTable->apstStaticBuckets[0] = pstTable->apstStaticBuckets[1] = 0; 
  pstTable->apstStaticBuckets[2] = pstTable->apstStaticBuckets[3] = 0; 
  pstTable->i32NumBuckets = HASH_SMALL_HASH_TABLE; 
  pstTable->i32NumEntries = 0; 
  pstTable->i32RebuildSize = HASH_SMALL_HASH_TABLE*REBUILD_MULTIPLIER; 
  pstTable->i32DownShift = 28; 
  pstTable->i32Mask = 3; 
  pstTable->i32KeyType = i32KeyType; 
  if (i32KeyType == HASH_STRING_KEYS) { 
    pstTable->pfFindProc = StringFind; 
    pstTable->pfCreateProc = StringCreate; 
  } 
  else if (i32KeyType == HASH_ONE_WORD_KEYS) { 
    pstTable->pfFindProc = OneWordFind; 
    pstTable->pfCreateProc = OneWordCreate; 
  } 
  else { 
    pstTable->pfFindProc = ArrayFind; 
    pstTable->pfCreateProc = ArrayCreate; 
  }; 
} 
 

/* 
 * @doc FUNC 
 * @func Remove a single entry from a hash table. 
 *	The entry given by pstEntry is deleted from its table and 
 *	should never again be used by the caller.  It is up to the 
 *	caller to free the pvClientData field of the entry, if that 
 *	is relevant. 
 */ 
 
void HASH_DeleteHashEntry(HASH_tstHashEntry *pstEntry
			  /* @parm The entry to delete. */) 
{ 
  HASH_tstHashEntry *pstPrev; 
  
  if (*pstEntry->ppstBucket == pstEntry) { 
    *pstEntry->ppstBucket = pstEntry->pstNext; 
  } 
  else { 
    for (pstPrev = *pstEntry->ppstBucket; ; pstPrev = pstPrev->pstNext) { 
      if (pstPrev == NULL) { 
	panic0("malformed bucket chain in HASH_DeleteHashEntry"); 
      } 
      if (pstPrev->pstNext == pstEntry) { 
	pstPrev->pstNext = pstEntry->pstNext; 
	break; 
      } 
    } 
  } 
  pstEntry->pstTable->i32NumEntries--; 
  sys_free((char *) pstEntry); 
} 
 

/* 
 * @doc FUNC 
 * @func 
 *	Free up everything associated with a hash table except for 
 *	the record for the table itself. 
 *	The hash table is no longer useable. 
 */ 
 
void HASH_DeleteHashTable(HASH_tstHashTable *pstTable	
			  /* @parm Table to delete. */ ) 
{ 
  HASH_tstHashEntry *pstHashEntry, *pstNext; 
  int32 i32; 
  
  /* 
   * Free up all the entries in the table. 
   */ 
  
  for (i32 = 0; i32 < pstTable->i32NumBuckets; i32++) { 
    pstHashEntry = pstTable->papstBuckets[i32]; 
    while (pstHashEntry != NULL) { 
      pstNext = pstHashEntry->pstNext; 
      sys_free((char *) pstHashEntry); 
      pstHashEntry = pstNext; 
    } 
  } 
  
  /* 
   * Free up the bucket array, if it was dynamically allocated. 
   */ 
  
  if (pstTable->papstBuckets != pstTable->apstStaticBuckets) { 
    sys_free((char *) pstTable->papstBuckets); 
  } 
  
  /* 
   * Arrange for panics if the table is used again without 
   * re-initialization. 
   */ 
  
  pstTable->pfFindProc = BogusFind; 
  pstTable->pfCreateProc = BogusCreate; 
} 

/* 
 * @doc FUNC 
 * @func 
 *	Locate the first entry in a hash table and set up a record 
 *	that can be used to step through all the remaining entries 
 *	of the table. 
 * 
 * @rdesc 
 *	The return value is a pointer to the first entry in pstTable, 
 *	or NULL if pstTable has no entries in it.  The memory at 
 *	*pstSearch is initialized so that subsequent calls to 
 *	HASH_NextHashEntry will return all of the entries in the table, 
 *	one at a time. 
 * 
 */ 
 
HASH_tstHashEntry* HASH_FirstHashEntry(HASH_tstHashTable *pstTable,
				       /* @parm Table to search. */ 
				       HASH_tstHashSearch *pstSearch
				       /* @parm Place to store information 
					  about progress through the table */) 
{ 
  pstSearch->pstTable = pstTable; 
  pstSearch->i32NextIndex = 0; 
  pstSearch->pstNextEntry = NULL; 
  return HASH_NextHashEntry(pstSearch); 
} 
 

/* 
 * @doc FUNC 
 * @func 
 *	Once a hash table enumeration has been initiated by calling 
 *	HASH_FirstHashEntry, this procedure may be called to return 
 *	successive elements of the table. 
 * 
 * @rdesc 
 *	The return value is the next entry in the hash table being 
 *	enumerated, or NULL if the end of the table is reached. 
 */ 
 
HASH_tstHashEntry* HASH_NextHashEntry(HASH_tstHashSearch *pstSearch
				      /* @parm Place to store information 
					 about progress through the table. 
					 Must have been initialized by calling
					 HASH_FirstHashEntry. */) 
{ 
  HASH_tstHashEntry *pstHashEntry; 
  
  while (pstSearch->pstNextEntry == NULL) { 
    if (pstSearch->i32NextIndex >= pstSearch->pstTable->i32NumBuckets) { 
      return NULL; 
    } 
    pstSearch->pstNextEntry = 
      pstSearch->pstTable->papstBuckets[pstSearch->i32NextIndex]; 
    pstSearch->i32NextIndex++; 
  } 
  pstHashEntry = pstSearch->pstNextEntry; 
  pstSearch->pstNextEntry = pstHashEntry->pstNext; 
  return pstHashEntry; 
} 


/* 
 * @doc FUNC 
 * @func 
 *	Return statistics describing the layout of the hash table 
 *	in its hash papstBuckets. 
 * 
 * @rdesc 
 *	The return value is a malloc-ed string containing information 
 *	about pstTable.  It is the caller's responsibility to free 
 *	this string. 
 */ 
 
char* HASH_HashStats(HASH_tstHashTable *pstTable
		     /* @parm Table for which to produce stats. */) 
{ 
#define NUM_COUNTERS 10 
  int32 ai32Count[NUM_COUNTERS], i32Overflow, i32, i32Tmp; 
  double dAverage, dTmp; 
  HASH_tstHashEntry *pstHashEntry; 
  char *pszResult, *psz; 
  
  /* 
   * Compute a histogram of bucket usage. 
   */ 
  
  for (i32 = 0; i32 < NUM_COUNTERS; i32++) { 
    ai32Count[i32] = 0; 
  } 
  i32Overflow = 0; 
  dAverage = 0.0; 
  for (i32 = 0; i32 < pstTable->i32NumBuckets; i32++) { 
    i32Tmp = 0; 
    for (pstHashEntry = pstTable->papstBuckets[i32];
	 pstHashEntry != NULL; 
	 pstHashEntry = pstHashEntry->pstNext) { 
      i32Tmp++; 
    } 
    if (i32Tmp < NUM_COUNTERS) { 
      ai32Count[i32Tmp]++; 
    } 
    else { 
      i32Overflow++; 
    } 
    dTmp = i32Tmp; 
    dAverage += (dTmp + 1.0) * (dTmp/pstTable->i32NumEntries) / 2.0; 
  } 
  
  /* 
   * Print out the histogram and a few other pieces of information. 
   */ 
  
  pszResult = (char *) sys_malloc((unsigned) ((NUM_COUNTERS * 60) + 300)); 
  sprintf(pszResult, "%d entries in table, %d papstBuckets\n", 
	  (int) pstTable->i32NumEntries, (int) pstTable->i32NumBuckets); 
  psz = pszResult + strlen(pszResult); 
  for (i32 = 0; i32 < NUM_COUNTERS; i32++) { 
    sprintf(psz, "number of papstBuckets with %i entries: %i\n", 
	    (int) i32, (int) ai32Count[i32]); 
    psz += strlen(psz); 
  } 
  sprintf(psz, "number of papstBuckets with %d or more entries: %d\n", 
	  NUM_COUNTERS, (int) i32Overflow); 
  psz += strlen(psz); 
  sprintf(psz, "dAverage search distance for entry: %.1f", dAverage); 
  return pszResult; 
} 

/* 
 * @doc FUNC 
 * @func 
 *	Compute a one-word summary of a text string, which can be 
 *	used to generate a hash index. 
 * 
 * @rdesc 
 *	The return value is a one-word summary of the information in 
 *	psz. 
 */ 
 
static uint32 HashString(char *psz	
			 /* @parm String from which to compute hash value. */) 
{ 
  uint32 ui32Result; 
  char c; 
  
  /* 
   * I tried a zillion different hash functions and asked many other 
   * people for advice.  Many people had their own favorite functions, 
   * all different, but no-one had much idea why they were good ones. 
   * I chose the one below (multiply by 9 and add new character) 
   * because of the following reasons: 
   * 
   * 1. Multiplying by 10 is perfect for keys that are decimal strings, 
   *    and multiplying by 9 is just about as good. 
   * 2. Times-9 is (shift-left-3) plus (old).  This means that each 
   *    character's bits hang around in the low-order bits of the 
   *    hash value for ever, plus they spread fairly rapidly up to 
   *    the high-order bits to fill out the hash value.  This seems 
   *    works well both for decimal and non-decimal strings. 
   */ 
  
  ui32Result = 0; 
  while (1) { 
    c = *psz; 
    psz++; 
    if (c == 0) { 
      break; 
    } 
    ui32Result += (ui32Result<<3) + c; 
  } 
  return ui32Result; 
} 


/* 
 * @doc FUNC 
 * @func 
 *	Given a hash table with psz keys, and a psz pchKey, find 
 *	the entry with a matching pchKey. 
 * 
 * @rdesc 
 *	The return value is a token for the matching entry in the 
 *	hash table, or NULL if there was no matching entry. 
 */ 
 
static HASH_tstHashEntry* StringFind(HASH_tstHashTable *pstTable,	
				     /* @parm Table in which to lookup
					entry. */ 
				     char *pchKey
				     /* @parm Key to use to find matching 
					entry. */) 
{ 
  HASH_tstHashEntry *pstHashEntry; 
  char *pch1, *pch2; 
  int32 i32Index; 
  
  i32Index = HashString(pchKey) & pstTable->i32Mask; 
  
  /* 
   * Search all of the entries in the appropriate bucket. 
   */ 
  for (pstHashEntry = pstTable->papstBuckets[i32Index]; pstHashEntry != NULL; 
       pstHashEntry = pstHashEntry->pstNext) { 
    for (pch1 = pchKey, pch2 = pstHashEntry->uniKey.achString; 
	 /* */; 
	 pch1++, pch2++) { 
      if (*pch1 != *pch2) { 
	break; 
      } 
      if (*pch1 == '\0') { 
	return pstHashEntry; 
      } 
    } 
  } 
  return NULL; 
} 
 

#ifdef HASH_INTEGRITY_CHECK 
#include "vm/cplist.h" 
extern  tCPListElement* pstTreeTop; 
 
void HASH_IntegrityCheck(HASH_tstHashTable *pstTable) 
{  
  HASH_tstHashSearch pstSearch; 
  HASH_tstHashEntry *pstHashEntry; 
  
  
  // First check if java/lang/Object is in CPLIST tree 
  if (pstTreeTop != NULL) {
    assert(CPLIST_Find("java/lang/Object")); 
  }
  
  pstHashEntry = HASH_FirstHashEntry(pstTable, &pstSearch); 
  
  while(pstHashEntry != NULL)	{ 
    assert(pstHashEntry->pstTable == pstTable); 
    pstHashEntry = HASH_NextHashEntry(&pstSearch); 
  } 
} 
#endif 

/* 
 * @doc FUNC 
 * @func 
 *	Given a hash table with psz keys, and a psz pchKey, find 
 *	the entry with a matching pchKey.  If there is no matching entry, 
 *	then create a new entry that does match. 
 * 
 * @rdesc 
 *	The return value is a pointer to the matching entry.  If this 
 *	is a newly-created entry, then *pi32New will be set to a non-zero 
 *	value;  otherwise *pi32New will be set to 0.  If this is a new 
 *	entry the value stored in the entry will initially be 0. 
 */ 

static HASH_tstHashEntry* StringCreate(HASH_tstHashTable *pstTable,
				       /* @parm Table in which to lookup
					  entry. */ 
				       char *pchKey,	
				       /* @parm Key to use to find or create
					  matching entry. */ 
				       int32 *pi32New
				       /* @parm Store info here telling
					  whether a new entry was created. */) 
{ 
  HASH_tstHashEntry *pstHashEntry; 
  char *pch1, *pch2; 
  int32 i32Index; 
  
  i32Index = HashString(pchKey) & pstTable->i32Mask; 
  
  /* 
   * Search all of the entries in this bucket. 
   */ 
  for (pstHashEntry = pstTable->papstBuckets[i32Index]; 
       pstHashEntry != NULL; 
       pstHashEntry = pstHashEntry->pstNext) { 
    //	    int dummy = (int) (*pstHashEntry).uniKey.achString; 
    for (pch1 = pchKey, pch2 = pstHashEntry->uniKey.achString; 
	 /* */; 
	 pch1++, pch2++) { 
      if (*pch1 != *pch2) { 
	break; 
      } 
      if (*pch1 == '\0') { 
	*pi32New = 0; 
	return pstHashEntry; 
      } 
    } 
  } 
  
  /* 
   * Entry not found.  Add a new one to the bucket. 
   */ 
  *pi32New = 1; 
  pstHashEntry = (HASH_tstHashEntry *) 
    sys_malloc((unsigned) (sizeof(HASH_tstHashEntry) + strlen(pchKey) - 
			   (sizeof(pstHashEntry->uniKey) - 1))); 
  
  assert(pstHashEntry);
  pstHashEntry->pstTable = pstTable; 
  pstHashEntry->ppstBucket = &(pstTable->papstBuckets[i32Index]); 
  pstHashEntry->pstNext = *pstHashEntry->ppstBucket; 
  pstHashEntry->pvClientData = 0; 
  strcpy(pstHashEntry->uniKey.achString, pchKey); 
  *pstHashEntry->ppstBucket = pstHashEntry; 
  pstTable->i32NumEntries++; 
  
  /* 
   * If the table has exceeded a decent size, rebuild it with many 
   * more papstBuckets. 
   */ 
  if (pstTable->i32NumEntries >= pstTable->i32RebuildSize) { 
    //	    fprintf(stderr, "Rebuilding hash table\n"); 
    RebuildTable(pstTable); 
  } 
  return pstHashEntry; 
} 


/* 
 * @doc FUNC 
 * @func 
 *	Given a hash table with one-word keys, and a one-word pchKey, find 
 *	the entry with a matching pchKey. 
 * 
 * @rdesc 
 *	The return value is a token for the matching entry in the 
 *	hash table, or NULL if there was no matching entry. 
 */ 
 
static HASH_tstHashEntry* OneWordFind(HASH_tstHashTable *pstTable,
				      /* @parm Table in which to lookup 
					 entry. */ 
				      char *pchKey
				      /* @parm Key to use to find matching 
					 entry. */) 
{ 
  HASH_tstHashEntry *pstHashEntry; 
  int32 i32Index; 
  
  i32Index = RANDOM_INDEX(pstTable, pchKey); 
  
  /* 
   * Search all of the entries in the appropriate bucket. 
   */ 
  
  for (pstHashEntry = pstTable->papstBuckets[i32Index]; pstHashEntry != NULL; 
       pstHashEntry = pstHashEntry->pstNext) { 
    if (pstHashEntry->uniKey.pchOneWordValue == pchKey) { 
      return pstHashEntry; 
    } 
  } 
  return NULL; 
} 
 

/* 
 * @doc FUNC 
 * @func 
 *	Given a hash table with one-word keys, and a one-word pchKey, find 
 *	the entry with a matching pchKey.  If there is no matching entry, 
 *	then create a new entry that does match. 
 * 
 * @rdesc 
 *	The return value is a pointer to the matching entry.  If this 
 *	is a newly-created entry, then *pi32New will be set to a non-zero 
 *	value;  otherwise *pi32New will be set to 0.  If this is a new 
 *	entry the value stored in the entry will initially be 0. 
 */ 
 
static HASH_tstHashEntry* OneWordCreate(HASH_tstHashTable *pstTable,
					/* @parm Table in which to lookup 
					   entry. */ 
					char *pchKey,
					/* @parm Key to use to find or
					   create matching entry. */ 
					int32 *pi32New
					/* @parm Store info here telling 
					   whether a new entry was created. */)
{ 
  HASH_tstHashEntry *pstHashEntry; 
  int32 i32Index; 
  
  i32Index = RANDOM_INDEX(pstTable, pchKey); 
  
  /* 
   * Search all of the entries in this bucket. 
   */ 
  for (pstHashEntry = pstTable->papstBuckets[i32Index]; 
       pstHashEntry != NULL; 
       pstHashEntry = pstHashEntry->pstNext) { 
    if (pstHashEntry->uniKey.pchOneWordValue == pchKey) { 
      *pi32New = 0; 
      return pstHashEntry; 
    } 
  } 
  
  /* 
   * Entry not found.  Add a new one to the bucket. 
   */ 
  *pi32New = 1; 
  pstHashEntry = (HASH_tstHashEntry *) sys_malloc(sizeof(HASH_tstHashEntry)); 
  pstHashEntry->pstTable = pstTable; 
  pstHashEntry->ppstBucket = &(pstTable->papstBuckets[i32Index]); 
  pstHashEntry->pstNext = *pstHashEntry->ppstBucket; 
  pstHashEntry->pvClientData = 0; 
  pstHashEntry->uniKey.pchOneWordValue = pchKey; 
  *pstHashEntry->ppstBucket = pstHashEntry; 
  pstTable->i32NumEntries++; 
  
  /* 
   * If the table has exceeded a decent size, rebuild it with many 
   * more papstBuckets. 
   */ 
  if (pstTable->i32NumEntries >= pstTable->i32RebuildSize) { 
    RebuildTable(pstTable); 
  } 
  return pstHashEntry; 
} 


/* 
 * @doc FUNC 
 * @func 
 *	Given a hash table with array-of-int32 keys, and a pchKey, find 
 *	the entry with a matching pchKey. 
 * 
 * @rdesc 
 *	The return value is a token for the matching entry in the 
 *	hash table, or NULL if there was no matching entry. 
 */ 
static HASH_tstHashEntry* ArrayFind(HASH_tstHashTable *pstTable,
				    /* @parm Table in which to lookup entry. */
				    char *pchKey
				    /* @parm Key to use to find matching 
				       entry. */) 
{ 
  HASH_tstHashEntry *pstHashEntry; 
  int32 *pai32Key = (int32 *) pchKey; 
  int32 *pi32Tmp1, *pi32Tmp2; 
  int32 i32Index, i32Count; 
  
  for (i32Index = 0, i32Count = pstTable->i32KeyType, pi32Tmp1 = pai32Key; 
       i32Count > 0; 
       i32Count--, pi32Tmp1++) { 
    i32Index += *pi32Tmp1; 
  } 
  i32Index = RANDOM_INDEX(pstTable, i32Index); 
  
  /* 
   * Search all of the entries in the appropriate bucket. 
   */ 
  for (pstHashEntry = pstTable->papstBuckets[i32Index]; 
       pstHashEntry != NULL; 
       pstHashEntry = pstHashEntry->pstNext) { 
    for (pi32Tmp1 = pai32Key, pi32Tmp2 = pstHashEntry->uniKey.ai32Words, 
	   i32Count = pstTable->i32KeyType; 
	 /* */; 
	 i32Count--, pi32Tmp1++, pi32Tmp2++) { 
      if (i32Count == 0) { 
	return pstHashEntry; 
      } 
      if (*pi32Tmp1 != *pi32Tmp2) { 
	break; 
      } 
    } 
  } 
  return NULL; 
} 
 

/* 
 * @doc FUNC 
 * @func 
 *	Given a hash table with one-word keys, and a one-word pchKey, find 
 *	the entry with a matching pchKey.  If there is no matching entry, 
 *	then create a new entry that does match. 
 * 
 * @rdesc 
 *	The return value is a pointer to the matching entry.  If this 
 *	is a newly-created entry, then *pi32New will be set to a non-zero 
 *	value;  otherwise *pi32New will be set to 0.  If this is a new 
 *	entry the value stored in the entry will initially be 0. 
 */ 
 
static HASH_tstHashEntry* ArrayCreate(HASH_tstHashTable *pstTable,	
				      /* @field Table in which to lookup
					 entry. */ 
				      char *pchKey,	
				      /* @field Key to use to find or 
					 create matching entry. */ 
				      int32 *pi32New
				      /* @field Store info here telling
					 whether a new entry was created. */)
{ 
  HASH_tstHashEntry *pstHashEntry; 
  int32 *pai32Array = (int32 *) pchKey; 
  int32 *pi32Tmp1, *pi32Tmp2; 
  int32 i32Index, i32Count; 
  
  for (i32Index = 0, i32Count = pstTable->i32KeyType, pi32Tmp1 = pai32Array; 
       i32Count > 0; 
       i32Count--, pi32Tmp1++) { 
    i32Index += *pi32Tmp1; 
  } 
  i32Index = RANDOM_INDEX(pstTable, i32Index); 
  
  /* 
   * Search all of the entries in the appropriate bucket. 
   */ 
  for (pstHashEntry = pstTable->papstBuckets[i32Index];
       pstHashEntry != NULL; 
       pstHashEntry = pstHashEntry->pstNext) { 
    for (pi32Tmp1 = pai32Array, pi32Tmp2 = pstHashEntry->uniKey.ai32Words, 
								
	   i32Count = pstTable->i32KeyType; 
	 /* */;
	 i32Count--, pi32Tmp1++, pi32Tmp2++) { 
      if (i32Count == 0) { 
	*pi32New = 0; 
	return pstHashEntry; 
      } 
      if (*pi32Tmp1 != *pi32Tmp2) { 
	break; 
      } 
    } 
  } 
  
  /* 
   * Entry not found.  Add a new one to the bucket. 
   */ 
  *pi32New = 1; 
  pstHashEntry = (HASH_tstHashEntry *) 
    sys_malloc((unsigned) (sizeof(HASH_tstHashEntry) +
			   (pstTable->i32KeyType * sizeof(int32)) - 4)); 
  pstHashEntry->pstTable = pstTable; 
  pstHashEntry->ppstBucket = &(pstTable->papstBuckets[i32Index]); 
  pstHashEntry->pstNext = *pstHashEntry->ppstBucket; 
  pstHashEntry->pvClientData = 0; 
  for (pi32Tmp1 = pai32Array, pi32Tmp2 = pstHashEntry->uniKey.ai32Words, 
	 i32Count = pstTable->i32KeyType; 
       i32Count > 0;
       i32Count--, pi32Tmp1++, pi32Tmp2++) { 
    *pi32Tmp2 = *pi32Tmp1; 
  } 
  *pstHashEntry->ppstBucket = pstHashEntry; 
  pstTable->i32NumEntries++; 
  
  /* 
   * If the table has exceeded a decent size, rebuild it with many 
   * more papstBuckets. 
   */ 
  if (pstTable->i32NumEntries >= pstTable->i32RebuildSize) { 
    RebuildTable(pstTable); 
  } 
  return pstHashEntry; 
} 
 

/* 
 * @doc FUNC 
 * @func 
 *	This procedure is invoked when an HASH_FindHashEntry is called 
 *	on a table that has been deleted. 
 *	Generates a panic. 
 * 
 * @rdesc 
 *	If panic returns (which it shouldn't) this procedure returns 
 *	NULL. 
 */ 
 
/* ARGSUSED */ 
static HASH_tstHashEntry* BogusFind(HASH_tstHashTable *pstTable,
				    /* @parm Table in which to lookup 
				       entry. */ 
				    char *pchKey 			
				    /* @parm Key to use to find matching
				       entry. */ 
				    ) 
{ 
  panic0("called HASH_FindHashEntry on deleted table"); 
  return NULL; 
} 
 

/* 
 * @doc FUNC 
 * @func 
 *	This procedure is invoked when an HASH_CreateHashEntry is called 
 *	on a table that has been deleted. 
 *	Generates a panic. 
 * 
 * @rdesc 
 *	If panic returns (which it shouldn't) this procedure returns 
 *	NULL. 
 */ 
/* ARGSUSED */ 
static HASH_tstHashEntry* BogusCreate(HASH_tstHashTable *pstTable,
				      /* @parm Table in which to lookup
					 entry. */ 
				      char *pchKey,
				      /* @parm Key to use to find or create 
					 matching entry. */ 
				      int32 *pi32New
				      /* @parm Store info here telling 
					 whether a new entry was created. */)
{ 
  panic0("called HASH_CreateHashEntry on deleted table"); 
  return NULL; 
} 
 

/* 
 * @doc FUNC 
 * @func 
 *	This procedure is invoked when the ratio of entries to hash 
 *	papstBuckets becomes too large.  It creates a new table with a 
 *	larger bucket array and moves all of the entries into the 
 *	new table. 
 *	Entries get re-hashed to new 
 *	papstBuckets. 
 */ 
static void RebuildTable(HASH_tstHashTable *pstTable
			 /* @parm Table to enlarge. */) 
{ 
  int32 i32OldSize, i32Count, i32Index; 
  HASH_tstHashEntry **papstOldBuckets; 
  HASH_tstHashEntry **ppstOldChain, **ppstNewChain; 
  HASH_tstHashEntry *pstHashEntry; 
  
  i32OldSize = pstTable->i32NumBuckets; 
  papstOldBuckets = pstTable->papstBuckets; 
  
  /* 
   * Allocate and initialize the new bucket array, and set up 
   * hashing constants for new array size. 
   */ 
  pstTable->i32NumBuckets *= 4; 
  pstTable->papstBuckets = (HASH_tstHashEntry **)
    sys_malloc((unsigned) (pstTable->i32NumBuckets * 
			   sizeof(HASH_tstHashEntry *))); 
  for (i32Count = pstTable->i32NumBuckets,
	 ppstNewChain = pstTable->papstBuckets; 
       i32Count > 0;
       i32Count--, ppstNewChain++) { 
    *ppstNewChain = NULL; 
  } 
  pstTable->i32RebuildSize *= 4; 
  pstTable->i32DownShift -= 2; 
  pstTable->i32Mask = (pstTable->i32Mask << 2) + 3; 
  
  /* 
   * Rehash all of the existing entries into the new bucket array. 
   */ 
  for (ppstOldChain = papstOldBuckets; 
       i32OldSize > 0; 
       i32OldSize--, ppstOldChain++) { 
    for (pstHashEntry = *ppstOldChain;
	 pstHashEntry != NULL; 
	 pstHashEntry = *ppstOldChain) { 
      *ppstOldChain = pstHashEntry->pstNext; 
      if (pstTable->i32KeyType == HASH_STRING_KEYS) { 
	i32Index = HashString(pstHashEntry->uniKey.achString) & 
	  pstTable->i32Mask; 
      } 
      else if (pstTable->i32KeyType == HASH_ONE_WORD_KEYS) { 
	i32Index = RANDOM_INDEX(pstTable, 
				pstHashEntry->uniKey.pchOneWordValue); 
      } 
      else { 
	int32 *iPtr; 
	int32 i32Count; 
	
	for (i32Index = 0, i32Count = pstTable->i32KeyType, 
	       iPtr = pstHashEntry->uniKey.ai32Words;
	     i32Count > 0; 
	     i32Count--, iPtr++) { 
	  i32Index += *iPtr; 
	} 
	i32Index = RANDOM_INDEX(pstTable, i32Index); 
      } 
      pstHashEntry->ppstBucket = &(pstTable->papstBuckets[i32Index]); 
      pstHashEntry->pstNext = *pstHashEntry->ppstBucket; 
      *pstHashEntry->ppstBucket = pstHashEntry; 
    } 
  } 
  
  /* 
   * Free up the old bucket array, if it was dynamically allocated. 
   */ 
  if (papstOldBuckets != pstTable->apstStaticBuckets) { 
    sys_free((char *) papstOldBuckets); 
  } 
} 

 
