#include "config.h"

#ifdef KISSME_LINUX_USER
#include <stdio.h>
#include <pthread.h>
#include <string.h>
#endif

#include "vm/interp.h"
#include "vm/jni.h"
#include "vm/kni.h"
#include "vm/kni_methods.h"
#include "vm/initialize.h"
#include "vm/interp_methods.h"
#include "vm/classfile_methods.h"

#include "vm/garbage.h"

#include "vm/newobject.h"
#include "lib/indigenous/java.lang/Class.h"

#include "vm/wrappers.h"

extern tOBREF OOMExceptionObject;

tClassLoaderTuple* pstObjectType; // this can be globally used to get the type for java.lang.Object
tClassLoaderTuple* pstStringType; // this can be globally used to get the type for java.lang.String
tClassLoaderTuple* pstClassType; // this can be globally used to get the type for java.lang.Class
tClassLoaderTuple* pstVMClassType; // this can be globally used to get the type for java.lang.VMClass

extern tAllocatorHeap* system_heap;

tOBREF INTERP_wrapPrimitiveByte(JNIEnv* env, jbyte b)
{
  jfieldID fieldID;
  jclass clazz;
  int failure;
  tOBREF ret = INTERP_NewObjectFromName(env, "java/lang/Byte", &failure);
  if (ret == NULL) {
    return OOMExceptionObject;
  }

  clazz = CLASS_GetClass(env, ret);

  fieldID = (*env)->GetFieldID(env, clazz, "value", "B");

  if (fieldID == NULL) {
    (*env)->Throw(env, 
		  INTERP_ExceptionObjectFromNameAndMessage(env, 
							   "java/lang/InternalError", 
							   "could not find value field for java/lang/Byte"));
      return NULL;
    }

  (*env)->SetByteField(env, ret, fieldID, b);
  return ret;
}


tOBREF INTERP_wrapPrimitiveInt(JNIEnv* env, jint i)
{
  jfieldID fieldID;
  jclass clazz;
  int failure;
  tOBREF ret = INTERP_NewObjectFromName(env, "java/lang/Integer", &failure);
  if (ret == NULL) {
    return OOMExceptionObject;
  }
  
  clazz = CLASS_GetClass(env,ret);
  
  fieldID = (*env)->GetFieldID(env, clazz, "value", "I");
  
  if (fieldID == NULL) {
    (*env)->Throw(env, 
		  INTERP_ExceptionObjectFromNameAndMessage(env, 
							   "java/lang/InternalError", 
							   "could not find value field for java/lang/Integer"));
    return NULL;
  }
  
  (*env)->SetIntField(env, ret, fieldID, i);
  return ret;
}


tOBREF INTERP_wrapPrimitiveFloat(JNIEnv* env, jfloat f)
{
  jfieldID fieldID;
  jclass clazz;
  int failure;
  tOBREF ret = INTERP_NewObjectFromName(env, "java/lang/Float", &failure);
  if (ret == NULL) {
    return OOMExceptionObject;
  }

  clazz = CLASS_GetClass(env,ret);

  fieldID = (*env)->GetFieldID(env, clazz, "value", "F");

  if (fieldID == NULL) {
    (*env)->Throw(env, 
		  INTERP_ExceptionObjectFromNameAndMessage(env, 
							   "java/lang/InternalError", 
							   "could not find value field for java/lang/Float"));
    return NULL;
  }
  
  (*env)->SetFloatField(env, ret, fieldID, f);
  return ret;
}


tOBREF INTERP_wrapPrimitiveDouble(JNIEnv* env, jdouble d)
{
  jfieldID fieldID;
  jclass clazz;
  int failure;
  tOBREF ret = INTERP_NewObjectFromName(env, "java/lang/Double", &failure);
  if (ret == NULL) {
    return OOMExceptionObject;
  }
  
  clazz = CLASS_GetClass(env,ret);

  fieldID = (*env)->GetFieldID(env, clazz, "value", "D");

  if (fieldID == NULL) {
    (*env)->Throw(env, 
		  INTERP_ExceptionObjectFromNameAndMessage(env, 
							   "java/lang/InternalError", 
							   "could not find value field for java/lang/Double"));
    return NULL;
  }
  
  (*env)->SetDoubleField(env, ret, fieldID, d);
  return ret;
}

tOBREF INTERP_wrapPrimitiveChar(JNIEnv* env, jchar c)
{
  jfieldID fieldID;
  jclass clazz;
  int failure;
  tOBREF ret = INTERP_NewObjectFromName(env, "java/lang/Character", &failure);
  if (ret == NULL) {
    return OOMExceptionObject;
  }

  clazz = CLASS_GetClass(env,ret);

  fieldID = (*env)->GetFieldID(env, clazz, "value", "C");

  if (fieldID == NULL) {
    (*env)->Throw(env, 
		  INTERP_ExceptionObjectFromNameAndMessage(env, 
							   "java/lang/InternalError", 
							   "could not find value field for java/lang/Character"));
      return NULL;
    }

  (*env)->SetCharField(env, ret, fieldID, c);
  return ret;
}


tOBREF INTERP_wrapPrimitiveShort(JNIEnv* env, jshort s)
{
  jfieldID fieldID;
  jclass clazz;
  int failure;
  tOBREF ret = INTERP_NewObjectFromName(env, "java/lang/Short", &failure);
  if (ret == NULL) {
    return OOMExceptionObject;
  }

  clazz = CLASS_GetClass(env,ret);

  fieldID = (*env)->GetFieldID(env, clazz, "value", "S");

  if (fieldID == NULL) {
    (*env)->Throw(env, 
		  INTERP_ExceptionObjectFromNameAndMessage(env, 
							   "java/lang/InternalError", 
							   "could not find value field for java/lang/Short"));
    return NULL;
  }

  (*env)->SetShortField(env, ret, fieldID, s);
  return ret;
}


tOBREF INTERP_wrapPrimitiveLong(JNIEnv* env, jlong l)
{
  jfieldID fieldID;
  jclass clazz;
  int failure;
  tOBREF ret = INTERP_NewObjectFromName(env, "java/lang/Long", &failure);
  if (ret == NULL) {
    return OOMExceptionObject;
  }
  
  clazz = CLASS_GetClass(env,ret);

  fieldID = (*env)->GetFieldID(env, clazz, "value", "J");

  if (fieldID == NULL) {
    (*env)->Throw(env, 
		  INTERP_ExceptionObjectFromNameAndMessage(env, 
							   "java/lang/InternalError", 
							   "could not find value field for java/lang/Long"));
    return NULL;
  }
  
  (*env)->SetLongField(env, ret, fieldID, l);
  return ret;
}


tOBREF INTERP_wrapPrimitiveBoolean(JNIEnv* env, jboolean b)
{
  jfieldID fieldID;
  jclass clazz;
  int failure;
  tOBREF ret = INTERP_NewObjectFromName(env, "java/lang/Boolean", &failure);
  if (ret == NULL) {
    return OOMExceptionObject;
  }

  clazz = CLASS_GetClass(env,ret);

  fieldID = (*env)->GetFieldID(env, clazz, "value", "Z");

  if (fieldID == NULL) {
    (*env)->Throw(env,
		  INTERP_ExceptionObjectFromNameAndMessage(env, 
							   "java/lang/InternalError", 
							   "could not find value field for java/lang/Boolean"));
    return NULL;
  }
  
  (*env)->SetBooleanField(env, ret, fieldID, b);
  return ret;
}


/*
 * @doc FUNC
 * @func
 * This function creates a new object of the specified class type,
 * allocating memory for the instance variables.
 *
 * @rdesc Returns a pointer to the object that was created.
 *
 * @ex Example showing creation of object |
 *
 * tClassLoaderTuple*  pstAnimalType;
 * tOBREF pstAnimal;
 *
 * pstAnimalType = CPLIST_Find("java/zoo/Wombat");
 * pstAnimal = INTERP_Object(pstAnimalType);
 *
 */
tOBREF INTERP_NewObject(JNIEnv* env,
			/* @parm Pointer to class to create instance of */
			tClassLoaderTuple* pstType, 
			int* failure)
{
  KNIApi *kniPtr = KNI_getKNIApiPtr();
  tOBREF hObjectTemp;
  
#ifdef DEBUG
  assert(pstType);
#endif
  *failure = 0;
  assert(pstType->uidName[0] != '[');
  if (pstType->pstClass->bInitialised == 0 && 
      (pstType != pstClassType && pstType != pstVMClassType)) {
    tOBREF pstExOb = INITIALIZE_InitializeClass(env, pstType->classObject);
    if (pstExOb) {
      traceExceptions("Exception %s thrown in static initialisation of %s", 
		      ODEREF(pstExOb)->pstType->uidName,
		      pstType->uidName);
      *failure = 1;
      return pstExOb;
    }
  }

  hObjectTemp = 
    (tOBREF) GARBAGE_Malloc((ROUND32BIT(sizeof(tObject)) + 
			     sizeof(int32) * pstType->pstClass->u16InstSize), 
			    kniPtr->GetHeap(env));
  
  if (hObjectTemp) {
#ifdef GARBAGE
    DEREF(hObjectTemp)->i32Flags = GARBAGE_OBJECT;
#endif

#ifdef PERSIST
#ifndef REVRPOT
    /* mark all new objects with NULL pid */
    DEREF(hObjectTemp)->pid = (PID) NULL;
#endif
#ifdef UPDATEBITS
    assert(pstType->u16InstCount <= 32);
    hObjectTemp->i32UpdateBits = 0;
#endif
#endif

    DEREF(hObjectTemp)->pstType = pstType;
    if (pstType->pstClass->u16InstSize) {
      DEREF(hObjectTemp)->pi32Vars = 
	(int32*) (((byte*) DEREF(hObjectTemp)) + ROUND32BIT(sizeof(tObject)));
      memset(DEREF(hObjectTemp)->pi32Vars, 0,
	     sizeof(int32) * pstType->pstClass->u16InstSize);
    }
    else {
      DEREF(hObjectTemp)->pi32Vars = NULL;
    }

#ifndef IBM_LOCKS
    DEREF(hObjectTemp)->pstWaitSet = NULL;
    DEREF(hObjectTemp)->i32WaitCount = 0;
    DEREF(hObjectTemp)->pstLockThread = NULL;
    DEREF(hObjectTemp)->i32LockCount = 0;
#else
    DEREF(hObjectTemp)->lock_word = 0;
#ifdef IN_PLACE_CONDITION
    DEREF(hObjectTemp)->condVariable = 
      sys_condition_create_inplace(&DEREF(hObjectTemp)->actualCondVariable);
#else
    DEREF(hObjectTemp)->condVariable = sys_condition_create();
#endif
    assert(DEREF(hObjectTemp)->condVariable);
#endif
#ifdef DEBUG_OBJECT_FOR_GC
    ODEREF(hObjectTemp)->magic1 = OMAGIC1;
    ODEREF(hObjectTemp)->magic2 = OMAGIC2;
#endif

    if (pstType->pstClass->bHasFinalizer) {
      GARBAGE_RegisterFinalizable(hObjectTemp, kniPtr->GetHeap(env));
    }
  }
  return hObjectTemp;
}


/*
 * @doc FUNC
 * @func
 * This function creates a new object of the specified class type,
 * allocating memory for the instance variables.  Its behaviour is the
 * same as that of INTERP_NewObject() but it takes the name of the class
 * as a parameter instead of a tClass pointer.
 *
 * @rdesc Returns a pointer to the object that was created.
 *
 * @ex Example showing creation of object |
 *
 * tOBREF pstAnimal;
 *
 * pstAnimal = INTERP_NewObjectFromName(env, "java/zoo/Wombat");
 *
 */
tOBREF INTERP_NewObjectFromName(JNIEnv* env,
				char* pszClassName,  
				/* @parm Name of class to create an 
				   instance of */
				int* failure)
{
  tClassLoaderTuple *pstType = NULL;
#ifdef LOADING_VERBOSE
  traceLoading("NewObjectFromName %s", pszClassName);
#endif
  pstType = CLASSFILE_FindOrLoad(env, pszClassName, NULL); //XXX we need a loading class
  
  if (pstType == NULL) {
    return NULL;
  }
#ifdef DEBUG
  assert(pstType);
#endif
  
  return INTERP_NewObject(env, pstType, failure);
}


/* Jewel - May 2000
 *
 * I'm thinking of making this routine look at the java.lang.String's
 * hashtable and then returning the object in there if it exists.
 *
 * That implies constructing the String, and then returning what
 * String.intern() gives us.
 */


/*
 * @ex Example showing creating of a java.lang.String object |
 *
 * tOBREF pstSentence;
 *
 * pstSentence = INTERP_NewStringFromAsciz("Wombats are marsupials.");
 *
 */
tOBREF INTERP_NewStringFromAsciz(JNIEnv* env,
				 char* pszString /* @parm Asciz string from
						    which to create the 
						    string */)
{
  uint16 i;  /* for keeping place in Asciz string */
  uint16 j;  /* for keeping place in Java String array */
  uint16 u16Temp;
  tOBREF hObjectTemp;
  tARREF pstArrayTemp;
  unsigned char* pszUString = (unsigned char*) pszString;
  int failure;

  // XXX - Handle failure cases for newobject and newarray ...
  if (pstStringType == NULL) {
    hObjectTemp = 
      INTERP_NewObject(env, 
		       CLASSFILE_FindOrLoad(env, "java/lang/String", NULL),
		       &failure);
  }
  else {
    hObjectTemp = INTERP_NewObject(env, pstStringType, &failure);
  }

  if (hObjectTemp == NULL) {
    return NULL;
  }

  u16Temp = strlen(pszUString);
  // XXX - Check this loading class
  pstArrayTemp = INTERP_NewArray(env, T_CHAR, u16Temp, NULL, NULL, &failure); 

  if (pstArrayTemp == NULL) {
    return NULL;
  }
  
  for (i = 0, j = 0; i < u16Temp; i++, j++) {
    if (pszUString[i] > 0x7f) {
      /* if 2-byte format - first byte starts with 110 */
      if ((pszUString[i] & 0xE0) == 0xC0) {
        ((uint16*) ADEREF(pstArrayTemp)->pvElements)[j] = 
	  ((pszUString[i] & 0x1f) << 6) + (pszUString[i + 1] & 0x3f);
        i++;
      }
      /* if 3-byte format - first byte starts with 1110 */
      else if ((pszUString[i] & 0xF0) == 0xE0) {
        ((uint16*) ADEREF(pstArrayTemp)->pvElements)[j] = 
	  ((pszUString[i] & 0xf) << 12) + ((pszUString[i + 1] & 0x3f) << 6) + 
	  (pszUString[i + 2] & 0x3f);
        i += 2;
      }
      /* else error */
      else {
        eprintf("Error in UTF8 encoding of string %s\n", pszUString);
	(*env)->ThrowNew(env, 
			 (*env)->FindClass(env, "java/lang/InternalError"), 
			 "Error in UTF8 encoding of string");
	return NULL;
      }
    }
    else {
      ((uint16*) ADEREF(pstArrayTemp)->pvElements)[j] = pszUString[i];
    }
  }
#ifdef PERSIST
  /* references and other fields are updated here, so mark them */
  /* actually we don't need to do this... because this object will be
     transient and will thus be promoted to persistence if it is
     reachable, whether or not it has been flagged as updated

     ACTUALLY, we do coz stabilise won't look at the references
     if they haven't been flagged even if it's transient */
  DEREF(hObjectTemp)->i32Flags |= (PERSIST_Dirty | PERSIST_RefsDirty);
#endif
  u16Temp = CLASSFILE_NativeInstOffset(env, pstStringType,
				       "java/lang/String", "value"); 
  DEREF(hObjectTemp)->pi32Vars[u16Temp] = (int32) pstArrayTemp;
#ifdef PERSIST
#ifdef UPDATEBITS
  DEREF(hObjectTemp)->i32UpdateBits |= (1 << u16Temp);
#endif
#endif
  u16Temp = CLASSFILE_NativeInstOffset(env, pstStringType, 
				       "java/lang/String", "offset");
  DEREF(hObjectTemp)->pi32Vars[u16Temp] = 0;
  u16Temp = CLASSFILE_NativeInstOffset(env, pstStringType, 
				       "java/lang/String", "count");
  DEREF(hObjectTemp)->pi32Vars[u16Temp] = j;
  
  assert(hObjectTemp);
  
  // Now return the value of String.intern
  {
    jobject ex;
    int32 pi32Args[2];
    jmethodID mid = CLASSFILE_FindMethod(env, ODEREF(hObjectTemp)->pstType, 
					 "intern", "()Ljava/lang/String;");

    if (mid == NULL) {
       panic0("method String.intern() not found");
    }
    // Previously, we used CallObjectMethod, but that is wrong because
    // it calls (*env)->CreateLocalRef() on the result.  This leads to
    // build up of local references.
    //
    // return (*env)->CallObjectMethod(env, hObjectTemp, mid);

    pi32Args[0] = (int32) hObjectTemp;
    ex = INTERP_RunVirtualMethodFromPtr(env, mid, pi32Args);
    if (ex) {
      (*env)->Throw(env, ex);
    }
    return (jobject) pi32Args[0];
  }
}


/*
 * @doc FUNC
 * @func
 * This function returns an Asciz representation of a Java string object. It
 * will be in UTF8 format so any Unicode characters will be correctly stored.
 *
 * You must call sys_free(psz) when you're finished with the string
 */

char* INTERP_AscizFromString(JNIEnv* env, tOBREF pstStringH)
{
  tARREF pstArray;
  int32 i, k, i32Offset, i32Count;
  char* pszBuf;
  int32 i32NameOff = CLASSFILE_NativeInstOffset(env, pstStringType, 
						"java/lang/String","value");

  if (pstStringH == NULL) {
    // This should never happen though ...
    return NULL;
  }

  /* just reusing variables here... */
  i32Offset = CLASSFILE_NativeInstOffset(env, pstStringType, 
					 "java/lang/String", "offset");
  i32Offset = DEREF(pstStringH)->pi32Vars[i32Offset];
  i32Count = CLASSFILE_NativeInstOffset(env, pstStringType, 
					"java/lang/String", "count");
  i32Count = DEREF(pstStringH)->pi32Vars[i32Count];

  pstArray = (tARREF) DEREF(pstStringH)->pi32Vars[i32NameOff];
  pszBuf = (char*) sys_malloc(sizeof(char) * (3 * i32Count + 1));
  if (pszBuf == NULL) {
    eprintf("Out of memory in INTERP_AscizFromString\n");
    return NULL;
  }
  
  for (i = 0, k = 0; i < i32Count; i++, k++) {
    if (((uint16*) ADEREF(pstArray)->pvElements)[i32Offset + i] > 0x7f) {
      /* two-byte format */
      if (((uint16*) ADEREF(pstArray)->pvElements)[i32Offset + i] < 0x7ff) {
	pszBuf[k++] = 0xc0 | 
	  ((((uint16*) ADEREF(pstArray)->pvElements)[i32Offset + i] & 
	    0x7c0) >> 6);
	pszBuf[k] = 0x80 | 
	  (((uint16*) ADEREF(pstArray)->pvElements)[i32Offset + i] & 0x3f);
      }
      /* three-byte format */
      else {
	pszBuf[k++] = 0xe0 | 
	  ((((uint16*) ADEREF(pstArray)->pvElements)[i32Offset + i] & 0xf000) >> 12);
	pszBuf[k++] = 0x80 | 
	  ((((uint16*) ADEREF(pstArray)->pvElements)[i32Offset + i] & 0xfc0) >> 6);
	pszBuf[k] = 0x80 | 
	  (((uint16*) ADEREF(pstArray)->pvElements)[i32Offset + i] & 0x3f);
      }
    }
    else {
      pszBuf[k] = 
	((uint16*) ADEREF(pstArray)->pvElements)[i32Offset + i] & 0x7f;
    }
  }
  pszBuf[k] = 0;
  return pszBuf;
}


/*
 * @doc FUNC
 * @func
 * This function creates a new array of the specified type with the
 * specified number of elements.
 *
 * @rdesc Returns a pointer to the array that was created.
 *
 * @ex Example showing creation of array |
 *
 * tARREF pstMonths;
 *
 * pstMonths = INTERP_NewArray(env, T_SHORT, 12, NULL);
 *
 */

tARREF INTERP_NewArray(JNIEnv* env,
		       tAType enType,   /* @parm Type of elements in array */
		       int32 i32Number, /* @parm Number of elements in array */
		       char* pszSig,    /* @parm Signature of elements if they
					   are of type array or object */
		       tClassLoaderTuple* loadingClass,
		       int* failure)
{
  KNIApi *kniPtr = KNI_getKNIApiPtr();
  tARREF pstArrayTemp;
  int32 i32ElSize, i32AllocAmount;
  long long i64AllocAmount;
  char* pszThisSig = NULL;
  int iSigLen;

  *failure = 0;
  
  if (i32Number < 0) {
    /* this should have been checked by our caller ... */
    panic("INTERP_NewArray: negative array size (%i)", i32Number);
  }
  
  switch (enType) {

    /* pointer */
  case T_ARRAY:
    i32ElSize = sizeof(tARREF);
    pszThisSig = (char*) sys_malloc(strlen(pszSig) + 2);
    assert(pszThisSig);
    pszThisSig[0] = '[';
    strcpy(pszThisSig + 1, pszSig);
    break;

  case T_OBJECT:
    i32ElSize = sizeof(tOBREF);
    iSigLen = strlen(pszSig);
    pszThisSig = (char*) sys_malloc(strlen(pszSig) + 4);
    assert(pszThisSig);
    if (pszSig[0] == '[' || pszSig[iSigLen - 1] == ';') {
      pszThisSig[0] = '[';
      strcpy(pszThisSig + 1, pszSig);
    }
    else {
      /* This deals with the case where pszSig is a class name without
	 the 'L' ... ';' wrapper. */
      pszThisSig[0] = '[';
      pszThisSig[1] = 'L';
      strcpy(pszThisSig + 2, pszSig);
      pszThisSig[iSigLen + 2] = ';';
      pszThisSig[iSigLen + 3] = '\000';
    }
    break;

    /* 8 bits */
  case T_BOOLEAN:
    i32ElSize = 1;
    pszThisSig = "[Z";
    break;

  case T_BYTE:
    i32ElSize = 1;
    pszThisSig = "[B";
    break;

    /* 16 bits */
  case T_CHAR:
    i32ElSize = 2;
    pszThisSig = "[C";
    break;

  case T_SHORT:
    i32ElSize = 2;
    pszThisSig = "[S";
    break;

    /* 32 bits */
  case T_INT:
    i32ElSize = 4;
    pszThisSig = "[I";
    break;

  case T_FLOAT:
    i32ElSize = 4;
    pszThisSig = "[F";
    break;

    /* 64 bits */
  case T_LONG:
    i32ElSize = 8;
    pszThisSig = "[J";
    break;

  case T_DOUBLE:
    i32ElSize = 8;
    pszThisSig = "[D";
    break;

  default:
    panic("Error in NewArray function, invalid element type: %i", enType);
  }
    
  i64AllocAmount = sizeof(tArray) + ((long long) i32ElSize) * i32Number;
  i32AllocAmount = ROUND32BIT(i64AllocAmount);
  if (i64AllocAmount > 0x7fffffff || i32AllocAmount < 0) {
    // XXX - this leaks pszThisSig in some cases
    return NULL;
  }

  pstArrayTemp = (tARREF) GARBAGE_Malloc(i32AllocAmount, kniPtr->GetHeap(env));
  if (pstArrayTemp == NULL) {
    // XXX - this leaks pszThisSig in some cases
    return NULL;
  }
  assert(pstArrayTemp);
  
  ADEREF(pstArrayTemp)->i32Flags = GARBAGE_ARRAY;
  
#ifdef PERSIST
#ifndef REVRPOT
  /* mark all new arrays with NULL pid */
  ADEREF(pstArrayTemp)->pid = (PID) NULL;
#endif
#ifdef UPDATEBITS
  assert(pstArrayTemp->i32Number <= 32);
  pstArrayTemp->i32UpdateBits = 0;
#endif
#endif

  /* set object fields */
#ifdef LOADING_VERBOSE
  traceLoading("INTERP_NewArray %s", pszThisSig);
#endif
  // XXX - deal with failure cases ...
  ADEREF(pstArrayTemp)->pstType = 
    CLASSFILE_FindOrLoad(env, pszThisSig, loadingClass);

  switch (enType) {
  case T_ARRAY:
  case T_OBJECT:
    sys_free(pszThisSig);
    break;
  default:
    break;
  }

  ADEREF(pstArrayTemp)->pi32Vars = NULL;
  ADEREF(pstArrayTemp)->lock_word = 0;
#ifdef IN_PLACE_CONDITION
  ADEREF(pstArrayTemp)->condVariable = 
    sys_condition_create_inplace(&ADEREF(pstArrayTemp)->actualCondVariable);
#else
  ADEREF(pstArrayTemp)->condVariable = sys_condition_create();
#endif
#ifndef IBM_LOCKS
  ADEREF(pstArrayTemp)->pstWaitSet = NULL;
  ADEREF(pstArrayTemp)->i32WaitCount = 0;
  ADEREF(pstArrayTemp)->pstLockThread = NULL;
  ADEREF(pstArrayTemp)->i32LockCount = 0;
#endif

  /* set array-specific fields */
  ADEREF(pstArrayTemp)->i32Number = i32Number;
  ADEREF(pstArrayTemp)->enType = enType;
  if (i32Number > 0) {
    ADEREF(pstArrayTemp)->pvElements = 
      (((byte*) ADEREF(pstArrayTemp)) + ROUND32BIT(sizeof(tArray)));
    memset(ADEREF(pstArrayTemp)->pvElements, 0, i32ElSize * i32Number);
  }
  else {
    ADEREF(pstArrayTemp)->pvElements = NULL;
  }

#ifdef DEBUG_OBJECT_FOR_GC
  ADEREF(pstArrayTemp)->magic1 = OMAGIC1;
  ADEREF(pstArrayTemp)->magic2 = OMAGIC2;
#endif

  return pstArrayTemp;
}


/*
 * @doc FUNC
 * @func
 *
 * This function is used to created multi-dimensional Java arrays.  It is a
 * recursive function and calls itself to create each level. It uses the
 * signature that is passed to it to decide what to do at each level.
 *
 * @rdesc Returns a pointer to the multi-dimensional array that was created.
 *
 * @ex Example of creation of a multi-dimensional array.  Here the dimension
 *     size array is defined manually, but when this function is used these
 *     values are taken from the Java operand stack and don't have to be
 *     created manually.  i32CurrDim must always start at 0 because it is
 *     used to keep track of recursion. |
 *
 * tARREF pstWeekDays;
 * int32   pi32Sizes[2] = {52, 7};
 *
 * pstWeekDays = MultiNewArray(env, pi32Sizes, 0, 2, "[[I");
 *
 */

tARREF INTERP_MultiNewArray(JNIEnv* env,
			    int32*  pi32Sizes,  /* @parm Pointer to array of 
						   dimension sizes */
			    int32   i32CurrDim, /* @parm Which dimension we're 
						   currently doing */
			    int32   i32NumDims, /* @parm Total number 
						   of dimensions */
			    char*   pszSig,     /* @parm Signature of this 
						   dimension (whole signature
						   in beginning) */
			    tClassLoaderTuple* loadingClass,
			    int*    failure)
{
  int32 i;
  tARREF pstArray = NULL;

  assert(pszSig[0] == '[');
  
  switch (pszSig[1])
  {
  case 'Z':
    pstArray = INTERP_NewArray(env, T_BOOLEAN, pi32Sizes[i32CurrDim], 
			       NULL, loadingClass, failure);
    break;

  case 'B':
    pstArray = INTERP_NewArray(env, T_BYTE, pi32Sizes[i32CurrDim],
			       NULL, loadingClass, failure);
    break;

  case 'C':
    pstArray = INTERP_NewArray(env, T_CHAR, pi32Sizes[i32CurrDim], 
			       NULL, loadingClass, failure);
    break;
    
  case 'S':
    pstArray = INTERP_NewArray(env, T_SHORT, pi32Sizes[i32CurrDim],
			       NULL, loadingClass, failure);
    break;
    
  case 'I':
    pstArray = INTERP_NewArray(env, T_INT, pi32Sizes[i32CurrDim],
			       NULL, loadingClass, failure);
    break;
    
  case 'F':
    pstArray = INTERP_NewArray(env, T_FLOAT, pi32Sizes[i32CurrDim], 
			       NULL, loadingClass, failure);
    break;
    
  case 'J':
      pstArray = INTERP_NewArray(env, T_LONG, pi32Sizes[i32CurrDim], 
				 NULL, loadingClass, failure);
      break;
      
  case 'D':
    pstArray = INTERP_NewArray(env, T_DOUBLE, pi32Sizes[i32CurrDim],
			       NULL, loadingClass, failure);
    break;
    
  case 'L':
#ifdef DEBUG_ARRAY
    eprintf("Doing new array T_OBJECT %x dim %i %i %s\n", 
	    pi32Sizes, i32CurrDim, pi32Sizes[i32CurrDim], pszSig);
#endif
    pstArray = INTERP_NewArray(env, T_OBJECT, pi32Sizes[i32CurrDim],
			       pszSig + 1, loadingClass, failure);
    break;
    
  case '[':
#ifdef DEBUG_ARRAYS
    eprintf("Doing multianewarray dim %i size: %i (%s)\n",
	    i32CurrDim, pi32Sizes[i32CurrDim], pszSig);
#endif
    pstArray = INTERP_NewArray(env, T_ARRAY, pi32Sizes[i32CurrDim], 
			       pszSig + 1, loadingClass, failure); 
    
    if (i32CurrDim + 1 == i32NumDims) {
#ifdef DEBUG_ARRAYS
      eprintf("Breaking\n");
#endif
      break;
    }

    if (pstArray == NULL || *failure) {
      // No array allocated / exception thrown
      break;
    }  

    if (pi32Sizes[i32CurrDim] > 0) {
      // Recursively initialise the elements of this array with subarrays
      for (i = 0; i < pi32Sizes[i32CurrDim]; i++){
#ifdef DEBUG_ARRAYS
	eprintf("Sub array %i, size %i\n", i, pi32Sizes[i32CurrDim + 1]);
#endif
	((tARREF*) ADEREF(pstArray)->pvElements)[i] = 
	  INTERP_MultiNewArray(env, pi32Sizes, i32CurrDim + 1, i32NumDims, 
			       pszSig + 1, loadingClass, failure);
      }
    }
    else {
      // This array has size == 0.  No need to allocate subarrays, but
      // the JVM spec says that we must check that the subarray sizes
      // are all non-negative.
      for (i = i32CurrDim + 1; i < i32NumDims; i++) {
	if (pi32Sizes[i] < 0) {
	  // XXX - the exception should be returned, not thrown
	  (*env)->ThrowNew(env, 
			   (*env)->FindClass(env, 
					     "java/lang/NegativeArraySizeException"),
			   "");
	  return NULL;
	}
      }
    }
    break;

  default:
    panic("INTERP_NewMultiArray: unknown type character (%c)", pszSig[1]);
  }

  return pstArray;
}


#ifdef DEBUG_OBJECT_FOR_GC
void INTERP_CheckObject(void* o) 
{
  tObject* obj = (tObject*) o;

  assert(obj->magic1 == OMAGIC1);
  if ((obj->i32Flags & GARBAGE_TYPEMASK) == GARBAGE_OBJECT) {
    assert(obj->magic2 == OMAGIC2);
  }
  else if ((obj->i32Flags & GARBAGE_TYPEMASK) == GARBAGE_ARRAY) {
    tArray* array = (tArray*) o;
    assert(array->magic2 == OMAGIC2);
  }
  else {
    panic("Object has unrecognized garbage type flags (0x%x)", 
	  (obj->i32Flags & GARBAGE_TYPEMASK));
  }
  assert(*(obj->hHandle) == obj);

}
#endif
