# Copyright (c) 2008 Hideki Ikemoto
#
# Permission is hereby granted, free of charge, to any person obtaining
# a copy of this software and associated documentation files (the
# "Software"), to deal in the Software without restriction, including
# without limitation the rights to use, copy, modify, merge, publish,
# distribute, sublicense, and/or sell copies of the Software, and to
# permit persons to whom the Software is furnished to do so, subject to
# the following conditions:
#
# The above copyright notice and this permission notice shall be
# included in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
# LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
# OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
# WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

require 'writedialogbase.rb'
require 'singleton'
require 'net/http'

class WriteDialog < WriteDialogBase
  include Singleton

  slots 'post()', 'set_sage(bool)', 'cancel()', 'check_body()'

  def initialize
    super

    connect(@ok_button, SIGNAL("clicked()"), self, SLOT("post()"))
    connect(@cancel_button, SIGNAL("clicked()"), self, SLOT("cancel()"))
    connect(@sage_checkbox, SIGNAL("toggled(bool)"), self, SLOT("set_sage(bool)"))
    connect(@body_edit, SIGNAL("textChanged()"), self, SLOT("check_body()"))
  end

  def show
    super

    index_path = Util.dat_url_to_index_path(@dat_url)
    config = KDE::Config.new(index_path)
    names = config.readListEntry("Names")
    kotehan = Config.instance.kotehan

    @name_combo.clear
    @name_combo.insertItem(Config.instance.default_name)
    @name_combo.insertStringList(kotehan)
    @name_combo.insertStringList(names)

    @mail_line.text = Config.instance.default_mail
    @body_edit.text = ""
    @sage_checkbox.setChecked(false)
    @ok_button.setEnabled(false)
  end

  def dat_url=(dat_url)
    @dat_url = dat_url
    @board_url = Util.dat_url_to_board_url(dat_url)
    @thread_id = Util.dat_url_to_thread_id(dat_url)
    @board_id = Util.dat_url_to_board_id(dat_url)
    @write_url = Util.dat_url_to_write_url(dat_url)
  end

  def board_name
    @board_name_label.text
  end

  def board_name=(board_name)
    @board_name_label.text = board_name
  end

  def thread_name
    @thread_name
  end

  def thread_name=(thread_name)
    @thread_name = thread_name
    @thread_name_label.text = Util.truncate_str(thread_name, 20)
    Qt::ToolTip::add(@thread_name_label, thread_name)
  end

  def post
    name = @name_combo.currentText
    mail = @mail_line.text
    body = @body_edit.text
    server_time = Util.server_time(@dat_url)

    save_name(name)
    save_log(board_name(), thread_name(), Util.dat_url_to_thread_url(@dat_url),
             name, mail, body)

    post_str = ""
    case Util.board_type(@board_url)
    when Util::TYPE_2CH, Util::TYPE_MACHIBBS, Util::TYPE_UNKNOWN
      post_str = Util.build_post_str(name, mail, body,
                                     @board_id, @thread_id, server_time,
                                     Account.instance.session_id)
      post_2ch(post_str)
    when Util::TYPE_JBBS
      post_str = Util.build_jbbs_post_str(name, mail, body,
                                          @board_id, @thread_id, server_time)
      post_jbbs(post_str)
    end
  end

  def post_jbbs(post_str)
    response_body = nil
    cookies = []
    Net::HTTP.start(URI.parse(@write_url).host, 80) {|http|
      header = {
        "Content-Type" => "application/x-www-form-urlencoded",
        "Referer" => @board_url
      }
      response = http.post(URI.parse(@write_url).path,
                           post_str, header)

      response_body = NKF.nkf("-Ewx", response.body) # EUC -> UTF-8 with JISX0201
    }

    accept
  end

  def post_2ch(post_str)
    response_body = nil
    cookies = []
    Net::HTTP.start(URI.parse(@write_url).host, 80) {|http|
      header = {
        "Content-Type" => "application/x-www-form-urlencoded",
        "Referer" => @board_url,
        "User-Agent" => USER_AGENT
      }
      response = http.post(URI.parse(@write_url).path,
                           post_str, header)

      response_body = NKF.nkf("-Swx", response.body) # SJIS -> UTF-8 with JISX0201

      response.get_fields("Set-Cookie").each {|cookie|
        value = cookie[0...cookie.index(';')]
        cookies.push(value)
      }
    }

    return unless response_body && response_body.match(/<!-- 2ch_X:cookie -->/)

    name_cmp = "<input type=hidden name=\""
    name_index = response_body.index(name_cmp)
    name_end_index = response_body.index("\"", name_index + name_cmp.size)
    name = response_body[name_index + name_cmp.size...name_end_index]

    value_cmp = "value=\""
    value_index = response_body.index(value_cmp, name_end_index)
    value_end_index = response_body.index("\"", value_index + value_cmp.size)
    value = response_body[value_index + value_cmp.size...value_end_index]

    post_str += "&#{name}=#{value}"

    Net::HTTP.start(URI.parse(@write_url).host, 80) {|http|
      header = {
        "Content-Type" => "application/x-www-form-urlencoded",
        "Referer" => @board_url,
        "Cookie" => cookies.join(";"),
        "User-Agent" => USER_AGENT
      }
      response = http.post(URI.parse(@write_url).path,
                           post_str, header)

      response_body = NKF.nkf("-Swx", response.body) # SJIS -> UTF-8 with JISX0201
    }

    accept
  end

  def save_log(board_name, thread_name, thread_url, name, mail, body)
    log_path = ENV['HOME'] + "/.kita/write.log"

    open(log_path, "a") {|io|
      io.puts "Date   : #{Time::now.to_s}"
      io.puts "Board  : #{board_name}"
      io.puts "Thread : #{thread_name}"
      io.puts "URL    : #{thread_url}"
      io.puts "Name   : #{name}"
      io.puts "Mail   : #{mail}"
      io.puts
      io.puts body
      io.puts "--------------------"
    }
  end

  def save_name(name)
    return if name == nil || name.empty?

    index_path = Util.dat_url_to_index_path(@dat_url)
    config = KDE::Config.new(index_path)

    names = config.readListEntry("Names") || []
    names.push(name) unless names.include?(name)
    config.writeEntry("Names", names)
    config.sync
  end

  def set_sage(on)
    if on == true
      @mail_stack = @mail_line.text
      @mail_line.text = "sage"
      @mail_line.setEnabled(false)
    else
      @mail_line.text = @mail_stack
      @mail_line.setEnabled(true)
    end
  end

  def cancel
    if @body_edit.text.length > 0 then
      if Qt::MessageBox::warning(self,
                                 "Kita",
                                 "このダイアログを閉じると、テキストが破棄されます。\n" +
                                 "閉じてもいいですか？",
                                 Qt::MessageBox::Ok,
                                 Qt::MessageBox::Cancel | Qt::MessageBox::Default) ==
        Qt::MessageBox::Ok then
        reject
      end
    else
      reject
    end
  end

  def check_body
    body = @body_edit.text
    if body.empty? then
      @ok_button.setEnabled(false)
    else
      @ok_button.setEnabled(true)
    end
  end
end
