# Copyright (c) 2008 Hideki Ikemoto
#
# Permission is hereby granted, free of charge, to any person obtaining
# a copy of this software and associated documentation files (the
# "Software"), to deal in the Software without restriction, including
# without limitation the rights to use, copy, modify, merge, publish,
# distribute, sublicense, and/or sell copies of the Software, and to
# permit persons to whom the Software is furnished to do so, subject to
# the following conditions:
#
# The above copyright notice and this permission notice shall be
# included in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
# LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
# OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
# WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

class Comment
  attr_accessor :num, :name, :mail, :date, :body, :id

  def to_html
    html = "<div class='res_title'><a name='res#{num}' href='#write#{num}'>#{num}</a> "
    if mail != nil && mail.length != 0 then
      html += "名前:<a href='mailto:#{mail}' title='#{mail}'><b>#{name}</b></a> : "
    else
      html += "名前:<b>#{name}</b> : "
    end
    html += "#{date}</div>"
    html += "<div class='res_body'>#{parse_body}</div>"

    html
  end

  def to_plain
    text = "#{num} 名前:#{Util::to_plain(name)} #{date}\n"
    text += Util::to_plain(@body)

    text
  end

  def parse_body
    body = @body.gsub(/h?t?tp:\/\/([-_.!~*\'()a-zA-Z0-9;\/?:\@&=+\$,%#]+)/) {|matched|
      "<a href=\"http://#{$1}\">#{matched}</a>"
    }
    body = body.gsub(/<a href="\.\.\/[^"]*\/([0-9]+)" target="_blank">([^<]+)<\/a>/) {|matched|
      "<a href=\"#res#{$1}\">#{$2}</a>"
    }
    body = body.gsub(/<a href="\.\.\/[^"]*\/([0-9]+-[0-9]+)" target="_blank">([^<]+)<\/a>/) {|matched|
      "<a href=\"#res#{$1}\">#{$2}</a>"
    }
    body
  end
end

class DatInfo
  attr_reader :thread_name

  def initialize(dat_url)
    @comments = []
    @thread_name = nil
    @dat_url = dat_url
  end

  def load
    begin
      str = Downloader::load_dat(@dat_url)
    rescue
      begin
        str = Downloader::download_dat(@dat_url)
      rescue Net::ProtoRetriableError
        str = Downloader::download_maru(@dat_url)
      end
    end

    parse(str)
  end

  def reload
    str = Downloader::download_dat(@dat_url)

    parse(str)
  end

  def parse(str)
    case Util.board_type(@dat_url)
    when Util::TYPE_2CH, Util::TYPE_UNKNOWN
      parse_2ch(str)
    when Util::TYPE_MACHIBBS
      parse_machibbs(str)
    when Util::TYPE_JBBS
      parse_jbbs(str)
    end
  end

  def parse_2ch(str)
    str.each_with_index {|line, i|
      line.chomp!
      items = line.split(/<>/)

      comment = Comment.new
      comment.num = i + 1
      comment.name = items[0]
      comment.mail = items[1]
      comment.date = items[2]
      comment.body = items[3]
      @thread_name = items[4] if items[4] != nil && i == 0

      @comments.push(comment)
    }

    save_index
  end

  def parse_machibbs(str)
    str.each_with_index {|line, i|
      line.chomp!
      items = line.split(/<>/)

      comment = Comment.new
      comment.num = items[0]
      comment.name = items[1]
      comment.mail = items[2]
      comment.date = items[3]
      comment.body = items[4]
      @thread_name = items[5] if items[5] != nil && items[5] != ""

      @comments.push(comment)
    }

    save_index
  end

  def parse_jbbs(str)
    str.each_with_index {|line, i|
      line.chomp!
      items = line.split(/<>/)

      comment = Comment.new
      comment.num = items[0]
      comment.name = items[1]
      comment.mail = items[2]
      comment.date = items[3]
      comment.body = items[4]
      @thread_name = items[5] if items[5] != nil && items[5] != ""
      comment.id = items[6]

      @comments.push(comment)
    }

    save_index
  end

  def each
    @comments.each {|comment|
      yield comment
    }
  end

  def save_index
    index_path = Util.dat_url_to_index_path(@dat_url)
    config = KDE::Config.new(index_path).group("")

    config.writeEntry("ThreadName", @thread_name) if @thread_name
    config.writeEntry("ResNum", @comments.size)
    config.sync
  end

  def body(num)
    @comments[num-1].body
  end

  def to_plain(num)
    if @comments[num-1] == nil then return "" end
    @comments[num-1].to_plain
  end
end

class DatManager
  def self.delete_dat(dat_url)
    dat_path = Util.dat_url_to_dat_path(dat_url)
    index_path = Util.dat_url_to_index_path(dat_url)

    File::delete(dat_path)
    File::delete(index_path)
  end
end
