/**************************************************************************/
/*  Klavaro - a flexible touch typing tutor                               */
/*  Copyright (C) 2005 - 2007  Felipe Castro                              */
/*                                                                        */
/*  This program is free software, licensed under the terms of the GNU    */
/*  General Public License as published by the Free Software Foundation,  */
/*  which is also included in this package, in a file named "COPYING".    */
/**************************************************************************/

/*
 * Fluidness exercise
 */
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <glib.h>
#include <glib/gstdio.h>
#include <gtk/gtk.h>

#include "support.h"
#include "main.h"
#include "translation.h"
#include "tutor.h"
#include "velocity.h"
#include "fluidness.h"

/*
 * Variables
 */
extern GtkWidget *window_tutor_;

typedef struct
{
	gchar *buffer;
	gint len;
	gchar name[21];
} Paragraph;

Paragraph par = { NULL, 0, "" };

/*******************************************************************************
 * Interface functions
 */
gchar *
fluid_get_paragraph_name ()
{
	return par.name;
}

void
fluid_reset_paragraph ()
{
	g_free (par.buffer);
	par.buffer = NULL;
	par.len = 0;
	par.name[0] = '\0';
}

/*
 * Get from the structure 'par' the paragraph defined by 'index'
 *
 */
gchar *
get_par (gint index)
{
	gint i;
	gint size;
	gchar *par_1;
	gchar *par_2;
	gchar *par_i;

	par_1 = par.buffer;
	par_2 = strchr (par_1, '\n') + 1;
	if (par_2 == NULL)
		par_2 = par_1;
	for (i = 0; i < index && i < par.len; i++)
	{
		par_1 = par_2;
		par_2 = strchr (par_1, '\n') + 1;
		if (par_2 == NULL)
			par_2 = par_1;
	}
	size = par_2 - par_1;
	if (size < 1)
	{
		g_message ("internal error while picking the paragraph %i.",
			   index);
		par_i = g_strdup_printf ("#%i\n", index);
	}
	else
		par_i = g_strndup (par_1, size);

	size = strlen (par_i);
	if (size > 0)
		par_i[size - 1] = '\n';
	return (par_i);
}

/**********************************************************************
 * Initialize the fluid exercise window.
 */
void
fluid_init ()
{
	gchar *tmp_name;
	gchar *tmp_str;
	FILE *fh;
	GtkWidget *wg;

	if (!main_preferences_exist ("tutor", "fluid_paragraphs"))
		main_preferences_set_int ("tutor", "fluid_paragraphs", 3);

	wg = lookup_widget (window_tutor_, "spinbutton_lesson");
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (wg),
				   main_preferences_get_int ("tutor",
							     "fluid_paragraphs"));

	if (par.len == 0)
	{
		if (main_preferences_exist ("tutor", "paragraph_list"))
		{
			tmp_str =
				main_preferences_get_string ("tutor",
							     "paragraph_list");
			tmp_name =
				g_strconcat (main_get_user_dir (), tmp_str,
					     ".paragraphs", NULL);
			if ((fh = (FILE *) g_fopen (tmp_name, "r")))
			{
				fluid_init_paragraph_list (tmp_str);
				fclose (fh);
			}
			g_free (tmp_str);
			g_free (tmp_name);
		}
	}
	if (par.len == 0)
		fluid_init_paragraph_list (NULL);
}

/**********************************************************************
 * Reads paragraphs from the text file.
 */
void
fluid_init_paragraph_list (gchar * list_name)
{
	guint len;
	gchar *memory_ok;
	gchar *tmp_name;
	gchar *tmp_code;
	gchar str_1000[1001];
	FILE *fh;

	if (list_name && strcmp (list_name, _("--> Default")) != 0)
	{
		main_preferences_set_string ("tutor", "paragraph_list",
					     list_name);
		tmp_name =
			g_strconcat (main_get_user_dir (), list_name,
				     ".paragraphs", NULL);
		g_message ("loading text file: %s.paragraphs", list_name);
		strncpy (par.name, list_name, 20);
		par.name[20] = '\0';
	}
	else
	{
		main_preferences_remove ("tutor", "paragraph_list");
		tmp_code =
			main_preferences_get_string ("interface", "language");
		tmp_name =
			g_strconcat (main_get_data_path (), tmp_code,
				     ".paragraphs", NULL);
		g_message ("loading text file: %s.paragraphs", tmp_code);
		strcpy (par.name, "Default");
		g_free (tmp_code);
	}

	fh = (FILE *) g_fopen (tmp_name, "r");
	if (fh == NULL && strcmp (par.name, "Default") == 0)
		fh = trans_lang_get_similar_file (".paragraphs");

	if (fh)
	{
		g_free (par.buffer);
		par.buffer = g_strdup ("");
		par.len = 0;
		g_print ("Paragraphs:\n0");
		while (fgets (str_1000, 1001, fh))
		{
			if ((len = strlen (str_1000)) < 2)
				continue;
			len += strlen (par.buffer) + 1;
			memory_ok = g_try_renew (gchar, par.buffer, len);
			if (memory_ok)
				par.buffer = memory_ok;
			else
			{
				g_print ("\nThere was truncation: memory restrictions...");
				break;
			}
			strcat (par.buffer, str_1000);
			par.len++;
			g_print (" - %i", par.len);
		}
		fclose (fh);
		g_print ("\nText file loaded!\n\n");
	}
	else
	{
		g_message ("could not open the file: %s", tmp_name);
		g_free (tmp_name);
		tmp_code =
			main_preferences_get_string ("interface", "language");
		if (strcmp (tmp_code, "C") == 0)
			g_error ("so, we must quit!");
		main_preferences_set_string ("interface", "language", "C");
		fluid_init_paragraph_list (list_name);
		main_preferences_set_string ("interface", "language",
					     tmp_code);
		g_free (tmp_code);
		return;
	}
	g_free (tmp_name);
}

/**********************************************************************
 * Draw random sentences selected from a '.paragraphs' file
 */
void
fluid_draw_random_paragraphs ()
{
	gint i, j;
	gint par_num;
	gint rand_i[10];
	static gchar *text[10] =
		{ NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
		NULL
	};

	par_num = main_preferences_get_int ("tutor", "fluid_paragraphs");

	for (i = 0; (i < par_num) && (i < par.len); i++)
		g_free (text[i]);

	for (i = 0; (i < par_num) && (i < par.len); i++)
	{
		do
		{
			rand_i[i] = rand () % par.len;
			for (j = 0; j < i; j++)
			{
				if (rand_i[i] == rand_i[j])
					rand_i[i] = par.len;
			}
		}
		while (rand_i[i] == par.len);

		text[i] = get_par (rand_i[i]);
		tutor_draw_paragraph (text[i]);
	}
}

/**********************************************************************
 * Copy the file 'file_name' so that it can be used as a customized
 * exercise.
 */
void
fluid_copy_text_file (gchar * file_name)
{
	gchar tmp_str[MAX_PAR_LEN];
	gint nlin;
	gboolean empty_file;
	gchar *tab;
	gchar *pars_path;
	gchar *pars_name;
	FILE *fh_source;
	FILE *fh_destiny;

	if (!file_name)
	{
		gdk_beep ();
		g_warning
			("fluid_copy_text_file(): null file name as argument.");
		return;
	}

	if (!(fh_source = (FILE *) g_fopen (file_name, "r")))
	{
		gdk_beep ();
		g_warning ("couldn't open the file:\n %s\n", file_name);
		return;
	}

	pars_name = g_strdup (strrchr (file_name, DIRSEP) + 1);
	pars_path =
		g_strconcat (main_get_user_dir (), pars_name, ".paragraphs",
			     NULL);

	assert_user_dir ();
	if (!(fh_destiny = (FILE *) g_fopen (pars_path, "w")))
	{
		gdk_beep ();
		g_warning ("couldn't create the file:\n %s\n", pars_path);
		fclose (fh_source);
		g_free (pars_path);
		g_free (pars_name);
		return;
	}
	g_free (pars_path);

	/********************
	 * Process the file
	 */
	empty_file = TRUE;
	nlin = 0;
	g_print ("Processing file:\n %s\nParagraphs:\n0", file_name);
	while (fgets (tmp_str, MAX_PAR_LEN, fh_source) != NULL)
	{
		if (nlin++ > MAX_PARAGRAPHS)
		{
			g_print (" - truncated!");
			break;
		}
		g_print (" - %u", nlin);

		if (g_utf8_validate (tmp_str, -1, NULL) == FALSE)
		{
			g_print (": NOT UTF8!");
			continue;
		}
		if (strcmp (tmp_str, "\n") == 0)
		{
			g_print (": BLANK!");
			continue;
		}

		tab = tmp_str;
		while ((tab = strchr (tab, '\t')))
			*tab = ' ';

		fputs (tmp_str, fh_destiny);
		empty_file = FALSE;
	}
	if (empty_file)
		fprintf (fh_destiny, "%i.\n", rand () % 9000);
	else if (tmp_str[strlen (tmp_str) - 1] != '\n')
		fputc ('\n', fh_destiny);

	fclose (fh_source);
	fclose (fh_destiny);

	g_print (" - end.\n");

	fluid_init_paragraph_list (pars_name);
	tutor_set_query (QUERY_INTRO);
	tutor_process_touch ('\0');

	velo_create_dict (file_name, FALSE);
	g_free (pars_name);
}

/**********************************************************************
 * Put on the screen the final comments
 */
void
fluid_comment (gdouble accuracy, gdouble velocity, gdouble fluidness)
{
	gchar str[1000];
	GtkWidget *wg;
	GtkTextBuffer *buf;

	/*
	 * Comments
	 */
	if (accuracy < 98)
		sprintf (str,
			 _(" Your accuracy rate is too low.\n"
			   " I want it greater than 98 %%.\n"));
	else if (velocity < 50)
		sprintf (str, _(" You type accurately but not fast.\n"
				" Can you reach 50 WPM?\n"));
	else if (fluidness < 70)
		sprintf (str,
			 _(" Your rhythm is not so constant. Calm down.\n"
			   " For now, try to make the fluidness greater than 70 %%.\n"));
	else if (fluidness < 90)
		sprintf (str,
			 _
			 (" You are almost getting there. Type more fluently.\n"
			  " I want a fluidness greater than 90 %%.\n"));
	else if (velocity < 80)
		sprintf (str,
			 _(" Congratulations!\n"
			   " It seems to me that you are a professional.\n"
			   " You don't need this program (me) anymore.\n"
			   " Hope you have enjoyed. Thanks and be happy!\n"));
	else
		sprintf (str,
			 _(" How can you type so fast?\n"
			   " You have exceeded all my expectations.\n"
			   " Are you a machine? Could you teach me?\n"
			   " I can not help you anymore. Go to an expert!\n"));

	wg = lookup_widget (window_tutor_, "text_tutor");
	buf = gtk_text_view_get_buffer (GTK_TEXT_VIEW (wg));
	gtk_text_buffer_insert_at_cursor (buf, str, strlen (str));
}
