/**************************************************************************/
/*  Klavaro - a flexible touch typing tutor                               */
/*  Copyright (C) 2005 - 2007  Felipe Castro                              */
/*                                                                        */
/*  This program is free software, licensed under the terms of the GNU    */
/*  General Public License as published by the Free Software Foundation,  */
/*  which is also included in this package, in a file named "COPYING".    */
/**************************************************************************/

/*
 * Klavaro is a touch typing tutor and exercise program.
 * Copyright (C) 2007 - Felipe E. F. de Castro
 */
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <glib.h>
#include <glib/gstdio.h>
#include <gtk/gtk.h>

#include "support.h"
#include "interface.h"
#include "translation.h"
#include "keyboard.h"
#include "tutor.h"
#include "main.h"

/*******************************************************************************
 * Variables
 */
GtkWidget *window_main_;

GKeyFile *preferences = NULL;

struct
{
	gchar *user_dir;
	gchar *data_path;
} general;


/*******************************************************************************
 * Interface functions
 */
gchar *
main_get_data_path ()
{
	return (general.data_path);
}

void
main_set_data_path (gchar * path)
{
	g_free (general.data_path);
	general.data_path = g_strdup (path);
}

gchar *
main_get_user_dir ()
{
	return (general.user_dir);
}

gboolean
main_preferences_exist (gchar * group, gchar * key)
{
	return (g_key_file_has_key (preferences, group, key, NULL));
}

void
main_preferences_remove (gchar * group, gchar * key)
{
	g_key_file_remove_key (preferences, group, key, NULL);
}

gchar *
main_preferences_get_string (gchar * group, gchar * key)
{
	return (g_key_file_get_string (preferences, group, key, NULL));
}

void
main_preferences_set_string (gchar * group, gchar * key, gchar * value)
{
	g_key_file_set_string (preferences, group, key, value);
}

gint
main_preferences_get_int (gchar * group, gchar * key)
{
	return (g_key_file_get_integer (preferences, group, key, NULL));
}

void
main_preferences_set_int (gchar * group, gchar * key, gint value)
{
	g_key_file_set_integer (preferences, group, key, value);
}

gboolean
main_preferences_get_boolean (gchar * group, gchar * key)
{
	return (g_key_file_get_boolean (preferences, group, key, NULL));
}

void
main_preferences_set_boolean (gchar * group, gchar * key, gboolean value)
{
	g_key_file_set_boolean (preferences, group, key, value);
}

/*******************************************************************************
 * Program
 */
int
main (int argc, char *argv[])
{
#ifdef ENABLE_NLS
	bindtextdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);
#endif

	gtk_disable_setlocale ();	/* The locale will be set according to user preference... */
	gtk_init (&argc, &argv);

	add_pixmap_directory (PACKAGE_DATA_DIR DIRSEP_S PACKAGE DIRSEP_S
			      "pixmaps");
	add_pixmap_directory (".." DIRSEP_S "pixmaps");

	main_initialize_global_variables ();	/* Here the locale is set. */
	window_main_ = create_window_main ();	/* So we must postpone gtk function calls */
	main_window_init ();	/* and initialize its parameters only as a third step... */

	gtk_widget_show (window_main_);

	gtk_main ();

	return 0;
}

/**********************************************************************
 * Temp function to preserve compatibility with the old "preferences.desktop" file,
 * used before version 1.0.3
 */
void
change_oldprefs ()
{
	gchar *tmp_old = NULL;
	gchar *tmp_new = NULL;

	tmp_old = g_strconcat (general.user_dir, "preferences.desktop", NULL);
	if (g_file_test (tmp_old, G_FILE_TEST_EXISTS))
	{
		tmp_new =
			g_strconcat (general.user_dir, "preferences.ini",
				     NULL);
		g_rename (tmp_old, tmp_new);
		g_free (tmp_new);
	}
	g_free (tmp_old);
}

/**********************************************************************
 * Initialize the value of the global variables
 */
void
main_initialize_global_variables ()
{
	gchar *tmp_name;
	FILE *fh;

	/*
	 * Set the home user dir 
	 */
	general.user_dir =
		g_strconcat (g_get_home_dir (), DIRSEP_S ".klavaro" DIRSEP_S,
			     NULL);
	/*
	 * Get a valid data path.
	 * First searches the path at the source directory.
	 */
	general.data_path = g_strconcat (".." DIRSEP_S "data" DIRSEP_S, NULL);
	tmp_name = g_strconcat (general.data_path, "basic_lessons.txt", NULL);
	fh = (FILE *) g_fopen (tmp_name, "r");
	if (fh == NULL)
	{
		g_free (general.data_path);
		g_free (tmp_name);
		general.data_path =
			g_strdup (PACKAGE_DATA_DIR DIRSEP_S PACKAGE DIRSEP_S);
		tmp_name =
			g_strconcat (general.data_path, "basic_lessons.txt",
				     NULL);
		fh = (FILE *) g_fopen (tmp_name, "r");
	}
	if (fh == NULL)
		g_error ("couldn't find a data file at the data path:\n %s",
			 tmp_name);
	fclose (fh);
	g_free (tmp_name);

	change_oldprefs ();

	/*
	 * Retrieve initial or saved preferences
	 */
	preferences = g_key_file_new ();
	tmp_name = g_strconcat (general.user_dir, "preferences.ini", NULL);
	if (g_file_test (tmp_name, G_FILE_TEST_EXISTS))
		g_key_file_load_from_file (preferences, tmp_name,
					   G_KEY_FILE_NONE, NULL);
	else
		g_key_file_load_from_file (preferences,
					   "/etc/klavaro_preferences.ini",
					   G_KEY_FILE_NONE, NULL);
	g_free (tmp_name);

	/*
	 * Apply the language environmental value
	 */
	trans_init_language_env ();

	/*
	 * Other initializations
	 */
	srand (time (0));

	tutor_init_timers ();
}

/*******************************************************************************
 * Initialize keyboard entry and language combo 
 */
void
main_window_init ()
{
	gchar *tmp_str;
	GtkWidget *wg;

	/*
	 * Set the initial keyboard to use
	 */
	wg = lookup_widget (window_main_, "entry_keyboard");
	if (!main_preferences_exist ("tutor", "keyboard"))
	{
		gtk_entry_set_text (GTK_ENTRY (wg), DEFAULT_KEYBOARD);
		g_key_file_set_string (preferences, "tutor", "keyboard",
				       DEFAULT_KEYBOARD);
	}
	else
	{
		tmp_str =
			g_key_file_get_string (preferences, "tutor",
					       "keyboard", NULL);
		gtk_entry_set_text (GTK_ENTRY (wg), tmp_str);
		g_free (tmp_str);
	}

	keyb_init_name (gtk_entry_get_text (GTK_ENTRY (wg)));
	keyb_get_layout ();

	/*
	 * Set the language
	 */
	trans_set_combo_language ();
}

/*******************************************************************************
 * Quit the application
 */
void
main_window_pass_away ()
{
	gchar *tmp_name;
	FILE *fh;

	/*
	 * Save preferences
	 */
	assert_user_dir ();
	tmp_name = g_strconcat (general.user_dir, "preferences.ini", NULL);
	fh = (FILE *) g_fopen (tmp_name, "w");
	if (fh == NULL)
		g_warning
			("couldn't save your preferences at the user folder:\n %s",
			 tmp_name);
	else
	{
		g_free (tmp_name);
		tmp_name = NULL;
		tmp_name = g_key_file_to_data (preferences, NULL, NULL);
		if (tmp_name != NULL)
			fputs (tmp_name, fh);
		fclose (fh);
	}
	g_free (tmp_name);

	/*
	 * Good bye
	 */
	g_message ("adiaux!");
	gtk_exit (0);
}

/**********************************************************************
 * General functions
 */
/**********************************************************************
 * Search for the user directory and create it if not found
 */
void
assert_user_dir ()
{
	GDir *dh;

	dh = g_dir_open (main_get_user_dir (), 0, NULL);
	if (dh == NULL)
	{
		g_message ("creating an empty user folder:\n %s",
			   main_get_user_dir ());

#ifdef G_OS_UNIX
		g_mkdir (main_get_user_dir (),
			 S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH);
#endif

#ifdef G_OS_WIN32
		g_mkdir (main_get_user_dir (), 0xFFFF);
#endif

		dh = g_dir_open (main_get_user_dir (), 0, NULL);
		if (dh == NULL)
			g_error ("could not creat a user folder, so we must quit!");
	}
	g_dir_close (dh);
}

/**********************************************************************
 * Compare two strings, so that it applies to other sorting functions.
 */
gint
comp_str (gconstpointer a, gconstpointer b)
{
	return (strcasecmp (a, b));
}
