#ifndef __CALC_STACK_CLASS__
#define __CALC_STACK_CLASS__

#include <iostream>
#include <vector>
#include <qstring.h>

using namespace std;

// ----------------------------------
// defines all known klogic operators
// ----------------------------------
class Operator
{
public:
	static bool isOP(char c)
	{
		return (c == AND || c == OR || c == XOR ||
			c == NOT || c == BIN || c == BIT ||
			c == NONE || c == INTERFACE);
	}

	static bool isReserved(char c)
	{
		return (c == DELIMITER || c == BRACKET || c == BRACKETC || c == 0 || isOP(c));
	}

	// return type of operation:
	//   OP_LR operator needs left and right operand
	//   OP_R operator needs right operand
	static int opType(char op)
	{
		switch(op) {
			case AND:
			case OR:
			case XOR:
			case BIN:
			case BIT:
				return TYPE_LR;
				break;
			case NOT:
				return TYPE_R;
				break;
			case BRACKET:
				return TYPE_PARENTHESIS;
		}
		return UNDEF;
	}

	static int containsOp(const char *str)
	{
		for (unsigned int i = 0; i < strlen(str); ++i) {
			if (isOP(str[i]))
				return i;
		}
		return NOOP;
	}

	// operations
	static const char DELIMITER = ' ';
	static const char BRACKET = '(';
	static const char BRACKETC = ')';
	static const char AND = '&';
	static const char OR = '+';
	static const char XOR = '#';
	static const char NOT = '/';
	static const char BIN = '';
	static const char BIT = '%';
	static const char NONE = '\10';
	static const char INTERFACE = '\11';

	// operation types
	enum OperatorType {UNDEF, TYPE_PARENTHESIS, TYPE_LR, TYPE_R };

	static const int NOOP = -1;
};

// ----------------------------------
// Calculator parsing result
// ----------------------------------
class QueueInfo
{
public:
	QueueInfo()
	{
		clear();
	}

	void clear()
	{
		m_cOperator = 0;
		m_iOutputID = 0;
		m_iValue = 0;
		m_iType = VAL_NOTYPE;
		m_iPos = 0;
		m_iLen = 0;

		m_iError = NOERR;
	}

	void setOperator(const QString& cOperator)
	{
		char c = cOperator[0].latin1();
		if (Operator::isOP(c))
			m_cOperator = c;
		else
			m_cOperator = 0;
	}

	enum ValType { VAL_NOTYPE, VAL_INPUT = 1, VAL_NUMBER, VAL_OUTPUT, VAL_THIS };
	enum ErrType { NOERR, ERR_NTERM, ERR_PARENTHESIS };

	QString m_sName;
	char m_cOperator;
	QString m_sOutputName;
	int m_iOutputID;
	int m_iValue;
	ValType m_iType;
	int m_iPos;
	int m_iLen;

	ErrType m_iError;
};

// ----------------------------------
// Calculator parsing results
// ----------------------------------
class CalcRules
{
public:
	CalcRules()
	{
		m_cStaticOperator = 0;
	}

	inline unsigned int push(const QueueInfo& oInfo)
	{
		m_vQueueInfo.push_back(oInfo);
		return m_vQueueInfo.size() - 1;
	}

	inline QueueInfo& operator [] (int idx)
	{
		return m_vQueueInfo[idx];
	}

	inline const char getOperator(int idx) const
	{
		if (!m_cStaticOperator)
			return m_vQueueInfo[idx].m_cOperator;
		return m_cStaticOperator;
	}

	inline const QString& getItem(int idx) const
	{
		return m_vQueueInfo[idx].m_sName;
	}

	inline bool checkStatic(const QString& str)
	{
		m_cStaticOperator = 0;
		if (str.length() != 1)
			return false;

		char c = str[0].latin1();
		if (!Operator::isOP(c))
			return false;

		m_cStaticOperator = c;
		return true;
	}

	inline char isStatic() const
	{
		return m_cStaticOperator;
	}

	inline const int size() const
	{
		return m_vQueueInfo.size();
	}

	inline void dump() const
	{
		int sz = size();
		for (int i = 0; i < sz; ++i)
			cout << (const char *)m_vQueueInfo[i].m_sName << " at offset " << m_vQueueInfo[i].m_iPos << m_vQueueInfo[i].m_iLen << endl;
	}

	inline void clear()
	{
		m_vQueueInfo.clear();
		m_cStaticOperator = 0;
	}


	static QString ITEM_THIS;

private:
	char m_cStaticOperator;
	vector<QueueInfo> m_vQueueInfo;
};

#endif
