/*************************************************/
/* member methods for class Calculator           */
/*                                               */
/* equation analyzer and calculator              */
/*                                               */
/* Andreas Rostin                                */
/* 15.10.99                                      */
/*************************************************/
#include <qstring.h>
#include <qvalidator.h>

#include <device.h>
#include <tool.h>

#include <calculator.h>

int Calculator::instance = 0;
QIntValidator Calculator::intValidator(0, 64, 0);

Calculator::Calculator(Device *dev, int undef, int delay)
	:DelayedValue(delay)
{
	instance++;
	setUndefinedValue(undef);
	m_dev = dev;
	m_iStackItems = 0;
	m_iStackSize = 20;
	m_poStack = new int[m_iStackSize];
}

Calculator::Calculator()
	:DelayedValue(0)
{
	instance++;
	m_dev = (Device *)NULL;
	m_iStackItems = 0;
}

Calculator::~Calculator()
{
	instance--;
}

// set a new equation
void Calculator::setEquation(char equation)
{
	m_sEquation = equation;
}

// set a new equation
void Calculator::setEquation(const QChar& equation)
{
	m_sEquation = equation;
}

// set a new equation
void Calculator::setEquation(const QString& sEquation)
{
	m_sEquation = sEquation;
	// replace double-inversions by spaces
	m_sEquation.replace("//", "");
}

// an empty equation has a content
int Calculator::hasEquation()
{
	if (m_sEquation.length() && m_sEquation[0] != Operator::NONE) return 1;
	return 0;
}

const QString & Calculator::getEquation() const
{
	return m_sEquation;
}

// build equation with all non-terminal qualified by a string
QString Calculator::getEquation(const QString& sQualifier) const
{
	QString sResultingEquation;
	bool bNonTerminalCanStart = true;

	unsigned int idx = 0;
	while(idx < m_sEquation.length()) {
		bool bThisFound = false;
		if (!Operator::isReserved(m_sEquation[idx].latin1()) && bNonTerminalCanStart) {
			if (m_sEquation.mid(idx, CalcRules::ITEM_THIS.length()) == CalcRules::ITEM_THIS) {
				if (sQualifier.length()) {
					sResultingEquation.append(sQualifier.left(sQualifier.length() - 1)); // remove dot
					bNonTerminalCanStart = true;
					bThisFound = true;
				} else {
					sResultingEquation.append(CalcRules::ITEM_THIS);
				}
				idx += CalcRules::ITEM_THIS.length();
			} else {
				sResultingEquation.append(sQualifier);
			}
			bNonTerminalCanStart = false;
		} else {
			bNonTerminalCanStart = true;
		}

		if (!bThisFound) {
			sResultingEquation.append(m_sEquation[idx]);
			idx++;
		}
	}
	return sResultingEquation;
}

bool Calculator::isStatic()
{
	return m_oCalcRules.isStatic();
}

// debugging method!
void Calculator::stackDump()
{
	printf("\n**********\n");
	printf("** Equation: %s\n", (const char *)m_sEquation);
	m_oCalcRules.dump();
	printf("**********\n");
}

bool Calculator::isNumber(const QString& str)
{
	int i = 0;
	return QValidator::Acceptable == intValidator.validate((QString&)str, i);
}

// public parse method
// transform equation into reverse polish notation and create a calc stack
const QueueInfo& Calculator::parse()
{
	m_oParseInfo.clear();
	m_oCalcRules.clear();

	if (m_sEquation.length() == 0)
		return m_oParseInfo;
	if (m_oCalcRules.checkStatic(m_sEquation))
		return m_oParseInfo;

	return parse(m_sEquation, 0);
}

// parse the equation and resolve it
// push it to the computation stack in RPN
const QueueInfo& Calculator::parse(const char *equation, int current_offset)
{
	char *op;		// operation
	char *left;		// left operand
	int left_length;	// length of left operand
	char *right;		// right operand
	int right_length;	// length of right operand
	int new_offset;		// new current offset relative to the original equation

	char *sEQ = new char[strlen(equation) + 1];
	strcpy(sEQ, equation);

	// find the next operator/bracket
	while (NULL != (op = getOP(sEQ))) {
		int iOpType = Operator::opType(*op);

		// parenthesis found
		if (iOpType == Operator::TYPE_PARENTHESIS) {
			char *cBracketStart = op;
			int iBracketLen = getBracket(op);
			if (iBracketLen < 0) {
				m_oParseInfo.m_iError = QueueInfo::ERR_PARENTHESIS;
				m_oParseInfo.m_iPos = cBracketStart - sEQ + 1 + current_offset;
				m_oParseInfo.m_iLen = 1;
				return m_oParseInfo;
			}
			*(cBracketStart + iBracketLen) = '\0';

			// if parenthesis contains no operands, drop it
			if (Operator::containsOp(cBracketStart + 1) == Operator::NOOP) {
				*cBracketStart = ' ';
				*(cBracketStart + iBracketLen) = ' ';
				continue;
			}

			new_offset = current_offset + (cBracketStart  + 1 - sEQ);
			const QueueInfo& oQI = parse(cBracketStart + 1, new_offset);
			if (oQI.m_iError != QueueInfo::NOERR)
				return oQI;

			// replace the complete bracket with spaces
			memset(cBracketStart, ' ', iBracketLen + 1);
			continue;
		}

		// determine operands and their length
		if (iOpType == Operator::TYPE_LR) {
			left = op - 1;
			left_length = getLeft(sEQ, &left);
		} else left_length = 0;
		right = op + 1;
		right_length = getRight(sEQ + strlen(sEQ), &right);

		// push operands to the calculation stack
		if (left_length) {
			int iPos = pushOperand(left, left_length, current_offset + (left - sEQ));
			if (m_oCalcRules[iPos].m_iError != QueueInfo::NOERR)
				return m_oCalcRules[iPos];
		}
		if (right_length) {
			int iPos = pushOperand(right, right_length, current_offset + (right - sEQ));
			if (m_oCalcRules[iPos].m_iError != QueueInfo::NOERR)
				return m_oCalcRules[iPos];
		}

		// push operation to the calculation stack
		pushOperation(op, 1,  current_offset + (op - sEQ));

		// replace the operation and its operands with spaces
		*op = ' ';
		if (right_length)
			memset(right, ' ', right_length);
		if (left_length)
			memset(left, ' ', left_length);
	}
	// find operand without operator
	right = sEQ;
	right_length = getRight(sEQ + strlen(sEQ), &right);
	if (right_length) {
		int iPos = pushOperand(right, right_length, current_offset + (right - sEQ));
		if (m_oCalcRules[iPos].m_iError != QueueInfo::NOERR)
			return m_oCalcRules[iPos];
	}

	delete sEQ;
	return m_oParseInfo;
}

// returns length of the bracket
int Calculator::getBracket(char *__equation)
{	char *start = __equation;
	char *end = start;

	while (NULL != (end = strchr(end + 1, ')'))) {
		start = strchr(start + 1, '(');
		if (!start || start > end) {
			return end - __equation;
		}
	}
	return -1;
}

// return a pointer to the next operation
char * Calculator::getOP(char *equation)
{	char *pt0, *pt1, *pt2, *pt3, *pt4, *pt5, *pt6;
	char *bet;

	pt0 = strchr(equation, Operator::BRACKET);
	pt1 = strchr(equation, Operator::NOT);
	pt2 = strchr(equation, Operator::XOR);
	pt3 = strchr(equation, Operator::AND);
	pt4 = strchr(equation, Operator::OR);
	pt5 = strchr(equation, Operator::BIN);
	pt6 = strchr(equation, Operator::BIT);

	// find two operators to the left for which is valid: op1 left of op2 left of op0, then return op1
	// (numbers: precedency!)
	if (pt0) {
		if (pt1 && pt1 < pt0) {
			bet = strchr(pt1, Operator::XOR);
			if (pt1 < bet && bet < pt0) return pt1;
			bet = strchr(pt1, Operator::AND);
			if (pt1 < bet && bet < pt0) return pt1;
			bet = strchr(pt1, Operator::OR);
			if (pt1 < bet && bet < pt0) return pt1;
		}
		if (pt2 && pt2 < pt0) {
			bet = strchr(pt2, Operator::AND);
			if (pt2 < bet && bet < pt0) return pt2;
			bet = strchr(pt2, Operator::OR);
			if (pt2 < bet && bet < pt0) return pt2;
		}
		if (pt3 && pt3 < pt0) {
			bet = strchr(pt3, Operator::OR);
			if (pt3 < bet && bet < pt0) return pt3;
		}
		return pt0;
	}

	if (pt1) {
		if (pt2 && pt2 < pt1) {
			bet = strchr(pt2, Operator::AND);
			if (pt2 < bet && bet < pt1) return pt2;
			bet = strchr(pt2, Operator::OR);
			if (pt2 < bet && bet < pt1) return pt2;
		}
		if (pt3 && pt3 < pt1) {
			bet = strchr(pt3, Operator::OR);
			if (pt3 < bet && bet < pt1) return pt3;
		}
		return pt1;
	}

	if (pt2) {
		if (pt3 && pt3 < pt2) {
			bet = strchr(pt3, Operator::OR);
			if (pt3 < bet && bet < pt1) return pt3;
		}
		return pt2;
	}

	if (pt3) return pt3;
	if (pt4) return pt4;

	// special internal operators without interaction
	if (pt5) return pt5;
	return pt6;
}

// sets pointer to the start of the left operand nonterminal
// returns length of the nonterminal
int Calculator::getLeft(char *start, char **left)
{	int length = 0;

	// jump over spaces
	while (**left == ' ' && *left > start)
		(*left)--;

	// reverse find next space, operand, ( or )
	while (	*left >= start && !Operator::isReserved(**left)) {
		(*left)--;
		length++;
	}
	if (length) (*left)++;

	return length;
}

// sets pointer to the start of the right operand nonterminal
// returns length of the nonterminal
int Calculator::getRight(char *end, char **right)
{	int length = 0;
	char *pt;

	// jump over spaces
	while (**right == ' ' && *right < end)
		(*right)++;

	pt = *right;
	// find next space, operand, ( or )
	while (	pt <= end && !Operator::isReserved(*pt)) {
		pt++;
		length++;
	}

	return length;
}

// add operator to rules
void Calculator::pushOperation(const QString& str, int length, int current_offset)
{
	QString sOperator;
	sOperator.setAscii(str, length);

	QueueInfo oInfo;
	oInfo.setOperator(sOperator);
	oInfo.m_iPos = current_offset;
	oInfo.m_iLen = length;
	m_oCalcRules.push(oInfo);
}

// add operand to rules
int Calculator::pushOperand(const QString& str, int length, int current_offset)
{
	QString sOperand;
	sOperand.setAscii(str, length);

	QueueInfo oInfo;
	oInfo.m_sName = sOperand;
	oInfo.m_iPos = current_offset;
	oInfo.m_iLen = length;

	if (m_dev) {
		const Connector *poConnector;
		// check if item exists as named input, output, const or itself
		if (sOperand == CalcRules::ITEM_THIS) {
			oInfo.m_sOutputName = m_dev->getName();
			oInfo.m_iOutputID = m_dev->getID();
			oInfo.m_iType = QueueInfo::VAL_THIS;	// own output
		} else if (NULL != (poConnector = m_dev->getInputConnector(sOperand))) {
			oInfo.m_sOutputName = poConnector->getName();
			oInfo.m_iOutputID = poConnector->getConnectorID();
			oInfo.m_iType = QueueInfo::VAL_INPUT;
		} else if (NULL != (poConnector = m_dev->getOutputConnector(sOperand))) {
			oInfo.m_sOutputName = poConnector->getName();
			oInfo.m_iOutputID = poConnector->getConnectorID();
			oInfo.m_iType = QueueInfo::VAL_OUTPUT;
		} else if (isNumber(sOperand)) {
			oInfo.m_iValue = sOperand.toInt();
			oInfo.m_iType = QueueInfo::VAL_NUMBER;
		} else {
			oInfo.m_iError = QueueInfo::ERR_NTERM;
		}
	}
	return m_oCalcRules.push(oInfo);
}

// push an input value to the stack
void Calculator::push(int val)
{
	checkStackSize(m_iStackItems);
	if (val != Global::Device::TRISTATE) {
		m_poStack[m_iStackItems] = val;
		m_iStackItems++;
	} else {
		if (m_oCalcRules.isStatic() && m_oCalcRules.getOperator(0) == Operator::INTERFACE) {
			m_poStack[m_iStackItems] = val;
			m_iStackItems++;
		}
	}
}

int Calculator::getInputValue(int idx)
{
	if (m_oCalcRules[idx].m_iType == QueueInfo::VAL_THIS)
		return m_dev->output();
	else if (m_oCalcRules[idx].m_iType == QueueInfo::VAL_OUTPUT)
		return m_dev->output(m_oCalcRules[idx].m_iOutputID);
	else if (m_oCalcRules[idx].m_iType == QueueInfo::VAL_INPUT)
		return m_dev->input(m_oCalcRules[idx].m_iOutputID);
	else if (m_oCalcRules[idx].m_iType == QueueInfo::VAL_NUMBER)
		return m_oCalcRules[idx].m_iValue;
	return undefined_result;
}

// stack calculation
// returns the result
// changed in 12-02-2002: _NO_ tristate values at the stack anymore!!
int Calculator::calculate(bool tristate_active)
{	int bin_pow = 1;
	bool hadTristate = false;

	int iPlane = m_iStackItems - 1;
	m_iStackItems = 0;

	if (tristate_active && isTristate()) {
		setValue(Global::Device::TRISTATE);
		return Global::Device::TRISTATE;
	}

	setValue(undefined_result);

	unsigned int iTotal;
	if (m_oCalcRules.isStatic())
		iTotal = iPlane + 1;
	else
		iTotal = m_oCalcRules.size();
	if (iTotal <= 0) return undefined_result;		// nothing to do

	for (unsigned int i = 0; i < iTotal; ++i) {
		switch (m_oCalcRules.getOperator(i)) {
			case Operator::AND:
				if (!iPlane) break;
				m_poStack[iPlane - 1] = m_poStack[iPlane - 1] & m_poStack[iPlane];
				iPlane--;
				break;
			case Operator::OR:
				if (!iPlane) break;
				m_poStack[iPlane - 1] = m_poStack[iPlane - 1] | m_poStack[iPlane];
				iPlane--;
				break;
			case Operator::XOR:
				if (!iPlane) break;
				m_poStack[iPlane - 1] = m_poStack[iPlane - 1] ^ m_poStack[iPlane];
				iPlane--;
				break;
			case Operator::BIN:
				if (!iPlane) break;
				m_poStack[iPlane - 1] += m_poStack[iPlane] * Tool::pow2(bin_pow);
				bin_pow++;
				iPlane--;
				break;
			case Operator::BIT:
				if (!iPlane) break;
				m_poStack[iPlane - 1] = (m_poStack[iPlane - 1] & Tool::pow2(m_poStack[iPlane])) > 0;
				iPlane--;
				break;
			case Operator::NOT:
				if (m_poStack[iPlane] == 1) m_poStack[iPlane] = 0;
				else m_poStack[iPlane] = 1;
				break;
			case Operator::NONE:
				break;
			case Operator::INTERFACE:
				// use OR for interfaces; propagate TRISTATE
				// but do no calculation with TRISTATE values!
				if (!iPlane) break;
				if (m_poStack[iPlane - 1] == Global::Device::TRISTATE)
					m_poStack[iPlane - 1] = m_poStack[iPlane];
				if (m_poStack[iPlane] != Global::Device::TRISTATE && m_poStack[iPlane - 1] != Global::Device::TRISTATE)
					m_poStack[iPlane - 1] |= m_poStack[iPlane];
				else
					hadTristate = true;
				iPlane--;
				if (iPlane == 0 && m_poStack[0] == Global::Device::TRISTATE)
					hadTristate = true;
				break;
			default:
				iPlane++;
				checkStackSize(iPlane);
				m_poStack[iPlane] = getInputValue(i);
				if (m_poStack[iPlane] == Global::Device::TRISTATE)
					hadTristate = true;
				break;
		}
	}

	if (iPlane != 0) {
		if (iPlane > 0) {
			// should never happen
			warning("Calculator::calculate: stack iPlane is not zero!\n");
			return undefined_result;
		} else if (hadTristate) {
			// special case: interface m_dev
			if (m_oCalcRules.isStatic() == Operator::INTERFACE) {
				setValue(Global::Device::TRISTATE);
				return Global::Device::TRISTATE;
			}
			setValue(undefined_result);
			return undefined_result;
		} else {
			// should never happen
			warning("Calculator::calculate: stack iPlane is not zero!\n");
			return undefined_result;
		}
	}

	int iVal = m_poStack[iPlane];
	setValue(iVal);
	return iVal;
}

void Calculator::checkStackSize(int iNeededSize)
{
	if ((iNeededSize + 1) >= m_iStackSize) {
		int *poStack = new int[m_iStackSize + 10];
		memcpy(poStack, m_poStack, m_iStackSize * sizeof(int));
		m_poStack = poStack;
		m_iStackSize += 10;
	}
}

unsigned int Calculator::getCount() const
{
	return m_oCalcRules.size();
}
