/*******************************************************/
/* methods for a net-widget                            */
/*                                                     */
/* this contains the current active circuit widget,    */
/* manages various modes, status, drawing and io       */
/*                                                     */
/* Andreas Rostin                                      */
/* 27.10.97                                            */
/*******************************************************/
#include <qpainter.h>
#include <qlistbox.h>
#include <qstring.h>
#include <qdir.h>
#include <qmessagebox.h>
#include <qpushbutton.h>
#include <qpopupmenu.h>
#include <qtimer.h>
#include <qwidget.h>
#include <qframe.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qdialog.h>
#include <qbuttongroup.h>
#include <qradiobutton.h>
#include <qpixmap.h>
#include <qlabel.h>
#include <qcombobox.h>

#include "mainw.h"
#include "netw.h"
#include "klogic.h"
#include "klogicIO.h"
#include "circuit.h"
#include "xdevice.h"
#include "deviceTypes.h" 
#include "selection.h"
#include "propOsz.h"
#include "propDev.h"
#include "propName.h"
#include "propPwr.h"
#include "propText.h"
#include "propSwitch.h"
#include "dlgEqu.h"
#include "xmlImport.h"
#include "grid.h"

#include "netw.moc"
//********************************************
//* static methods of NetWidget              *
//********************************************

int NetWidget::STATsimStepTime = 40;
int NetWidget::STATsimBurst = 1;
QTimer NetWidget::simTimer;
QTimer NetWidget::shortcutWarningTimer;
int NetWidget::actualDevFunc = 0;
int NetWidget::simmode = 0;

// static method
int NetWidget::simTime()
{
	return STATsimStepTime;
}

// static method
void NetWidget::setSimTime(int newi)
{
	if ((newi >= MIN_SIMSTEPTIME) && (newi <=  MAX_SIMSTEPTIME)) {
		STATsimStepTime = newi;
		if (simTimer.isActive())
			simTimer.changeInterval(STATsimStepTime);
	}
}

// static method
int NetWidget::simBurst()
{
	return STATsimBurst;
}

// static method
void NetWidget::setSimBurst(int newi)
{
	if ((newi >= MIN_SIMBURST) && (newi <=  MAX_SIMBURST))
		STATsimBurst = newi;
}

// static method
// set current object for dropping
void NetWidget::setDevice(int newdev)
{
	actualDevFunc = newdev;
}

// static method
// change between permanent and stepping simulation mode
void NetWidget::setSimMode(int newmode)
{
	simmode = newmode;

	// permanent simulation mode
	if (simmode == MODE_SIM_MULT) {
		simTimer.start(STATsimStepTime);
	}
	else {
		simTimer.stop();
	}
}

// static method
// return current simulation mode
int NetWidget::getSimMode()
{
	return simmode;
}

//********************************************
//* methods of NetWidget                     *
//********************************************
NetWidget::NetWidget(QWidget *hparent, QWidget *parent) : QFrame(hparent)
{
	client = false;

	setBackgroundMode(PaletteBase);    // set widgets background
	setBackgroundColor(white);
	setFrameStyle( QFrame::Panel | QFrame::Raised );
	resize(Global::Screen::VIRT_SCREEN_SIZE_X, Global::Screen::VIRT_SCREEN_SIZE_Y);                  // set default size in pixels

	m_poActiveCircuit = new Circuit();
	m_poActiveDevice = (XDevice *)NULL;
	m_poActiveWire = (XWire *)NULL;
	m_poActiveSelection = new Selection();
	m_oParent = parent;

	// drawing mode
	setSimMode(NetWidget::MODE_SIM_MULT);
	// device to drop when in drop-mode
	actualDevFunc = 0;

	// popup for the right mouse key (on devices)
	rpop = new QPopupMenu();
	connect(rpop, SIGNAL (activated (int)), SLOT (rmenuCallback(int)));

	// popup for the right mouse key (on selections)
	rselpop = new QPopupMenu();
	rselpop->insertItem(i18n("Copy"), RPOPSEL_COPY);
	rselpop->insertItem(i18n("Paste"), RPOPSEL_PASTEAT);
	rselpop->insertItem(i18n("Cut"), RPOPSEL_CUT);
	connect(rselpop, SIGNAL (activated (int)), SLOT (rmenuSelCallback(int)));

	changed = false;
	m_bSelMoveCursor = false;
	m_bRMenuActive = false;

	runtimeShortcutWarning = true;

	connect(&simTimer, SIGNAL(timeout()), this, SLOT(simStep()));
	connect(&shortcutWarningTimer, SIGNAL(timeout()), this, SLOT(enableShortcutWarning()));
	connect(this, SIGNAL(netContentChanged()), parent, SLOT(repaintClients()));
	connect(this, SIGNAL(netContentUnchanged()), parent, SLOT(circuitUnchanged()));
}

NetWidget::NetWidget(QWidget *hparent, QWidget *parent, Circuit *net)
	: QFrame(hparent)
{
	client = true;

	setBackgroundMode(PaletteBase);    // set widgets background
	setBackgroundColor(white);
	setFrameStyle( QFrame::Panel | QFrame::Raised );
	resize(Global::Screen::VIRT_SCREEN_SIZE_X, Global::Screen::VIRT_SCREEN_SIZE_Y);                // set default size in pixels

	m_poActiveCircuit = net;
	m_poActiveDevice = (XDevice *)NULL;
	m_poActiveWire = (XWire *)NULL;
	m_poActiveSelection = new Selection();
	m_oParent = hparent;

	// device to drop when in drop-mode
	actualDevFunc = 0;

	// popup for the right mouse key (on devices)
	rpop = new QPopupMenu();
	connect(rpop, SIGNAL (activated (int)), SLOT (rmenuCallback(int)));

	// popup for the right mouse key (on selections)
	rselpop = new QPopupMenu();
	rselpop->insertItem(i18n("Copy"), RPOPSEL_COPY);
	rselpop->insertItem(i18n("Paste"), RPOPSEL_PASTEAT);
	rselpop->insertItem(i18n("Cut"), RPOPSEL_CUT);
	connect(rselpop, SIGNAL (activated (int)), SLOT (rmenuSelCallback(int)));

	changed = false;
	m_bSelMoveCursor = false;
	m_bRMenuActive = false;

	runtimeShortcutWarning = true;

	connect(&simTimer, SIGNAL(timeout()), this, SLOT(simStep()));
	connect(&shortcutWarningTimer, SIGNAL(timeout()), this, SLOT(enableShortcutWarning()));
	connect(this, SIGNAL(netContentChanged()), parent, SLOT(repaintClients()));
	connect(this, SIGNAL(netContentUnchanged()), parent, SLOT(circuitUnchanged()));
}

NetWidget::~NetWidget()
{
	if (client) return;

	delete m_poActiveCircuit;
	delete m_poActiveSelection;
}

void NetWidget::mousePressEvent( QMouseEvent *e )
{	QPainter p;

	if (e->button() == LeftButton) {
		if (m_bRMenuActive) {
			m_bRMenuActive = false;
			return;
		}

		if (MainWidget::getMode() == MainWidget::MAIN_MODE_DRAW) {
			// device clicked? set active device
			// devices have priority - since we have bus devices
			m_poActiveDevice = m_poActiveCircuit->containsDev(e->pos());
			if (m_poActiveDevice) {
				m_poActiveDevice->setOrigin();

				// if device is a push button device,  toggle its status
				if (m_poActiveDevice->isInteractive() && m_poActiveDevice->isToggle()) {
					m_poActiveDevice->toggleStaticInput();
					p.begin(this);
					m_poActiveCircuit->draw(this, &p);
					p.end();
				}

				return;
			}

			// wire clicked? set active wire
			m_poActiveWire = m_poActiveCircuit->containsWire(e->pos());
			if (m_poActiveWire) {
				return;
			}

			// nothing clicked? new wire
			p.begin(this);
			m_poActiveWire = m_poActiveCircuit->newWire(&p, e->pos());
			p.end();
			if (m_poActiveWire) {
				emit netContentChanged();
				return;
			}
		}
		if (MainWidget::getMode() == MainWidget::MAIN_MODE_DROP) {
			// move device
			m_poActiveDevice = m_poActiveCircuit->containsDev(e->pos());
			if (m_poActiveDevice) {
				m_poActiveDevice->setOrigin();

				// if device is a push button device,  toggle its status
				if (m_poActiveDevice->isInteractive() && m_poActiveDevice->isToggle()) {
					m_poActiveDevice->toggleStaticInput();
					p.begin(this);
					m_poActiveCircuit->draw(this, &p);
					p.end();
				} else {
					p.begin(this);
					m_poActiveCircuit->drawDev(this, &p, m_poActiveDevice);
					p.end();

					// redraw client widgets with the same content as this
					emit netContentChanged();
					changed = true;
				}
				return;
			}
			// create new device
			m_poActiveDevice = m_poActiveCircuit->newDevice(actualDevFunc, e->pos());
			if (!m_poActiveDevice && (actualDevFunc == DeviceType::fIN)) {
				QMessageBox::warning(m_oParent, i18n("device failure"), i18n("unable to add more named inputs\nresize this sub circuit"));
				emit netContentChanged();
				changed = true;
				return;
			}
			if (!m_poActiveDevice && (actualDevFunc == DeviceType::fOUT)) {
				QMessageBox::warning(m_oParent, i18n("device failure"), i18n("unable to add more named outputs\nresize this sub circuit"));
				emit netContentChanged();
				changed = true;
				return;
			}
			if (!m_poActiveDevice) {
				QMessageBox::warning(m_oParent, i18n("device failure"), i18n("panic: cannot create device!"));
				return;
			}
			if (m_poActiveCircuit->containsDev(m_poActiveDevice)) {
				deleteDev(0);
				emit netContentChanged();
				changed = true;
				return;
			}
			m_poActiveDevice->setOrigin();
			p.begin(this);
			m_poActiveCircuit->drawDev(this, &p, m_poActiveDevice);
			p.end();
			emit netContentChanged();
			if (m_poActiveDevice->drawGraph() && m_poActiveDevice->graphEnabled())
				emit graphChanged();
			return;
		}
		if (MainWidget::getMode() == MainWidget::MAIN_MODE_SELECT) {
			if (m_poActiveSelection->getStatus() == SEL_EXISTS && m_poActiveSelection->onFrame(e->pos())) {
				// selection move mode
				m_poActiveSelection->beginMove(e->pos());
			} else if (m_poActiveSelection->getStatus() == SEL_EXISTS &&  NULL != (m_poActiveDevice = m_poActiveCircuit->containsDev(e->pos()))) {
				if (m_poActiveSelection->contains(m_poActiveDevice)) {
					// selection move mode
					m_poActiveSelection->beginMove(e->pos());
				} else {
					// select device, then selection move mode
					m_poActiveSelection->addTemp(m_poActiveDevice);
					m_poActiveSelection->beginMove(e->pos());
					p.begin(this);
					m_poActiveSelection->erase(&p);
					m_poActiveCircuit->draw(this, &p);
					m_poActiveSelection->draw(&p);
					p.end();
				}
			} else {
				// new selection?
				m_poActiveSelection->setNet(m_poActiveCircuit);

				// select/deselect single device
				m_poActiveDevice = m_poActiveCircuit->containsDev(e->pos());
				if (m_poActiveDevice) {
					if (m_poActiveSelection->contains(m_poActiveDevice)) m_poActiveSelection->remove(m_poActiveDevice);
					else m_poActiveSelection->add(m_poActiveDevice);
					m_poActiveDevice = (XDevice *)NULL;
					m_poActiveWire = (XWire *)NULL;
					p.begin(this);
					m_poActiveSelection->erase(&p);
					m_poActiveCircuit->draw(this, &p);
					m_poActiveSelection->draw(&p);
					p.end();
					return;
				}

				// select/deselect single wire
				m_poActiveWire = m_poActiveCircuit->containsWire(e->pos());
				if (m_poActiveWire) {
					if (m_poActiveSelection->contains(m_poActiveWire)) m_poActiveSelection->remove(m_poActiveWire);
					else m_poActiveSelection->add(m_poActiveWire);
					m_poActiveDevice = (XDevice *)NULL;
					m_poActiveWire = (XWire *)NULL;
					p.begin(this);
					m_poActiveSelection->erase(&p);
					m_poActiveCircuit->draw(this, &p);
					m_poActiveSelection->draw(&p);
					p.end();
					return;
				}

				// group selections
				p.begin(this);
				m_poActiveSelection->erase(&p);
				p.end();
				m_poActiveSelection->beginFrame(e->pos());
			}
			p.begin(this);
			m_poActiveSelection->erase(&p);
			m_poActiveCircuit->draw(this, &p);
			m_poActiveSelection->draw(&p);
			p.end();
			return;
		}
	}

	if (e->button() == RightButton) {
		// right clicked in selection mode? open popup to choose operation
		if (MainWidget::getMode() == MainWidget::MAIN_MODE_SELECT) {
			m_poActiveSelection->at(e->pos());
			selRightSelMenu(e);
			m_bRMenuActive = true;
			return;
		}

		// device right clicked? open popup to chooce operation
		if (NULL != (m_poActiveDevice = m_poActiveCircuit->containsDev(e->pos()))) {
			selRightMenu(e);
			m_bRMenuActive = true;
			return;
		}

		// right clicked twice?
		if (m_bRMenuActive) {
			m_bRMenuActive = false;
			return;
		}

		// wire right clicked? remove node 
		p.begin(this);
		if (m_poActiveCircuit->removeNode(&p, e->pos())) {
			m_poActiveCircuit->draw(this, &p);
			m_poActiveWire = (XWire *)NULL;
			emit netContentChanged();
			changed = true;
			p.end();
			return;
		}
		p.end();
	}
}

void NetWidget::mouseReleaseEvent( QMouseEvent *e)
{	int ret;
	QPainter p;

	if (e->button() == RightButton) return;

	if (MainWidget::getMode() == MainWidget::MAIN_MODE_SELECT) {
		if (m_poActiveSelection->getStatus() == SEL_START) {	// making frames
			m_poActiveSelection->endFrame();
			return;
		}
		if (m_poActiveSelection->getStatus() == SEL_MOVE) {	// moving frames or select devices
			m_poActiveSelection->endMove();

			if (! m_poActiveSelection->hasMoved()) {
				// select/deselect single device when group was not moved
				m_poActiveDevice = m_poActiveCircuit->containsDev(e->pos());
				if (m_poActiveDevice) {
					if (m_poActiveSelection->isTemp(m_poActiveDevice)) m_poActiveSelection->fixTemp(m_poActiveDevice);
					else m_poActiveSelection->remove(m_poActiveDevice);
					m_poActiveDevice = (XDevice *)NULL;
					m_poActiveWire = (XWire *)NULL;
					p.begin(this);
					m_poActiveSelection->erase(&p);
					m_poActiveCircuit->draw(this, &p);
					m_poActiveSelection->draw(&p);
					p.end();
					m_poActiveDevice = (XDevice *)NULL;
					return;
				}
			}
			m_poActiveDevice = m_poActiveCircuit->containsDev(e->pos());
			if (m_poActiveDevice) {
				// look if some wires want to drop nodes..
				m_poActiveDevice->garbageCollection();
				m_poActiveDevice = (XDevice *)NULL;
			}
			return;
		}
		return;
	}

	// check for collision with another device
	if (m_poActiveDevice) {
		//look for devices lying in the device 
		if (m_poActiveCircuit->containsDev(m_poActiveDevice)) {
			m_poActiveDevice->toOrigin();
			p.begin(this);
			m_poActiveCircuit->drawDev(this, &p, m_poActiveDevice);
			p.end();
		}

		// if device was not moved and it is interactive, toggle its status
		if (m_poActiveDevice->isInteractive() && m_poActiveDevice->isOrigin(e->pos())) {
			m_poActiveDevice->toggleStaticInput();
			p.begin(this);
			m_poActiveCircuit->draw(this, &p);
			p.end();
		}

		// look if some wires want to drop nodes..
		m_poActiveDevice->garbageCollection();
		m_poActiveDevice = (XDevice *)NULL;

		emit netContentChanged();
		changed = true;
	}
	if (m_poActiveWire) {
		// check for connection to device or another wire
		ret = m_poActiveCircuit->checkConnection(m_poActiveWire);
		switch (ret) {
		case NFAIL:
			QMessageBox::warning(m_oParent,
				i18n("connection failure"),
				i18n("only one output per wire allowed"));
			break;
		case DFAILMAXI:
			QMessageBox::warning(m_oParent,
				i18n("connection failure"),
				i18n("maximum number of inputs exeeded"));
			break;
		case DFAILMAXO:
			QMessageBox::warning(m_oParent,
				i18n("connection failure"),
				i18n("maximum number of outputs exeeded"));
			break;
		case WSHORT:
			QMessageBox::warning(m_oParent,
				i18n("connection failure"),
				i18n("shortcut\n"));
			break;
		default:
			break;
		}
		p.begin(this);
		m_poActiveWire->erase(&p);
		m_poActiveCircuit->releaseWire(&p, m_poActiveWire);
		m_poActiveWire = (XWire *)NULL;
		m_poActiveCircuit->draw(this, &p);
		p.end();
		emit netContentChanged();
		changed = true;
	}
}

void NetWidget::mouseMoveEvent( QMouseEvent *e )
{	QPainter p;

	if (MainWidget::getMode() == MainWidget::MAIN_MODE_SELECT) {
		if (!hasMouseTracking())
			setMouseTracking(true);
		if (m_poActiveSelection->getStatus() == SEL_EXISTS && m_poActiveSelection->onFrame(e->pos())) {
			if (!m_bSelMoveCursor) {
				m_bSelMoveCursor = true;
				setCursor(sizeAllCursor);
				return;
			}
		} else {
			if (m_bSelMoveCursor) {					// switch selecion cursor
				m_bSelMoveCursor = false;
				setCursor(arrowCursor);
				return;
			}
			if (m_poActiveSelection->getStatus() == SEL_START) {	// make selections
				emit(autoMove(e->pos()));	// move circuit when nessecary
				m_poActiveSelection->changeFrame(e->pos());
				p.begin(this);
				m_poActiveSelection->erase(&p);
				m_poActiveCircuit->draw(this, &p);
				m_poActiveSelection->draw(&p);
				p.end();
				emit netContentChanged();
				return;
			}
			if (m_poActiveSelection->getStatus() == SEL_MOVE) {		// move selections
				emit(autoMove(e->pos()));	// move circuit when nessecary
				p.begin(this);
				m_poActiveSelection->moveFrame(e->pos(), &p);
				m_poActiveCircuit->draw(this, &p);
				p.end();
				emit netContentChanged();
				return;
			}
		}
		return;
	}

	if (hasMouseTracking()) {
		m_poActiveDevice = (XDevice *)NULL;
		m_poActiveWire = (XWire *)NULL;
		m_bSelMoveCursor = false;
		setCursor(arrowCursor);
		setMouseTracking(false);
	}

	if (m_bRMenuActive) return;

	// move wire/device
	if (m_poActiveDevice) {
		emit(autoMove(e->pos()));	// move circuit when nessecary

		if (m_poActiveDevice->moveTo(e->pos())) {
			p.begin(this);
			m_poActiveCircuit->drawDev(this, &p, m_poActiveDevice);
			p.end();
			emit netContentChanged();
		}
	}
	if (m_poActiveWire) {
		emit(autoMove(e->pos()));	// move if nessecary

		p.begin(this);
		m_poActiveWire->erase(&p);
		m_poActiveWire->updateNode(e->pos());
		m_poActiveCircuit->draw(this, &p);
		p.end();
		emit netContentChanged();
	}
}

void NetWidget::mouseDoubleClickEvent( QMouseEvent *e )
{
	// device clicked?
	m_poActiveDevice = m_poActiveCircuit->containsDev(e->pos());
	if (m_poActiveDevice) {
		if (m_poActiveDevice->type() == DeviceType::fSWI) {
			m_poActiveDevice = (XDevice *)NULL;
			return;
		}

		// switch to sub circuit?
		if (toSubNet()) {
			emit netContentChanged();
			return;
		}

		// open device properties?
		if (openProp()) {
			emit netContentChanged();
			return;
		}
	}
}

// always call whith repaint=FALSE
void NetWidget::paintEvent(QPaintEvent *)
{	int x, y;
	int x1s, y1s, x2s, y2s;
	QPainter p;

	x1s = m_oVisibleRect.x() / Grid::GRID * Grid::GRID - Grid::GRID;
	x2s = (m_oVisibleRect.width() + m_oVisibleRect.x()) / Grid::GRID * Grid::GRID + Grid::GRID;
	y1s = m_oVisibleRect.y() / Grid::GRID * Grid::GRID - Grid::GRID;
	y2s = (m_oVisibleRect.height() +  m_oVisibleRect.y()) / Grid::GRID * Grid::GRID + Grid::GRID;

	if (!isActiveWindow()) erase();

	p.begin(this);
	p.setPen(black);
	for(x = x1s ; x < x2s ;x += Grid::GRID)
		for(y = y1s; y < y2s; y += Grid::GRID)
			p.drawPoint(x, y);
	m_poActiveCircuit->draw(this, &p);
	m_poActiveSelection->draw(&p);
	p.end();
}

int NetWidget::printNet(QPainter *p)
{
	m_poActiveCircuit->draw((QPaintDevice *)NULL, p);
	return 1;
}

// right mouse button popup entries
// menu depends on device type
void NetWidget::selRightMenu(QMouseEvent *e)
{
	if (!m_poActiveDevice) return;

	rpop->clear();
	rpop->insertItem(i18n("properties"), RPOP_PROP);
	if (m_poActiveDevice->type() == DeviceType::fNET) {
		rpop->insertItem(i18n("open"), RPOP_SUB);
		rpop->insertItem(i18n("open in window"), RPOP_SUBW);
		rpop->insertItem(i18n("display graph"), RPOP_SHOWGRAPH);
	}
	if (m_poActiveDevice->type() == DeviceType::fEQU) {
		rpop->insertItem(i18n("equations"), RPOP_EQU);
	}
	if (m_poActiveDevice->drawGraph()) {
		if (m_poActiveDevice->graphEnabled()) {
			rpop->insertItem(i18n("hide graph"), RPOP_GRAPH);
		} else {
			rpop->insertItem(i18n("show graph"), RPOP_GRAPH);
		}
	}
	if (m_poActiveDevice->type() == DeviceType::fEQU || m_poActiveDevice->type() == DeviceType::fNET) {
		rpop->insertItem(i18n("add to lib"), RPOP_ADD_LIB);
	}

	rpop->insertSeparator();
	rpop->insertItem(i18n("remove"), RPOP_DEL);

	rpop->popup(mapToGlobal(e->pos()));
}

// callback for right mouse button popup
void NetWidget::rmenuCallback(int val)
{
	switch(val) {
		case RPOP_DEL:
			deleteDev(1);
			break;
		case RPOP_PROP:
			openProp();
			break;
		case RPOP_EQU:
			showDeviceEquations();
			break;
		case RPOP_SUB:
			toSubNet();	// to sub circuit, same window
			break;
		case RPOP_SUBW:
			toSubNetW();	// to sub circuit, new window
			break;
		case RPOP_GRAPH:
			if (!m_poActiveDevice) return;
			if (m_poActiveDevice->graphEnabled()) m_poActiveDevice->enableGraph(0);
			else m_poActiveDevice->enableGraph(1);
			emit graphChanged();
			break;
		case RPOP_SHOWGRAPH:
			if (m_poActiveDevice->devIsCircuit())
				emit showGraph(m_poActiveDevice->devIsCircuit());
			break;
		case RPOP_ADD_LIB:
			// send signal to main widget
			if (m_poActiveDevice)
				emit addToLib(m_poActiveDevice);
			break;
		default:
			break;
	}
	m_bRMenuActive = false;
}

// right mouse button popup entries on selections
void NetWidget::selRightSelMenu(QMouseEvent *e)
{	//(entries are static)
	rselpop->popup(mapToGlobal(e->pos()));
}

//callback for right mouse button popup on selections
// also called vom outside (mainw.ccp/menu)
void NetWidget::rmenuSelCallback(int val)
{	QPainter p;

	switch(val) {
		case RPOPSEL_CUT:
			p.begin(this);
			m_poActiveSelection->cut(&p, m_poActiveCircuit);
			m_poActiveCircuit->draw(this, &p);
			p.end();
			emit netContentChanged();
			emit graphChanged();
			changed = true;
			break;
		case RPOPSEL_COPY:
			m_poActiveSelection->copy(m_poActiveCircuit);
			break;
		case RPOPSEL_PASTE:
			m_poActiveSelection->at(QPoint(40, 10));
		case RPOPSEL_PASTEAT:
			p.begin(this);
			m_poActiveSelection->paste(&p, m_poActiveCircuit);
			m_poActiveSelection->erase(&p);
			m_poActiveCircuit->drawStatus(this, &p);
			m_poActiveCircuit->draw(this, &p);
			m_poActiveSelection->draw(&p);
			p.end();
			emit netContentChanged();
			emit graphChanged();
			changed = true;
			break;
		default:
			break;
	}
	m_bRMenuActive = false;
}

// remove old selections
void NetWidget::removeSelection()
{	QPainter p;

	p.begin(this);
	m_poActiveSelection->remove(&p, 1);
	m_poActiveSelection->erase(&p);
	m_poActiveCircuit->draw(this, &p);
	m_poActiveSelection->draw(&p);
	p.end();
}

// remove device
void NetWidget::deleteDev(int ask)
{	QPainter p;
	int simw_rel = 1;

	if (m_poActiveDevice->type() == DeviceType::fNET) {
		if (ask) {
			if (1 == QMessageBox::warning(m_oParent, i18n("remove device"),
					i18n("remove entire sub-circuit?"), i18n("OK"), i18n("Cancel")))
			{
				m_poActiveDevice = (XDevice *)NULL;
				return;
			}
		}
		emit netContentChanged();
		emit netDeleted(m_poActiveDevice->devIsCircuit());
	}
	if (!m_poActiveDevice->drawGraph() ||  !m_poActiveDevice->graphEnabled())
		simw_rel = 0;

	p.begin(this);
	m_poActiveCircuit->deleteDevice(&p, m_poActiveDevice);
	m_poActiveCircuit->draw(this, &p);
	p.end();
	m_poActiveDevice = (XDevice *)NULL;

	if (simw_rel) emit graphChanged();
}

// open device properties?
int NetWidget::openProp()
{	QPainter p;

	// open properties for oszillators
	if (m_poActiveDevice->type() == DeviceType::fOSC) {
		PropOsz *dlg = new PropOsz(m_oParent, i18n("oscillator properties"), m_poActiveCircuit, m_poActiveDevice);
		if (simmode == MODE_SIM_MULT) simTimer.stop();
		dlg->exec();
		if (simmode == MODE_SIM_MULT) simTimer.start(STATsimStepTime);
		delete dlg;
		m_poActiveDevice = (XDevice *)NULL;
		emit netContentChanged();
		changed = true;
		emit graphChanged();
		return 1;
	}

	// open properties for text devices
	if (m_poActiveDevice->type() == DeviceType::fTXT) {
		PropText *dlg = new PropText(m_oParent, i18n("text label"), m_poActiveCircuit, m_poActiveDevice);
		dlg->exec();
		delete dlg;
		m_poActiveDevice = (XDevice *)NULL;
		emit netContentChanged();
		changed = true;
		return 1;
	}

	// open properties for power sources
	if (m_poActiveDevice->type() == DeviceType::fPWR) {
		PropPwr *dlg = new PropPwr(m_oParent, i18n("power device properties"), m_poActiveCircuit, m_poActiveDevice);
		dlg->exec();
		delete dlg;
		p.begin(this);
		m_poActiveCircuit->drawStatus(this, &p);
		p.end();
		m_poActiveDevice = (XDevice *)NULL;
		emit netContentChanged();
		changed = true;
		emit graphChanged();
		return 1;
	}

	// open properties for switch devices
	if (m_poActiveDevice->type() == DeviceType::fSWI) {
		PropSwitch *dlg = new PropSwitch(m_oParent, i18n("switch device properties"), m_poActiveCircuit, m_poActiveDevice);
		if (simmode == MODE_SIM_MULT) simTimer.stop();
		dlg->exec();
		if (simmode == MODE_SIM_MULT) simTimer.start(STATsimStepTime);
		
		delete dlg;
		m_poActiveDevice = (XDevice *)NULL;
		emit netContentChanged();
		changed = true;
		emit graphChanged();
		return 1;
	}

	// open properties for inputs and ouputs
	if ((m_poActiveDevice->type() == DeviceType::fIN) ||
	    (m_poActiveDevice->type() == DeviceType::fOUT)) {
		PropName *dlg = new PropName(m_oParent, i18n("interface device name"), this, m_poActiveCircuit, m_poActiveDevice);
		if (simmode == MODE_SIM_MULT) simTimer.stop();
		dlg->exec();
		if (simmode == MODE_SIM_MULT) simTimer.start(STATsimStepTime);
		
		delete dlg;
		m_poActiveDevice = (XDevice *)NULL;
		emit netContentChanged();
		changed = true;
		emit graphChanged();
		return 1;
	}

	// open dialog with common device properties
	PropDev *dlg = new PropDev(m_oParent, i18n("device properties"),this ,m_poActiveCircuit, m_poActiveDevice);
	if (simmode == MODE_SIM_MULT) simTimer.stop();
	dlg->exec();
	m_poActiveDevice->forceOutputChange();
	if (simmode == MODE_SIM_MULT) simTimer.start(STATsimStepTime);

	delete dlg;
	m_poActiveDevice = (XDevice *)NULL;
	emit netContentChanged();
	changed = true;
	emit graphChanged();
	return 1;
}

// switch to sub circuit in a new window?
int NetWidget::toSubNetW()
{	Circuit *new_m_poActiveCircuit;

	if (!m_poActiveDevice) return 0;

	new_m_poActiveCircuit = m_poActiveDevice->devIsCircuit();
	if (new_m_poActiveCircuit) {
		// remove the current selection before switching to a new circuit
		removeSelection();
		new_m_poActiveCircuit->forceOutputChange();
		emit createNewWidget(new_m_poActiveCircuit);	// signal goes to main widget (mainw.cpp)
		return 1;
	}
	return 0;
}

// switch to sub circuit?
int NetWidget::toSubNet()
{	Circuit *new_m_poActiveCircuit;

	new_m_poActiveCircuit = m_poActiveDevice->devIsCircuit();
	if (new_m_poActiveCircuit) {
		// remove the current selection before switching to a new circuit
		removeSelection();

		m_poActiveCircuit = new_m_poActiveCircuit;
		m_poActiveDevice = (XDevice *)NULL;
		m_poActiveWire = (XWire *)NULL;
		// signal mainw info about parent or not (changes toolbar)
		if (m_poActiveCircuit->parent())
			emit netChanged(1);
		else
			emit netChanged(0);
		m_poActiveCircuit->forceOutputChange();
		repaint(TRUE);
		return 1;
	}
	return 0;
}

// switch to parent of active net
void NetWidget::toParentNet()
{	Circuit *new_m_poActiveCircuit;

	// remove the current selection before switching to a new circuit
	removeSelection();

	new_m_poActiveCircuit = m_poActiveCircuit->parent();
	if (!new_m_poActiveCircuit) {
		warning("no parent net!?");
		return;
	}
	m_poActiveCircuit = new_m_poActiveCircuit;
	m_poActiveWire = (XWire *)NULL;
	m_poActiveDevice = (XDevice *)NULL;
	repaint(TRUE);
	// signal mainw info about having a parent (changes toolbar)
	if (m_poActiveCircuit->parent())
		emit netChanged(1);
	else
		emit netChanged(0);
	m_poActiveCircuit->forceOutputChange();
}

bool NetWidget::saveAsSubNet()
{	Circuit *_net = m_poActiveCircuit->rootParent();

	_net = m_poActiveCircuit->rootParent();
	if (_net != m_poActiveCircuit) {	
		int ret = QMessageBox::warning(m_oParent,
				i18n("save circuit"),
				i18n("the current circuit is a sub circuit.\nsave it as a.."),
				i18n("main circuit"),
				i18n("sub circuit"));
		if (ret) return true;
	}
	return false;
}

int NetWidget::saveNet(QString filename, bool as_sub_net)
{	Circuit *net;
	int ret;

	if (as_sub_net)
		net = m_poActiveCircuit;
	else
		net = m_poActiveCircuit->rootParent();

	emit netContentUnchanged();
	changed = false;

	klogicIO io(filename, net);
	if (as_sub_net) io.setSubFilename();

	ret = io.write(as_sub_net, false, 0, 0);

	if (ret <= 0) {
		QMessageBox::warning(m_oParent,
			i18n("save error"),
			i18n("unable to write file"));
		return 0;
	}

	return 1;
}

// load file
// init: refresh client widget
// return values:
// 0: error occured
// 1: ok, normal circuit
// 2: ok, subcircuit
int NetWidget::openNet(QString filename, bool main_circuit_changed)
{	Circuit *_net;
	klogicIO *_io;
	QPainter p;
	bool create_sub = false;

	// -----------------------------------------
	// check file format
	// -----------------------------------------
	_io = new klogicIO(filename);
	int file_type = _io->checkFileFormat();

	if (file_type == klogicIO::XML_SUB || file_type == klogicIO::KLOGIC_SUB) {
		//fprintf(stderr, "SUB import\n");
		// -----------------------------------------
		// is a sub circuit
		// -----------------------------------------
		create_sub = true;
		_net = m_poActiveCircuit;
	} else if (file_type == klogicIO::XML_MAIN || file_type == klogicIO::KLOGIC_MAIN) {
		//fprintf(stderr, "MAIN import\n");
		// -----------------------------------------
		// replace main circuit: ask!
		// -----------------------------------------
		if (main_circuit_changed && !m_poActiveCircuit->rootParent()->empty()) {
			if (QMessageBox::Yes != QMessageBox::warning(m_oParent,
				i18n("new circuit"), 
				i18n("delete current circuit?"), 
				QMessageBox::Yes, 
				QMessageBox::No))
				return 1;
		}
		_net = m_poActiveCircuit->rootParent();
		m_poActiveCircuit = _net;
		m_poActiveDevice = (XDevice *)NULL;
		m_poActiveWire = (XWire *)NULL;

		p.begin(this);
		m_poActiveCircuit->erase(&p);
		p.end();
		m_poActiveCircuit->deleteAll();
		emit netContentUnchanged();
		uniqueID::reset();
		changed = false;
	} else {
		QMessageBox::warning(m_oParent,
			i18n("read error"),
			i18n("unknown file format"));
		return 0;
	}

	if (simmode == MODE_SIM_MULT) simTimer.stop();
	_io->setNet(_net);

	// -----------------------------------------
	// read the file and create the circuit now
	// -----------------------------------------
	bool result = false;
	switch (file_type) {
		case klogicIO::XML_SUB:
		case klogicIO::XML_MAIN:
			result = _io->readXML();
			break;
		case klogicIO::KLOGIC_SUB:
		case klogicIO::KLOGIC_MAIN:
			result = _io->readNet(create_sub);
			break;
	}

	if (simmode == MODE_SIM_MULT) simTimer.start(STATsimStepTime);

	// -----------------------------------------
	// analyze the result
	// -----------------------------------------
	if (!result) {
		if (XMLImportHandler::errorsOccured()) {
			QMessageBox::warning(m_oParent,
				i18n("read error"),
				(const char *)XMLImportHandler::getErrors());
		} else {
			QMessageBox::warning(m_oParent,
			i18n("read error"),
			i18n("unable to read file"));
		}

		p.begin(this);
/*
		// remove useless nodes/wires
		m_poActiveCircuit->garbageCollection(&p);
*/
		// redraw everything
		m_poActiveCircuit->drawStatus(this, &p);
		p.end();

		return 0;
	}

	// -----------------------------------------
	// final steps
	// -----------------------------------------
	repaint(TRUE);

	// signal mainw info about having a parent or not (changes toolbar)
	if (m_poActiveCircuit->parent())
		emit netChanged(1);
	else
		emit netChanged(0);
	delete _io;
	emit graphChanged();
	emit netContentUnchanged();

	p.begin(this);
	// remove useless nodes/wires
	m_poActiveCircuit->garbageCollection(&p);
	// redraw everything
	m_poActiveCircuit->drawStatus(this, &p);
	p.end();

	if (create_sub) return 2;
	return 1;
}

int NetWidget::newNet()
{	QPainter p;

	if ((changed && !m_poActiveCircuit->empty() &&
			QMessageBox::Yes == (QMessageBox::warning(m_oParent,
			i18n("new circuit"), 
			i18n("delete current circuit?"), 
			QMessageBox::Yes,
			QMessageBox::No))) || (!changed)) {
		p.begin(this);
		m_poActiveCircuit->erase(&p);
		p.end();
		m_poActiveCircuit->deleteAll();
		repaint(TRUE);
		emit netContentUnchanged();
		changed = false;
		emit graphChanged();
		uniqueID::reset();
		return 1;
	}
	return 0;
}

// display dialog with the equations of the current circuit
void NetWidget::showCircuitEquations()
{
	if (!m_poActiveCircuit->checkCircuitNames()) {
		if (QMessageBox::No == (QMessageBox::warning(this,
				i18n("Parse circuit equations"),
				i18n("There are some non unique device names \nwithin your circuit ... \n\nrename them automaticly?"),
				QMessageBox::Yes,
				QMessageBox::No)))
			return;
		m_poActiveCircuit->unifyCircuitNames();
	}
	if (m_poActiveCircuit->hasImpossibleEquations()) {
		QMessageBox::information(this,
			i18n("Parse circuit equations"),
			i18n("Unable to display equations for circuits containing \nRAM or Tristate devices!"));
		return;
	} 
	DlgEqu *equation_dialog = new DlgEqu(this, i18n("Circuit Equation Editor"), m_poActiveCircuit);
	equation_dialog->exec();
	delete equation_dialog;
}

// display dialog with the equations of the active device
void NetWidget::showDeviceEquations()
{
	if (!m_poActiveDevice) return;
	DlgEqu * equation_dialog = new DlgEqu(this, i18n("Device Equation Editor"), m_poActiveDevice);
	equation_dialog->exec();
	delete equation_dialog;
}

// retrieve the active Circuit
Circuit *NetWidget::getActive()
{
	return m_poActiveCircuit;
}

// check if given net is this or a parent of this
int NetWidget::contained(Circuit *rootnet)
{	Circuit *_rootnet = m_poActiveCircuit;

	if (_rootnet == rootnet) return 1;
	while(NULL != (_rootnet = _rootnet->parent())) {
		if (_rootnet == rootnet) return 1;
	}
	return 0;
}

// private slot
// perform a simulation step
void NetWidget::simStep()
{	QPainter p;
	int ret;

	if (!client) {
		// process a simulation step
		m_poActiveCircuit->rootParent()->Burst(STATsimBurst);
		emit simStepped();
	}
	p.begin(this);
	// draw wires and devices with changed image
	ret = m_poActiveCircuit->drawStatus(this, &p);
	p.end();

	// runtime shortcut occured?
	if (ret == WSHORT && runtimeShortcutWarning) {
		if (simmode == MODE_SIM_MULT) simTimer.stop();
		QMessageBox::warning(m_oParent,
			i18n("runtime error"),
			i18n("shortcut\n"));
		if (simmode == MODE_SIM_MULT) simTimer.start(STATsimStepTime);
		runtimeShortcutWarning = false;
		shortcutWarningTimer.start(10000, true);	// single shot after 20 sec!
	}
}

// private slot
// enable runtime shortcut warning again, 10sec after the last one
void NetWidget::enableShortcutWarning()
{
	runtimeShortcutWarning = true;
}

// public slot
// update visible rect, emitted by MainWidget
void NetWidget::visible(const QRect& newvisi)
{
	m_oVisibleRect = newvisi;
}

// apply device defaults to all existing devices
void NetWidget::applyDefaults()
{	Circuit *toplevel = m_poActiveCircuit->rootParent();

	toplevel->applyDefaults();
	repaint(TRUE);
}

// returns wether net has changed or not
bool NetWidget::hasChanged()
{
	return changed;
}

// reset change flag, mainwidget is the caller
void NetWidget::resetChanges()
{
	changed = false;
}

