/***************************************************************************
                          kmess.h  -  description
                             -------------------
    begin                : Sun Jan  5 15:18:36 CST 2003
    copyright            : (C) 2003 by Mike K. Bennett
    email                : mkb137b@hotmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef KMESS_H
#define KMESS_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qptrlist.h>
#include <qdict.h>

#include <kapp.h>
#include "kmessinterface.h"
#include "kmessdebug.h"

// Forward declarations
class Account;
class AccountAction;
class AutologinView;
class ChatMaster;
class Contact;
class ContactList;
class NotificationManager;
class NotificationChat;
class NotificationContactStatus;
class NotificationNewEmail;
class CurrentAccount;
class IdleTimer;
class InitialView;
class KMessTest;
class KMessView;
class MsnNotificationConnection;
class SystemTrayWidget;
class NowListeningClient;

/**
 * @brief The main window of the application.
 *
 * This class initializes the application and handles
 * all menu actions of the main window.
 *
 * @author Mike K. Bennett
 * @ingroup Root
 */
class KMess : public KMessInterface
{
  Q_OBJECT

  friend class KMessTest;

  public: // Public methods
    // The construtor
                       KMess(QWidget* parent=0, const char *name=0);
    // The destructor
                      ~KMess();
    // Autologin with the first account that has autologin enabled
    void               checkAutologin(QString handle);
    // Initialize the class
    bool               initialize();

  protected:  // Protected methods
    // Read in account and other properties
    virtual void       readProperties(KConfig *config);
    // Save account and other properties
    virtual void       saveProperties(KConfig *config);

  private: // Private methods
    // Add the account and create the GUI elements
    void               addAccount(Account *account);
    // The application is closing, after queryExit() was approved
    void               applicationClosing();
    // Create the program's default directories in .kde/share/apps/
    bool               createDirectories();
    // Return the account for a given handle
    Account *          getAccountByHandle( const QString &handle );
    // Initialize the autologin view
    bool               initAutologinView();
    // Initialize the chat master
    bool               initChatMaster();
    // Initialize the emoticon manager
    bool               initEmoticonManager();
    // Initialize the idle timer
    bool               initIdleTimer();
    // Initialize the initial view
    bool               initInitialView();
    // Initialize the main view
    bool               initKMessView();
    // Initialize the MSN notification connection
    bool               initMsnNotificationConnection();
    // Initialize notification objects
    bool               initNotifications();
    // Initialize the now listening support.
    bool               initNowListening();
    // Initialize the system tray widget
    bool               initSystemTrayWidget();
    // Validate a contact email
    bool               isValidEmail( QString email );
    // Set the caption
    void               setCaptionToUser();

  private slots: // Private slots
    // The account changed it's status to invisible
    void               accountInvisible();
    // The account changed it's status to offline
    void               accountOffline();
    // The account changed it's status to online
    void               accountOnline();
    // "Add a new contact" was selected from the menu.
    void               addNewContact();
    // "Add a new group" was selected from the menu.
    void               addNewGroup();
    // An account's settings have been changed
    void               changedAccountSettings( Account *account, QString oldHandle, QString oldFriendlyName );
    // A status was selected from the menu.
    void               changeStatus(const QString &statusName);
    // Tray notifications settings were changed
    void               changedNotificationsSettings();
    // The current now listening settings have changed.
    void               changedNowListeningSettings();
    // The currently playing song was changed.
    void               changedSong( const QString &artist, const QString &album, const QString &track, bool playing );
    // The status was changed
    void               changedStatus( Account *account = 0 );
    // A view mode has been selected from the menu.
    void               changeViewMode(int mode);

    // Show a "Contact added you" dialog
    void               showContactAddedUserDialog(const Contact *contact);
    // Show a dialog before removing the contact
    void               showRemoveContactDialog(QString handle);
    // Show a dialog before removing the group
    void               showRemoveGroupDialog(QString groupId);
    // Show a "Rename group" dialog
    void               showRenameGroupDialog(QString groupId);
#ifdef KMESS_NETWORK_WINDOW
    // Open the network window
    void               showNetworkWindow();
#endif
    // Open the transfer manager
    void               showTransferWindow();
    // A connection has been made with the notification server.
    void               connected();
    // Connect to the server with the given account
    void               connectWithAccount(Account *account);
    // Connect to the server with the given account, possibly temporary or new.
    void               connectWithAccount(QString handle, QString password, bool rememberMe, QString initialStatus );
    // The current account changed its name, so set the corresponding account's name
    void               currentAccountChangedName();
    // "Add new account" has been selected from the menu.
    void               createNewAccount();
    // Delete the given account
    void               deleteAccount(Account *account);
    // Disconnect was selected from the menu.
    void               disconnectClicked();
    // The program is not connected (initially) or no longer connected (after
    //  a disconnect) to the notification server.
    void               disconnected();
    // Show the settings dialog for a given account
    void               showSettingsForAccount(Account *account);
    // Show the settings dialog for the current account.
    void               showSettingsForCurrentAccount();
    // Show the user's MSN profile.
    void               showUserProfile();
    // We're about to shutdown, apparently
    void               shutDown();
    // The "show allowed contacts" menu item has been toggled.
    void               toggleShowAllowed(bool show);
    // The "show offline contacts" menu item has been toggled.
    void               toggleShowOffline(bool show);
    // The "Show removed contacts" menu item has been toggled.
    void               toggleShowRemoved(bool show);
    // The user has gone idle
    void               userIsIdle();
    // The user is no longer idle
    void               userIsNotIdle();

  private: // Private attributes
    // A list of user accounts
    QPtrList<Account>  accounts_;
    // The autologin view widget
    AutologinView     *autologinView_;
    // The chat master - the object that controls the chat window
    ChatMaster        *chatMaster_;
    // The menu items of the connect menu
    QDict<AccountAction> connectMenuItems_;
    // The notifications manager instance
    NotificationManager        *notificationManager_;
    // The notification object for when a contact sends us a message
    NotificationChat           *notificationChat_;
    // The notification object for when a contact changes its status
    NotificationContactStatus *notificationContactStatus_;
    // The notification object for when a new email is received
    NotificationNewEmail       *notificationNewEmail_;
    // A pointer to the instance of the current account
    CurrentAccount    *currentAccount_;
    // The current account status
    QString            currentStatus_;
    // The idle timer
    IdleTimer         *idleTimer_;
    // Whether or not the object has been initialized
    bool               initialized_;
    // The initial login view widget
    InitialView       *initialView_;
    // The now listening client
    NowListeningClient* nowListeningClient_;
    // The connection to the msn notification server
    MsnNotificationConnection *msnNotificationConnection_;
    // The menu items of the settings menu
    QDict<AccountAction> settingsMenuItems_;
    // The system tray widget
    SystemTrayWidget  *systemTrayWidget_;
    // The main view widget
    KMessView         *view_;
};

#endif
