// Copyright(C) 2005,2006,2007 Stefan Siegl <stesie@brokenpipe.de>
// Copyright(C) 2006 Martin Albrecht <malb@informatik.uni-bremen.de>
// Copyright(C) 2007 Christian Dietrich <stettberger@brokenpipe.de>
//
// kopete_silc - silc plugin for kopete messenger
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <iostream>
#include <assert.h>

#include "silcaccount.h"
#include "silccontact.h"
#include "silcbuddycontact.h"
#include "silcchannelcontact.h"
#include "silcservercontact.h"
#include "silcmessagemanager.h"
#include "silcjoindlgwidget.h"
#include "silcfiletransfer.h"
#include "silcbuddyattributes.h"

#include <kopetemetacontact.h>
#include <kopeteonlinestatus.h>
#include <kopeteuiglobal.h>
#include <kopetechatsessionmanager.h>
#include <ui/kopetefileconfirmdialog.h>
#include <ui/kopeteview.h>

#include <kdebug.h>
#include <qcheckbox.h>

#include <kuser.h>
#include <klocale.h>
#include <kconfigbase.h>
#include <kaction.h>
#include <kpopupmenu.h>
#include <klineeditdlg.h>
#include <kcompletion.h>
#include <kglobalsettings.h>
#include <kmessagebox.h>
#include <klineedit.h>
#include <kstandarddirs.h>
#include <qdatastream.h>
#include <qregexp.h>
#include <kstandarddirs.h>
#include <kopetecontactlist.h>
#include <kpassdlg.h>
#include <qimage.h>
#include <qfile.h>

namespace Kopete {
  class KopeteView;
}

// sorry for this hack, unfortunately we need it for
// the macros of recent libsilc to work ...
typedef unsigned char SilcUInt8;

// set ourself's usermode
static void sendUmode(SilcTK::SilcClient client, 
		      SilcTK::SilcClientConnection conn,
		      SilcTK::SilcUInt32 umode);


const QString SilcAccount::CONFIG_NICKNAME = QString("NickName");
const QString SilcAccount::CONFIG_USERNAME = QString("UserName");
const QString SilcAccount::CONFIG_REALNAME = QString("RealName");
const QString SilcAccount::CONFIG_HOSTNAME = QString("HostName");
const QString SilcAccount::CONFIG_SIGN_CHAT_MESSAGE = QString("SignChatMsg");
const QString SilcAccount::CONFIG_SIGN_CHAT_ACTION = QString("SignChatAction");
const QString SilcAccount::CONFIG_SIGN_PRIV_MESSAGE = QString("SignPrivMsg");
const QString SilcAccount::CONFIG_SIGN_PRIV_ACTION = QString("SignPrivAction");
const QString SilcAccount::CONFIG_DISPLAY_IMAGES_INLINE = QString("InlineImg");
const QString SilcAccount::CONFIG_FT_USE_SILC_MIME = QString("UseSilcMime");
const QString SilcAccount::CONFIG_FT_AUTO_ACCEPT = QString("FtAutoAccept");
const QString SilcAccount::CONFIG_FT_BIND_SELECTION = QString("FtBindSelection");
const QString SilcAccount::CONFIG_QUIT_MESSAGE = QString("QuitMessage");
const QString SilcAccount::CONFIG_ATTR_MOOD = QString("AttrMood");
const QString SilcAccount::CONFIG_ATTR_CONTACT = QString("AttrContact");
const QString SilcAccount::CONFIG_ATTR_GEO_ALLOWED = QString("AttrGeoAllowed");
const QString SilcAccount::CONFIG_ATTR_GEO_LONG = QString("AttGeoLong");
const QString SilcAccount::CONFIG_ATTR_GEO_LAT = QString("AttrGeoLat");
const QString SilcAccount::CONFIG_ATTR_GEO_ALT = QString("AttrGeoAlt");
const QString SilcAccount::CONFIG_ATTR_MESSAGE = QString("AttrMessage");
const QString SilcAccount::CONFIG_ATTR_LANGUAGE = QString("AttrLanguage");
const QString SilcAccount::CONFIG_ATTR_TIMEZONE = QString("AttrTimezone");
const QString SilcAccount::CONFIG_ATTR_ALLOWED = QString("AttrAllowed");

const QString pubKeyName = QString("kopete_silc_public_key.pub");
const QString prvKeyName = QString("kopete_silc_private_key.prv");

/**
 *
 *
 */

SilcAccount::SilcAccount(SilcProtocol *proto, const QString &accountID,
			 const char *name)
  : Kopete::PasswordedAccount(proto, accountID, 0, name), _contactManager(this),
    _pubkey(NULL), _privkey(NULL),  _client(NULL), _conn(NULL),  
    libsilcTimerId(0), libsilcTimerLocked(false)
{
  KConfigGroup *config = configGroup();
  QString nickname = name ? QString::fromUtf8(name) :
    config->readEntry(CONFIG_NICKNAME);

  if(nickname == QString::null) {
    // account doesn't have a nickname assigned, therefore use some default
    const char *userenv = getenv("USER");
    if(! userenv) userenv = "SilcUser";
    nickname = QString(userenv);
  }
  
  setNickName(nickname);

  /* SilcClient */
  SilcTK::SilcClientParams params;
  memset((void *)&params, 0, sizeof(params));
  snprintf(params.nickname_format, 32, "%%n");

  _client = silc_client_alloc(&SilcAccount::ops, &params, this, NULL);

  if(! _client) {
    KMessageBox::error(Kopete::UI::Global::mainWidget(),
		       i18n("failed to create SILC client"),
		       i18n("Initialisation"));
    return;
  }

  if(! silc_client_init(_client, strdup(userName().utf8()), 
                        SilcTK::silc_net_localhost(), strdup(realName().utf8()),
                        NULL, NULL)) {
    KMessageBox::error(Kopete::UI::Global::mainWidget(),
		       i18n("failed to initialize SILC client"),
		       i18n("Initialisation"));
    SilcTK::silc_client_stop(_client, NULL, NULL);
    SilcTK::silc_client_free(_client);
    _client = NULL;
    return;
  }

  SilcTK::silc_hash_alloc((const unsigned char *)"sha1", &sha1hash);

  QString pubFile = locateLocal("appdata", pubKeyName);  
  QString prvFile = locateLocal("appdata", prvKeyName);  

  if(silc_load_key_pair(pubFile, prvFile, "", &_pubkey, &_privkey))
  {
    (void) 0;
    // keypair does not exist ... try silc  
  } else if(silc_load_key_pair(KUser().homeDir().append
                               ("/.silc/public_key.pub").latin1(),
                               KUser().homeDir().append
                               ("/.silc/private_key.prv").latin1(),
                               "", &_pubkey, &_privkey))
  {
    copyKey(KUser().homeDir().append("/.silc/public_key.pub"));
    copyKey(KUser().homeDir().append("/.silc/private_key.prv"),true);
    // keypair does not exist ... try silky 
  } else if(silc_load_key_pair(KUser().homeDir().append
                               ("/.silky/silky.pub").latin1(),                
                               KUser().homeDir().append
                               ("/.silky/silky.prv").latin1(),
                               "", &_pubkey, &_privkey))
  {
    copyKey(KUser().homeDir().append("/.silky/silky.pub"));
    copyKey(KUser().homeDir().append("/.silky/silky.prv"),true);
    // try new one 
  } else if(silc_create_key_pair("rsa", 2048, pubFile, prvFile, 
                                 NULL, "", &_pubkey, &_privkey, FALSE)) {
     (void) 0;
  } else {
    KMessageBox::error(Kopete::UI::Global::mainWidget(),
		       i18n("Could not generated key pair"),
		       i18n("SILC Key Generation"));
    SilcTK::silc_client_stop(_client, NULL, NULL);
    SilcTK::silc_client_free(_client);
    _client = 0;
    return;
  }


  SilcContact *myself = contactManager()->createBuddy(nickname);
  setMyself(myself);

  _myServer = contactManager()->createServer(hostName());

  menuJoinChannel = new KAction(i18n("Join Channel ..."), QString::null, 0,
				this, SLOT(slotJoinChannel()), this);
  menuShowServer = new KAction(i18n("Show Server Window"), QString::null, 0,
			       this, SLOT(slotShowServer()), this);
  menuShowFingerprint = new KAction(i18n("Show My Fingerprint"), QString::null,
				    0, this, SLOT(slotShowFingerprint()), this);
  QObject::connect(Kopete::ContactList::self(), 
                   SIGNAL(globalIdentityChanged(const QString&, const QVariant&)), 
                   SLOT(slotGlobalIdentityChanged(const QString&, const QVariant&)));
  QObject::connect(this, SIGNAL(disconnected()), SLOT(slotStopTimer()));

  SilcTK::silc_client_run_one(_client);
}


SilcAccount::~SilcAccount() 
{ 
  if(_client) {
    SilcTK::silc_client_stop(_client, NULL, NULL);
    SilcTK::silc_client_free(_client);
    _client = 0;
  }

  if(_myServer) {
    delete _myServer;
    _myServer = 0;
  }
  if(_pubkey)
    SilcTK::silc_pkcs_public_key_free(_pubkey);
  if(_privkey)
    SilcTK::silc_pkcs_private_key_free(_privkey);
}

void
SilcAccount::timerEvent(QTimerEvent *)
{
  if(! _client)
    return;

  if(libsilcTimerLocked)
    return; // libsilcTimerLocked held, never ever call libsilc recursively.

  // this isn't of course a good locking technique, but it's perfectly
  // okay, since Kopete always runs single threaded.
  libsilcTimerLocked = true;
  
  SilcTK::silc_client_run_one(_client);

  libsilcTimerLocked = false;
}

void 
SilcAccount::connect(const Kopete::OnlineStatus &status)
{
  if(! _client) {
     return;
  }
  if(! _conn) {
    setOnlineStatus(SilcProtocol::protocol()->statusConnecting);
    kdDebug() << "connecting to " << hostName() << " now" << endl;

    int collonpos = hostName().find(":");
    QString host;
    int port = 706;  
    if(collonpos < 0) {
      host = hostName();
    } else {
      //split
      host = hostName().left(collonpos);
      port = hostName().mid(collonpos + 1).toInt();    
    }

    SilcTK::SilcClientConnectionParams params;
    /* Set connection parameters */
    memset(&params, 0, sizeof(params));
    params.nickname = (char *)nickName().latin1();
    params.pfs = TRUE;

    
    SilcTK::SilcAsyncOperation asop = 
      SilcTK::silc_client_connect_to_server
      (_client, &params, _pubkey, _privkey, (char *)host.latin1(), port, 
       (SilcTK::SilcClientConnectCallback) silc_connection_cb, NULL);
    if(!asop) {
      setOnlineStatus(SilcProtocol::protocol()->statusOffline);
      return;
    }

    SilcTK::silc_client_run_one(_client);

    libsilcTimerId = startTimer(250); 
  }

  else {
    // we're connected already
    if(myself()->onlineStatus() != status) {
      setOnlineStatus(status);
    }
  }

  // if the user presses connect-all, status is set to unknown, 
  // use Online as a sane default ...
  if(status.status() == Kopete::OnlineStatus::Unknown)
    _wantedOnlineStatus = SilcProtocol::protocol()->statusOnline;
  else
    _wantedOnlineStatus = status;
}

void 
SilcAccount::disconnect(void)
{
  // kopete directly calls disconnect() when it shut's down. 
  // however we'd like to have setOnlineStatus called first, so ...

  if(! _client) 
    return;
	
  if(_conn) 
    sendSilcCommand(QString("QUIT %1").arg(quitMessage()));
}
void 
SilcAccount::slotStopTimer(void)
{
  killTimer(libsilcTimerId);
  libsilcTimerId = 0;
}

bool
SilcAccount::isBehindNat(void) const
{
  char *ip = localIp();

  if (_conn) {

    SilcTK::SilcUInt32 bin;
    if (SilcTK::silc_net_addr2bin(ip, &bin, 4)) {
      /* Now let's test for the private ip ranges */
      /* 192.168.255.255 */
      if((bin & 0xffff) == 0xa8c0) 
        return TRUE;
      /* 10.255.255.255 */
      else if ((bin & 0xff) == 0x0a)
        return TRUE;
      /* 172.16.0.0 - 172.31.255.255 */ 
      else if (((bin & 0xff) == 0xac) && (((bin >> 8) & 0xff) >= 16) 
               && (((bin >> 8 )  & 0xff) <= 31))
        return TRUE;
      /* 169.254.255.255 */
      else if ((bin & 0xffff) == 0xfea9)
        return TRUE;
    }
  }
  return FALSE;
}
char *
SilcAccount::localIp(void) const 
{
 if (_conn) {
   SilcTK::SilcSocket socket;
   SilcTK::SilcStream stream;
   char *ip;
   /* Yes SilcToolkit 1.1 is much easier than 1.0 */
   stream = SilcTK::silc_packet_stream_get_stream(_conn->stream);
   SilcTK::silc_socket_stream_get_info(stream, &socket, NULL, NULL, NULL);
   SilcTK::silc_net_check_local_by_sock(socket, NULL, &ip);
   return ip;
 } else return NULL;
};

void 
SilcAccount::setOnlineStatus(const Kopete::OnlineStatus& status,
			     const QString &)
{
  assert(myself()->account() == this);

  // set new status
  //myself()->setOnlineStatus(status);

  if(status == SilcProtocol::protocol()->statusOffline) {
    myself()->setOnlineStatus(status);
    disconnect();
    return;
  }

  if(status == SilcProtocol::protocol()->statusConnecting) {
     if(myself()->onlineStatus() == SilcProtocol::protocol()->statusOffline) {
      //it we are not online yet
      myself()->setOnlineStatus(status);
    }
    return;
  }

  if(! _conn) {
    connect(status);
    return;
  }

  // the connectionstatus plugin may kick-off an unknown status
  // which, when accepted, crashes Kopete. As this status
  // has no description assigned we'll just ignore it for now
  if(QString::null.compare(status.description())!=0)
    myself()->setOnlineStatus(status);

  // send the correct user mode ...
  if(status == SilcProtocol::protocol()->statusGone)
    sendUmode(_client, _conn, SILC_UMODE_GONE);

  else if(status == SilcProtocol::protocol()->statusIndisposed)
    sendUmode(_client, _conn, SILC_UMODE_INDISPOSED);

  else if(status == SilcProtocol::protocol()->statusBusy)
    sendUmode(_client, _conn, SILC_UMODE_BUSY);

  else if(status == SilcProtocol::protocol()->statusHyper)
    sendUmode(_client, _conn, SILC_UMODE_HYPER);

  else
    sendUmode(_client, _conn, 0);
}

void
SilcAccount::slotShowServer(void)
{
  myServer()->view();
}

/**
 * Shows the fingerprint of the current user to this user
 * Code taken from silcapputil.c:silc_show_public_key()
 */
void SilcAccount::slotShowFingerprint(QString nickName)
{ 
  SilcTK::SilcUInt32 pk_len;
  unsigned char *pk = SilcTK::silc_pkcs_public_key_encode(_pubkey,
							  &pk_len);
  char *fingerprint = SilcTK::silc_hash_fingerprint(0, pk, pk_len);

  if(nickName==QString::null) 
    nickName=this->nickName();   

  KMessageBox::information(Kopete::UI::Global::mainWidget(),
			   QString(fingerprint).replace(" ",":"),
			   i18n("Fingerprint of ").append(nickName),
                           QString::null, 0);  

  
  free(fingerprint); 
  free(pk); 
} 

/** 
 * create a new SilcContact (type depending on the first char of the
 * contactId). This function is used by SilcProtocol::deserializeContact
 * to create the contacts on the contactlist.
 */
bool
SilcAccount::createContact(const QString &contactId,
			   Kopete::MetaContact *meta)
{
  if(contactId[0] == '@')
    _contactManager.createBuddy(meta, contactId.mid(1));

  else if(contactId[0] == '#')
    _contactManager.createChannel(contactId.mid(1), meta);

  else {
    std::cerr << "invalid contactId, refusing to create new contact: "
	      << contactId.latin1() << std::endl;
    return false;
  }

  return true;
}

void
SilcAccount::connectWithPassword(const QString & /* password */)
{
  std::cout << "connectWithPassword called" << std::endl;
}

SilcChatSession *
SilcAccount::chatSession(Kopete::ContactPtrList others)
{
  Kopete::ChatSessionManager *csm = Kopete::ChatSessionManager::self();

  // check whether Kopete's ChatSessionManager knows about an existing 
  // instance for the group `others' we need ...
  SilcChatSession *chatSession = dynamic_cast<SilcChatSession *>
    (csm->findChatSession(myself(), others, protocol()));
  if(chatSession) return chatSession;

  chatSession = new SilcChatSession(myself(), others, protocol());
  return chatSession;							       
}

void
SilcAccount::setNickName(const QString &nickname)
{
  configGroup()->writeEntry(CONFIG_NICKNAME, nickname);

  if(myself())
    myself()->setNickName(nickname);

  QString host = hostName();
  if(host.isEmpty()) host = QString(i18n("<invalid-configuration>"));
  setAccountLabel(QString("%1@%2").arg(nickname).arg(host));
}

const QString
SilcAccount::nickName(void) const 
{
  return myself()->nickName();
}

void
SilcAccount::setUserName(const QString &name)
{
  configGroup()->writeEntry(CONFIG_USERNAME, name);
}

const QString
SilcAccount::userName(void) const 
{
  const QString &username = configGroup()->readEntry(CONFIG_USERNAME);
  
  if(username.isEmpty()) {
    // if there is no username set, use the user's login name.
    // In case this is not available as well, default to something sane,
    // since the SILC network requires us to specify such a name.
    //
    const char *userenv = getenv("USER");
    if(! userenv) userenv = "SilcUser";
    return QString(userenv);
  }

  return username;
}

void
SilcAccount::setRealName(const QString &name)
{
  configGroup()->writeEntry(CONFIG_REALNAME, name);
}

const QString
SilcAccount::realName(void) const 
{
  const QString &realname = configGroup()->readEntry(CONFIG_REALNAME);

  if(realname.isEmpty()) {
    // use some default value, since we're expected to have one ...
    return QString("I'm a lucky kopete_silc user.");
  }

  return realname;
}

void
SilcAccount::setHostName(const QString &name)
{
  configGroup()->writeEntry(CONFIG_HOSTNAME, name);
  setAccountLabel(QString("%1@%2").arg(nickName()).arg(name));
}

const QString
SilcAccount::hostName(void) const 
{
  return configGroup()->readEntry(CONFIG_HOSTNAME);
}

void
SilcAccount::setDisplayImagesInline(const bool flagStatus)
{
  configGroup()->writeEntry(CONFIG_DISPLAY_IMAGES_INLINE, flagStatus);
}

void
SilcAccount::setUseSilcMime(const bool flagStatus)
{
  configGroup()->writeEntry(CONFIG_FT_USE_SILC_MIME, flagStatus);
}

void
SilcAccount::setFtAutoAccept(const bool flagStatus)
{
  configGroup()->writeEntry(CONFIG_FT_AUTO_ACCEPT, flagStatus);
}

void
SilcAccount::setFtBind(const int flagStatus)
{
  configGroup()->writeEntry(CONFIG_FT_BIND_SELECTION, flagStatus);
}

int
SilcAccount::getFtBind(void) const 
{
  return configGroup()->readNumEntry(CONFIG_FT_BIND_SELECTION);
}

bool
SilcAccount::displayImagesInline(void) const
{
  return configGroup()->readBoolEntry(CONFIG_DISPLAY_IMAGES_INLINE);
}

bool
SilcAccount::useSilcMime(void) const
{
  return configGroup()->readBoolEntry(CONFIG_FT_USE_SILC_MIME);
}

bool
SilcAccount::ftAutoAccept(void) const
{
  return configGroup()->readBoolEntry(CONFIG_FT_AUTO_ACCEPT);
}

bool
SilcAccount::ftNoBind(void) const
{
  switch(configGroup()->readNumEntry(CONFIG_FT_BIND_SELECTION)) {
  case FT_BIND_ALWAYS:
    return FALSE;

  case FT_BIND_NEVER:
    return TRUE;

  default:
  case FT_BIND_NAT_TEST:
    return isBehindNat();
  }
}

void
SilcAccount::setQuitMessage(const QString &msg)
{
  configGroup()->writeEntry(CONFIG_QUIT_MESSAGE, msg);
}

const QString
SilcAccount::quitMessage(void) const 
{
  const QString &msg = configGroup()->readEntry(CONFIG_QUIT_MESSAGE);

  if(msg.isEmpty())
    return QString("");

  return msg;
}

void
SilcAccount::setSignChannelMessages(const bool flagStatus)
{
  configGroup()->writeEntry(CONFIG_SIGN_CHAT_MESSAGE, flagStatus);
}

bool
SilcAccount::signChannelMessages(void) const
{
  return configGroup()->readBoolEntry(CONFIG_SIGN_CHAT_MESSAGE);
}

void
SilcAccount::setSignChannelActions(const bool flagStatus)
{
  configGroup()->writeEntry(CONFIG_SIGN_CHAT_ACTION, flagStatus);
}

bool
SilcAccount::signChannelActions(void) const
{
  return configGroup()->readBoolEntry(CONFIG_SIGN_CHAT_ACTION);
}

void
SilcAccount::setSignPrivateMessages(const bool flagStatus)
{
  configGroup()->writeEntry(CONFIG_SIGN_PRIV_MESSAGE, flagStatus);
}

bool
SilcAccount::signPrivateMessages(void) const
{
  return configGroup()->readBoolEntry(CONFIG_SIGN_PRIV_MESSAGE);
}

void
SilcAccount::setSignPrivateActions(const bool flagStatus)
{
  configGroup()->writeEntry(CONFIG_SIGN_PRIV_ACTION, flagStatus);
}

bool
SilcAccount::signPrivateActions(void) const
{
  return configGroup()->readBoolEntry(CONFIG_SIGN_PRIV_ACTION);
}

void 
SilcAccount::setAttributeMood(SilcTK::SilcAttributeMood mood)
{
  configGroup()->writeEntry(CONFIG_ATTR_MOOD, mood);
}

SilcTK::SilcAttributeMood 
SilcAccount::getAttributeMood(void) const
{
  return (SilcTK::SilcAttributeMood) 
    configGroup()->readUnsignedNumEntry(CONFIG_ATTR_MOOD);
}

void 
SilcAccount::setAttributeContact(SilcTK::SilcAttributeContact contact)
{
  configGroup()->writeEntry(CONFIG_ATTR_CONTACT, contact);
}

SilcTK::SilcAttributeContact 
SilcAccount::getAttributeContact(void) const
{
  return (SilcTK::SilcAttributeContact) 
    configGroup()->readUnsignedNumEntry(CONFIG_ATTR_CONTACT);
}
void 
SilcAccount::setGeoInformations(bool allowed, double longitude, 
                                double latitude, int altitude)
{
  configGroup()->writeEntry(CONFIG_ATTR_GEO_ALLOWED, allowed);
  configGroup()->writeEntry(CONFIG_ATTR_GEO_LONG, longitude);
  configGroup()->writeEntry(CONFIG_ATTR_GEO_LAT, latitude);
  configGroup()->writeEntry(CONFIG_ATTR_GEO_ALT, altitude);
}

bool 
SilcAccount::getGeoAllowed(void) const
{
  return configGroup()->readBoolEntry(CONFIG_ATTR_GEO_ALLOWED);
}

double 
SilcAccount::getGeoLongitude(void) const
{
  return configGroup()->readDoubleNumEntry(CONFIG_ATTR_GEO_LONG);
}

double 
SilcAccount::getGeoLatitude(void) const
{
  return configGroup()->readDoubleNumEntry(CONFIG_ATTR_GEO_LAT);
}

int
SilcAccount::getGeoAltitude(void) const
{
  return configGroup()->readNumEntry(CONFIG_ATTR_GEO_ALT);
}

bool 
SilcAccount::getAttributesAllowed(void) const
{
  return configGroup()->readBoolEntry(CONFIG_ATTR_ALLOWED);
}
void 
SilcAccount::setAttributesAllowed(bool allowed)
{
  configGroup()->writeEntry(CONFIG_ATTR_ALLOWED, allowed);
}

bool 
SilcAccount::getAttributeTimezone(void) const
{
  return configGroup()->readBoolEntry(CONFIG_ATTR_TIMEZONE);
}
void 
SilcAccount::setAttributeTimezone(bool allowed)
{
  configGroup()->writeEntry(CONFIG_ATTR_TIMEZONE, allowed);
}

void 
SilcAccount::setAttributeMessage(const QString &message)
{
  configGroup()->writeEntry(CONFIG_ATTR_MESSAGE, message);
}

const QString 
SilcAccount::getAttributeMessage(void) const
{
  return configGroup()->readEntry(CONFIG_ATTR_MESSAGE);
}

void 
SilcAccount::setAttributeLanguage(const QString &language)
{
  configGroup()->writeEntry(CONFIG_ATTR_LANGUAGE, language);
}

const QString 
SilcAccount::getAttributeLanguage(void) const
{
  return configGroup()->readEntry(CONFIG_ATTR_LANGUAGE);
}

KActionMenu *
SilcAccount::actionMenu(void)
{
  const QString &status = myself()->onlineStatus().description();
  QString menuTitle = QString("%1 <%2>").arg(accountId()).arg(status);
  KActionMenu *menu = Kopete::Account::actionMenu(); 

  menu->popupMenu()->insertSeparator();

  menuJoinChannel->setEnabled(isConnected());
  menu->insert(menuJoinChannel);
  menu->insert(menuShowServer);
  menuShowFingerprint->setEnabled(isConnected()); 
  menu->insert(menuShowFingerprint); 

  return menu;
}

void 
SilcAccount::slotJoinChannel(void)
{
  assert(isConnected());

  QStringList chans = configGroup()->readListEntry("Recent Channel list");
  KCompletion comp;
  comp.insertItems(chans);

  SilcJoinDlgWidget dlg(Kopete::UI::Global::mainWidget(), 0);
  dlg.setCompletionList(comp);

  while(true) {
    if(dlg.exec() != QDialog::Accepted)
      break;

    QString chan = dlg.channel();
    if(chan.isNull())
      break;

    // FIXME make sure channel-name is valid, i.e. compare against
    // regexp, maximum length, etc.  There is no rule that the channel
    // must begin with a '#' !!
    {
      slotJoinChannel(chan, dlg.founder(), dlg.auth(), dlg.password());

      // push the joined channel to first in list
      chans.remove(chan);
      chans.prepend(chan);

      configGroup()->writeEntry( "Recent Channel list", chans );
      break;
    }

    KMessageBox::error(Kopete::UI::Global::mainWidget(),
		       i18n("\"%1\" is an invalid channel name.").arg(chan),
		       i18n("SILC Plugin"));
  }
}

void 
SilcAccount::slotJoinChannel(const QString &channel, bool founder, bool auth,
			     const QString &password)
{
  SilcChannelContact *contact = contactManager()->createChannel(channel);
  assert(contact);

  contact->join(founder, auth, password);
}

void 
SilcAccount::sendSilcCommand(const QString &command, 
                             SilcTK::SilcClientCommandReply reply, 
                             void *context)
{
  if(! _conn) return;
  kdDebug() << "SILCCommand: " << command.latin1() << endl;
  assert(_client);
  SilcTK::SilcUInt16 cmd_ident = 
    SilcTK::silc_client_command_call(_client, _conn, command.latin1());
  if (reply)
    /* I don't know why there is the SilcCommand argument here. it only change
     * the value of the SilcClientCommandReply SilcCommand argument. We know
     * what we send, so we ignore it.
     */
    SilcTK::silc_client_command_pending(_conn, 0, cmd_ident, reply, context);
}


void 
SilcAccount::slotJoinedChannel(const QString &channel)
{
  SilcChannelContact *contact = contactManager()->lookupChannel(channel);
  assert(contact);

  contact->view();
}

void 
SilcAccount::setAway(bool away, const QString &reason)
{
  if(away)
    setOnlineStatus(SilcProtocol::protocol()->statusGone, reason);
  
  else
    setOnlineStatus(SilcProtocol::protocol()->statusOnline, QString::null);
}




void 
SilcAccount::silc_connection_cb(SilcTK::SilcClient client,
                                SilcTK::SilcClientConnection conn,
                                SilcTK::SilcClientConnectionStatus status,
                                SilcTK::SilcStatus error,
                                const char *message,
                                void *context) 
{
  QString error_msg;
  SilcAccount *account = static_cast<SilcAccount *>(client->application);
  SilcBuddyContact *me = static_cast<SilcBuddyContact *>(account->myself());

  switch(status) {
  case SilcTK::SILC_CLIENT_CONN_SUCCESS:
  case SilcTK::SILC_CLIENT_CONN_SUCCESS_RESUME:
    kdDebug() << "client: connected" << endl;
    if(! conn->local_entry)
      return; // obviously failed to connect (key exchange failure)
    // save connection context
    account->_conn = conn;
    me->setClientEntry(conn->local_entry);

    account->setOnlineStatus(account->_wantedOnlineStatus);
    // Set UserPhoto if set
    if(! account->_globalIdentityPicture.isEmpty()) { 
      account->setAttributePicture();
      me->setProperty(Kopete::Global::Properties::self()->photo(), 
                      account->_globalIdentityPicture);
    }
    account->updateAttributes();

    // emit connected signal ...
    emit account->connected();
    break;
  case SilcTK::SILC_CLIENT_CONN_DISCONNECTED:
    kdDebug() << "client: disconnected" << endl;
    account->_conn = 0;
    account->setOnlineStatus(SilcProtocol::protocol()->statusOffline);

    emit account->disconnected();

    if(error)
      KMessageBox::queuedMessageBox
        (Kopete::UI::Global::mainWidget(), KMessageBox::Sorry, 
         SilcTK::silc_get_status_message(error),
         i18n("Kopete SilcTK::SILC Plugin"));
    break;
  case SilcTK::SILC_CLIENT_CONN_ERROR:
    error_msg = i18n("unknown error");
    goto display_error;
  case SilcTK::SILC_CLIENT_CONN_ERROR_KE:
    error_msg = i18n("key exchange failed");
    goto display_error;
  case SilcTK::SILC_CLIENT_CONN_ERROR_AUTH:
    error_msg = i18n("authenitication failed");
    goto display_error;
  case SilcTK::SILC_CLIENT_CONN_ERROR_RESUME:
    error_msg = i18n("resume error");
    goto display_error;
  case SilcTK::SILC_CLIENT_CONN_ERROR_TIMEOUT:
    error_msg = i18n("timeout error");
display_error:
    KMessageBox::queuedMessageBox
      (Kopete::UI::Global::mainWidget(), KMessageBox::Sorry, 
       i18n("Unable to establish connection to the remote host: %1")
       .arg(error_msg), i18n("Kopete SilcTK::SILC Plugin"));
    account->_conn = 0;
    break;
  }
}

///
/// callback, telling that libsilc received the command and is now
/// going to try and execute it ...
///
///////////////////////////////////////////////////////////////////////////////
void
SilcAccount::silc_command(SilcTK::SilcClient client, 
			  SilcTK::SilcClientConnection,
			  SilcTK::SilcBool /* success */,
			  SilcTK::SilcCommand command, 
			  SilcTK::SilcStatus status,
                          SilcTK::SilcUInt32 /* argc */,
                          unsigned char ** /* argv */)
{
  if (status != SILC_STATUS_OK) {
    std::cerr << "MyBot: COMMAND " 
	      << SilcTK::silc_get_command_name(command) << ": "
	      << SilcTK::silc_get_status_message(status) << std::endl;
    return;
  }
}


///
/// callback function, telling that a command was executed
///
///////////////////////////////////////////////////////////////////////////////
void 
SilcAccount::silc_command_reply(SilcTK::SilcClient client, 
				SilcTK::SilcClientConnection,
				SilcTK::SilcCommand command, 
                                SilcTK::SilcStatus status,
				SilcTK::SilcStatus error, 
                                va_list va)
{
  SilcAccount *account = static_cast<SilcAccount *>(client->application);

  if(command == SILC_COMMAND_WATCH && status != SILC_STATUS_OK) 
    return;

  kdDebug() << SilcTK::silc_get_command_name(command) << status << " " << error << endl;
  if(status >=  SILC_STATUS_ERR_NO_SUCH_NICK) {
    // warn, that a command has failed, but don't do so for 
    // no-such-nick errors 'caused by /whois requests, since these
    // are usually initiated by ourselves (when logging on)
    account->myServer()->appendMessage
      (QString("SILC Command %1 failed: %2")
       .arg(SilcTK::silc_get_command_name(command))
       .arg(SilcTK::silc_get_status_message(error)));

    // simply ignore some error messages ...
    if(command == SILC_COMMAND_WHOIS
       && error == SILC_STATUS_ERR_NO_SUCH_NICK)
      return;

    if(command == SILC_COMMAND_WATCH
       && error == SILC_STATUS_ERR_NICKNAME_IN_USE)
      return;

    // warn finally ....
    KMessageBox::queuedMessageBox
      (Kopete::UI::Global::mainWidget(), KMessageBox::Sorry,
       i18n(SilcTK::silc_get_status_message(error)),
       QString(i18n("SILC Command %1 failed"))
       .arg(SilcTK::silc_get_command_name(command)));
    return;
  }

  if(command == SILC_COMMAND_JOIN) {
    (void) va_arg(va, char *); /* channelname */
    SilcTK::SilcChannelEntry ce = va_arg(va, SilcTK::SilcChannelEntry);
    /* SilcUInt32 channel_mode */ (void) va_arg(va, SilcTK::SilcUInt32); 
    SilcTK::SilcHashTableList *user_list = 
      va_arg(va, SilcTK::SilcHashTableList *);
    
    // we successfully have joined the channel, open a chat window ...
    SilcContactManager *cm = account->contactManager();
    SilcChannelContact *ch = cm->lookupChannel(QString(ce->channel_name));
    assert(ch);

    ch->setChannelEntry(ce);
    ch->view();


    // now insert the list of joined buddies ...
    SilcTK::SilcChannelUser chu;
    while (SilcTK::silc_hash_table_get(user_list, NULL, (void **)&chu)) {
      if(!chu->client->nickname[0]) continue;

      SilcTK::SilcClientEntry e = (SilcTK::SilcClientEntry) chu->client;

      // either lookup buddy or create a new one ....
      SilcBuddyContact *buddy = e->context 
        ? (SilcBuddyContact *) e->context 
        : account->contactManager()->createBuddy(e->nickname, NULL, e);

      // this is to suppress useless `user is now online' messages ...
      if(buddy->onlineStatus() == SilcProtocol::protocol()->statusOffline)
        buddy->setOnlineStatus(SilcProtocol::protocol()->statusOnline);

      // add buddy to list of chat members ...
      ch->updateBuddyOnlineStatus(buddy);

      // send a whois request ...
      buddy->whoami();
    }
  }

  if(command == SILC_COMMAND_WHOIS) {
    SilcTK::SilcClientEntry client_entry = va_arg(va, SilcTK::SilcClientEntry);
    QString nick = QString::fromUtf8(va_arg(va, char*));
    QString username = QString::fromUtf8(va_arg(va, char*));
    QString realname = QString::fromUtf8(va_arg(va, char*));
    SilcTK::SilcDList channels = va_arg(va, SilcTK::SilcDList);
    /* SilcTK::SilcUInt32 usermode */ (void) va_arg(va, SilcTK::SilcUInt32);
    /* SilcTK::SilcUInt32 idletime */ (void) va_arg(va, SilcTK::SilcUInt32);
    unsigned char *fingerprint = va_arg(va, unsigned char*);
    /* SilcTK::SilcUInt32 *ch_umodes */ (void) va_arg(va, SilcTK::SilcUInt32);
    SilcTK::SilcDList attrs = va_arg(va, SilcTK::SilcDList);
    SilcBuddyContact *buddy = (SilcBuddyContact *) client_entry->context;
    SilcContactManager *cm = account->contactManager();


    if(! buddy)
      buddy = cm->createBuddy(QString::fromUtf8(client_entry->nickname),
			      NULL, client_entry);
    
    // Update Attributes, if some arrived
    if(attrs) 
      buddy->attributes->updateAttributes(attrs);
    

   //handle added buddies which do not have a valid fingerprint yet
    SilcBuddyContactData *old_buddy =
      cm->popPendingBuddy(QString::fromUtf8(client_entry->nickname));

    if(old_buddy) {
      buddy = new SilcBuddyContact(old_buddy->account,
				   old_buddy->nickName(),
				   buddy->convFingerprint((const char*)
							  fingerprint),
				   old_buddy->meta);

      buddy->setClientEntry(client_entry);
      cm->addBuddy(buddy);
      delete old_buddy;

      //online status
      buddy->watchme(true);
    }
    
    // store channel list for later use (i.e. display in userinfo widget)
    if(channels) {
      QStringList chlist;
      SilcTK::SilcChannelPayload e;
      SilcTK::silc_dlist_start(channels);
	
	while ((e = (SilcTK::SilcChannelPayload) SilcTK::silc_dlist_get(channels))
	       != SILC_LIST_END) {
	  SilcTK::SilcUInt32 len;
	  const char *name = (char *) SilcTK::silc_channel_get_name(e, &len);
	  chlist.append(QString::fromUtf8(name));
	}
      buddy->setChannelList(chlist);
    }

    // updateWhois should be one of the latest (since the info-widgets
    // updates channellist etc. upon updateWhois signal ...)
    buddy->updateWhois(username,realname);
    setBuddyOnlineStatus(cm,buddy,client_entry->mode);
  }

  if(command == SILC_COMMAND_GETKEY) {
    SilcTK::SilcUInt32 id_type = va_arg(va, SilcTK::SilcUInt32);
    void *entry = va_arg(va, void *);
    SilcTK::SilcPublicKey pubkey = va_arg(va, SilcTK::SilcPublicKey);

    // we store client key's only, server and router keys are not of any
    // interest for us at the moment ...
    if(id_type == SILC_ID_CLIENT) {
      SilcTK::SilcUInt32 pk_len;
      unsigned char *pk = SilcTK::silc_pkcs_public_key_encode(pubkey, &pk_len);
      char *fpKey = SilcTK::silc_hash_fingerprint(0, pk, pk_len);
      QString fp = QString(fpKey).replace(QChar(' '), QChar(':'));
      SilcTK::silc_free(fpKey);

      SilcBuddyContact *buddy = (SilcBuddyContact *)
	((SilcTK::SilcClientEntry) entry)->context;

      if(buddy->fingerprint().compare(fp))
	KMessageBox::queuedMessageBox
	  (Kopete::UI::Global::mainWidget(), KMessageBox::Sorry,
	   QString(i18n("Received client public key of %1. \n"
			"Fingerprint: %2\n\n"
			"However this does not match the cached client's "
			"fingerprint which appears to be `%3'. "
			"This probably should not happen"))
	   .arg(buddy->nickName()).arg(fp).arg(buddy->fingerprint()),
	   QString(i18n("Incoming Public Key")));

      else {
	// fingerprint of public key does match ...
        if(!buddy->fpTrusted()) { //if not trusting already
	  int answer = KMessageBox::questionYesNo
	    (Kopete::UI::Global::mainWidget(),
	     QString(i18n("Received client public key of %1. \n"
		  	  "Fingerprint: %2\n"
			  "Would you like to accept the key?"))
	     .arg(buddy->nickName()).arg(fp),
	     i18n("Incoming Public Key"));
	  if(answer == KMessageBox::Yes) {
            buddy->setFpTrusted(true);
          } else {
            SilcTK::silc_free(pk);
            return;
          }
         }
         QString fn = buddy->publicKeyPath(fp);
         if(! SilcTK::silc_pkcs_save_public_key(fn.latin1(), pubkey, 
                                                SilcTK::SILC_PKCS_FILE_BASE64))
           KMessageBox::queuedMessageBox
             (Kopete::UI::Global::mainWidget(), KMessageBox::Sorry,
              QString(i18n("Unable to save received client public key of %1."))
              .arg(buddy->nickName()),
              QString(i18n("Incoming Public Key")));
       }
      SilcTK::silc_free(pk);
    }
  }

  va_end(va);
}


QString
SilcAccount::pubKeyPath(void)
{
  return locateLocal("appdata", pubKeyName);
}

QString
SilcAccount::privKeyPath(void)
{
  return locateLocal("appdata", prvKeyName);
}

void
SilcAccount::copyKey(QString keyPath, bool prvKey)
{
  QFile input(keyPath);
  QFile output; 
  char tmpByte;

  if(!input.open(IO_ReadOnly)) {
    KMessageBox::error(Kopete::UI::Global::mainWidget(),
                       QString(i18n("Cannot open %1 for reading").arg(input.name())),
                       i18n("Opening Key File"));
    return;
  }
  if(prvKey) { 
    output.setName(locateLocal("appdata",prvKeyName));
  } else {  
    output.setName(locateLocal("appdata",pubKeyName));
  }  
  if(!output.open(IO_WriteOnly)){
     input.close();  
     KMessageBox::error(Kopete::UI::Global::mainWidget(),
                       QString(i18n("Cannot open %1 for writing").arg(output.name())),
                       i18n("Opening Key File"));

     return;
  }

  QDataStream inputstream(&input);
  QDataStream outputstream(&output);
  while(!inputstream.atEnd()) {
    inputstream.readRawBytes(&tmpByte,1);
    outputstream.writeRawBytes(&tmpByte,1) ;
  }
  output.close();
  input.close();
}


static QString
id_with_type_to_name(SilcTK::SilcIdType id_type, void *entry)
{
  if(! entry)
    return QString::null;

  switch(id_type) {
  case SILC_ID_CLIENT:
    return QString::fromUtf8(((SilcTK::SilcClientEntry) entry)->nickname);

  case SILC_ID_SERVER:
    return QString::fromUtf8(((SilcTK::SilcServerEntry) entry)->server_name);

  case SILC_ID_CHANNEL:
    return QString::fromUtf8(((SilcTK::SilcChannelEntry) entry)->channel_name);

  default:
    return QString::null;
  }
}

void 
SilcAccount::silc_notify(SilcTK::SilcClient client, 
			 SilcTK::SilcClientConnection,
			 SilcTK::SilcNotifyType type, ...)
{
  SilcAccount *account = (SilcAccount *) client->application;
  va_list va;
  va_start(va, type);

  switch(type) {

    ///////////////////////////////////////////////////////////////////////////
  case SILC_NOTIFY_TYPE_NONE:
    // received some generic text, usually not too important ...

  case SILC_NOTIFY_TYPE_MOTD: {
    // Recieved motion of the day ...
    // ... simply dump these to the server window
    const char *str = va_arg(va, char *);
    account->myServer()->appendMessage(QString(str));
    break;
  }



    ///////////////////////////////////////////////////////////////////////////
  case SILC_NOTIFY_TYPE_JOIN: {
    // Received 'user has joined' notification
    SilcTK::SilcClientEntry ce = va_arg(va, SilcTK::SilcClientEntry);

    SilcBuddyContact *buddy = ce->context
      ? (SilcBuddyContact *) ce->context
      : account->contactManager()->createBuddy(ce->nickname, NULL, ce);

    SilcChannelContact *channel = 
      (SilcChannelContact *) va_arg(va, SilcTK::SilcChannelEntry)->context;
    if(!channel) break; /* channel is nil, when we are the joined client */ 

    // this is to suppress useless `user is now online' messages ...
    if(buddy->onlineStatus() == SilcProtocol::protocol()->statusOffline)
      buddy->setOnlineStatus(SilcProtocol::protocol()->statusOnline);

    // let updateBuddyOnlineStatus happily add the buddy to the channel list
    channel->updateBuddyOnlineStatus(buddy);

    // send a whois request ...
    buddy->whoami();
    break;
  }




    ///////////////////////////////////////////////////////////////////////////
  case SILC_NOTIFY_TYPE_LEAVE: {
    // received `user has left' notification 
    SilcBuddyContact *buddy = 
      (SilcBuddyContact *) va_arg(va, SilcTK::SilcClientEntry)->context;
    assert(buddy);

    SilcChannelContact *channel = 
      (SilcChannelContact *) va_arg(va, SilcTK::SilcChannelEntry)->context;
    assert(channel);

    // FIXME, is there really no leave message available ??

    if(! channel->isJoined(buddy))
      channel->manager()->removeContact(buddy);
    break;
  }

    ///////////////////////////////////////////////////////////////////////////
  case SILC_NOTIFY_TYPE_KICKED: {
    SilcBuddyContact *buddy =
        (SilcBuddyContact *) va_arg(va, SilcTK::SilcClientEntry)->context;
    assert(buddy);
    
    /* Kick Message, maybe NULL */
    const char *str = va_arg(va, char *);

    /* Kicker Maybe NULL, maybe somebody we don't have on our buddylist */
    SilcTK::SilcClientEntry kicker =
        va_arg(va, SilcTK::SilcClientEntry );

    SilcChannelContact *channel = 
      (SilcChannelContact *) va_arg(va, SilcTK::SilcChannelEntry)->context;
    assert(channel);

    QString r = i18n("Kicked by %1.").arg(kicker ? QString(kicker->nickname) 
                                          : QString("annonymous"));
    if(str)
      r.append(i18n(" Reason: %2").arg(QString(str)));
    
    if(account->myself() != buddy) {
      // don't remove the SilcBuddyContact from the chat members list,
      // if another ClientEntry is joined to this channel
      // (unfortunately we cannot tell about the kick in this case)
      if(! channel->isJoined(buddy))
	channel->manager()->removeContact(buddy, r);
    }
    else {
        // FIXME: This is not the fine british way ( a /leave will be sent )
        KMessageBox::queuedMessageBox
	  (Kopete::UI::Global::mainWidget(), KMessageBox::Sorry, r,
	   i18n("You were kicked"));
        channel->manager()->view()->closeView();
    }
    break;
  }


    ///////////////////////////////////////////////////////////////////////////
  case SILC_NOTIFY_TYPE_KILLED: {
    // received `user has been killed from network' notification
    SilcTK::SilcClientEntry ce = va_arg(va, SilcTK::SilcClientEntry);
    SilcBuddyContact *buddy = (SilcBuddyContact *) ce->context;
    assert(buddy);

    kdDebug() << "received kill notification for "
	      << buddy->nickName() << endl;
    
    // kill message, might be NULL
    const char *kill_message = va_arg(va, char *);

    SilcTK::SilcIdType killer_type = (SilcTK::SilcIdType) va_arg(va, int);
    void *killer_id = (void *) va_arg(va, void *);
    QString killer = id_with_type_to_name(killer_type, killer_id);

    // what for?  the buddy is killed from the network, not a channel !?
    (void) va_arg(va, SilcTK::SilcChannelEntry);

    QString reason = kill_message
      ? i18n("Given reason: %1").arg(QString::fromUtf8(kill_message))
      : i18n("No reason given.");

    if(account->myself() == buddy) {
      // we have been killed ...
      KMessageBox::queuedMessageBox
	(Kopete::UI::Global::mainWidget(), KMessageBox::Sorry,
	 i18n("You have been killed from the SILC network by %1. ")
	 .arg(killer).append(reason), i18n("You have been killed"));
      //account->setOnlineStatus(SilcProtocol::protocol()->statusOffline);
    }
    else {
      // someone else has been killed ...
      buddy->removeClientEntry(ce);
      account->contactManager()->buddySignedOff(buddy, i18n("Killed by %1. ")
						.arg(killer).append(reason));
    }

    break;
  }


    ///////////////////////////////////////////////////////////////////////////
  case SILC_NOTIFY_TYPE_SIGNOFF: {
    // received `user has quit' notification 
    SilcTK::SilcClientEntry ce = va_arg(va, SilcTK::SilcClientEntry);
    const char *msg = va_arg(va, char *);

    SilcBuddyContact *buddy = (SilcBuddyContact *) ce->context;
    
    // if we've received a watch-notification for this quit, the
    // SilcBuddyContact is already marked offline, therefore do nothing.
    if (!buddy)
      break;

    QString quitMsg = msg ? QString::fromUtf8(msg) : QString::null;

    buddy->removeClientEntry(ce);
    account->contactManager()->buddySignedOff(buddy, quitMsg);
    break;
  }

    
    ///////////////////////////////////////////////////////////////////////////
  case SILC_NOTIFY_TYPE_NICK_CHANGE: {
    SilcTK::SilcClientEntry ce = va_arg(va, SilcTK::SilcClientEntry);

    assert(ce->context);

    SilcBuddyContact *buddy = (SilcBuddyContact *) ce->context;
    buddy->setNickName(QString::fromUtf8(ce->nickname));
    break;
  }


    ///////////////////////////////////////////////////////////////////////////
  case SILC_NOTIFY_TYPE_CUMODE_CHANGE: {
    int idtype = va_arg(va, int);
    (void) va_arg(va, void *);
    (void) va_arg(va, SilcTK::SilcUInt32);
    SilcBuddyContact *buddy = 
      (SilcBuddyContact *) va_arg(va, SilcTK::SilcClientEntry)->context;
    SilcChannelContact *channel = 
      (SilcChannelContact *) va_arg(va, SilcTK::SilcChannelEntry)->context;

    if (idtype == SILC_ID_CLIENT)
      channel->updateBuddyOnlineStatus(buddy);
    break;
  }

	

    ///////////////////////////////////////////////////////////////////////////
  case SILC_NOTIFY_TYPE_CMODE_CHANGE: {
    (void) va_arg(va,int); //changer_id_type ignored
    (void) va_arg(va,void *); //changer_entry ignored
    int mode = va_arg(va,SilcTK::SilcUInt32);
    (void) va_arg(va,char *); //cipher_name ignored
    (void) va_arg(va,char *); //hmac_name ignored
    (void) va_arg(va,char *); //passphrase ignored
    (void) va_arg(va,SilcTK::SilcPublicKey); //founder_key ignored
    (void) va_arg(va,SilcTK::SilcDList); //channel_pubkeys ignored
    SilcChannelContact *channel = 	
      (SilcChannelContact *) va_arg(va,SilcTK::SilcChannelEntry)->context; 
    channel->setNickNameForMode(mode);
    break;
  }


    ///////////////////////////////////////////////////////////////////////////
  case SILC_NOTIFY_TYPE_INVITE: {
    (void) va_arg(va, SilcTK::SilcChannelEntry);
    char *str = va_arg(va, char *); // @todo do we have to free `str' ??
    SilcTK::SilcClientEntry ce = va_arg(va, SilcTK::SilcClientEntry);
    
    SilcBuddyContact *buddy;
    if(ce->context)
      buddy = (SilcBuddyContact *) ce->context;
    else
      buddy = account->contactManager()->createBuddy(ce->nickname, NULL, ce);
    assert(buddy);

    if(! str) {
      KMessageBox::queuedMessageBox
	(Kopete::UI::Global::mainWidget(), KMessageBox::Sorry,
	 QString(i18n("Invitation from %1 received, however no "
		      "channel name has been specified. Ignoring request."))
	 .arg(buddy->nickName()), QString(i18n("Incoming Invitation")));
      return;
    }
    
    int answer = KMessageBox::questionYesNo
      (Kopete::UI::Global::mainWidget(), 
       QString(i18n("'%1' invited you into channel '%2'. "
		    "Do you want to follow this invitation?"))
       .arg(buddy->nickName()).arg(str), i18n("Incoming Invitation"));

    if(answer == KMessageBox::Yes)
      account->slotJoinChannel(str, false, true);
	       
    break;
  }

  case SILC_NOTIFY_TYPE_WATCH: {  
    SilcTK::SilcClientEntry ce = va_arg(va,SilcTK::SilcClientEntry);
    char *new_nick = va_arg(va,char*);
    SilcTK::SilcUInt32 umode = va_arg(va,SilcTK::SilcUInt32);
    SilcTK::SilcNotifyType type = va_arg(va,SilcTK::SilcUInt32);

    SilcContactManager *cm = account->contactManager();
    SilcBuddyContact *buddy = (SilcBuddyContact *)ce->context;
    if(! buddy) buddy = cm->createBuddy(ce->nickname, NULL, ce);

    switch(type) {
      case SILC_NOTIFY_TYPE_NICK_CHANGE:
	if(new_nick)
	  buddy->setNickName(new_nick);

	else
	  // new_nick is NULL, do a whois inquiry to find it out ...
	  // buddy->setNickName(ce->nickname); <- ce->nickname is the old nick
	  buddy->whoami();
	
	break;

      case SILC_NOTIFY_TYPE_UMODE_CHANGE:
        setBuddyOnlineStatus(cm, buddy, umode);
	break;

      case SILC_NOTIFY_TYPE_NONE:
        setBuddyOnlineStatus(cm, buddy, umode);
        break;

      case SILC_NOTIFY_TYPE_SIGNOFF:
      case SILC_NOTIFY_TYPE_SERVER_SIGNOFF:
      // ignore SILC_NOTIFY_TYPE_KILLED notification for the moment, since
      // libsilc tends to notify WHO killed, not WHO WAS killed, therefore
      // we'll remove the wrong person from the list ...
      //case SILC_NOTIFY_TYPE_KILLED: 
	buddy->removeClientEntry(ce);
	cm->buddySignedOff(buddy);
        break;
    }
    break;
  }

  }
}



void 
SilcAccount::silc_get_auth_method(SilcTK::SilcClient /* client */, 
				  SilcTK::SilcClientConnection /* conn */,
				  char * /* hostname */, 
				  SilcTK::SilcUInt16 /* port */,
                                  SilcTK::SilcAuthMethod, /* authmethod */
				  SilcTK::SilcGetAuthMeth completion,
				  void *context)
{
  std::cerr << "silc_get_auth_method not fully implemented yet" << std::endl;
  completion(SILC_AUTH_NONE, NULL, 0, context);
}



void 
SilcAccount::silc_verify_public_key(SilcTK::SilcClient client, 
				    SilcTK::SilcClientConnection conn,
				    SilcTK::SilcConnectionType conn_type, 
				    SilcTK::SilcPublicKey public_key,
				    SilcTK::SilcVerifyPublicKey completion, 
				    void *context)
{
  SilcAccount *account = static_cast<SilcAccount *>(client->application);
  
  SilcTK::SilcUInt32 pk_len; 
  unsigned char* pk;
  /* encode the publickkey to revert the simplification */
  pk = SilcTK::silc_pkcs_public_key_encode(public_key, &pk_len);
  if(!pk) {
    completion(FALSE, context);
    return;
  }

  if (conn_type == SilcTK::SILC_CONN_SERVER ||
      conn_type == SilcTK::SILC_CONN_ROUTER) {
    char remote_ip[64] = "\0\0";
    SilcTK::silc_net_gethostbyname(conn->remote_host, false, 
                                   remote_ip, sizeof(remote_ip));
    const QString &host = QString("%2_%3:%4")
      .arg(conn->remote_host ? conn->remote_host : "none")
      .arg(remote_ip).arg(conn->remote_port);
    const QString &key = QString("%1key_%2")
      .arg(conn_type == SilcTK::SILC_CONN_SERVER ? "server" : "router")
      .arg(host);
    const QString &result = account->configGroup()->readEntry(key);
    

    QString finger(SilcTK::silc_hash_fingerprint(NULL, pk, pk_len));
    finger.replace(QChar(' '), QChar(':'));
    
    if(result.isEmpty()) {
      // ask user what to do
      int answer = KMessageBox::questionYesNo
	(Kopete::UI::Global::mainWidget(), 
	 QString(i18n("The authenticity of host '%1 (%2)' can't be "
		      "established. Key fingerprint is %3.\n\n"
		      "Are you sure you want to continue connecting?"))
	 .arg(conn->remote_host).arg(remote_ip).arg(finger), 
	 i18n("Kopete SILC Plugin"));

      if(answer == KMessageBox::Yes) {
	account->configGroup()->writeEntry(key, finger);
	
	// add the new host to the list of known hosts ...
	const QString &key_hostlist = QString("%1key_hostlist")
	  .arg(conn_type == SilcTK::SILC_CONN_SERVER 
	       ? "server" : "router");
	QStringList hostlist =
	  account->configGroup()->readListEntry(key_hostlist);
	hostlist.append(host);
	account->configGroup()->writeEntry(key_hostlist, hostlist);

	completion(TRUE, context);
      }
      else
	completion(FALSE, context);
    } 
    else {
      if(result.compare(finger)) {
	// warn user ...
	KMessageBox::queuedMessageBox
	  (Kopete::UI::Global::mainWidget(), KMessageBox::Sorry, 
	   QString(i18n("The public key of the remote host %1 has changed. "
			"According to my config it used to have the "
			"fingerprint %2, however I have received the "
			"fingerprint %3 this time. This may mean that "
			"there is a man in the middle. \n"
			"Be careful, resetting connection now."))
	   .arg(conn->remote_host).arg(result).arg(finger),
	   i18n("Kopete SILC Plugin"));

	completion(FALSE, context); // key-mismatch ...
      } 
      else
	completion(TRUE, context);
    }
  }

  else if(conn_type == SilcTK::SILC_CONN_CLIENT) {
    QString finger = QString(SilcTK::silc_hash_fingerprint(NULL, pk, pk_len))
      .replace(QChar(' '), QChar(':'));
    SilcContactManager *cm = account->contactManager();
    SilcBuddyContact *buddy = cm->lookupBuddyByFingerprint(finger);
    if (buddy) {
      if (!buddy->fpTrusted()) {
        int answer = KMessageBox::questionYesNo
          (Kopete::UI::Global::mainWidget(),
           QString(i18n("Received client public key of %1. \n"
                        "Fingerprint: %2\n"
                        "Would you like to accept the key?"))
           .arg(buddy->nickName()).arg(finger),
           i18n("Incoming Public Key"));
        if (answer == KMessageBox::Yes) {
          buddy->setFpTrusted(true);
          completion(TRUE, context);
        }
        else
          completion(FALSE, context);
      }
      else 
        /* The buddy is already trusted */
        completion(TRUE, context);
    }
    else {
      KMessageBox::queuedMessageBox
        (Kopete::UI::Global::mainWidget(), KMessageBox::Sorry,
         QString(i18n("Recieved client public key, but no key was requested. \n"
                      "Fingerprint: %1\n"
                      "This shouldn't happen, so the request will be ignored."))
         .arg(finger), QString(i18n("Incoming Public Key")));
      completion(FALSE, context);
    }
  }
  
  else {
    std::cerr << "verify_public_key called for unknown conn_type" << std::endl;
    completion(FALSE, context);
  }
}

/**
 * asking a password from the user in the least memory effecting and thus most
 * secure way is the task here.
 * 
 * @todo memlock, overriding, disabling core dumps (?)
 */

void 
SilcAccount::silc_ask_passphrase(SilcTK::SilcClient /* client */, 
				 SilcTK::SilcClientConnection /* conn */,
				 SilcTK::SilcAskPassphrase completion, 
				 void *context)
{
  
 QCString password;
 int result = KPasswordDialog::getPassword(password, 
        i18n("Please enter the passphrase.\n \
              The server window may contain additional information."));
 if (result == KPasswordDialog::Accepted) {
    completion((unsigned char*)(const char*)password, password.length(),
	       context);
  } else {
    completion(NULL, 0, context); //we failed
  }
}


void 
SilcAccount::silc_say(SilcTK::SilcClient client, 
		      SilcTK::SilcClientConnection /* conn */,
		      SilcTK::SilcClientMessageType /* type */, 
		      char *msg, ...)
{
  char str[200];
  va_list va;
  va_start(va, msg);
  vsnprintf(str, sizeof(str) - 1, msg, va);
  va_end(va);

  SilcAccount *account = static_cast<SilcAccount *>(client->application);
  account->myServer()->appendMessage(QString(str));
}


void
SilcAccount::silc_key_agreement(SilcTK::SilcClient,
				SilcTK::SilcClientConnection,
				SilcTK::SilcClientEntry, 
				const char *, SilcTK::SilcUInt16,
                                SilcTK::SilcUInt16)
{
  std::cerr << "silc_key_agreement called." << std::endl;
}



void
SilcAccount::silc_ftp(SilcTK::SilcClient client,
		      SilcTK::SilcClientConnection conn,
		      SilcTK::SilcClientEntry client_entry,
		      SilcTK::SilcUInt32 session_id,
		      const char *hostname, SilcTK::SilcUInt16 port)
{
  kdDebug() << "Filetransfer from " << client_entry->nickname << endl;

  SilcAccount *account = static_cast<SilcAccount *>(client->application);
  SilcBuddyContact *buddy = (SilcBuddyContact *) client_entry->context;

  if(!buddy) {
    /* Buddy is not in our buddylist, create temporary buddy */
    SilcContactManager *cm = account->contactManager();
    buddy = cm->createBuddy(QString::fromUtf8(client_entry->nickname),
                            NULL, client_entry);
    client_entry->context = buddy;
  }

  if(account->ftNoBind() && !hostname) {
    KMessageBox::queuedMessageBox
      (Kopete::UI::Global::mainWidget(), KMessageBox::Sorry,
       QString(i18n("Received incoming file transfer request from %1, "
		    "however the sender requested, that the receiver "
		    "shall bind a listening port. Unfortunately the latter is "
		    "forbidden according this plugin's configuration.\n\n"
		    "The file transfer request has been rejected."))
       .arg(buddy->nickName()), i18n("Incoming File Transfer"));
    SilcTK::silc_client_file_close(client, conn, session_id);
    return;
  }

  if(account->ftAutoAccept()) {
    new SilcFileTransfer(account, buddy, session_id, false);
  }

  // else, we ought not auto-accept the file, therefore block and ask the user
  else {
    int answer = KMessageBox::questionYesNo
      (Kopete::UI::Global::mainWidget(),
       QString(i18n("Received incoming file transfer request from %1 "
		    "(%2:%3). Do you want to receive and store "
		    "this file?"))
       .arg(buddy->nickName()).arg(hostname).arg(port),
       i18n("Incoming File Transfer"));


    if(answer != KMessageBox::Yes) {
      SilcTK::silc_client_file_close(client, conn, session_id);
      return;
    }

    new SilcFileTransfer(account, buddy, session_id);
  }
}



///
/// update the ourself's user mode
///
///////////////////////////////////////////////////////////////////////////////
static void
sendUmode(SilcTK::SilcClient client, 
	  SilcTK::SilcClientConnection conn,
	  SilcTK::SilcUInt32 umode)
{
  if(! conn) return;

  // calculate the new umode flags ...
  SilcTK::SilcUInt32 mode = conn->local_entry->mode;
  mode &= ~(SILC_UMODE_GONE | SILC_UMODE_HYPER | 
	    SILC_UMODE_BUSY | SILC_UMODE_INDISPOSED);
  mode |= umode;

  // convert id payload ...
  SilcTK::SilcBuffer idp = 
    SilcTK::silc_id_payload_encode(conn->local_id, SILC_ID_CLIENT);

  // convert user mode ...
  unsigned char mb[4];
  SILC_PUT32_MSB(mode, mb);

  // send the command ...
  SilcTK::silc_client_command_send(client, conn, SILC_COMMAND_UMODE,
				   NULL, NULL, 2,
				   1, idp->data,
                                   (SilcTK::SilcUInt32)(idp->tail - idp->data),
				   2, mb, sizeof(mb));

  // get rid of id payload ...
  SilcTK::silc_buffer_free(idp);
}

void
SilcAccount::setBuddyOnlineStatus(SilcContactManager *cm, 
				  SilcBuddyContact *buddy, 
				  SilcTK::SilcUInt32 mode)
{
  if(mode & (SILC_UMODE_DETACHED))
    cm->setOnlineStatus(buddy, SilcProtocol::protocol()->statusDetached);

  else if(mode & (SILC_UMODE_GONE))
    cm->setOnlineStatus(buddy, SilcProtocol::protocol()->statusGone);

  else if(mode & (SILC_UMODE_BUSY))
    cm->setOnlineStatus(buddy, SilcProtocol::protocol()->statusBusy);

  else if(mode & (SILC_UMODE_HYPER))
    cm->setOnlineStatus(buddy, SilcProtocol::protocol()->statusHyper);

  else if(mode & (SILC_UMODE_INDISPOSED))
    cm->setOnlineStatus(buddy, SilcProtocol::protocol()->statusIndisposed);

  else
    cm->setOnlineStatus(buddy, SilcProtocol::protocol()->statusOnline); 
}

void 
SilcAccount::slotGlobalIdentityChanged(const QString &key, const QVariant &value)
{
  SilcContact *me = static_cast<SilcContact *>(myself());

  if(key == Kopete::Global::Properties::self()->photo().key()) {
    _globalIdentityPicture = value.toString();
    if (me->account()->conn()) 
      setAttributePicture();
  }
}
void 
SilcAccount::setAttributePicture()
{
  SilcContact *me = static_cast<SilcContact *>(myself());
  SilcTK::silc_client_attribute_del(_client, _conn, SILC_ATTRIBUTE_USER_ICON, NULL);
  
  QFile photo(_globalIdentityPicture);
  photo.open(IO_ReadOnly);
  QByteArray data = photo.readAll();
  photo.close();

  SilcTK::SilcMime mime = SilcTK::silc_mime_alloc();

  SilcTK::silc_mime_add_field(mime, "MIME-Version", "1.0");
  SilcTK::silc_mime_add_field(mime, "Content-Transfer-Encoding", "binary");
  SilcTK::silc_mime_add_field(mime, "Content-Type", "image/png");
  SilcTK::silc_mime_add_data(mime, (unsigned char *)data.data(), data.size());


  SilcTK::silc_client_attribute_add(_client, _conn, SILC_ATTRIBUTE_USER_ICON,
                                      (void *)mime, sizeof(*mime));
  
  me->setProperty(Kopete::Global::Properties::self()->photo(), _globalIdentityPicture);
}

  void 
SilcAccount::updateAttributes(void)
{
  if(conn()) {
    // Deleting all
    SilcTK::silc_client_attribute_del(_client, _conn, 
                                      SILC_ATTRIBUTE_STATUS_MOOD, NULL);
    SilcTK::silc_client_attribute_del(_client, _conn, 
                                      SILC_ATTRIBUTE_PREFERRED_CONTACT, NULL);
    SilcTK::silc_client_attribute_del(_client, _conn, 
                                      SILC_ATTRIBUTE_PREFERRED_LANGUAGE, NULL);
    SilcTK::silc_client_attribute_del(_client, _conn, 
                                      SILC_ATTRIBUTE_GEOLOCATION, NULL);
    SilcTK::silc_client_attribute_del(_client, _conn, 
                                      SILC_ATTRIBUTE_STATUS_MESSAGE, NULL);
    if(getAttributesAllowed()) {
      // mood
      SilcTK::SilcAttributeMood mood = getAttributeMood();
      SilcTK::silc_client_attribute_add(_client, _conn, SILC_ATTRIBUTE_STATUS_MOOD,
                                        (void *)mood, sizeof(mood));
      // contact
      SilcTK::SilcAttributeContact contact = getAttributeContact();
      if(contact)
        SilcTK::silc_client_attribute_add(_client, _conn, 
                                          SILC_ATTRIBUTE_PREFERRED_CONTACT,
                                          (void *)contact, sizeof(contact));
      // geolocation
      if(getGeoAllowed()) {
        SilcTK::SilcAttributeObjGeo geo;

        QString longitude =  QString("%1").arg(getGeoLongitude());
        QString latitude = QString("%1").arg(getGeoLatitude());
        QString altitude =  QString("%1 m").arg(getGeoAltitude());

        geo.longitude = (char *) longitude.latin1();
        geo.latitude = (char *) latitude.latin1();
        geo.altitude = (char *) altitude.latin1(); 
        silc_client_attribute_add(_client, _conn,
                                  SILC_ATTRIBUTE_GEOLOCATION, &geo,
                                  sizeof(geo));
      }
      // Language
      if(!getAttributeLanguage().isEmpty()) {
        QStringList list = QStringList::split(QRegExp("[^a-zA-Z0-9.]+"), 
                                              getAttributeLanguage());
        for(QStringList::Iterator it = list.begin(); it != list.end(); it ++) {
          silc_client_attribute_add(_client, _conn, 
                                    SILC_ATTRIBUTE_PREFERRED_LANGUAGE, 
                                    (void *)(*it).latin1(), 
                                    sizeof((*it).latin1()));
        }
      }
      if(!getAttributeMessage().isEmpty()) {
        SilcTK::SilcMime mime = SilcTK::silc_mime_alloc();

        SilcTK::silc_mime_add_field(mime, "MIME-Version", "1.0");
        SilcTK::silc_mime_add_field(mime, "Content-Transfer-Encoding", "binary");
        SilcTK::silc_mime_add_field(mime, "Content-Type",
                                    "text/plain; charset=utf-8");

        QCString content = QString(getAttributeMessage()).utf8();
        SilcTK::silc_mime_add_data(mime, (unsigned char *)(const char *)content,
                                   content.length());

        silc_client_attribute_add(_client, _conn, 
                                  SILC_ATTRIBUTE_STATUS_MESSAGE, 
                                  mime, sizeof(*mime));
      }
    }
  }
}


SilcTK::SilcClientOperations SilcAccount::ops = {
  SilcAccount::silc_say,
  SilcChannelContact::silc_channel_message,
  SilcBuddyContact::silc_private_message,
  SilcAccount::silc_notify,
  SilcAccount::silc_command,
  SilcAccount::silc_command_reply,
  SilcAccount::silc_get_auth_method,
  SilcAccount::silc_verify_public_key,
  SilcAccount::silc_ask_passphrase,
  SilcAccount::silc_key_agreement,
  SilcAccount::silc_ftp 
};

#include "silcaccount.moc"
