/*************************************************************************
 *  Copyright (C) 2008, 2010 by Volker Lanz <vl@fidra.de>                *
 *                                                                       *
 *  This program is free software; you can redistribute it and/or        *
 *  modify it under the terms of the GNU General Public License as       *
 *  published by the Free Software Foundation; either version 3 of       *
 *  the License, or (at your option) any later version.                  *
 *                                                                       *
 *  This program is distributed in the hope that it will be useful,      *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *  GNU General Public License for more details.                         *
 *                                                                       *
 *  You should have received a copy of the GNU General Public License    *
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.*
 *************************************************************************/

#include "gui/partwidget.h"

#include "core/partition.h"
#include "fs/filesystem.h"
#include "util/capacity.h"

#include <QPainter>
#include <QStyleOptionButton>
#include <QApplication>
#include <QFontDatabase>

/** Creates a new PartWidget
    @param parent pointer to the parent widget
    @param p pointer to the Partition this widget will show. must not be nullptr.
*/
PartWidget::PartWidget(QWidget* parent, const Partition* p) :
    PartWidgetBase(parent),
    m_Partition(nullptr),
    m_Active(false)
{
    setFont(QFontDatabase::systemFont(QFontDatabase::SmallestReadableFont));
    init(p);
    m_fileSystemColorCode = FileSystem::defaultColorCode;
}

void PartWidget::init(const Partition* p)
{
    m_Partition = p;

    if (partition())
        setToolTip(partition()->deviceNode() + QStringLiteral("\n") + partition()->fileSystem().name() + QStringLiteral(" ") + QString(Capacity::formatByteSize(partition()->capacity())));
    else
        setToolTip(QString());

    updateChildren();
}

/** Updates the widget's children */
void PartWidget::updateChildren()
{
    if (partition()) {
        for (const auto &w : childWidgets()) {
            w->setVisible(false);
            w->deleteLater();
            w->setParent(nullptr);
        }

        for (const auto &child : partition()->children()) {
            QWidget* w = new PartWidget(this, child);
            w->setVisible(true);
        }

        positionChildren(this, partition()->children(), childWidgets());
    }
}

void PartWidget::setFileSystemColorCode(const std::array< QColor, FileSystem::__lastType >& colorCode)
{
    m_fileSystemColorCode = colorCode;
    repaint();
}

void PartWidget::resizeEvent(QResizeEvent*)
{
    if (partition())
        positionChildren(this, partition()->children(), childWidgets());
}

QColor PartWidget::activeColor(const QColor& col) const
{
    return isActive() ? col.darker(190) : col;
}

void PartWidget::paintEvent(QPaintEvent*)
{
    if (partition() == nullptr)
        return;

    const int usedPercentage = static_cast<int>(partition()->used() * 100 / partition()->capacity());
    const int w = width() * usedPercentage / 100;

    QPainter painter(this);
    painter.setRenderHints(QPainter::Antialiasing);

    if (partition()->roles().has(PartitionRole::Extended)) {
        drawGradient(&painter, activeColor(
                         m_fileSystemColorCode[ partition()->fileSystem().type() ]),
                     QRect(0, 0, width(), height()));
        return;
    }

    const QColor base = activeColor(m_fileSystemColorCode[ partition()->fileSystem().type() ]);

    if (!partition()->roles().has(PartitionRole::Unallocated)) {
        const QColor dark = base.darker(105);
        const QColor light = base.lighter(120);

        // draw free space background
        drawGradient(&painter, light, QRect(0, 0, width(), height()), isActive());

        // draw used space in front of that
        drawGradient(&painter, dark, QRect(0, 0, w, height() - 1));
    } else
        drawGradient(&painter, base, QRect(0, 0, width(), height()), isActive());

    // draw name and size
    QString text = partition()->deviceNode().remove(QStringLiteral("/dev/")) + QStringLiteral("\n") + QString(Capacity::formatByteSize(partition()->capacity()));

    const QRect textRect(0, 0, width() - 1, height() - 1);
    const QRect boundingRect = painter.boundingRect(textRect, Qt::AlignVCenter | Qt::AlignHCenter, text);
    if (boundingRect.x() > PartWidgetBase::borderWidth() && boundingRect.y() > PartWidgetBase::borderHeight()) {
        if (isActive())
            painter.setPen(QColor(255, 255, 255));
        painter.drawText(textRect, Qt::AlignVCenter | Qt::AlignHCenter, text);
    }
}

void PartWidget::drawGradient(QPainter* painter, const QColor& color, const QRect& rect, bool active) const
{
    if (rect.width() < 8)
        return;

    QStyleOptionButton option;
    option.initFrom(this);
    option.rect = rect;
    option.palette.setColor(QPalette::Button, color);
    option.palette.setColor(QPalette::Window, color);
    option.state |= QStyle::State_Raised;
    if (!active)
        option.state &= ~QStyle::State_MouseOver;
    else
        option.state |= QStyle::State_MouseOver;

    style()->drawControl(QStyle::CE_PushButtonBevel, &option, painter, this);
}
