//
// C++ Implementation: kpgqueryresultwindow
//
// Description: 
//
//
// Author: Lumir Vanek <lvanek@users.sourceforge.net>, (C) 2006
//
// Copyright: See COPYING file that comes with this distribution
//
//
#include "kpgqueryresultwindow.h"


// include files for Qt
#include <qtable.h>
#include <qwidgetstack.h> 
#include <qtextcodec.h>
#include <qsplitter.h>
#include <qlabel.h>
#include <qcheckbox.h>
#include <qlayout.h>
#include <qregexp.h>
#include <qtimer.h>

// include files for KDE
#include <kcombobox.h>
#include <kpushbutton.h>
#include <ktextedit.h>
#include <kcursor.h>
#include <kmessagebox.h>
#include <kdebug.h>
#include <klocale.h>
#include <kfinddialog.h>
#include <kfind.h>

// include files for KatePart
#include <kate/document.h>  // KatePart document
#include <ktexteditor/encodinginterface.h>
#include <ktexteditor/viewstatusmsginterface.h>

// application specific includes
#include "../kpogreview.h"
#include "kpgqueryresultchildview.h"
#include "../DbObjects/kpgconnection.h"
#include "../DbObjects/kpgserver.h"
#include "../DbObjects/kpgdatabase.h"
#include "../kpgutil.h"
#include "kpgsyntaxhighlighter.h"
#include "kpgsqlparser.h"

#define NUM_OF_RECORDS_PER_PAGE 500 // How many records are loaded to result table 
#define UPDATE_TIMER_INTERVAL 1000 // 1 sec for updatiing code completion

KPGQueryResultWindow::KPGQueryResultWindow(KPGQueryResultChildView *parent, 
	KPoGreView *pPoGreView,
	KParts::Factory *pKatePartFactory,
	KXMLGUIFactory *pXmlGuiFactory,
	const QString & strNamespaceName,
	const QString & strSelectedName
	)
	: 
	KXMLGUIClient(pPoGreView),
	KPGQueryResultWindowBase(parent, "KPGQueryResultWindow"),
 	m_pPoGreView(pPoGreView),
	m_bCodeCompletionListUpdated(true),
	m_pUpdateCodeCompletionListTimer(0),
	m_pFind(0)
{
	// Setup GUI
	setXMLFile("kpgqueryreslt.rc", true);
	m_pXmlGuiFactory = pXmlGuiFactory;
	m_bIsAddedToGuiFactory = false;

	//-----------------------------------------------------------------
	// Query/result actions
	m_pActRunQuery = new KAction(i18n("&Run query"), "fork", Key_F8, this, SLOT(slotRunQuery()), actionCollection(), "queryresult_run_query");
	
	m_pActStopQuery = new KAction(i18n("&Stop query"), "stop", 0, this, SLOT(slotStopQuery()), actionCollection(), "queryresult_stop_query");
	
	m_pActFetchNext = new KAction(i18n("Fetch &next"), "1downarrow", 0, this, SLOT(slotFetchNext()), actionCollection(), "queryresult_fetch_next");
	
	m_pActFetchAll = new KAction(i18n("Fetch &all"), "2downarrow", 0, this, SLOT(slotFetchAll()), actionCollection(), "queryresult_fetch_all");
		
	m_pActFindInResult = new KAction(i18n("&Find in result..."), "find", 0, this, SLOT(slotFindInResult()), actionCollection(), "edit_find_in_result");
	m_pActFindInResult->setWhatsThis(i18n("Find in result table."));
	
	m_pActFindNextInResult = new KAction(i18n("Find next in result"), "next", 0, this, SLOT(slotFindNextInResult()), actionCollection(), "edit_find_next_in_result");
	m_pActFindNextInResult->setWhatsThis(i18n("Find next in result table."));
    
	m_pActRunQuery->setEnabled(false);
	m_pActStopQuery->setEnabled(false);
	m_pActFetchNext->setEnabled(false);
  	m_pActFetchAll->setEnabled(false);
  	
  	//-----------------------------------------------------------------
	// Common clipboard actions
	m_pActCopyCell = new KAction(i18n("&Copy cell"), "editcopy", 0, this, SLOT(slotCopyCell()), actionCollection(), "copy_cell");
	
	m_pActCopyRow = new KAction(i18n("Copy &row"), 0, 0, this, SLOT(slotCopyRow()), actionCollection(), "copy_row");
	
	m_pActCopyTableCsv = new KAction(i18n("Copy table to C&SV"), 0, 0, this, SLOT(slotCopyTableCsv()), actionCollection(), "copy_table_csv");
	
	m_pActCopyTableXml = new KAction(i18n("Copy table to &XML"), 0, 0, this, SLOT(slotCopyTableXml()), actionCollection(), "copy_table_xml");
	
	//-----------------------------------------------------------------
	// Create Kate part editor
    m_pKateView = createKatePart(pKatePartFactory);
	m_pFrameEditQueryLayout->addWidget(m_pKateView, 0, 0 );
			
	// install a working kate part popup dialog thingy
  	if (static_cast<Kate::View*>(m_pKateView->qt_cast("Kate::View")))
  	{
    	static_cast <Kate::View*> (m_pKateView->qt_cast("Kate::View"))->installPopup ((QPopupMenu*)(pXmlGuiFactory->container("ktexteditor_popup", pPoGreView)) );
  	}
    
    m_pKateView->setLineNumbersOn(true);
    m_pKateView->setIconBorder(true);
	
	KTextEditor::HighlightingInterface *pHighlightIface = KTextEditor::highlightingInterface(m_pKateView->getDoc());
        
    m_iHighlightMode = 0;
    for(uint i = 0; i < pHighlightIface->hlModeCount(); i++) 
    {
    	//kdDebug() << "hlmode("<<i<<"): " << pHighlightIface->hlModeName(i) << endl;
        if (pHighlightIface->hlModeName(i).contains("SQL (PostgreSQL)", false))
        {
        	pHighlightIface->setHlMode(i);
        	m_iHighlightMode = i;
        	break;
        }
     }
	
	QString strSetSearchPath;
	
	if(strNamespaceName != QString::null)
	{
		strSetSearchPath = "SET search_path TO " + strNamespaceName + ";\n";
	}
	
	setEditorText(strSetSearchPath + "SELECT * FROM ;");
	 		
 	if(strSelectedName != QString::null)
 	{
 		KTextEditor::EditInterface *pEditIface = KTextEditor::editInterface(m_pKateView->document());
		pEditIface->insertText((strNamespaceName == QString::null) ? 0 : 1, 14, strSelectedName);
 	}
 		
 	viewCursorInterface(m_pKateView)->setCursorPosition((strNamespaceName == QString::null) ? 0 : 1, 14);	
 		
	connect(m_pKateView->document(), SIGNAL(textChanged()), this, SLOT(slotTextEditQueryTextChanged()));
	
	connect(m_pKateView->document(), SIGNAL(fileNameChanged()), this, SLOT(slotFileNameChanged()));
		
	connect(m_pTableResult, SIGNAL(contextMenuRequested(int, int, const QPoint &)), this, SLOT(slotContextMenuRequested(int, int, const QPoint &)));
	
	m_ConnectionInThread.setEventReceiver(this, &m_pqxxResult);
	
	m_nTotalRows = 0;
	m_nTotalCols = 0;
	m_nFetchedRows = 0;
   
    m_pFrameOptions->hide();
    
    //-----------------------------------------------------------------
	// Setup status message handling
    //KTextEditor::ViewStatusMsgInterface *pViewStatusMsgIface = KTextEditor::viewStatusMsgInterface(m_pKateView);
    
    connect(m_pKateView, SIGNAL(viewStatusMsg(const QString &)), this, SLOT(slotTextEditStatusMsg(const QString &)));
    
    //-----------------------------------------------------------------
    // Add SQL keywords to code completion list
    clearListOfObjectsForCodeCompletion();
    m_eCompletionMode = modeNone;
        
    connect(m_pKateView, SIGNAL(completionDone()), this, SLOT(slotCompletionDone()));
    connect(m_pKateView, SIGNAL(completionAborted()), this, SLOT(slotCompletionAborted()));
    
    // Setup timer
	m_pUpdateCodeCompletionListTimer = new QTimer(this);
    connect(m_pUpdateCodeCompletionListTimer, SIGNAL(timeout()), this, SLOT(slotUpdateCodeCompletionList()));
	m_pUpdateCodeCompletionListTimer->start( UPDATE_TIMER_INTERVAL ); 
}

KPGQueryResultWindow::~KPGQueryResultWindow()
{
	m_pUpdateCodeCompletionListTimer->stop(); // stop timer, but don't delete it
	
	if(m_pFind != 0) 
    {
    	disconnect( m_pFind, 0, this, 0);
    	delete m_pFind;
    }
	
	if(m_pKateView)
    {
    	m_pKateView->removeEventFilter(this);
        removeFromGuiFactory();

        // remove the view, then the doc
        Kate::Document *pDoc = m_pKateView->getDoc();
        delete m_pKateView;
        delete pDoc;
    }
}

// Add yourself and Kate view to GUI factory
void KPGQueryResultWindow::addToGuiFactory()
{
	if(m_pKateView && !m_bIsAddedToGuiFactory)
    {
        m_pXmlGuiFactory->addClient(this);
        m_pXmlGuiFactory->addClient(m_pKateView);
        m_bIsAddedToGuiFactory = true;
        m_pKateView->setFocus();
    }
}
    
// Remove yourself and Kate view from GUI factory
void KPGQueryResultWindow::removeFromGuiFactory()
{
    if(m_pKateView && m_bIsAddedToGuiFactory)
    {
        m_pXmlGuiFactory->removeClient(this);
        m_pXmlGuiFactory->removeClient(m_pKateView);
        m_bIsAddedToGuiFactory = false;
    }
}

// Create Kate part view
Kate::View* KPGQueryResultWindow::createKatePart(KParts::Factory* pFactory)
{
    // The library was found, so create the Kate::Document
    KTextEditor::Document *doc = (KTextEditor::Document *) pFactory->createPart(m_pFrameEditQuery, "", this, "", "KTextEditor::Document");

    // The document only represents the document, to view
    // the document's content
    // we have to create a view for the document.
    Kate::View *pView = (Kate::View *) doc->createView( m_pFrameEditQuery, 0L );
	pView->installEventFilter(this); // we need watch kaybord events in our eventFilter
	
    // all went well, so return the view
    return pView;
}

// Display popup menu
void KPGQueryResultWindow::slotContextMenuRequested(int, int, const QPoint &pos)
{
	QWidget * pContainer = m_pXmlGuiFactory->container("popupQueryResultTable", this);

	if ( ! pContainer )
	{
		kdError() << k_funcinfo << " Couldn't get a container widget for the given menu name popupQueryResultTable" << endl;
		return;
	}

	if ( ! pContainer->inherits("KPopupMenu") )
	{
		kdError() << k_funcinfo << " Wrong container widget" << endl;
		return;
	}

	KPopupMenu * pMenu = static_cast <KPopupMenu*> (pContainer);
	pMenu->popup( pos );
}

// Set editor text
void KPGQueryResultWindow::setEditorText(const QString &strTest)
{
	// Setup edited text
	KTextEditor::EditInterface *pEditIface = KTextEditor::editInterface(m_pKateView->document());
	pEditIface->setText(strTest);
		
	KTextEditor::UndoInterface *pUndoIface = KTextEditor::undoInterface(m_pKateView->document());
	pUndoIface->clearUndo();
	pUndoIface->clearRedo();

	m_pKateView->document()->setModified(false);
}

// Open document from given URL
void KPGQueryResultWindow::loadURL(const KURL &url)
{
  	m_pKateView->document()->openURL(url);
  	
  	KTextEditor::HighlightingInterface *pHighlightIface = KTextEditor::highlightingInterface(m_pKateView->getDoc());
  	
  	pHighlightIface->setHlMode(m_iHighlightMode);
}

// Get editor text
const QString KPGQueryResultWindow::editorText() const 
{
	KTextEditor::EditInterface *pEditIface = KTextEditor::editInterface(m_pKateView->document());
    return pEditIface->text();
}

// Get document encoding
const QString KPGQueryResultWindow::encoding() const
{
	if(KTextEditor::encodingInterface(m_pKateView->getDoc()))
	{
		return KTextEditor::encodingInterface(m_pKateView->document())->encoding();
	}
	
	return QString::null;
}

// Set document encoding
void KPGQueryResultWindow::setEncoding(const QString& strEncoding)
{
	if(KTextEditor::encodingInterface(m_pKateView->document()))
	{
		KTextEditor::encodingInterface(m_pKateView->document())->setEncoding(strEncoding);
	}
}

// Get document URL
const KURL KPGQueryResultWindow::url() const
{
  	return m_pKateView->document()->url();
}

// Set flag indicating changed query text
void KPGQueryResultWindow::setQueryTextChanged(bool bModified) 
{ 
   m_pKateView->document()->setModified(bModified);
}

// Is SQL query changed after last open/save ?
bool KPGQueryResultWindow::isModified() const 
{ 
	return m_pKateView->document()->isModified();
}

// Set state of action Run query
void KPGQueryResultWindow::setEnableRunQuery(bool bEnable) 
{
    m_pActRunQuery->setEnabled(bEnable);
}

void KPGQueryResultWindow::slotServerChanged(const QString& strServer)
{
    m_pActRunQuery->setEnabled(false);
	emit sigRequestDatabasesList(this, strServer);
}

// Send SQL query to DB
void KPGQueryResultWindow::slotRunQuery()
{
    // search selected server and database
	QListViewItem * pItem = m_pPoGreView->getTreeView()->firstChild();
	KPGDatabase *pDatabase = lookupDatabase(pItem);
	if(pDatabase)
		runQuery(pDatabase);
	else
		kdError() << k_funcinfo << "Can't obtain database !" << endl;   
}

// Stop SQL operation
void KPGQueryResultWindow::slotStopQuery()
{
    stopQuery(); 
}

// Fetch next X rows from large result
void KPGQueryResultWindow::slotFetchNext()
{
	fetchNext();
}

// Fetch all rows from large result    
void KPGQueryResultWindow::slotFetchAll()
{
    fetchAll(); 
}

// Find first occurence of text in result table    
void KPGQueryResultWindow::slotFindInResult()
{
	findInResultFirst(m_listOfResultQuerySearchHistory); 
}

// Find next occurence of text in result table    
void KPGQueryResultWindow::slotFindNextInResult()
{
	findInResultNext(); 
}

void KPGQueryResultWindow::slotCopyCell()
{
	KPGUtil::copyCell(m_pTableResult);
}

void KPGQueryResultWindow::slotCopyRow()
{
	KPGUtil::copyRow(m_pTableResult);
}

void KPGQueryResultWindow::slotCopyTableCsv()
{
	KPGUtil::copyTableCsv(m_pTableResult);
}

void KPGQueryResultWindow::slotCopyTableXml()
{
	KPGUtil::copyTableXml(m_pTableResult);
}

void KPGQueryResultWindow::slotDatabaseChanged(const QString& strDatabase)
{
    if(strDatabase != QString::null)
	{
    	kdDebug() << k_funcinfo << "Database selected: " << strDatabase << endl;
    	emit sigRequestListOfDatabaseObjectsForCodeCompletion(this, strDatabase);
    }
    
    KTextEditor::EditInterface *pEditIface = KTextEditor::editInterface(m_pKateView->document());
    
    bool bEnableRunQuery = ((m_pComboBoxDatabase->count() > 0) && 
	   (pEditIface->text().length() > 0));
	   
	m_pActRunQuery->setEnabled(bEnableRunQuery);
}

void KPGQueryResultWindow::slotTextEditQueryTextChanged()
{
	KTextEditor::EditInterface *pEditIface = KTextEditor::editInterface(m_pKateView->document());
	
	bool bEnableRunQuery = ((m_pComboBoxDatabase->count() > 0) && 
	   (pEditIface->text().length() > 0));
	   

	m_pActRunQuery->setEnabled(bEnableRunQuery);
	   
	updateTabToolTip();
		
	m_bCodeCompletionListUpdated = false;
}

void KPGQueryResultWindow::slotFileNameChanged()
{
	//kdDebug() << k_funcinfo << m_pKateView->getDoc()->docName() << endl;
	emit sigFileNameChanged(m_pKateView->getDoc()->docName());
}

// Set KATE editor status message
void KPGQueryResultWindow::slotTextEditStatusMsg(const QString &strMessage)
{
	m_pLabelKateStatus->setText(strMessage);
}

bool KPGQueryResultWindow::resultEmpty() const
{ 
	return m_pTableResult->numRows() == 0; 
}

KPGDatabase *KPGQueryResultWindow::lookupDatabase(QListViewItem * pItem)
{
	// get selected server and database
	QString strServer(m_pComboBoxServer->currentText());
	QString strDatabase(m_pComboBoxDatabase->currentText());
		
	if(strServer.length() == 0)
	{
		kdError() << k_funcinfo << " Server is not selected " << endl;
		return 0;
	}
		
	if(strDatabase.length() == 0)
	{
		kdError() << k_funcinfo << " Database is not selected " << endl;
		return 0;
	}
		
	while(pItem)
	{
		if((static_cast <KPGTreeItem *> (pItem))->type() == KPGTreeItem::nodeServer)
		{
            KPGServer *pServer = static_cast <KPGServer *> (pItem);
            if(pServer->text(0) != strServer)
            { 
                pItem = pItem->nextSibling(); continue; }
                        
                //--- traverse list of connected databases
                QListViewItem * pItem2 = pServer->firstChild();
                
                while(pItem2)
                {
                if((static_cast <KPGTreeItem *> (pItem2))->type() == KPGTreeItem::nodeDatabase)
                {
                    KPGDatabase *pDatabase = static_cast <KPGDatabase *> (pItem2);
                    if(pDatabase->text(0) != strDatabase)
                    { 
                        pItem2 = pItem2->nextSibling(); continue; 
                    }
                
                    if(pDatabase->connection())
                    { 
                        return pDatabase;      
                    }
                    else
                    {
                        kdError() << k_funcinfo << " Database is disconected " << endl;
                        return 0;
                    }
                }     
                pItem2 = pItem2->nextSibling();
			}  
		}
		pItem = pItem->nextSibling();
	}
	
	return 0;
}

void KPGQueryResultWindow::runQuery(KPGDatabase *pDatabase)
{
	// clear previous result
	m_pTableResult->setNumRows(0);
	m_pTableResult->setNumCols(0);
	m_pTextEditResult->clear();
	m_pTextLabelExecutionTime->setText("");
	
	// clear error marks
	KTextEditor::MarkInterface *pMarkIface = KTextEditor::markInterface(m_pKateView->document());
	KTextEditor::EditInterface *pEditIface = KTextEditor::editInterface(m_pKateView->document());
							
	unsigned int uiNumLines = pEditIface->numLines();
	for(unsigned int line = 0; line < uiNumLines; line++)
	{						
		pMarkIface->removeMark(line, KTextEditor::MarkInterface::Error); 
	}
	
	// free memory
	m_pqxxResult.clear(); 
		
	int iTransType = m_pComboBoxTransaction->currentItem();
	KPGConnection::ETransType transType = KPGConnection::eTransNormal;
	switch(iTransType)
	{
		case 0: transType = KPGConnection::eTransNormal;
				break;
				
		case 1: transType = KPGConnection::eTransRobust;
				break;
				
		case 2: transType = KPGConnection::eTransNone;
				break;
	}
	
	QString strConectionString( );
	
	m_pKateView->setEnabled(false);
	m_pComboBoxServer->setEnabled(false);
	m_pComboBoxDatabase->setEnabled(false);
	m_pComboBoxTransaction->setEnabled(false);
	emit sigSetTerminalIcon(this, 1); // red terminal icon
	((QWidget *) parent())->setCursor(KCursor::waitCursor());
	
	m_pActRunQuery->setEnabled(false);
	m_pActStopQuery->setEnabled(true);
	m_pActFetchNext->setEnabled(false);
  	m_pActFetchAll->setEnabled(false);
  	m_pActFindInResult->setEnabled(false); 
    m_pActFindNextInResult->setEnabled(false);
    
	// Run connection in thread
	m_ConnectionInThread.disconnectFromServer(); // close connection if any
	
	try
	{
		m_ConnectionInThread.connectToServer(pDatabase->connection()->options());
	}
	catch (const std::exception &e)
    {
        kdError() << "Failed to open connection " << e.what() << endl;
        KMessageBox::sorry(this, e.what());
        return;
    }
    
    QString strSql;
    
    if(m_pCheckBoxExplain->isChecked())
    {
    	strSql.append("EXPLAIN ");
    
    	if(m_pCheckBoxAnalyze->isChecked())
    		strSql.append("ANALYZE ");
    		
		if(m_pCheckBoxVerbose->isChecked())
			strSql.append("VERBOSE ");
    }
    
    strSql.append(editorText());
    
	m_ConnectionInThread.setSQL(strSql, transType);
	m_ConnectionInThread.start();
}

void KPGQueryResultWindow::customEvent(QCustomEvent *pEvent)
{
	KPGQueryResultEvent *pQueryResultEvent = static_cast <KPGQueryResultEvent *> (pEvent);
	
	//kdDebug() << QString("Result size is: %1").arg(m_pqxxResult.size()) << endl;
	
	m_nFetchedRows = 0;
		
	// Get codec for connection client encoding
 	m_pTextCodec = pQueryResultEvent->connection()->textCodec();
			
	// Display result
	if(m_pqxxResult.size() > 0)
	{
		displayExecutionTime(pQueryResultEvent->executionMiliseconds());   
		displayResult(); 
		m_pWidgetStackResults->raiseWidget(0);
	}
	else
	{
		m_nTotalRows = 0;
		m_nTotalCols = 0;
	
		if(pQueryResultEvent->error().isEmpty() == false)
		{
			QString strErrorText(pQueryResultEvent->error());
			
			
			if(pQueryResultEvent->connection()->notifications().count() > 0)
			{
				m_pTextEditResult->setText(pQueryResultEvent->connection()->notifications().join("") + strErrorText);
				pQueryResultEvent->connection()->clearNotifications();
			}
			else
			{
				m_pTextEditResult->setText(strErrorText);
			}
			
			
			if(strErrorText.contains("syntax error at or near") > 0)
			{
				// Parse error message and obtain number of character position
				QRegExp patternNumber = QRegExp("\\d+\n", false);
				int iPos = patternNumber.search(strErrorText, 0);
				
				if(iPos > 0)
				{
					QString strNumber = strErrorText.mid(iPos);
					kdDebug() << "Error at character: " << strNumber << endl;
					
					bool bConversionOk;
					int iErrorPosition = strNumber.toInt(&bConversionOk);
					
					if(bConversionOk)
					{
						if(iErrorPosition > 1) iErrorPosition--; // 1 based -> 0 based
						
						// Place cursor and error mark in editor to error position
						
						KTextEditor::EditInterface *pEditIface = KTextEditor::editInterface(m_pKateView->document());
						
						KTextEditor::MarkInterface *pMarkIface = KTextEditor::markInterface(m_pKateView->getDoc());
						
						unsigned int uiNumLines = pEditIface->numLines();
												
						for(unsigned int line = 0; line < uiNumLines; line++)
						{
							int iLineLength = pEditIface->lineLength(line);
							if(iErrorPosition <= iLineLength)
							{
								viewCursorInterface(m_pKateView)->setCursorPosition(line, iErrorPosition);
								pMarkIface->addMark(line, KTextEditor::MarkInterface::Error);
								
								break;
							}
							else
							{
								iErrorPosition -= iLineLength;
							}
						}
					}
				}
			}
		}
		else
		{
			displayExecutionTime(pQueryResultEvent->executionMiliseconds());      
			
			if(pQueryResultEvent->connection()->notifications().count() > 0)
			{
				m_pTextEditResult->setText(pQueryResultEvent->connection()->notifications().join(""));
				pQueryResultEvent->connection()->clearNotifications();
			}
			else  
			{
				m_pTextEditResult->setText(i18n("Operation successfull, no rows returned"));
			}
		}
		m_pWidgetStackResults->raiseWidget(1); 
	}
	
	m_pKateView->setEnabled(true);
	m_pKateView->setFocus();
	m_pComboBoxServer->setEnabled(true);
	m_pComboBoxDatabase->setEnabled(true);
	m_pComboBoxTransaction->setEnabled(true);
	emit sigSetTerminalIcon(this, 2); // blue terminal icon
	((QWidget *) parent())->setCursor(KCursor::arrowCursor());
	
	m_ConnectionInThread.wait();
		
	m_pActRunQuery->setEnabled(true);
	m_pActStopQuery->setEnabled(false);
	m_pActFetchNext->setEnabled((fetchedAllRows()) ? false : true);
  	m_pActFetchAll->setEnabled((fetchedAllRows()) ? false : true);
  	m_pActFindInResult->setEnabled((resultEmpty()) ?  false : true);
}

void KPGQueryResultWindow::stopQuery()
{
	if(m_ConnectionInThread.running ())
	{
		m_ConnectionInThread.terminate();
		m_ConnectionInThread.wait();
		m_ConnectionInThread.leaveConnection();
		
		m_pTextEditResult->setText(i18n("Operation terminated"));
		m_pWidgetStackResults->raiseWidget(1); 
		
		m_pKateView->setEnabled(true);
		m_pComboBoxServer->setEnabled(true);
		m_pComboBoxDatabase->setEnabled(true);
		m_pComboBoxTransaction->setEnabled(true);
		emit sigSetTerminalIcon(this, 0); // black terminal icon 
        
		m_pActRunQuery->setEnabled(true);
		m_pActStopQuery->setEnabled(false);
		m_pActFetchNext->setEnabled(false);
		m_pActFetchAll->setEnabled(false);
		
		m_pqxxResult.clear(); // free memory
		m_nTotalRows = 0;
		m_nTotalCols = 0;
			
		((QWidget *) parent())->setCursor(KCursor::arrowCursor());
	} 
}


void KPGQueryResultWindow::displayExecutionTime(int iMiliseconds)
{
    int hour = 0; int min = 0; int sec = 0;
    int milis = iMiliseconds;
    while(milis >= 1000)
    {
      milis -= 1000;
      sec++;
    }
    while(sec >= 60)
    {
      sec -=60;
      min++;
    }
    while(min >= 60)
    {
      min -= 60;
      hour++;
    }
    
    QString execTime;
    if(hour > 0) execTime.append(QString("%1 h :").arg(hour));
    execTime.append(QString("%1 m : ").arg(min));
    execTime.append(QString("%1 s : ").arg(sec));
    execTime.append(QString("%1 ms").arg(milis));
      
    m_pTextLabelExecutionTime->setText(execTime);
}

// Display SQL result
void KPGQueryResultWindow::displayResult()
{
	bool bAllRowsFetched = true;
	
	// Fill output area with query result
	m_nTotalRows = m_pqxxResult.size();
	m_nTotalCols = m_pqxxResult.columns();
		
	m_pTableResult->setNumRows(min(m_nTotalRows, NUM_OF_RECORDS_PER_PAGE));
	m_pTableResult->setNumCols(m_nTotalCols);
	
	// Field names
	QHeader* hHeader = m_pTableResult->horizontalHeader();
	
	for(unsigned int nCol = 0; nCol < m_nTotalCols; nCol++)
	{
		hHeader->setLabel(nCol, m_pqxxResult.column_name(nCol));
	}
	
	
	// Data
	for(unsigned int nRow = 0; nRow < m_nTotalRows; nRow++)
	{
		for(unsigned int nCol = 0; nCol < m_nTotalCols; nCol++)
		{
			QString strValue(m_pTextCodec->toUnicode(m_pqxxResult[nRow][nCol].c_str()));
			m_pTableResult->setText(nRow, nCol, strValue);
		}
		
		m_nFetchedRows++;
		if(nRow + 1 >= NUM_OF_RECORDS_PER_PAGE)
		{
			bAllRowsFetched = false;
			break;
		}
	}
	
	for(unsigned int nCol = 0; nCol < m_nTotalCols; nCol++)
	{
		m_pTableResult->adjustColumn(nCol);
	}
	
	if(bAllRowsFetched) m_pqxxResult.clear(); // free memory
}

// Fetch next X rows from large result
void KPGQueryResultWindow::fetchNext()
{
	unsigned int nRows = m_nTotalRows - m_nFetchedRows;
	nRows = min(nRows, NUM_OF_RECORDS_PER_PAGE);
	m_pTableResult->setNumRows(nRows + m_pTableResult->numRows());
	
	// Data
	unsigned int nRow = m_nFetchedRows;
	unsigned int nFetchedRows = 0;
	
	for(; nRow < m_nTotalRows; nRow++)
	{
		for(unsigned int nCol = 0; nCol < m_nTotalCols; nCol++)
		{
			QString strValue(m_pTextCodec->toUnicode(m_pqxxResult[nRow][nCol].c_str()));
			m_pTableResult->setText(nRow, nCol, strValue);
		}
		
		m_nFetchedRows++;
		nFetchedRows++;
		if(nFetchedRows >= NUM_OF_RECORDS_PER_PAGE)
		{
			m_pActFetchNext->setEnabled(true);
			m_pActFetchAll->setEnabled(true);
            break;
		}
	}
	
	if(nRow == m_nTotalRows)
	{
		m_pActFetchNext->setEnabled(false);
		m_pActFetchAll->setEnabled(false);
		m_pqxxResult.clear(); // free memory
	}
	
	
}
  
// Fetch all rows from large result  
void KPGQueryResultWindow::fetchAll()
{
	unsigned int nRows = m_nTotalRows - m_nFetchedRows;
	
	if(nRows > 10000)
	{ 
		if( KMessageBox::questionYesNo(this, QString("There are %1 rows in result. Fething them may take long time and much anmount of memory. Continue ?").arg(nRows)) != KMessageBox::Yes )
		{
			return;
		}
	}
	
	((QWidget *) parent())->setCursor(KCursor::waitCursor());
	m_pTableResult->setNumRows(nRows + m_pTableResult->numRows());
	
	// Data
	unsigned int nRow = m_nFetchedRows;
	for(; nRow < m_nTotalRows; nRow++)
	{
		for(unsigned int nCol = 0; nCol < m_nTotalCols; nCol++)
		{
			QString strValue(m_pTextCodec->toUnicode(m_pqxxResult[nRow][nCol].c_str()));
			m_pTableResult->setText(nRow, nCol, strValue);
		}
		
		m_nFetchedRows++;
	}
	
	((QWidget *) parent())->setCursor(KCursor::arrowCursor());
	
	m_pActFetchNext->setEnabled(false);
	m_pActFetchAll->setEnabled(false);
    
	m_pqxxResult.clear(); // free memory
}  

void KPGQueryResultWindow::updateTabToolTip()
{
	QString strText = editorText().left(150);
	emit sigUpdateTabTooltip(strText);
}

void KPGQueryResultWindow::slotExplainToggled(bool bToggled)
{
	m_pCheckBoxAnalyze->setEnabled(bToggled);
	m_pCheckBoxVerbose->setEnabled(bToggled);
}

void KPGQueryResultWindow::slotButtonOptionsToggleg(bool bToggled)
{
	if(bToggled)
	{
		m_pFrameOptions->show();
	}
	else
	{
		m_pFrameOptions->hide();
	}
}

// Find first occurence of text in result table
void KPGQueryResultWindow::findInResultFirst(QStringList &listOfSearchHistory)
{
    KFindDialog dlg(this, "", 0, listOfSearchHistory, false);
    dlg.setHasCursor(false);
        
    int c = dlg.exec();
    
    if(c != QDialog::Accepted)
        return;

    listOfSearchHistory = dlg.findHistory();
        
    
    if(m_pFind != 0) 
    {
    	disconnect( m_pFind, 0, this, 0);
    	delete m_pFind;
    }
    m_pFind = new KFind(dlg.pattern(), dlg.options(), this);

    // Connect highlight signal to code which handles highlighting
    // of found text.
    connect( m_pFind, SIGNAL( highlight( const QString &, int, int ) ),
             this, SLOT( slotHighlight( const QString &, int, int ) ) );
             
    // Connect findNext signal - called when pressing the button in the dialog
    connect( m_pFind, SIGNAL( findNext() ), this, SLOT( slotFindInResultNext() ) );

    m_iRowToSearch = (m_pFind->options() & KFindDialog::FindBackwards) ? m_pTableResult->numRows() - 1 : 0;
    m_iColToSearch = (m_pFind->options() & KFindDialog::FindBackwards) ? m_pTableResult->numCols() - 1 : 0;
    m_pFind->setData(m_pTableResult->text(m_iRowToSearch, m_iColToSearch));

    slotFindInResultNext();
}

// Find next occurence of text in result table
void KPGQueryResultWindow::findInResultNext()
{
    m_iRowToSearch = m_pTableResult->currentRow();
	m_iColToSearch = m_pTableResult->currentColumn();
	if(m_iRowToSearch < 0) return;
	if(m_iColToSearch < 0) return;
	
    slotFindInResultNext();
}

// Find next occurence of text
void KPGQueryResultWindow::slotFindInResultNext()
{
	if (!m_pFind) // shouldn't be called before find is activated
        return;

    KFind::Result res = KFind::NoMatch;
    while ( res == KFind::NoMatch &&
    		((m_pFind->options() & KFindDialog::FindBackwards) ? 
    				(m_iRowToSearch >= 0) : (m_iRowToSearch < m_pTableResult->numRows())
    		) 
          ) 
    	{
    		//kdDebug() << "searching  1:" << m_iParaToSearch << " 2: " << m_iEndPara << " 3: " << m_iStartPara << endl;
        	
        	Q_ASSERT(m_iRowToSearch >= 0);
            Q_ASSERT(m_iRowToSearch < m_pTableResult->numRows());
            Q_ASSERT(m_iColToSearch >= 0);
            Q_ASSERT(m_iColToSearch < m_pTableResult->numCols());
            	
        	if(m_pFind->needData()) 
        	{
            	m_pFind->setData(m_pTableResult->text(m_iRowToSearch, m_iColToSearch));
        	}
        
        	// Let KFind inspect the text fragment, and display a dialog if a match is found
        	res = m_pFind->find();
        
        	if( res == KFind::NoMatch ) 
        	{
            	if((m_pFind->options() & KFindDialog::FindBackwards))
            	{
            		if(m_iColToSearch == 0)
            		{
            			m_iColToSearch = m_pTableResult->numCols() - 1;
            			m_iRowToSearch--;
            		}
            		else
            		{
            			m_iColToSearch--;
            		}
            	}
            	else
            	{
            		if(m_iColToSearch == m_pTableResult->numCols() - 1)
            		{
            			m_iColToSearch = 0;
            			m_iRowToSearch++;
            		}
            		else
            		{
            			m_iColToSearch++;
            		}
            	}
        	}
        	//kdDebug() << "2searching  1:" << m_iParaToSearch << " 2: " << m_iEndPara << " 3: " << m_iStartPara << endl;
		}

    if( res == KFind::NoMatch ) 
    { // i.e. at end
        m_pFind->displayFinalDialog();
        m_pFind->resetCounts();
        m_pTableResult->removeSelection(0);
    }
}

// Highligth found text
void KPGQueryResultWindow::slotHighlight( const QString &, int, int)
{
    //kdDebug() << "highlight: " << index << " " << length << endl;
    m_pTableResult->setCurrentCell(m_iRowToSearch, m_iColToSearch);
}

// Re-route signal from KPGSqlEdit
void KPGQueryResultWindow::slotRequestTableColumnsListsForCodeCompletion(pqxx::oid oidTable)
{
	emit sigRequestTableColumnsListsForCodeCompletion(this, oidTable);
}

// Re-route signal from KPGSqlEdit
void KPGQueryResultWindow::slotRequestSchemaChildsListsForCodeCompletion(pqxx::oid oidSchema)
{
    emit sigRequestSchemaChildsListsForCodeCompletion(this, oidSchema);
}

// Re-route signal from KPGSqlEdit
void KPGQueryResultWindow::slotRequestFunctionReturnTypeAttributesListsForCodeCompletion(pqxx::oid oidFunction)
{
	emit sigRequestFunctionReturnTypeAttributesListsForCodeCompletion(this, oidFunction);
}

// Fired from m_pUpdateCodeCompletionListTimer. Updates aliases 
// in m_listOfCodeCompletionObjects according to current SQL
void KPGQueryResultWindow::slotUpdateCodeCompletionList()
{
	if(m_bCodeCompletionListUpdated) return;
	kdDebug() << k_funcinfo << endl;
	m_pUpdateCodeCompletionListTimer->stop(); 

	KPGSqlParser::updateAliases(editorText(), m_listOfCodeCompletionObjects); 

	m_bCodeCompletionListUpdated = true;
	m_pUpdateCodeCompletionListTimer->start( UPDATE_TIMER_INTERVAL );
}

// Fired when the completion list disappears and a completion has been inserted into text.
void KPGQueryResultWindow::slotCompletionDone()
{
	kdDebug() << k_funcinfo << endl;
	m_eCompletionMode = modeNone;
}
	
// Fired when the completion list disappears and no completion has been done
void KPGQueryResultWindow::slotCompletionAborted()
{
	kdDebug() << k_funcinfo << endl;
	m_eCompletionMode = modeNone;
}

// Return the string to complete (the letters behind the cursor)
QString KPGQueryResultWindow::word()
{
	uint cline, ccol;
	viewCursorInterface(m_pKateView)->cursorPositionReal(&cline, &ccol);
	if (! ccol) return QString::null; // no word
	
	KTextEditor::EditInterface *pEditIface = KTextEditor::editInterface(m_pKateView->document());
	
	//m_RegExp.setWildcard(false);
	m_RegExp.setPattern( "\\b(\\w+)$" );
	if ( m_RegExp.searchRev(pEditIface->text( cline, 0, cline, ccol )) < 0 )
		return QString::null; // no word
		
	kdDebug() << m_RegExp.cap( 1 ) << endl;
	return m_RegExp.cap( 1 );
}

// Return the string to complete (the letters behind the cursor)
QString KPGQueryResultWindow::wordBeforeDot()
{
	uint cline, ccol;
	viewCursorInterface(m_pKateView)->cursorPositionReal(&cline, &ccol);
	if (! ccol) return QString::null; // no word
	
	KTextEditor::EditInterface *pEditIface = KTextEditor::editInterface(m_pKateView->document());
				
	//m_RegExp.setWildcard(true);
	m_RegExp.setPattern( "\\b(\\w+\\.)$" );
	if ( m_RegExp.searchRev(pEditIface->text( cline, 0, cline, ccol )) < 0 )
		return QString::null; // no word
		
	kdDebug() << m_RegExp.cap( 1 );
	return m_RegExp.cap( 1 );
}

// Set list of main DB objects for code completion
void KPGQueryResultWindow::setListOfObjectsForCodeCompletion(const KPGOidNameList &listOfCodeCompletionObjects)
{
	if(m_pUpdateCodeCompletionListTimer) m_pUpdateCodeCompletionListTimer->stop(); 
    clearListOfObjectsForCodeCompletion();
    
    // Add given DB objects
    for(KPGOidNameList::const_iterator cit = listOfCodeCompletionObjects.begin(); cit != listOfCodeCompletionObjects.end(); ++cit)
    {
        m_listOfCodeCompletionObjects.append(KPGOidNameAliases(*cit));
    }
    
    // Sort list of code completions
    m_listOfCodeCompletionObjects.sort();
    
    // Create list of all completions
    /*for(KPGOidNameAliasesList::const_iterator cit = m_listOfCodeCompletionObjects.begin(); cit != m_listOfCodeCompletionObjects.end(); ++cit)
	{
        KTextEditor::CompletionEntry e;
    	e.text = ((*cit).name());
        m_completionList.append(e);
    }*/
                
    m_bCodeCompletionListUpdated = false;
    if(m_pUpdateCodeCompletionListTimer) m_pUpdateCodeCompletionListTimer->start( UPDATE_TIMER_INTERVAL ); 
}

// Set list of schema childs for code completion
void KPGQueryResultWindow::setListOfSchemaChildsForCodeCompletion(const KPGOidNameList &listOfCodeCompletionObjects)
{
	if(m_pUpdateCodeCompletionListTimer) m_pUpdateCodeCompletionListTimer->stop(); 
    clearListOfSchemaChildsForCodeCompletion();
    
    // Add given DB objects
    for(KPGOidNameList::const_iterator cit = listOfCodeCompletionObjects.begin(); cit != listOfCodeCompletionObjects.end(); ++cit)
    {
        m_listOfCodeCompletionSchemaChilds.append(*cit);
    }
    
    // Sort list of code completions
    m_listOfCodeCompletionSchemaChilds.sort();
    
    // Create list of all completions
    /*for(KPGOidNameList::const_iterator cit = m_listOfCodeCompletionSchemaChilds.begin(); cit != m_listOfCodeCompletionSchemaChilds.end(); ++cit)
	{
        KTextEditor::CompletionEntry e;
    	e.text = ((*cit).name());
        m_completionList.append(e);
    }*/
                
    m_bCodeCompletionListUpdated = false;
    if(m_pUpdateCodeCompletionListTimer) m_pUpdateCodeCompletionListTimer->start( UPDATE_TIMER_INTERVAL ); 
}

// Clear list of DB objects for code completion
void KPGQueryResultWindow::clearListOfObjectsForCodeCompletion()
{
	m_listOfCodeCompletionObjects.clear();
    // @deprecated m_completionList.clear();
    
    // Add SQL keywords to list
    /*  @deprecated QStringList strKeyWords = KPGSyntaxHighlighter::listOfKeyWords();
    
    for(QStringList::const_iterator cit = strKeyWords.begin(); cit != strKeyWords.end(); ++cit)
    {
    	KTextEditor::CompletionEntry e;
    	e.text = (*cit);
        m_completionList.append(e);
    }*/
}

// Clear list of schema childs for code completion
void KPGQueryResultWindow::clearListOfSchemaChildsForCodeCompletion()
{
    m_listOfCodeCompletionSchemaChilds.clear();
    // @deprecated m_completionList.clear();
}

// Set list of table columns/type attributes for code completion
void KPGQueryResultWindow::setListOfColumnsForCodeCompletion(const KPGListTableColumns &listOfTableColumns)
{
	clearListOfColumnsForCodeCompletion();
    
    // Add given table columns, create list of all completions
    for(KPGListTableColumns::const_iterator cit = listOfTableColumns.begin(); cit != listOfTableColumns.end(); ++cit)
    {
        m_listOfCodeCompletionColumns.append(*cit);
        
        /* @deprecated KTextEditor::CompletionEntry e;
    	e.text = ((*cit).name());
        m_completionList.append(e);*/
    }
}

// Clear list of table columns/type attributes for code completion
void KPGQueryResultWindow::clearListOfColumnsForCodeCompletion()
{
	m_listOfCodeCompletionColumns.clear();
    // @deprecated m_completionList.clear();
}

// Filters events - for watching Ctrl+SPACE and others keys, that starts code completion
bool KPGQueryResultWindow::eventFilter(QObject *pObject, QEvent *pEvent)
{
	if(m_pUpdateCodeCompletionListTimer) m_pUpdateCodeCompletionListTimer->stop(); 
	
	if(pObject == m_pKateView)
	{	
		switch(pEvent->type())
		{
			case QEvent::KeyPress: 
				{
					QKeyEvent *pKeyEvent = static_cast <QKeyEvent*> (pEvent);
							
					if((pKeyEvent->key() == Key_Space) && (pKeyEvent->state() & ControlButton))
					{
						kdDebug() << k_funcinfo << "updateCtrlSpaceCompletion " << endl;
						updateCtrlSpaceCompletion(); 
						return true;
					}
				}
				break;
				
			case QEvent::KeyRelease: 
				{
					QKeyEvent *pKeyEvent = static_cast <QKeyEvent*> (pEvent);
							
					if(pKeyEvent->key() == Key_Period)
					{
						kdDebug() << k_funcinfo << "updatePeriodCompletion " << endl;
						updatePeriodCompletion(); 
						return true;
					}
				}
				break;
			
			default: 
				break;
		}
	}
	
	bool bRetVal = KPGQueryResultWindowBase::eventFilter(pObject, pEvent);
	
	if(m_pUpdateCodeCompletionListTimer) m_pUpdateCodeCompletionListTimer->start( UPDATE_TIMER_INTERVAL ); 
	
	return bRetVal;
}

// Get uncompleted word, set it to completion and set completed items to box, eventually show it
void KPGQueryResultWindow::updateCtrlSpaceCompletion()
{
    QString strWord(word());
		
	QStringList strKeyWords = KPGSyntaxHighlighter::listOfKeyWords();
    QValueList<KTextEditor::CompletionEntry> listMatches;
    for(QStringList::const_iterator cit = strKeyWords.begin(); cit != strKeyWords.end(); ++cit)
    {
    	KTextEditor::CompletionEntry e;
    	e.text = (*cit);
    	e.comment = "KEYWORD";
        listMatches.append(e);
    }
    
    for(KPGOidNameAliasesList::const_iterator cit = m_listOfCodeCompletionObjects.begin(); cit != m_listOfCodeCompletionObjects.end(); ++cit)
	    {
            KTextEditor::CompletionEntry e;
			//e.type = (*i).type;
			e.text = (*cit).name();
			
			switch((*cit).type())
			{
				case KPGTreeItem::nodeTable:  
						e.comment = i18n("Table");
						break;
													   	
				case KPGTreeItem::nodeFunction:  
						e.comment = i18n("Function");
						e.postfix = (*cit).argumentTypes();
						break;
													   						   	
				default: break;							   						   	
			}
			
			if(((*cit).description() != QString::null) && ((*cit).description().length() > 0))
			{
				e.comment += ": " + (*cit).description();
			}
			
			listMatches.append(e);
        }
		
	m_eCompletionMode = modeMainObjects;
	updateCompletion(strWord, listMatches);
}

// Set columns into completion if table/function name is left before period: aaa.
void KPGQueryResultWindow::updatePeriodCompletion()
{
    QString strWord(wordBeforeDot());
    QString strWordWithoutDot(strWord.left(strWord.length() - 1));	
		
	if(strWord.length() > 0)
	{											
		// If code completion is not up-to date, update it first
        if(!m_bCodeCompletionListUpdated)
        {
        	slotUpdateCodeCompletionList();
        }
        
        kdDebug() << k_funcinfo << " for: " << strWordWithoutDot << endl;
		
		// Uncompleted word is table or alias
		      
        // Find table or function, 1st by name, 2nd by alias
        const KPGOidNameAliases * pOidNameAliases = m_listOfCodeCompletionObjects.getConstItemByName(strWordWithoutDot); 
        if(pOidNameAliases == 0)
        {
            pOidNameAliases = m_listOfCodeCompletionObjects.getConstItemByAlias(strWordWithoutDot); 
        }
            
        if(pOidNameAliases && (pOidNameAliases->type() == KPGTreeItem::nodeSchema))
        {
            emit sigRequestSchemaChildsListsForCodeCompletion(this, pOidNameAliases->oid());
            QValueList<KTextEditor::CompletionEntry> listMatches;
                
            for(KPGOidNameList::const_iterator cit = m_listOfCodeCompletionSchemaChilds.begin(); cit != m_listOfCodeCompletionSchemaChilds.end(); ++cit)
            {
                KTextEditor::CompletionEntry e;
				//e.type = "";
				e.text = (*cit).name();
				//e.comment = (*i).comment;
				//e.userdata = (*i).userdata;
					
				switch((*cit).type())
				{
					case KPGTreeItem::nodeTable:  
							e.comment = i18n("Table");
							break;
													   	
					case KPGTreeItem::nodeFunction:  
							e.comment = i18n("Function");
							e.postfix = (*cit).argumentTypes();
							break;
													   						   	
					default: break;					   						   	
				}
					
				if(((*cit).description() != QString::null) && ((*cit).description().length() > 0))
				{
					e.comment += ": " + (*cit).description();
				}
			
				listMatches.append(e);
            }
                
            m_eCompletionMode = modeSchemaChilds;
            updateCompletion(strWordWithoutDot, listMatches);
        }
        else if(pOidNameAliases && (pOidNameAliases->type() == KPGTreeItem::nodeTable))
        {
          	// Uncompleted word is table column
		    
            emit sigRequestTableColumnsListsForCodeCompletion(this, pOidNameAliases->oid());
                
            QValueList<KTextEditor::CompletionEntry> listMatches;
            for(KPGListTableColumns::const_iterator cit = m_listOfCodeCompletionColumns.begin(); cit != m_listOfCodeCompletionColumns.end(); ++cit)
            {
                KTextEditor::CompletionEntry e;
				e.type = "COLUMN";
				e.text = (*cit).name();
				e.comment = (*cit).typName() + i18n(" column");
				//e.userdata = (*i).userdata;
					
				if(((*cit).description() != QString::null) && ((*cit).description().length() > 0))
				{
					e.comment += ": " + (*cit).description();
				}
					
				listMatches.append(e);
            }
                
            m_eCompletionMode = modeTableColumns;
            updateCompletion(listMatches);
        }
        else if(pOidNameAliases && (pOidNameAliases->type() == KPGTreeItem::nodeFunction))
        {
            // Uncompleted word is function return type column
            emit sigRequestFunctionReturnTypeAttributesListsForCodeCompletion(this, pOidNameAliases->oid());
                
            QValueList<KTextEditor::CompletionEntry> listMatches;
            for(KPGListTableColumns::const_iterator cit = m_listOfCodeCompletionColumns.begin(); cit != m_listOfCodeCompletionColumns.end(); ++cit)
            {
                KTextEditor::CompletionEntry e;
				e.type = "COLUMN";
				e.text = (*cit).name();
				e.comment = i18n("Function column");
				//e.userdata = (*i).userdata;
				listMatches.append(e);
            }
                                                
            m_eCompletionMode = modeTableColumns;
            updateCompletion(listMatches);
        }
	}
}

// Set completed items to box, eventually show it
void KPGQueryResultWindow::updateCompletion(const QString &strWord, const QValueList<KTextEditor::CompletionEntry> & completionList)
{	
	if(completionList.count() > 0)
	{
		//m_completionList = completionList;
		
		KTextEditor::CodeCompletionInterface * pCodeComplIface = KTextEditor::codeCompletionInterface(m_pKateView);
		
		pCodeComplIface->showCompletionBox(completionList, strWord.length(), true);
	}
	else
	{
	    m_eCompletionMode = modeNone;
	}
}

// Set completed items to box, eventually show it
void KPGQueryResultWindow::updateCompletion(const QValueList<KTextEditor::CompletionEntry> & completionList)
{	
	if(completionList.count() > 0)
	{
		//m_completionList = completionList;
		
		KTextEditor::CodeCompletionInterface * pCodeComplIface = KTextEditor::codeCompletionInterface(m_pKateView);
		
		pCodeComplIface->showCompletionBox(completionList, 0, true);
	}
	else
	{
	    m_eCompletionMode = modeNone;
	}
}

// Scan for possible completions, ignoring any dublets
/*QValueList<KTextEditor::CompletionEntry> KPGQueryResultWindow::allMatches(const QString &word)
{
	QValueList<KTextEditor::CompletionEntry> completionList;
	
	QValueList < KTextEditor::CompletionEntry >::const_iterator i = m_completionList.begin();
	for (++i; i != m_completionList.end(); ++i)
	{
			if((*i).text.startsWith(word))
			{
				KTextEditor::CompletionEntry e;
				e.type = (*i).type;
				e.text = (*i).text;
				e.prefix = (*i).prefix;
				e.postfix = (*i).postfix;
				e.comment = (*i).comment;
				e.userdata = (*i).userdata;
				completionList.append(e);
			}
	}
	
	return completionList;
}*/
