 /**************************************************************************
 *   Copyright (C) 2004 by  Thomas Renninger                               *
 *                            <trenn@suse.de> and                          *
 *                          Danny Kukawka                                  *
 *                            <dkukawka@suse.de>, <danny.kukawka@web.de>   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.          *
 ***************************************************************************/


#ifndef DBUS_API_SUBJECT_TO_CHANGE
#define DBUS_API_SUBJECT_TO_CHANGE
#endif

// include global header
#include <dbus/dbus.h>
#include <fcntl.h>

// include powersave specific header
#include <powersave_dbus.h>

// include QT header
#include <qtimer.h>
#include <qobject.h>
#include <qtimer.h>
#include <qdir.h>

// include KDE header
#include <klocale.h>

// include own header
//#include "kpowersave_debug.h"
#include "pdaemon.h"


/*! 
*  \file 	pdaemon.cpp
*  \brief 	all functionality to get information from the powersave daemon located here.
*  \todo 	check if we need a new timer for this: powersaved died/stopped, poll (only) for 
*		AC Adapter. There is atm a delay from 5-10 seconds
*/

/*!
 * The default constructor for this class. Within all values get initialise and
 * set up this first time.
 */
pDaemon::pDaemon(){
	myDebug ("pDaemon::pdaemon");
	
	dbus_conn = new dbusPowersaveConnection();

	on_AC_power = AC_UNKNOWN;
	perc = -1;
	left = -1;
	numOfCPUs = -1;
	standby_allowed = 0;
	suspend2disk_allowed = 0;
	suspend2ram_allowed = 0;
	
	// update everything the first time 
	update_info_ac_changed = true;
	update_info_sleep_state_changed = true;
	update_info_cpufreq_policy_changed = true;
	update_info_battery_state_changed = true;
	update_info_battery_charge_changed = true;
	update_info_battery_perc_changed = true;
	update_info_cpufreq_speed_changed = true;
	update_info_scheme_info_changed = true;

	// to store the state of our central services
	powersaved_terminated = false;
	dbus_terminated =  false;
	hal_terminated = false;

	cpufreq_policy = CPU_UNSUPP;
	battery_state = -1;

	// assume it is running in the beginning for msg to user if it is stopped
	daemon_running = 0;
	
	send_battery_state_change_message = FALSE;
	send_no_daemon_message = FALSE;

	check_system_timer = new QTimer(this);
	connect(check_system_timer, SIGNAL(timeout()), this, SLOT(checkSystemTimeout()));
	BAT_PdOffAcOn_Timer = new QTimer(this);
	connect(BAT_PdOffAcOn_Timer, SIGNAL(timeout()), this, SLOT(checkBatteryProcInfo()));
	BAT_PdOffAcOff_Timer = new QTimer(this);
	connect(BAT_PdOffAcOff_Timer, SIGNAL(timeout()), this, SLOT(checkBatteryProcInfo()));

	// receive event from daemon
	connect(dbus_conn, SIGNAL(msgReceived_withStringInteger( msg_type, QString, int )),
		this, SLOT(processClientMessage( msg_type, QString, int )));

	daemon_running = checkDaemon();
	if (daemon_running < 0 ){
		// daemon is not running, wait 20 seconds to recheck. We don't display
		// errormessages on startup for this time to prevent races with autologin
		// and parallel booting
		daemon_running = 0;
		switchToNonDaemonMode();
		QTimer::singleShot(20000, this, SLOT(recheckDaemon()));
	}
	else
		switchToDaemonMode();

	supported_sleeping_states = getSupportedSleepStates();
	// we always support suspend to disk via swsusp
	suspend2disk_supported = 1;
	if (( supported_sleeping_states & APM_SUSPEND) || (supported_sleeping_states & ACPI_S3 ))
		suspend2ram_supported = 1;
	else
		suspend2ram_supported = 0;
	if ((supported_sleeping_states & APM_STANDBY) || (supported_sleeping_states & ACPI_S1))
		standby_supported = 1;
	else
		standby_supported = 0;
}

/*!
 * The default destructor of the class. Within the scheme_list destroyed.
 */
pDaemon::~pDaemon(){
	myDebug ("pDaemon::~pdaemon");

	delete dbus_conn;
}

/*!
 * The function checks if the machine is a laptop. Currently we ask
 * HAL for this information.
 * \return boolean with info if machine is a laptop 
 * \retval true 	if a laptop
 * \retval false 	else/if not a laptop
 */
bool pDaemon::isLaptop() {
	myDebug ("pDaemon::isLaptop");

	QString ret = ps_hal_get_property_string("/org/freedesktop/Hal/devices/computer","system.formfactor");

	if (ret == "laptop")
		return true;
	else 
		return false;
}

/*!
 * Requests available schemes from the powersave daemon. If there are changes
 * the function store the new schemes in scheme_list.
 * \return A integer value with information about the update and the changes.
 * \retval -2 if we cannot connect to powersaved over dbus
 * \retval -1 if there have never been any schemes accessible
 * \retval  0 if there are no changes, but schemes available
 * \retval  1 if ac, battery or current have switched 
 * \retval  2 if whole entries have changed
 */
int pDaemon::updateSchemeInfo(){
	myDebug ("pDaemon::updateSchemeInfo");

	DBusMessage *reply;

	int ret = dbusSendMessageWithReply( REQUEST_MESSAGE,
					   &reply,
					   "SchemesGet",
					   DBUS_TYPE_INVALID );

	if ( ret != REPLY_SUCCESS ) {
		schemes.clear();
		return -2;
	}
	int changes = 0;
	int current_scheme;
	int battery_scheme;
	int ac_power_scheme;
	Schemes temp_schemes;

	if ( dbusGetMessageInteger(reply, &current_scheme, 0)
	     || dbusGetMessageInteger(reply, &battery_scheme, 1)
	     || dbusGetMessageInteger(reply, &ac_power_scheme, 2)) {
		dbus_message_unref( reply );
		return -2;
	}

	char *scheme;
	for ( int x = 0; !dbusGetMessageString( reply, &scheme, x ); x++ ) {
		temp_schemes.append( scheme );

		if ( x == current_scheme ) {
			temp_schemes.setCurrentScheme( scheme );
		}

		if ( x == battery_scheme ) {
			temp_schemes.setBatteryScheme( scheme );
		}

		if ( x == ac_power_scheme ) {
			temp_schemes.setAcScheme( scheme );
		}
	}

	dbus_message_unref( reply );

	if ( schemes.count() > 0 ){
		changes += 1;
	}
	if ( temp_schemes.count() > 0 ){
		changes += 2;
	}
	if (!changes)
		return -1;

	// no schemes any more
	if (changes == 1){
		schemes.clear();
		update_info_scheme_info_changed = true;
		return 1;
	}

	// now we have schemes
	if (changes == 2){
		schemes = temp_schemes;
		update_info_scheme_info_changed = true;
		return 1;
	}

	changes = 0;
	if (schemes.currentScheme().isEmpty() || schemes.acScheme().isEmpty()
	    || schemes.batteryScheme().isEmpty() || temp_schemes.currentScheme().isEmpty()
	    || temp_schemes.acScheme().isEmpty() || temp_schemes.batteryScheme().isEmpty()){
		// either ac, battery or current scheme not defined, this should not happen.
		// Better delete all schemes to avoid segfaults later...
		temp_schemes.clear();
		schemes.clear();
		return -1;
	}

	if ( schemes.differ( temp_schemes.schemeList() ) ) {
		changes = 2;
	}

	if (changes == 0 && 
	    ( schemes.currentScheme() != temp_schemes.currentScheme() 
	      || schemes.acScheme() != temp_schemes.acScheme()
	      || schemes.batteryScheme() != temp_schemes.batteryScheme() ) )
		changes = 1;

	schemes.clear();

	schemes = temp_schemes;

	if (changes == 1)
		update_info_scheme_info_changed = true;

	// scheme info needs graphical update in menu 
	if (changes > 0)
		emit schemeDataChanged();
	return changes;
}


/*!
 * Change the CPUFreqPolicy by sending the request to the powersave daemon.
 * \param policy ID of the CPUFreqPolicy as a integer value
 * \return A integer value with the status about set the requestet CPUFreqPolicy.
 * \retval -1 if policy could not be set
 * \retval  0 if this policy has already been set
 * \retval  1 if policy could be set
 */
int pDaemon::setCPUFreqPolicy(int policy){
	myDebug ("pDaemon::setCPUFreqPolicy");

	// be careful policy coming from the param and cpufreq_policy are different! Policy is the define to 
	// set a new policy (POWERSAVE_ACTION_CPUFREQ_*) and cpufreq_policy is the define of the return value 
	// coming from evaluate_States(daemon_system_states, POWERSAVED_REQUEST_CPUFREQ_STATE_INFO); e.g. 
	// POWERSAVED_CPUFREQ_STATE_*) setting(ACTION) and getting(REQUEST) are different values/defines!
	if (policy == cpufreq_policy)
		// already set to wanted policy, ignore
		return 0;

	int ret = 1;
	int err_code;

	switch( policy ) {
	case CPU_HIGH:
		err_code = dbusSendSimpleMessage(ACTION_MESSAGE,
						 "CpufreqPerformance");
		break;
	case CPU_AUTO:
		err_code = dbusSendSimpleMessage(ACTION_MESSAGE,
						 "CpufreqDynamic");
		break;
	case CPU_LOW:
		err_code = dbusSendSimpleMessage(ACTION_MESSAGE,
						 "CpufreqPowersave");
		break;
	default:
		return -1;
		break;
	}

	if ( err_code != REPLY_SUCCESS && err_code != REPLY_ALREADY_SET ) {
		cpufreq_policy = CPU_UNSUPP;
		update_info_cpufreq_policy_changed = true;
		ret = -1;
	}

	switch(err_code){
	case REPLY_HW_NOT_SUPPORTED:
		ret=-1;
		break;
	case REPLY_ALREADY_SET:
		return 0;
		break;
	case REPLY_SUCCESS:
                cpufreq_policy = policy;
		update_info_cpufreq_policy_changed = true;
		break;
	case REPLY_NO_RIGHTS:
		ret=-1;			
		break;
	default:
		break;
	}

	emit generalDataChanged();
	return ret;
}

/*!
 * Change the current scheme by sending the request to the powersave daemon.
 * \param schemename untranslated name of the scheme
 * \return 
 * \retval -1 if scheme could not be set
 * \retval  0 if scheme policy has already been set
 * \retval  1 if scheme was set
 */
int pDaemon::setActiveScheme( QString schemename ){
	myDebug ("pDaemon::setActiveScheme");
	
	int ret_code, ret_val = 0;
	char *scheme = qstrdup(schemename);
	ret_code = dbusSendMessage( ACTION_MESSAGE, "SchemesSet",
				    DBUS_TYPE_STRING, &scheme , DBUS_TYPE_INVALID);
	
	switch(ret_code){
	case REPLY_HW_NOT_SUPPORTED:
		myDebug ("pDaemon::setActiveScheme -- ERROR==REPLY_HW_NOT_SUPPORTED");
		ret_val=-1;
		break;
	case REPLY_ALREADY_SET:
		myDebug ("pDaemon::setActiveScheme -- ERROR==REPLY_HW_NOT_SUPPORTED");
		ret_val=0;
		break;
	case REPLY_SUCCESS:
		myDebug ("pDaemon::setActiveScheme -- REPLY_SUCCESS");
		ret_val=1;
		break;
	case REPLY_INVALID_PARAM:
		myDebug ("pDaemon::setActiveScheme -- ERROR==REPLY_INVALID_PARAM");
		ret_val=-1;
		break;
	default:
		myDebug ("pDaemon::setActiveScheme -- ERROR==UNKNOWN");
		break;
	}

	return ret_val;
}


/*! 
 * Check whether we can access the daemon via client or data socket 
 * \return A integer value with the result of the check
 * \retval 1 if daemon is available
 * \retval -1 if daemon is not available
 */
int pDaemon::checkDaemon(){
	myDebug("pDaemon::checkDaemon");

	DBusMessage *reply;
	int err_code = dbusSendMessageWithReply( REQUEST_MESSAGE,
						 &reply,
						 "AcPower",
						 DBUS_TYPE_INVALID );

	if ( err_code != REPLY_SUCCESS ) {
		if (daemon_running){
			switchToNonDaemonMode();
		}

		return -1;
	}
	else{
		dbus_message_unref( reply );
		if (dbus_conn->isConnected() || dbus_conn->reconnect() >= 0){
			return 1;
		}
		else{
			if (daemon_running){
				daemon_running = 0;
				switchToNonDaemonMode();
			}
			return -1;
		}
	}

}

/*!
 * This function queries the powersave daemon through local UNIX socket connection and obtains 
 * the current: 
 * \li power source (AC/battery)
 * \li rem Battery in percent
 * \li rem Mins in percent
 * \li battery state (NORMAL, WARNING, LOW or CRITICAL)
 * \li battery charging state
 * \li cpu freq policy
 *
 * \return Information about the query and if something changed as a integer value.
 * \retval -1 if daemon could not be queried
 * \retval 0  if daemon could be queried, but nothing has changed
 * \retval 1  if daemon could be queried and something has changed
 */		
int pDaemon::readDaemonData(){
	myDebug ("pDaemon::readDaemonData");

	DBusMessage *reply;
	BatteryGeneral bg;
	int ret, new_value;

	ret = dbusSendMessageWithReply( REQUEST_MESSAGE,
					     &reply,
					     "AcPower",
					     DBUS_TYPE_INVALID );

	if ( ret == REPLY_SUCCESS ) {
		char *value;
		dbusGetMessageString( reply, &value, 0 );

		if ( !strcmp( value, "on" ) ) {
			new_value = AC_ONLINE;
		}
		else if ( !strcmp( value, "off" ) ) {
			new_value = AC_OFFLINE;
		}
		else {
			new_value = AC_UNKNOWN;
		}

		if (new_value != on_AC_power){
			on_AC_power = new_value;
			
			// tells update function of kpowersave.cpp to change and redraw picture
			update_info_ac_changed = true;
			updateSchemeInfo();
		}
		dbus_message_unref( reply );
	}

	// enable suspend/standy by if we are ran as root
	if (getuid() == 0)
		new_value = 1;
	else
		new_value = (REPLY_SUCCESS == dbusSendSimpleMessage(REQUEST_MESSAGE, "AllowedSuspendToDisk"));

	if (new_value != suspend2disk_allowed){
		// update standby menu things in kpowersave.cpp update function
		update_info_sleep_state_changed = true;
		suspend2disk_allowed = new_value;
	}

	if (getuid() == 0)
		new_value = 1;
	else
		new_value = (REPLY_SUCCESS == dbusSendSimpleMessage(REQUEST_MESSAGE, "AllowedSuspendToRam"));

	if (new_value != suspend2ram_allowed){
		// update standby menu things in kpowersave.cpp update function
		update_info_sleep_state_changed = true;
		suspend2ram_allowed = new_value;
	}

	if (getuid() == 0)
		new_value = 1;
	else
		new_value = (REPLY_SUCCESS == dbusSendSimpleMessage(REQUEST_MESSAGE, "AllowedStandby"));

	if (new_value != standby_allowed){
		// update standby menu things in kpowersave.cpp update function
		update_info_sleep_state_changed = true;
		standby_allowed = new_value;
	}

	ret = dbusSendMessageWithReply( REQUEST_MESSAGE,
					&reply,
					"CpufreqPolicy",
					DBUS_TYPE_INVALID );
	if ( ret == REPLY_SUCCESS ) {
		char *value;
		dbusGetMessageString( reply, &value, 0 );

		if ( !strcmp( value, "performance" ) ) {
			new_value = CPU_HIGH;
		}
		else if ( !strcmp( value, "powersave" ) ) {
			new_value = CPU_LOW;
		}
		else if ( !strcmp( value, "dynamic" ) ) {
			new_value = CPU_AUTO;
		}
		else {
			new_value = CPU_UNSUPP;
		}

		if (new_value != cpufreq_policy){
			// update cpufreq menu things 
			update_info_cpufreq_policy_changed = true;
			cpufreq_policy = new_value;
		}
		dbus_message_unref( reply );
	}

	ret = dbusSendMessageWithReply( REQUEST_MESSAGE,
					&reply,
					"BatteryState",
					DBUS_TYPE_INVALID );
	if ( ret == REPLY_SUCCESS ) {
		char *value;
		dbusGetMessageString( reply, &value, 0 );

		if ( !strcmp( value, "critical" ) ) {
			new_value = BAT_CRIT;
		}
		else if ( !strcmp( value, "low" ) ) {
			new_value = BAT_LOW;
		}
		else if ( !strcmp( value, "warning" ) ) {
			new_value = BAT_WARN;
		}
		else if ( !strcmp( value, "normal" ) ) {
			new_value = BAT_NORM;
		}
		else {
			new_value = BAT_NONE;
		}

		if (new_value != battery_state){
			// update graphics in update func 
			update_info_battery_state_changed = true;
			send_battery_state_change_message = true;
			battery_state = new_value;
		}
		dbus_message_unref( reply );
	}

	ret = getBatteriesInfo(&bg);
	if (ret < 0) {
		handleGetBatteryInfoError(ret);
	} else {
		if (bg.remaining_percent != perc){
			update_info_battery_perc_changed = true;
			perc = bg.remaining_percent;
		}
	
		if (bg.remaining_minutes != left){
			update_info_battery_perc_changed = true;
			left = bg.remaining_minutes;
		}

		if (bg.charging_state != charge_state){
			update_info_battery_charge_changed = true;
			charge_state = bg.charging_state;
		}
	}

	if (update_info_ac_changed || update_info_sleep_state_changed || 
	    update_info_cpufreq_policy_changed || update_info_battery_state_changed || 
	    update_info_battery_perc_changed || update_info_battery_charge_changed){
		// something needs to be graphically updated
		emit generalDataChanged();
		return 1;
	}
	else
		return 0;
}


/*!
 * This function request the powersave daemon to return the current brightness
 * level (or percentage) of the display, if supported
 * \param percentage boolean which is true if the value should get a percentage
 *		     instead of levels
 * \return Information about the brightness level.
 * \retval -1 if not supported by hardware or powersave
 * \retval else current level
 */
int pDaemon::getBrightness( bool percentage ){
	myDebug ("pDaemon::getBrightness");

	if (percentage)
		return simpleGetIntegerValue ("BrightnessGetPercent");
	else 	
		return simpleGetIntegerValue ("BrightnessGet");
}

/*!
 * This function request the powersave daemon to return the max. available 
 * brightness levels of the display, if supported
 * \return Information about the number brightness levels.
 * \retval -1 if not supported by hardware or powersave
 * \retval else max. available levels
 */
int pDaemon::getBrightnessLevels(){
	myDebug ("pDaemon::getBrightnessLevels");
	
	availableBrightnessLevels = simpleGetIntegerValue ("BrightnessLevelsGet");
	return availableBrightnessLevels;
}


/*!
 * This function set the brightness to the given value.
 * \param value a Integer with a supported level
 * \param percentage boolean which is true if the value is in percentage
 *		     instead of levels
 * \param force_min  boolean which is true if force to lowest value.
 *		     This mean: ignore possible display off.
 * \return Information about the success of the operation.
 * \retval -1 if not supported by hardware or other error
 * \retval else current set level (should be the same as given value)
 */	
int pDaemon::setBrightness( int value, bool percentage, bool force_min){
	myDebug ("pDaemon::setBrightness");

	int ret_code = -1;

	if (percentage) {
		// find the lowest percentage before off
		int _min = (int)( 100 / availableBrightnessLevels );

		// be sure not to off the display --> use BrightnessMin which should 
		// prevent display off per definition
		if ((value <= _min) && !force_min )
			ret_code = dbusSendSimpleMessage ( ACTION_MESSAGE, "BrightnessMin" );
		else
			ret_code = dbusSendMessage( ACTION_MESSAGE, "BrightnessSetPercent",
						    DBUS_TYPE_INT32, &value , DBUS_TYPE_INVALID);

	} else {
		// be sure not to off the display
		if ((value < 1) && !force_min )	
			ret_code = dbusSendSimpleMessage ( ACTION_MESSAGE, "BrightnessMin" );
		else 
			ret_code = dbusSendMessage( ACTION_MESSAGE, "BrightnessSet",
						    DBUS_TYPE_INT32, &value , DBUS_TYPE_INVALID);

	}	

	switch(ret_code){
		case REPLY_SUCCESS:
			ret_code = getBrightness(percentage);
			break;
		case REPLY_HW_NOT_SUPPORTED:
		case REPLY_GENERAL_ERROR:
		default:
			ret_code = -1;
			break;
	}
		
	return ret_code;
}

/*!
 * This function queries the proc directly to gain system info. DO NOT USE THIS IF THERE IS 
 * A RUNNING POWERSAVE DAEMON ! You can test whether \ref pDaemon::checkDaemon() returns lower than zero. \n \n
 * This and the function queryACProcInfo are splitted because reading out battery from proc increases 
 * the processor usage significantly on most acpi systems, therefore the checkBatteryProcInfo func 
 * should not as often used as reading out e.g. the current power source.
 * \return Information if something changed as a integer value.
 * \retval 0 if nothing changed
 * \retval 1 if something changed 
 */
int pDaemon::checkBatteryProcInfo(){
	myDebug ("pDaemon::checkBatteryProcInfo");

	BatteryGeneral bg;
	int state = 0;

	state = getBatteriesInfo(&bg);

	if (state < 0){
		handleGetBatteryInfoError(state);
	}
	else {
		if ( left != bg.remaining_minutes){
			left = bg.remaining_minutes;
			update_info_battery_perc_changed = true;
		}
		if (charge_state != bg.charging_state){
			charge_state = bg.charging_state;
			update_info_battery_charge_changed = true;
		}
		if ( perc != bg.remaining_percent){
			perc = bg.remaining_percent;
			update_info_battery_perc_changed = true;
		}
		if (charge_state != bg.charging_state){
			charge_state = bg.charging_state;
			update_info_battery_charge_changed = true;
		}

		// set battery limits if no daemon is running: this would override the battery 
		// state limits from daemon !
		
		// Critical limit fixed at one percent and lower 
		if (perc <= 1){
			if (battery_state != BAT_CRIT){
				battery_state = BAT_CRIT;
				update_info_battery_state_changed = true;
				send_battery_state_change_message = true;
			}
		}
		// Low battery limit set fixed between 1 an 7 rem perc capac. 
		else if(perc > 1 && perc <= 7){
			// only send message to user when state was lower before 
			if (battery_state != BAT_CRIT)
				send_battery_state_change_message = true;
			if (battery_state != BAT_LOW){
				battery_state = BAT_LOW;
				update_info_battery_state_changed = true;
			}
		}
		else if (perc > 7 && perc <= 12){
			// only send message to user when state was lower before
			if (battery_state != BAT_CRIT || 
			    battery_state != BAT_LOW)
				send_battery_state_change_message = true;
			if (battery_state != BAT_WARN){
				battery_state = BAT_WARN;
				update_info_battery_state_changed = true;
			}
		}
		else if (perc > 12 && perc <= 100){
			if (battery_state != BAT_NORM){
				battery_state = BAT_NORM;
				update_info_battery_state_changed = true;
			}
		}
		else
			if (battery_state != BAT_NONE){
				battery_state = BAT_NONE;
				update_info_battery_state_changed = true;
			}
	}
	
	if ( update_info_battery_state_changed || update_info_battery_perc_changed || 
	     update_info_battery_charge_changed || send_battery_state_change_message ){
		/* something needs to be graphically updated */
		emit generalDataChanged();
		return 1;
	}
	else
		return 0;
}


/*!
 * The function checks if something changes at the AC-Adapter status if the daemon
 * isn't running or present.
 * \return Information if something changed as a integer value.
 * \retval 	0 if nothing changed
 * \retval 	1 if something changed
 */
int pDaemon::checkACProcInfo(){
	myDebug ("pDaemon::checkACProcInfo");

	int new_value = getACAdapterStatus();
	if (new_value != on_AC_power){
		update_info_ac_changed = true;

		if ( new_value < 0) {
			if (new_value == HAL_ERROR)
				on_AC_power = HAL_ERROR;
			else 
				on_AC_power = UNKNOWN;
			
			emit generalDataChanged();
		}	
		else if (new_value == AC_OFFLINE ) {
			on_AC_power = AC_OFFLINE;
			emit ACStatus(false);
			emit generalDataChanged();
		} else {
			on_AC_power = AC_ONLINE;
			emit ACStatus(true);
			emit generalDataChanged();
		}
		return 1;
	}
	return 0;
}

/*!
 * This function counts all online/offline CPUS.
 * Returns the total count of CPUs - _not_ the last CPU ID!
 */

int pDaemon::getCPUNum() {
	myDebug("pDaemon::getCPUNum");

	int cpu_id=0;
	QDir tmp_dir;
	QString cpu_path = "/sys/devices/system/cpu/cpu0/";
#ifdef FAKE_CPU
	cpu_path.prepend("/tmp/foo");
#endif
	QString tmp_path = tmp_dir.absFilePath(cpu_path, true);

	while (tmp_dir.exists(tmp_path)) {
		int tmp = cpu_id;

		cpu_id++;
                cpu_path.replace(QString::number(tmp), QString::number(cpu_id));
		tmp_path = tmp_dir.absFilePath(cpu_path, true);
        }
	return cpu_id;
}

/*!
 * The function checks the current CPU Speed. The current cpu speed needs to be read out from 
 * sysfs and currently not be obtained through the daemon. If the CPUFreg changed the new value
 * is set to \ref cpufreq_speed .
 * \return Information if something changed or if there are errors as an interger value
 * \retval 	-1 if there are error by reading from /sys/..
 * \retval	 0 if nothing changed
 * \retval 	 1 if something changed
 */
int pDaemon::checkCPUSpeed(){
	myDebug ("pDaemon::checkCPUSpeed");

	bool speed_changed = false;
	int new_value = -1;
	int fd;
	char buf[15];
	QString cpu_device = "/sys/devices/system/cpu/cpu0/cpufreq/scaling_cur_freq";
#ifdef FAKE_CPU
	cpu_device.prepend("/tmp/foo");
#endif

	// first check path for the kernel on-demand-govenour then 
	// for the use userspace case
	update_info_cpufreq_speed_changed = false;
	cpufreq_speed.clear();

	if (numOfCPUs == -1)	
		numOfCPUs = getCPUNum();

	for (int cpu_id=0; cpu_id < numOfCPUs; cpu_id++) {

		new_value = -1;

		fd = open(cpu_device, O_RDONLY);
		if (read(fd, buf, 14) > 0){
			new_value = strtol(buf, NULL, 10)/1000;
			close(fd);
		}
		else{
			close(fd);
			speed_changed = true;
			// CPU disabeld -> set Freq to -1
			cpufreq_speed.append(-1);
		}

		if (new_value != cpufreq_speed[cpu_id]) {
			speed_changed = true;
			cpufreq_speed.append(new_value);
		}

		cpu_device.replace(QString::number(cpu_id), QString::number(cpu_id+1));
	}
	
	if (speed_changed) {
		update_info_cpufreq_speed_changed = true;
		return 1;
	}

	return 0;
}

/*!
 * The function checks the Speed of throttling CPU(s). The cpu speed needs to be read out from 
 * /proc/cpuinfo.
 * \return Success or error while reading /proc/cpuinfo 
 * \retval	  0 successful
 * \retval 	 -1 reading problem
 */
int pDaemon::checkCPUSpeedThrottling() {
	myDebug("pDaemon::checkCPUSpeedThrottling()");

	QStringList lines;
	QString cpu_file = "/proc/cpuinfo";
#ifdef FAKE_CPU
	cpu_file.prepend("/tmp/foo");
#endif	
	QFile cpu_info(cpu_file);

	// clear cpufreq list
	cpufreq_speed.clear();

	if ( !cpu_info.open(IO_ReadOnly) ) {
		cpu_info.close();
		return -1;
	}

	QTextStream stream( &cpu_info );
	QString line;

	while ( !stream.atEnd() ) {
		line = stream.readLine();
		
		if (line.startsWith("cpu MHz		: ")) {
			line.remove("cpu MHz		: ");
			line = line.remove(line.length() - 4, 4);
			cpufreq_speed.append(line.toInt());
		}
	}
	
	while ((int) cpufreq_speed.count() < numOfCPUs) {
		cpufreq_speed.append(-1);
	}

	cpu_info.close();
	return 0;
}

/*!
 * The function gets the current throttling state of the CPU(s). The throttling state needs to be 
 * read out from /proc/acpi/processor/CPUX/throttling.
 * \return boolean with info if throttling is supported
 * \retval true if throttling is supported
 * \retval false if not supported or on any other error
 */
bool pDaemon::getCPUThrottlingState() {
	myDebug("pDaemon::getCPUThrottlingState()");

	int id = 0;
	QStringList lines;
	QFileInfo *fi;
	QString cpu_dirname;
	QString dir_acpi_processor = "/proc/acpi/processor/";
#ifdef FAKE_CPU
	dir_acpi_processor.prepend("/tmp/foo");
#endif

	QDir d_throttling(dir_acpi_processor);
	if (!d_throttling.exists())
		return false;

	d_throttling.setFilter( QDir::Dirs );
	d_throttling.setNameFilter("CPU*");

	const QFileInfoList *list = d_throttling.entryInfoList();
	QFileInfoListIterator it( *list );

	// clear throttling value list
	cpu_throttling.clear();

	while ((fi = it.current()) != 0 ) {
		cpu_dirname = fi->fileName();

		QString throttling_device = d_throttling.absPath();
		throttling_device.append("/").append(cpu_dirname).append("/throttling");

		myDebug("Throttling state file for CPU%i will be: %s", id, throttling_device.latin1());

		QFile throttling(throttling_device);

		// open throttling state file
		if ( throttling.open(IO_ReadOnly) ) {
			QTextStream stream( &throttling );
			QString line;

			do {
				line = stream.readLine();
			} while (!line.startsWith("   *T") && !stream.atEnd() );

			if (line.startsWith("   *T")) {
				line = line.right(3);
				line.remove("%");
				cpu_throttling.append(line.toInt());
				myDebug("CPU%i: cpu_throttling is set to %i", id, cpu_throttling[id]);
			} else {
				cpu_throttling.append(0);
			}
		}

		throttling.close();
		++it; // next entry
		id++; // count cpu id
	}

	return true;
}

/*!
 * The function gets the Max CPU Speed. The max cpu speed needs to be read out from 
 * sysfs and currently not be obtained through the daemon. 
 */
void pDaemon::getCPUMaxSpeed() {
	myDebug("pDaemon::getCPUMaxSpeed()");

	int fd;
	int maxfreq;
	char buf[15];
	QString cpu_device_max = "/sys/devices/system/cpu/cpu0/cpufreq/cpuinfo_max_freq";
#ifdef FAKE_CPU
	cpu_device_max.prepend("/tmp/foo");	
#endif

	cpufreq_max_speed.clear();

	if (numOfCPUs == -1)	
		numOfCPUs = getCPUNum();

//	while (!access(cpu_device_max, R_OK)) {
	for (int cpu_id=0; cpu_id < numOfCPUs; cpu_id++) {

		fd = open(cpu_device_max, O_RDONLY);
		if (read(fd, buf, 14) > 0){
			maxfreq = strtol(buf, NULL, 10)/1000;
			cpufreq_max_speed.append(maxfreq);
			close(fd);
		} else {
			cpufreq_max_speed.append(-1);
			close(fd);
		}

		cpu_device_max.replace(QString::number(cpu_id), QString::number(cpu_id+1));
	}
}

/*!
 * \b SLOT to recheck the powersave daemon if there is any problem to connect
 * to the powersave library. This SLOT called by \ref checkSystemTimeout() through a 
 * QTimer::singleShot with a 4 second intervall and within mousePressEvent().
 * \param onetime boolean to allow recheck without start the check_system_timer timer
 */
void pDaemon::recheckDaemon(){
	myDebug ("pDaemon::recheckDaemon");

	if (checkDaemon() < 0 || !dbus_conn->isConnected() && (dbus_conn->reconnect() < 0)){
		// daemon not running for more than 4 seconds, this means the daemon has not been restart, 
		// but died. Send message to user that daemon is not running. If daemon can still be queried 
		// for info, but client connecition to receive events cannot be established we still fall 
		// back to non-daemon mode
		if(!check_system_timer->isActive()) {
			check_system_timer->start(system_timer_interval);
			checkSystemTimeout();
		}
		// daemon is not running for at least xx seconds 
		if (powersaved_terminated) {
			emit powersaveNotRunningDialog();
		} else if (dbus_conn->noRights()) {
			dbus_terminated = false;	
			emit dbusErrorMsgDialog (DBUS_NO_RIGHTS);
		} else {
			emit dbusErrorMsgDialog (DBUS_NOT_RUNNING);
		}

		emit generalDataChanged();
	}
	else {
		// daemon seems to only got restarted 
		switchToDaemonMode();
	}
}


/*!
 * This funtion is used to parse a message from powersave for the different 
 * messagetypes and events. We also do the related things if we match a 
 * supported message.
 * \param type a \ref msg_type which should be parse/processed
 * \param signal the related signal
 * \param value a optional related integer value
 */
void pDaemon::processClientMessage( msg_type type, QString signal, int value ){
	myDebug ("pDaemon::processClientMessage");
	if( value == 0 ) ; // to prevent compiler warning

	switch(type) {
		
		case DBUS_EVENT:
			myDebug("D-BUS Event: '%s'", signal.ascii());
			if ( signal.startsWith("dbus.terminate")){
				QTimer::singleShot(4000, this, SLOT(recheckDaemon()));
				dbus_terminated = true;
				switchToNonDaemonMode();
			}
			else if ( signal.startsWith("hal.")) {
				if ( signal.startsWith("hal.terminate"))
					hal_terminated = true;
				if ( signal.startsWith("hal.started"))
					hal_terminated = false;
				
				emit halErrorMsgDialog();
				readDaemonData();
				emit generalDataChanged();
			}
			break;
		case ACPI_EVENT:
			// we don't use acpi events here
			break;
		case NOTIFICATION:
			myDebug("\"Notification for client\" Event: '%s'", signal.ascii());
			emit forwardClientMessage( signal );
			break;
		case POWERSAVE_EVENT:
			myDebug("Powersave Event: '%s'", signal.ascii());
			if ( signal.startsWith("daemon.terminate")){
				// DON NO RECHECK !!! We do this if we get signal: dbus.terminate
				powersaved_terminated = true;
				switchToNonDaemonMode();
			}
			else if ( signal.startsWith("daemon.start")){
				QTimer::singleShot(4000, this, SLOT(recheckDaemon()));
				switchToDaemonMode();
			}
			else if (signal.startsWith("battery.info")){
				readDaemonData();
				emit batteryInfoEvent();
			}
			
			else if (signal.startsWith("button.lid.closed")){
				emit lidclosed();
			}

			else if (signal.startsWith("button.lid.open")){
				emit lidopened();
			}	

			// scheme has changed -> reread the current active 
			else if (signal.startsWith("daemon.scheme.change")){
				updateSchemeInfo();
			}
			
			else if (signal.startsWith("acadapter.offline")){
				emit ACStatus(false);
				readDaemonData();
			}
			else if (signal.startsWith("acadapter.online")){
				emit ACStatus(true);
				readDaemonData();
			}
			
			else if (signal.startsWith("battery.normal")){
				readDaemonData();
			}
			else if (signal.startsWith("battery.warning")){
				emit batteryWARNState("battery.warning");
				readDaemonData();
			}
			else if (signal.startsWith("battery.low")){
				emit batteryWARNState("battery.low");
				readDaemonData();
			}
			else if (signal.startsWith("battery.critical")){
				emit batteryWARNState("battery.critical");
				readDaemonData();
			}
			
			// Those tell you about a powersave cpufreq policy change
			else if (signal.startsWith("processor.performance")){
				readDaemonData();
			}
			else if (signal.startsWith("processor.powersave")){
				readDaemonData();
			}
			else if (signal.startsWith("processor.dynamic")){
				readDaemonData();
			}
			else if (signal.startsWith("processor.busy")){
				readDaemonData();
				emit throttlingChanged();
			}
			else if (signal.startsWith("processor.idle")){
				readDaemonData();
				emit throttlingChanged();
			}

			// Those tell the client, that there is a suspend
			if (signal.startsWith("global.suspend2disk")){
				emit setSuspend("suspend2disk");
			}
			else if (signal.startsWith("global.suspend2ram")){
				emit setSuspend("suspend2ram");
			}
			else if (signal.startsWith("global.standby")){
				emit setSuspend("standby");
			}
			else if (signal.startsWith("global.resume")) {
				if (signal.startsWith("global.resume.suspend2disk"))
					emit resumedFrom("suspend2disk");
				else if (signal.startsWith("global.resume.suspend2ram"))
					emit resumedFrom("suspend2ram");
				else if (signal.startsWith("global.resume.standby" )) 
					emit resumedFrom("standby");

				emit setSuspend("NULL");
				emit resumeFromSuspend(true);
			}
			
			emit generalDataChanged();
			break;
			
		case SCREENLOCK:
			if (signal.startsWith("global.suspend2disk")){
				emit lockscreen();
			}
			else if (signal.startsWith("global.suspend2ram")){
				emit lockscreen();
			}
			else if (signal.startsWith("global.standby")){
				emit lockscreen();
			}
			break;
		case PROGRESS:
			//emit progressDialogRequest( signal, value );
			emit progressDialogRequest( signal );
			break;
		default:
			// unknown paket type 
			break;
	};

	/*else {
		if (!dbus_conn->isConnected()){
			// This was our last paket we received over this socket_fd. Probably the daemon broke 
			// away, recheck daemon in four seconds. If still no connection, fall back to non-daemon mode.
			if (checkDaemon() < 0){
				QTimer::singleShot(4000, this, SLOT(recheckDaemon()));
				switchToNonDaemonMode();
			}
		}
		else{
			emit forwardClientMessage( "Error - Received invalid packet from powersave daemon." );
		}
	}*/
}

/*!
 * \b SLOT for the QTimer \ref check_system_timer . Within the function kpowersave 
 * is in non-daemon mode. If the daemon can be contacted it is switched back
 * to daemon mode via switchToDaemonMode(). If not, data is read out from /proc.
 * This function is invoked every second through the check_system_timer.
 *
 * Trough the QTimer \ref BAT_PdOffAcOff_Timer and \ref BAT_PdOffAcOn_Timer we control the \b SLOT 
 * \ref checkBatteryProcInfo if the powersave daemon isn't running or if kpowersave can't connect to
 * the daemon by other reasons. In this case kpowersave poll information from the battery directly
 * from /proc . \sa BAT_PdOffAcOff_Timer \sa BAT_PdOffAcOn_Timer for more information.
 */
void pDaemon::checkSystemTimeout(){
	myDebug ("pDaemon::checkSystemTimeout");	

	// check whether we now hava daemon 
	if (checkDaemon() < 0){
		// POWERSAVED_BATTERY_STATE_NO_BAT -> assume there never will be a battery on this 
		// system, still have to check for a possible upcoming daemon 
		if (battery_state != BAT_NONE)
			checkACProcInfo();
	}
	else
		switchToDaemonMode();
}

/*!
 * Use this function to switch to mode without powersave client notification.
 * Here we close the socket and start all needed Timer for poll the battery.
 */
void pDaemon::switchToNonDaemonMode(){
	myDebug ("pDaemon::switchToNonDaemonMode");

	daemon_running = 0;
	dbus_conn->close();
	
	cpufreq_policy = CPU_UNSUPP;
	
	// no daemon, check from pro
	if (checkACProcInfo() > 0)
	{
		// if daemon not running and AC plugged out: set batterycheck to 20 seconds, it's a 
		// compromise between fresh infos and cost of batterycheck without daemon
		if (!BAT_PdOffAcOff_Timer->isActive()){
			BAT_PdOffAcOff_Timer->start(BAT_PdOffAcOff_timeout);
			BAT_PdOffAcOn_Timer->stop();
		}
	}
	else
	{
		// if daemon not running and AC plugged in: set batterycheck to 60 seconds
		// because: if AC On, user don't so often info update
		if (!BAT_PdOffAcOn_Timer->isActive()){
			BAT_PdOffAcOff_Timer->stop();
			BAT_PdOffAcOn_Timer->start(BAT_PdOffAcOn_timeout);
		}
	}

	if(!check_system_timer->isActive())
		check_system_timer->start(system_timer_interval);
	
	checkBatteryProcInfo();
	checkCPUSpeed();
	updateSchemeInfo();
	
	update_info_cpufreq_policy_changed = true;
	update_info_sleep_state_changed = true;
	
	emit generalDataChanged();
}


/*!
 * Use this function to switch to mode with powersave client notification.
 * Here we try to open a new socket to powersave and if this work we stop
 * all unneeded Timer for polling the battery.
 */
void pDaemon::switchToDaemonMode(){
	myDebug ("pDaemon::switchToDaemonMode");

	daemon_running = 1;
	if (dbus_conn->isConnected() || dbus_conn->reconnect() >= 0){
		// Deactivate all timers we get notified by the daemon about changes
		if (BAT_PdOffAcOn_Timer->isActive()) BAT_PdOffAcOn_Timer->stop();
		if (BAT_PdOffAcOff_Timer->isActive()) BAT_PdOffAcOff_Timer->stop();
		if (check_system_timer->isActive()) check_system_timer->stop();
		
		readDaemonData();
		updateSchemeInfo();
		
		update_info_cpufreq_policy_changed = true;
		update_info_sleep_state_changed = true;
		powersaved_terminated = false;
		dbus_terminated = false;

		emit generalDataChanged();
	}
	else switchToNonDaemonMode();
 }


// -------------- PRIVATE HELPER FUNCTIONS ------------------ //

/*!
 * This funtion is used to handle errors from libpower as e.g.
 * getBatteriesInfo() and set all related variables.
 * \param errortype a integer with the errorcode/returncode
 */
void pDaemon::handleGetBatteryInfoError(int errortype) {
	myDebug ("pDaemon::handleGetBatteryInfoError");

	if (errortype < 0){
		// Cannot get battery info, maybe I could before and have to update some graphics 
		if ( perc != 100){
			perc = 100;
			update_info_battery_perc_changed = true;
		}
		if (charge_state != CHARG_STATE_UNKNOWN){
			charge_state = CHARG_STATE_UNKNOWN;
			update_info_battery_charge_changed = true;
		}
		if (battery_state != BAT_NORM) {
			battery_state = BAT_NORM;
			update_info_battery_state_changed = true;
		}
		if (errortype == HAL_ERROR) {
			battery_state = BAT_HAL_ERROR;
			update_info_battery_state_changed = true;
		}
	}
}


/*!
 * This is a helper function to reduce the code. You can use the function to
 * read a simple integer value from dbus interface.
 * \param request a QString with the name of a dbus methode
 * \return A integer value with the requested information
 * \retval -1 on any failure 
 * \retval else the returned value of the former operation
 */
int pDaemon::simpleGetIntegerValue ( QString request ) {
	myDebug ("pDaemon::simpleGetIntegerValue");

	DBusMessage *reply;
	int value;

	int ret = dbusSendMessageWithReply( REQUEST_MESSAGE,
					    &reply,
					    request,
					    DBUS_TYPE_INVALID );

	switch (ret) {
		case REPLY_SUCCESS:
			if(dbusGetMessageInteger(reply, &value, 0)) {
				dbus_message_unref( reply );
			}
			break;
		case REPLY_HW_NOT_SUPPORTED:
		default:
			value = -1;
			break;
	}

	return value;
}
