/*
    Copyright (C) 2008  Tim Fechtner < urwald at users dot sourceforge dot net >

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of
    the License or (at your option) version 3 or any later version
    accepted by the membership of KDE e.V. (or its successor approved
    by the membership of KDE e.V.), which shall act as a proxy
    defined in Section 14 of version 3 of the license.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef CONSOLE_READER_H
#define CONSOLE_READER_H

#include <QObject>
#include <QPointer>
#include <KProcess>

/** \brief Use this abstract class when you want to interpretate
*          the output of a KProcess line-by-line.
*
*   This class is useful when you want to read and
*   interpretate the output of a command line program, and espacialy
*   when you want to write a graphical frontend to a command line program.
*
*   This class has a protected member #m_process  of type \e KProcess which
*   is used to start the external program.
*   When you inherit from this class, you have to add some methods to set up
*   and start this process.
*   In #m_process, \e stderr and \e stdout are merged.
*   Don't read the output from #m_process yourself. Instead, implement
*   interpretate_console_output(QStringList& stringList) to process the output
*   of #m_process line-by-line.
*
*   \e Attention: It is expected that no <tt>\\0x00</tt> is contained in the output of #m_process.
*/
class console_reader : public QObject
{

     Q_OBJECT

  public:
     //contructor(s) and destructor(s)
     /** Constructor of the class.
     * @param parent Sets the parent of this object. */
     console_reader(const QPointer<QObject> parent = 0);
     /** Destructor of the class.
     *
     *   Calling the destructor when the process of #m_process is still running
     *   kills of course this process. (It could maybe take very much time to just
     *   send a \e terminate
     *   signal to the process and than wait until the process terminates voluntarily,
     *   so this desctructor sends directly a \e kill signal.)
     *
     *   Of course, it's nicer when you make
     *   sure that the process is terminated \e before you delete an instance of this class. */
     virtual ~console_reader();

  protected:
     /** This abstract method is called when one or more new lines were written by m_process.
     *   It doesn't matter if the line breaks are done
     *   by \e CR+LF or by \e CR or by \e LF.
     *
     *   It is made sure that this method is called always with \e complete lines - that means
     *   with lines which are terminated by a line break. This method will never be called with
     *   with uncomplete lines.
     *
     *   @param stringList This parameter holds a QStringList, and each
     *   QString in this list will contain exactly one line of the output.
     *
     *   \e Attention: This function uses a \e by-reference parameter for performance
     *   reasons (instead of
     *   a \e by-value parameter). When you reimplement this function, you can safly change the
     *   parameter - the parameter/variable won't be accessed anymore by the calling method.
     *
     *   \sa internal_splitBehavior
     *   \sa QByteArray_to_QString(QByteArray& m_byteArray) */
     virtual void interpretate_console_output(QStringList & stringList) = 0;
     /** This function is used internally to convert the output of #m_process
     *   (a \e QByteArray) to a \e QString before splitting it into individual
     *   lines and providing it to interpretate_console_output(QStringList& stringList).
     *   To do so, you need information about the encoding. This standard implementation
     *   just uses \e QString::fromLocal8Bit() to convert. Reimplement this function if
     *   you want to handle the encoding differently.
     *
     *   @param m_byteArray The QByteArray that has to be converted to QString.
     *
     *   \return You have to return a QString containing the converted m_byteArray.
     *
     *   \e Attention: This function uses a \e by-reference parameter for performance
     *   reasons (instead of  a \e by-value parameter). When you reimplement this
     *   function, you can safly change the parameter - the parameter/variable won't be
     *   accessed anymore by the calling method. */
     virtual QString QByteArray_to_QString(QByteArray & m_byteArray);
     /** This member is used to determinate the handling of empty lines.
     *   By default, it is set to \e QString::KeepEmptyParts and the QStringList
     *   passed to interpretate_console_output(QStringList& stringList) contains the
     *   output "as is".
     *
     *   You can also set this member to \e QString::SkipEmptyParts and empty lines
     *   will be oppressed. */
     QString::SplitBehavior internal_splitBehavior;
     /** Use this member to start the external program. You can access and modify it
     *   like you want. However, \e stderr and \e stdout are merged - leave this at
     *   it is. And don't read data from the output - Instead, implement
     *   interpretate_console_output(QStringList& stringList). */
     KProcess m_process;   // the process

  private:
     /** This private member is used by read_console_output() to buffer the lines
     *   that #m_process has started to write to the console but which
     *   aren't yet finished (means: which aren't terminated by a line break).
     *   They are buffered until #m_process terminates them by a line break. */
     QString m_uncomplete_line;

  private slots:
     /** This private slot is connected to the signal readReady() from #m_process
     *   and reads the output of #m_process. This slot makes sure that the output
     *   is made available line-by-line to interpretate_console_output(QStringList&
     *   stringList).
     *
     *   If the last line of the output of #m_process isn't terminated by a line
     *   break when this slot is called, this line is buffered internally in
     *   #m_uncomplete_line, and when later the #m_process has completed the line,
     *   interpretate_console_output(QStringList& stringList) is called with the complete
     *   line.
     *
     *   You can use QByteArray_to_QString(QByteArray& m_byteArray) and
     *   #internal_splitBehavior to fine-tune the conversion from QByteArray to nicely
     *   separated lines.
     *   \sa interpretate_console_output(QStringList& stringList)
     *   \sa m_process
     *   \sa m_uncomplete_line
     *   \sa internal_splitBehavior
     *   \sa QByteArray_to_QString(QByteArray& m_byteArray) */
     void read_console_output();
};

#endif
