/*
    Copyright (C) 2008-2009  Tim Fechtner < urwald at users dot sourceforge dot net >

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of
    the License or (at your option) version 3 or any later version
    accepted by the membership of KDE e.V. (or its successor approved
    by the membership of KDE e.V.), which shall act as a proxy
    defined in Section 14 of version 3 of the license.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef STREAMDIRECTORYSERVICE_H
#define STREAMDIRECTORYSERVICE_H

#include <QPointer>
#include "stationdirectorytree.h"

/** \brief Abstract base class for stream directory plugins.
*
* This is the abstract base class for service plugins for the stationDirectoryTree widget.
* Each plugin makes a streams from a specific service (for example http://shoutcast.com)
* available for a stationDirectoryTree widget. With the property #enabled, you can
* enable or disable the plugin.
*
* To register plugins in stationDirectoryTree, please see the documentation of
* stationDirectoryTree.
*
* To implement plugins, you have to implement asynchroniously the processing. Reimplement
* <tt>virtual void kickOffStreamFetch() = 0;</tt> in a way that it just kicks off the fetching of
* the stream - and returns imediatly. (You can do this with the KIO library of maybe also with
* threads). Once the data has arrived, process it and use
* addStreamToWidget() to make the streams available. Furthermore, you have to implement
* <tt>virtual void stopStreamFetch() = 0;</tt>, which stopps all fetching and processing of data.
* You don't have to worry about removing existing items from the widget - this is done
* automatically. */
class streamDirectoryService : public QObject
{

  Q_OBJECT

  friend class stationDirectoryTree; // TODO wende das nur auf die absolut nötigsten Funktionen an!

  /** This property holds whether the plugin is enabled.
  *
  * If the plugin is disabled, it's streams will not be shown in stationDirectoryTree.
  *
  * \li \e read: \code bool isEnabled() const \endcode
  * \li \e write: \code void setEnabled(const bool value) \endcode
  *
  * \sa <tt> bool internal_enabled </tt> */
  Q_PROPERTY(bool enabled READ isEnabled WRITE setEnabled)

  public:
    /** Constructor of the class.
    * @param parent Sets the parent widget of this object. It is not necessary to use the
    * stationDirectoryTree object as parent, but it is possible. */
    explicit streamDirectoryService(QObject *parent = 0);
    /** The desctructor. */
    virtual ~streamDirectoryService();
    /** See property #enabled. */
    bool isEnabled() const;
    /** See property #enabled. */
    void setEnabled(const bool value);

  protected:
    /** Adds a stream entry with the specified values to the stationDirectoryTree widget where
    * this object is used as plugin.
    *
    * Use this function in your implementation of kickOffStreamFetch().
    * TODO Parameter dokumentieren */
    void addStreamToWidget(const QString & genre,
                           const QString & streamName,
                           const quint64 bitrate,
                           const QString & currentlyPlaying = QString());
    /** This function kicks of the fetching of the stream list.
    *
    * The fetching of the stream list \e must be asynchronious. This function only kicks of the
    * necessary processes, but it expected to return itself imediatly.
    *
    * Once the information that you have requested from the internet is available, process it
    * and use addStreamToWidget() to make it available. */
    virtual void kickOffStreamFetch() = 0;
    /** This function stopps all running file transfers from the internet and stopps all
    * processing of data. It is expected, that after calling this function, there will be no
    * call of addStreamToWidget() anymore (until kickOffStreamFetch() is called the next time).
    *
    * You don't have to worry about removing existing items from the widget - this is done
    * automatically. */
    virtual void stopStreamFetch() = 0;

  private:
    /** Used to remember if the object is fetching the stream directory data. This is only possible
    * if the widget is enabled \e and registered as plugin.
    *
    * It is used and managed by helper_start() and helper_stop(). */
    bool fetchIsRunning;
    /** Helper function to start the fetching of the stream list. \sa fetchIsRunning */
    void helper_start();
    /** Helper function to stop the fetching of the stream list and to remove all stream of this
    * plugin from #widget. \sa fetchIsRunning */
    void helper_stop();
    /** Used internally to store the property #enabled. */
    bool internal_enabled;
    /** A pointer to the stationDirectoryTree where this object is used as plugin. If you want to
    * register this object as plugin, don't set this directly, but use setWidget()! */
    QPointer<stationDirectoryTree> widget;
    /** Registers this object as plugin of the specified stationDirectoryTree widget.
    *
    * Although this function is private, it is acessible from stationDirectoryTree because of
    * a \e friend declaration.
    * @param newWidget A pointer to the widget where this object is registered as plugin.
    * Can also be set to 0 to unregister the object. */
    void setWidget(stationDirectoryTree *newWidget);
};

#endif
