/*
    Copyright (C) 2008  Tim Fechtner < urwald at users dot sourceforge dot net >

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of
    the License or (at your option) version 3 or any later version
    accepted by the membership of KDE e.V. (or its successor approved
    by the membership of KDE e.V.), which shall act as a proxy
    defined in Section 14 of version 3 of the license.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef RADIOSTATION_H
#define RADIOSTATION_H

#include "ripping.h"
#include "settings_stream_dialog.h"

/** \brief This class rips radio streams and uses the options available from a config file.
*
* It uses via #config_skeleton a config file (see property #configFileName) to
* determinate the URI of the stream and further options for the record (pregrap,
* postgrab, user agent string...).
*
* It provides the public slot <tt>void showSettingsDialog()</tt>, who shows
* a dialog that is a graphical interface to the config file. */
class radioStation : public ripping
{

     Q_OBJECT

     /** This property holds the path to the configuration file used by this object.
     *   You can determinate the path to the configuration file in the constructor (see there for
     *   further details). Once set there, it can't be changed anymore.
     *
     *   \li \e read: <tt> QString configFileName() const </tt>
     *
     *   Use #config_skeleton for comfortable access to the settings that are stored in the
     *   configuration file.
     *   \sa <tt> QString internal_configFileName </tt> */
     Q_PROPERTY(QString configFileName READ configFileName)

  public:
     //contructor(s) and destructor(s)
     /** Constructor of the class.
     * @param parent     Sets the parent of this object.
     * @param mainWidget When an options dialog gets displayed, this widget
     *                   is used as parent and the dialog
     *                   gets centerd to this Widget.
     * @param configFileName Here you can give a (local!) path to a configuration file. See
     *                   class \e QFile::setName for
     *                   details about which format is valid. If the file doesn't
     *                   exist, the object tries to create
     *                   it. If this isn't possible (for example if the path is invalid
     *                   or you don't have the rights
     *                   or you gave the name of a directory or if you didn't give any path),
     *                   then a new configuration file is created in the appdata directory
     *                   set in KDE, with
     *                   the name <em>stream_??????_rc</em> with the six <em>?</em> replaced by
     *                   random letters. You can obtain the path
     *                   of the really used file later trough the property #configFileName. */
     explicit radioStation(const QPointer<QObject> parent = 0,
                            const QPointer<QWidget> mainWidget = 0,
                            const QString &configFileName = QString());
     /** Conveniance constructor of the class.
     * @param parent     Sets the parent of this object.
     * @param mainWidget When an options dialog gets displayed, this widget is
     *                   used as parent and the dialog
     *                   gets centerd to this Widget.
     * @param configFileName Here you can give a (local!) path to a configuration file. See
     *                   class \e QFile::setName for
     *                   details about which format is valid. If the file doesn't
     *                   exist, the object tries to create
     *                   it. If this isn't possible (for example if the path is invalid
     *                   or you don't have the rights
     *                   or you gave the name of a directory or if you didn't give any path),
     *                   then a new configuration file is created in the appdata directory
     *                   set in KDE, with
     *                   the name <em>stream_??????_rc</em> with the six <em>?</em> replaced by
     *                   random letters. You can obtain the path
     *                   of the really used file later trough the property #configFileName.
     * @param index      Initializes the property #index. */
     explicit radioStation(const QPointer<QObject> parent,
                            const QPointer<QWidget> mainWidget,
                            const QString & configFileName,
                            const qlonglong & index);
     /** The destructor. */
     virtual ~radioStation();
     /** See property #configFileName. */
     QString configFileName() const;

     /** \name Reimplemented properties
     *   These functions for property access are reimplemented. They read from and write to
     *   the file <tt>configFileName</tt>. */
     // doxygen doesn't accept "#configFileName" in this special section.
     //@{
  public:
     /** Reimplemented virtual function that uses the file #configFileName. */
     virtual PropertyValue bitrate() const;
     /** Reimplemented virtual function that uses the file #configFileName. */
     virtual PropertyValue metaInterval() const;
     /** Reimplemented virtual function that uses the file #configFileName. */
     virtual PropertyValue serverName() const;
     /** Reimplemented virtual function that uses the file #configFileName. */
     virtual PropertyValue streamName() const;
  protected:
     /** Reimplemented virtual function that uses the file #configFileName. */
     virtual void setBitrate(qint64 bitrate);
     /** Reimplemented virtual function that uses the file #configFileName. */
     virtual void setMetaInterval(qint64 metaInterval);
     /** Reimplemented virtual function that uses the file #configFileName. */
     virtual void setServerName(const QString newServerName);
     /** Reimplemented virtual function that uses the file #configFileName. */
     virtual void setStreamName(const QString newStreamName);
     //@}

  public slots:
     /** Execs a configuration dialog.
     *   \sa #helper_displaySettingsDialog() */
     int execSettingsDialog();
     /** Shows a configuration dialog.
     *   \sa #helper_displaySettingsDialog() */
     void showSettingsDialog();

  protected:
     /** Provides comfortable access to configuration file (for stream settings).
     *   \sa #configFileName
     *   \sa #shared_config (used internally to access #configFileName) */
     QPointer<settings_stream> config_skeleton;
     /** Is set by the constructor. The settings dialog will be centered
     *   to this widget.
     *   \sa showSettingsDialog() */
     QPointer<QWidget> m_mainWidget;
     /** This function provides the necessary parameters for calling \e streamripper.
     *   Uses ripping::parameterList() and adds the parameters readed from #configFileName. */
     virtual QStringList parameterList() const;
     /** \returns The stream URI readed from #configFileName. */
     virtual QString serverUri() const;
     /** A pointer to the settings dialog (if actually exists one).
     *   \sa showSettingsDialog() */
     QPointer<settings_stream_dialog> settingsDialog;

  private:
     /** A helper function used in the constructors.
     *   \li Tests if #configFileName is working fine
     *       as config file. (If it doesn't exist, than
     *       a new file is created and #configFileName
     *       is set to this new file.)
     *   \li Sets up #shared_config.
     *   \li Sets up #config_skeleton. */
     void helper_setupConfigSystem(const QString & configFile);
     /** Displays a configuration dialog for the settings in the config file.
     *   The dialog is centered to #m_mainWidget. When the dialog becomes closed,
     *   it will be deleted automatically.
     *   \warning The parent of the dialog isn't this object (because it isn't a
     *            widget), but #m_mainWidget. So the destructor must delete the
     *            dialog manually (if it still exists because it hasn't been closed).
     *   \param returnImmediately If \e true, we show() the dialog. If \e false, we
     *                            exec() the dialog.
     *   \returns If using exec(), it returns the exit code (see QDialog::DialogCode).
     *            Else the return value is undefined.
     *   \sa #settingsDialog */
     int helper_displaySettingsDialog(bool returnImmediately);
     /** Used internally to store the property #configFileName. */
     QString internal_configFileName;
     /** This points to a KSharedConfig object that uses #configFileName. Use #config_skeleton
     *   to get comfortable access all these things.
     *
     *   <em>This is a <tt>KSharedConfig::Ptr</tt> (a <tt>KSharedPtr</tt> to a
     *   <tt>KSharedConfig</tt> object). The constructor will create a <tt>KSharedConfig</tt>
     *   object that uses #configFileName - and this pointer will point on it.
     *   When the pointer is deleted (=when the <tt>radiostation</tt>
     *   object it belongs to becomes deleted), it will automatically
     *   delete also the <tt>KSharedConfig</tt> object it points to.</em> */
     KSharedConfig::Ptr shared_config;
 };

#endif
