/*
    Copyright (C) 2008  Tim Fechtner < urwald at users dot sourceforge dot net >

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of
    the License or (at your option) version 3 or any later version
    accepted by the membership of KDE e.V. (or its successor approved
    by the membership of KDE e.V.), which shall act as a proxy
    defined in Section 14 of version 3 of the license.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef SETTINGS_STREAM_WIDGET_CONNECTION_H
#define SETTINGS_STREAM_WIDGET_CONNECTION_H

#include "ui_settings_stream_widget_connection.h"
#include "get_stream_info.h"
#include <QStackedWidget>

/** \brief Widget for the "connection" options in the stream configuration
*          (for use with a KConfigDialog).
*
* It uses the UI that is generated with Qt-Designer.
*
* <em>The interface design and also the internals of this class are unhandy and it is a dirty hack.
* I've spend some hours to improve this, but the result is still ugly - and I don't know how
* to implement the functionality in a more elegant way. Sorry.</em>
*
* This widget provides (through get_stream_info class) on-the-fly
* recognization of some information about the stream (stream
* information): get_stream_info::bitrate, get_stream_info::metaInterval,
* get_stream_info::serverName and get_stream_info::streamName.
* Always when the user writes something to the field with the URI
* of the stream, than the widget tries to recognize the stream
* information immediately. A (grayed out) message "Recognizing" is
* displayed. If it can't connect to the internet or the URI
* isn't valid or isn't a stream, a (grayed out) error message is
* displayed. If it can recognize the stream information, they are
* displayed.
*
* To make this possible, this class makes the widgets for the stream
* information that are connected to the kcfg system and automatically
* handled be <tt>KConfigDialog</tt> (<tt>kcfg_info_bitrate</tt>,
* <tt>kcfg_info_metaInterval</tt>, <tt>kcfg_info_serverName</tt> and
* <tt>kcfg_info_streamName</tt>) invisible. Visible for the user are
* other widgets (QLabels) - that makes nice displaying easier. This
* way, the bitrate must not be displayed in a QSpinBox, but can be
* nicly formated and displayed in a QLabel. This class takes care of
* keeping in sync the visible and the invisible widgets.
*
* However, not everything works automatically. <b>You \e must call
* <tt>load_info_from_kcfg_and_start_recognization()</tt>
* \e after adding the widget to a <tt>KConfigDialog</tt> with
* <tt>KConfigDialog::addWidget()</tt>.</b>
*
* \sa settings_stream */
class settings_stream_widget_connection
  : public QStackedWidget, private Ui::settings_stream_widget_connection
{

  Q_OBJECT

  public:
    /** The constructor. */
    settings_stream_widget_connection(QWidget *parent=0);
    /** The destructor. */
    virtual ~settings_stream_widget_connection();

  public slots:
    /** This function loads the stream infos from the invisible
    *   kcfg widgets. This can't be done in the constructor,
    *   because when construncting this widget, the fields are still
    *   empty. <b>You \e must call
    *   <tt>load_info_from_kcfg_and_start_recognization()</tt>
    *   \e after adding the widget to a <tt>KConfigDialog</tt> with
    *   <tt>KConfigDialog::addWidget()</tt>.</b> Example for code
    *   used inside a class that's derived from <tt>KConfigDialog</tt>:
\code
settings_stream_widget_connection *temp;
temp = new settings_stream_widget_connection(this);
addPage(temp, i18c("NAME"));
temp->load_info_from_kcfg_and_start_recognization();
\endcode
    *   After loading the infos, the function tries to actualise them.
    *   However, when this doesn't work because there's no
    *   connection (to the internet or because the stream server
    *   is down), then nothing is deleted.
    *
    *   Furthermore, this function enables the on-the-fly recognization of URIs
    *   that are typed by the user.
    *
    *   <em>It is necessary to request this manual call of
    *   <tt>load_info_from_kcfg_and_start_recognization()</tt>.
    *   It isn't possible to make it obsolete by connecting to the signal
    *   <tt>textChanged</tt> of the QLineEdit that contains the URL, because
    *   when KConfig sets </em>first<em> the stream information and
    *   </em>after that<em> the serverUri, this class would overwrite
    *   the yet displayed stream information because the URI has changed
    *   and a new recognization starts - and when in this moment no internet
    *   connection is available, all stream information would be deleted.</em> */
    void load_info_from_kcfg_and_start_recognization();

  private:
    /** The get_stream_info object that is used to perform the recognization
    *   of the stream information. */
    get_stream_info m_process;
    /** Deletes the connection between signals of #m_process and
    *   slots from this object and kills the recognization
    *   process.
    *
    *   It is a good idea to delete always the connection \e before killing
    *   streamripper - who knows which strange things could happen...?
    *   \sa helper_setServerUri_connectMProcess_startRecognization() */
    void helper_disconnect_m_process_and_kill();
    /** \returns the localized form of "Recognizing...". */
    QString helper_qstring_localized__recognizing();
    /** Uses the actual URI from the UI for #m_process, sets up
    *   the connection between signals of #m_process and
    *   slots from this object and starts the recognization.
    *
    *   Calling this function will propably cause #m_process to emit
    *   some signals because get_stream_info::setServerUri() resets
    *   all recognized \e streamripper properties.
    *   \sa helper_disconnect_m_process_and_kill() */
    void helper_setServerUri_connectMProcess_startRecognization();

  private slots:
    /** This slot will
    *   \li stop the possibly actually running recognization
    *   \li reset the values of the kcfg_... widgets for the stream info
    *   \li display "Recognizing..." (grayed out) in the corresponding user visible QLabels
    *   \li start a new recognization with for the actual URI */
    void delete_old_streamInfo_and_start_recognization();
    /** This function tests for each of the 4 streamInfos if it has
    *   the value ripping::unset. If so, it changes the corresponding user-visible
    *   QLabel to "Connection failed." (without changing if the QLabel is grayed out or not). */
    void change_empty_user_visible_widgets_to__connection_failed();
    /** This slot is used to set some info about the stream for this widget.
    *   It sets the value for both, the user visible QLabel
    *   (nicly formated - and grayed out if nothing was
    *   recognized an an error message is displayed instead)
    *   and for the widget kcfg_... which is relevant for
    *   the kcfg system.
    *
    *   This parameter \e index is ignored. It just present to
    *   make it possible to connect to the signals of get_stream_info. */
    void setBitrate(qlonglong index, PropertyValue bitrate);
    /** This slot is used to set some info about the stream for this widget.
    *   It sets the value for both, the user visible QLabel
    *   (nicly formated - and grayed out if nothing was
    *   recognized an an error message is displayed instead)
    *   and for the widget kcfg_... which is relevant for
    *   the kcfg system.
    *
    *   This parameter \e index is ignored. It just present to
    *   make it possible to connect to the signals of get_stream_info. */
    void setMetaInterval(qlonglong index, PropertyValue metaInterval);
    /** This slot is used to set some info about the stream for this widget.
    *   It sets the value for both, the user visible QLabel
    *   (nicly formated - and grayed out if nothing was
    *   recognized an an error message is displayed instead)
    *   and for the widget kcfg_... which is relevant for
    *   the kcfg system.
    *
    *   This parameter \e index is ignored. It just present to
    *   make it possible to connect to the signals of get_stream_info. */
    void setServerName(qlonglong index, PropertyValue serverName);
    /** This slot is used to set some info about the stream for this widget.
    *   It sets the value for both, the user visible QLabel
    *   (nicly formated - and grayed out if nothing was
    *   recognized an an error message is displayed instead)
    *   and for the widget kcfg_... which is relevant for
    *   the kcfg system.
    *
    *   This parameter \e index is ignored. It just present to
    *   make it possible to connect to the signals of get_stream_info. */
    void setStreamName(qlonglong index, PropertyValue streamName);
};

#endif
