/*
    Copyright (C) 2008  Tim Fechtner < urwald at users dot sourceforge dot net >

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of
    the License or (at your option) version 3 or any later version
    accepted by the membership of KDE e.V. (or its successor approved
    by the membership of KDE e.V.), which shall act as a proxy
    defined in Section 14 of version 3 of the license.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef STATIONLISTMODEL_H
#define STATIONLISTMODEL_H

#include <QAbstractTableModel>
#include <QPointer>
#include <QList>
#include "radiostation.h"

/** \brief A model (for model-view-programming) handling radio streams.
*
* This class handels radio stations. It reads the list of configuration
* files of radio stations from \e settings_general and represents these radio
* stations. Internally, it uses objects of the class \e radioStation
* to handle the stations. This class can also create new radio stations or
* delete existing ones - and takes care of creating/changing/removing the
* corresponding config files.
*
* This class inherits from \e QAbstractTableModel (and this way indirectly from
* \e QAbstractItemModel). So it can be used as model for Qt's model-view-programming.
* Use a \e QTableView object to watch the data of this model.
*
* \warning Use only one instance of this class at the same time! (Otherwise, the
* same stream could be recorded by more than one instance at the same time - and
* written by all instances to the same file! That's bad! Also this class uses the config
* files - and it is bad when more than one instance uses them... */
class stationlistModel : public QAbstractTableModel
{

     Q_OBJECT

     /** This property holds the total bandwidth that is actually
     *   really used by the streams (summation of the bitrate of all streams
     *   which are not ripping::idle).
     *
     * \warning The formated version contains the value in kbit/s (with SI prefix which means
     * 1000 bit/s, <em> not 1024 bit/s</em>).
     *
     * \li \e read: \code quint64 bandwidth() const \endcode
     * \li \e notify: \code void bandwidthChanged() \endcode
     *
     * \sa \code void recalculate_numberOfActiveStreams_and_bandwidth() \endcode
     * \sa \code quint64 internal_bandwidth \endcode */
     Q_PROPERTY(quint64 bandwidth READ bandwidth)

     /** This property holds the number of active streams (number of all streams
     *   which are not ripping::idle).
     *
     * \li \e read: \code int numberOfActiveStreams() const \endcode
     * \li \e notify: \code void numberOfActiveStreamsChanged() \endcode
     *     and also, if the new number is 0:
     * \li \e notify: \code void numberOfActiveIsZero() \endcode
     *
     * \sa \code void recalculate_numberOfActiveStreams_and_bandwidth() \endcode
     * \sa \code int internal_numberOfActiveStreams \endcode */
     Q_PROPERTY(int numberOfActiveStreams READ numberOfActiveStreams)

  public:
     /** The constructor.
     * \param parent the parent of this object
     * \param mainWidget the widget to which configuration dialogs will be centered. */
     explicit stationlistModel(const QPointer<QObject> parent = 0,
                                const QPointer<QWidget> mainWidget = 0);

     /** The destructor. */
     virtual ~stationlistModel();

     /** This enum is used in columnInfo() to determinate which action
     * will be performed:
     * \li \e stationlistModel::columnHeaderTitle:
     *     Get the header title of the given column (QString).
     * \li \e stationlistModel::columnHeaderToolTip:
     *     Get the tooltip of the header of the given column (QString).
     * \li \e stationlistModel::columnHeaderWhatsThis:
     *     Get the "What's this"-help of the header of the given column (QString).
     * \li \e stationlistModel::columnWidth:
     *     Get the width of the given column from settings_general (quint64). A view
     *     for this model should take care of this value and read it and apply it in
     *     his constructor. In the config file the values aren't saved as a simple list
     *     of values (ordered by column number), but each column has an individual entry
     *     name for the width. This way it is guaranteed that, also when the model changes
     *     the order of columns, each column guards its width. Furthermore, no changes at
     *     the "view" widget are necessary as this system works without that the view
     *     widget knows about the order of the columns and of there content.
     * \li \e stationlistModel::setColumnWidth:
     *     Set the width of the given column in settings_general. A view for this model
     *     should write this value back in his contructor to guarantee that the last state
     *     of the UI is saved.
     * \li \e stationlistModel::columnVisibility:
     *     Get the visibility of the given column from settings_general (bool). A view
     *     for this model should take care of this value and read it and apply it in his
     *     constructor. Like stationlistmodel::columnWidth, it is saved individually for
     *     each column.
     * \li \e stationlistModel::setColumnVisibility:
     *     Set the visibility of the given column in settings_general. A view for this
     *     model should write this value back in his contructor to guarantee that the
     *     last state of the UI is saved.
     * \li \e stationlistModel::columnData:
     *     Get the data of the field specified by column and row (QString).
     * \li \e stationlistModel::columnDataToolTip:
     *     Get the tooltip of the field specified by column and row (QString).
     * \li \e stationlistModel::columnDataWhatsThis:
     *     Get the "What's this"-help of the field specified by column and row (QString). */
     enum columnInfoType {
                           columnHeaderTitle,
                           columnHeaderToolTip,
                           columnHeaderWhatsThis,
                           columnWidth,
                           setColumnWidth,
                           columnVisibility,
                           setColumnVisibility,
                           columnData,
                           columnDataToolTip,
                           columnDataWhatsThis
                         };

     /** See property #bandwidth. */
     quint64 bandwidth() const;

     /** Reimplemented from QAbstractTableModel.
     * @param parent As this is a table model (non-hirarchical!),
     *               this parameter is ignored.
     * \returns the number of columns */
     virtual int columnCount(const QModelIndex & parent = QModelIndex()) const;

     /** This function is the central place to access all information
     *   about the data in the model and also about related things (column
     *   width, visibility, header title...).
     *
     *   \warning This function has the \e const qualifier, but this doesn't
     *   mean that nothing changes. You have not only \e read access, but
     *   also \e write access (to some values).
     *
     *   \param type Specifies the type of action that will be performed.
     *   See #columnInfoType for details.
     *
     *   \param column The column to access.
     *
     *   \param row The row to access. Is ignored if the action is independent
     *   from the row.
     *
     *   \param value The value to write. Is ignored if the action is not
     *   a writing action.
     *
     *   \warning Notice that this function takes as argument the column
     *   \e before the row (which is an unusual order).
     *
     *   \returns The requested data as QVariant. If the passed arguments
     *   were invalid, than an invalid QVariant is returned. */
     QVariant columnInfo(const columnInfoType type,
                          const int column,
                          const qint64 row = (-1),
                          const quint64 value = (-1)) const;

     /** Reimplemented from QAbstractTableModel.
     *
     * Uses columnInfo() to deliver the data.
     *
     * @param index As this is a table model (non-hirarchical!),
     *              only rows and columns of the index are relevant.
     *              The parent is ignored. If the index doesn't belong
     *              to this model, an invalid QVariant is returned.
     * @param role  The type of information that is requested. Supported are
     *              \li Qt::DisplayRole (the data itself)
     *              \li Qt::ToolTipRole
     *              \li Qt::WhatsThisRole.
     *
     * For other values, an invalid \e QVariant is returned.
     * \returns the data of a the spezified item.*/
     virtual QVariant data (const QModelIndex & index, int role = Qt::DisplayRole) const;

     /** Reimplemented from QAbstractTableModel.
     *
     * Uses columnInfo() if orientation is horizontal. */
     virtual QVariant headerData(int section, Qt::Orientation orientation, int role) const;

     /** See property #numberOfActiveStreams. */
     int numberOfActiveStreams() const;

     /** TODO */
     virtual bool queryClose();

     /** TODO */
     virtual void readProperties(const KConfigGroup & m_configGroup);

     /** Reimplemented from QAbstractTableModel.
     * @param parent As this is a table model (non-hirarchical!),
     *               this parameter is ignored.
     * \returns the number of rows */
     virtual int rowCount(const QModelIndex & parent = QModelIndex()) const;

     /** TODO */
     virtual void saveProperties(KConfigGroup & m_configGroup);

     /** Reimplemented from QAbstractTableModel.
     *
     * This is just a dummy function. It (still) does nothing. */
     virtual void sort (int column, Qt::SortOrder order = Qt::AscendingOrder);

  signals:
     /** See property #bandwidth. */
     void bandwidthChanged();
     /** See property #numberOfActiveStreams. */
     void numberOfActiveStreamsChanged();
     /** See property #numberOfActiveStreams. */
     void numberOfActiveStreamsIsZero();

  public slots:
     /** This slot adds a new radio station: It displays a configuration dialog
     *   for the new stream. If accepted by the user (= clicking OK), the new
     *   stream is inserted at the end of the list. */
     void addNewStation();
     /** Removes the specified radio station from this model and
     *   deletes the corresponding config file.
     *   @param index You must pass the index (= the row) of the stream which you want to delete.
     *                When you pass an invalid index, nothing happens. */
     void deleteStation(const int index);
     /** This slot starts the recording of a stream.
     *   @param index You must pass the index (= the row) of the stream which you want to record.
     *                When you pass an invalid index, nothing happens. */
     void record(const int index);
     /** If exactly one stream is selected, then this slot shows the config dialog.
     *   Else nothing happens. */
     void showConfigDialog(const int index);
     /** This slot stops the recording of a stream.
     *   @param index You must pass the index (= the row) of the stream
     *                which you want to stop the recording.
     *                When you pass an invalid index, nothing happens. */
     void stopRecording(const int index);

  private:
     /** This helper function sets up all the necessary connection of
     * signals and slots which are needed to keep this model up to date.
     * \warning Call this function only \e once for each station.
     * \param m_stream a pointer to the station for which you want
     * to set up the connection */
     void helper_connectSignalsAndSlots(QPointer<radioStation> m_stream);
     /** Writes the actual list of streams back to the configuration file. */
     void helper_writeStationListToGeneralSettings();
     /** Used internally to cache the value of the property #bandwidth. */
     quint64 internal_bandwidth;
     /** Used internally to cache the value of the property #numberOfActiveStreams. */
     int internal_numberOfActiveStreams;
     /** Here you can save a list of the streams for which
     * radioStation::doesTheUserWantsThatTheStreamIsRipping() is \e true.
     * Used for session management handling.
     *
     * By initialization, this pointer is \e null.
     *
     * \warning Keep this pointer always valid and set it back
     * to \e null when deleting the QStringList. The destructor
     * will delete a possibly remaining QStringlist, and this
     * will lead to a crash when the pointer is invalid. */
     QStringList * m_listOfStreamsOfWhichTheUserWantsThatTheyRip;
     /** A pointer to the widget to which configuration dialogs should be centered.
     * This member is initialized in the constructor. */
     QPointer<QWidget> m_mainWidget;
     /** This is a QList of objects of the radiostation class
     *   used internally to handle the stations. */
     QList< QPointer<radioStation> > m_stationlist;  /* we need a pointer as list member
                                                      instead of radioStation itself
                                                      because we need the construct
                                                      radioStation with the constructor
                                                      (no other way to set the config file),
                                                      and when radioStation's
                                                      constructor is called, the config file
                                                      is opened and so on. I
                                                      don't think it's save to copy this
                                                      class - what's necessary
                                                      when not using a pointer. */

  private slots:
     /** This slot recalculates the values for the properties
     * #bandwidth and #numberOfActiveStreams from scratch. It
     * caches them in the corresponding member. If for one of
     * the properties or for both of them, the new value differs
     * from the old value, the corresponding signal(s) is/are
     * emitted.
     *
     * <i>We recalculate the value each time from scratch. We
     * could have implemented a solution the reacts on signals
     * of the radiostation objects that the status has changed,
     * and than increment or decrement our properties. This
     * solution would be more efficient, but also more complicate:
     * We would have to worry about a correct initialization,
     * about session management.... The situation would be
     * confusing and complicate. So it's better to have a clean
     * solution, also when it's not so efficient.</i> */
     void recalculate_numberOfActiveStreams_and_bandwidth();
     /** Used internally to emit the signal <tt>dataChanged</tt>. */
     void reloadBitrate(const qlonglong stationIndex);
     /** Used internally to emit the signal <tt>dataChanged</tt>. */
     void reloadDataSize(const qlonglong stationIndex);
     /** Used internally to emit the signal <tt>dataChanged</tt>. */
     void reloadMetaInterval(const qlonglong stationIndex);
     /** Used internally to emit the signal <tt>dataChanged</tt>. */
     void reloadRelayPort(const qlonglong stationIndex);
     /** Used internally to emit the signal <tt>dataChanged</tt>. */
     void reloadServerName(const qlonglong stationIndex);
     /** Used internally to emit the signal <tt>dataChanged</tt>. */
     void reloadSong(const qlonglong stationIndex);
     /** Used internally to emit the signal <tt>dataChanged</tt>. */
     void reloadStatus(const qlonglong stationIndex);
     /** Used internally to emit the signal <tt>dataChanged</tt>. */
     void reloadStreamName(const qlonglong stationIndex);
     /** TODO */
     void rememberListOfStreamsWhichTheUserWantsToRip_ifNotYetDone();

};

#endif
