/***************************************************************************
                          calltreedlg.cpp  -  description
                             -------------------
    begin                : Fri Jul 18 2003
    copyright            : (C) 2003 by Elad Lahav
    email                : elad_lahav@users.sf.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qapplication.h>
#include <qclipboard.h>
#include <qfile.h>
#include <qpushbutton.h>
#include <qradiobutton.h>
#include <klocale.h>
#include "calltreedlg.h"
#include "cscopefrontend.h"
#include "queryresultsmenu.h"

#define FILE_VERSION	"VERSION=2"

/** File Name index for the file name generation */
int CallTreeDlg::m_iFileNameIndex = 0;

/**
 * Class constructor.
 * @param	sFunc	The name of the function that is the root of the tree
 * @param	pParent	The parent widget
 * @param	szName	The widget's name
 */
CallTreeDlg::CallTreeDlg(const QString& sFunc, QWidget* pParent, 
	const char* szName) :
	CallTreeLayout(pParent, szName),
	m_progress(m_pTree),
	m_pActiveCallingItem(NULL),
	m_pActiveCalledItem(NULL),
	m_sFileName(sFunc)
{
	// Make sure the dialog object is deleted when the dialog is closed
	setWFlags(getWFlags() | WDestructiveClose);

	// Create a persistent Cscope process
	m_pCscope = new CscopeFrontend();
	
	// Create the popup-menu
	m_pQueryMenu = new QueryResultsMenu(this);
	m_pQueryMenu->setColumns(1, 0, 2, 3);
	
	// Add records output by the Cscope process
	connect(m_pCscope, SIGNAL(dataReady(FrontendToken*)), this,
		SLOT(slotDataReady(FrontendToken*)));
	
	// Display query progress information
	connect(m_pCscope, SIGNAL(progress(int, int)), this,
		SLOT(slotProgress(int, int)));	
		
	// Makes an item child-less if the query returns no results
	connect(m_pCscope, SIGNAL(finished(uint)), this,
		SLOT(slotFinished(uint)));
	
	// Close the dialog when the "Close" button is clicked
	connect(m_pCloseButton, SIGNAL(clicked()), this, SLOT(close()));

	// Search for calling functions when an item is first expanded
	connect(m_pTree, SIGNAL(expanded(QListViewItem*)), this,
		SLOT(slotExpandFunc(QListViewItem*)));

	// Open an editor with the matching file and line number, when an item is
	// double-clicked
	connect(m_pTree, SIGNAL(doubleClicked(QListViewItem*)), this,
		SLOT(slotRequestLine(QListViewItem*)));

	// Switch between called and calling modes
	connect(m_pCalling, SIGNAL(toggled(bool)), this,
		SLOT(slotCallTreeModeChanged(bool)));
	connect(m_pCalled, SIGNAL(toggled(bool)), this,
		SLOT(slotCallTreeModeChanged(bool)));
		
	// Show the popup-menu when requested
	connect(m_pTree, 
		SIGNAL(contextMenuRequested(QListViewItem*, const QPoint&, int)),
		m_pQueryMenu, SLOT(slotShow(QListViewItem*, const QPoint&, int)));
		
	// Handle popup-menu commands
	connect(m_pQueryMenu, SIGNAL(copy(QListViewItem*, int)), this,
		SLOT(slotCopy(QListViewItem*, int)));
	connect(m_pQueryMenu, SIGNAL(search(QListViewItem*, const QRegExp&, int)),
		this, SLOT(slotSearch(QListViewItem*, const QRegExp&, int)));
	connect(m_pQueryMenu, SIGNAL(showAll()), this, SLOT(slotShowAll()));
	connect(m_pQueryMenu, SIGNAL(remove(QListViewItem*)), this,
		SLOT(slotRemoveItem(QListViewItem*)));

	// Initialise the tree's columns
	m_pTree->addColumn(i18n("Function"));
	m_pTree->addColumn(i18n("File"));
	m_pTree->addColumn(i18n("Line"));
	m_pTree->addColumn(i18n("Text"));
	m_pTree->hide();

	// Allocate shadow trees for both Calling and Called modes, hide them
	m_pCallingTreeSave = new QListView(this);
	m_pCallingTreeSave->hide();
	m_pCalledTreeSave = new QListView(this);
	m_pCalledTreeSave->hide();
	
	if (sFunc != QString::null && !sFunc.isEmpty()) {
		// Generate unique file name to save call tree later
		m_sFileName.replace(' ', '_');
		m_sFileName += QString::number(++m_iFileNameIndex);
	
		// Set the root item
		m_pActiveItem = new QListViewItem(m_pTree, sFunc);
	
		// Add the immediate called functions
		m_pTree->setOpen(m_pActiveItem, true);
		queryCallTree(sFunc);
	}
	m_pTree->show();
}

/**
 * Class destructor.
 */
CallTreeDlg::~CallTreeDlg()
{
	// Notify everybody interested
	emit closed(this);
	
	delete m_pCscope;
	delete m_pCallingTreeSave;
	delete m_pCalledTreeSave;
}

/**
 * Adds an entry to the tree, as the child of the active item.
 * Called by a CscopeFrontend object, when a new entry was received in its
 * whole from the Cscope back-end process. The entry contains the data of a
 * function calling the function described by the active item.
 * @param	pToken	The first token in the entry
 */
void CallTreeDlg::slotDataReady(FrontendToken* pToken)
{
	QString sFile, sFunc, sLine, sText;
	QListViewItem* pItem;

	// Get the file name
	sFile = pToken->getData();
	pToken = pToken->getNext();

	// Get the function name
	sFunc = pToken->getData();
	pToken = pToken->getNext();

	// Get the line number
	sLine = pToken->getData();
	if (!sLine.toInt()) {
		// Line number could not be 0!
		// means that function name was empty
		sLine = sFunc;
		sFunc = "<global>";
	} else 
		pToken = pToken->getNext();

	// Get the line's text
	sText = pToken->getData();
	pToken = pToken->getNext();

	// Add a list item
	pItem = new QListViewItem(m_pActiveItem, sFunc, sFile, sLine, sText);
	pItem->setExpandable(true);

	// Open the parent item, if not yet opened
	if (!m_pActiveItem->isOpen())
		m_pTree->setOpen(m_pActiveItem, true);
}

/**
 * Displays search progress information.
 * This slot is connected to the progress() signal emitted by a
 * CscopeFrontend object.
 * @param	nProgress	The current progress value
 * @param	nTotal		The expected final value
 */
void CallTreeDlg::slotProgress(int nProgress, int nTotal)
{
	m_progress.setProgress(nProgress, nTotal);
}

/**
 * Disables the expandability feature of an item, if no functions calling it
 * were found.
 * @param	nRecords	Number of records reported by the query
 */
void CallTreeDlg::slotFinished(uint nRecords)
{
	// Destroy the progress bar
	m_progress.finished();
		
	if (nRecords == 0)
		m_pActiveItem->setExpandable(false);
}

/**
 * Queries Cscope for functions calling the specified function.
 * @param	sFunc	Function name to query
 */
void CallTreeDlg::queryCallTree(const QString& sFunc)
{
	if (m_pCalling->isOn())
		m_pCscope->query(CscopeFrontend::Calling, sFunc);
	else
		m_pCscope->query(CscopeFrontend::Called, sFunc);
}

/**
 * Looks for calling functions to the function item being expanded.
 * @param	pItem	A tree item being expanded by the user
 */
void CallTreeDlg::slotExpandFunc(QListViewItem* pItem)
{
	// Do not re-query
	if (pItem->childCount() > 0)
		return;
		
	m_pActiveItem = pItem;
	queryCallTree(pItem->text(0));
}

/**
 * Emits a signal indicating a certain source file and line number are
 * requested.
 * This slot is connected to the dobuleClicked() signal of the tree. The
 * signal emitted by this slot is used to display an
 * editor page at the requested line.
 * @param	pItem	The list item selected by the user
 */
void CallTreeDlg::slotRequestLine(QListViewItem* pItem)
{
	QString sFilePath, sLine;

	// Get the file name and line number
	sFilePath = pItem->text(1);
	sLine = pItem->text(2);

	// The root item has no associated file
	if (sFilePath.isEmpty())
		return;
		
	// Emit the signal
	emit lineRequested(sFilePath, sLine.toUInt());
}

/**
 * Refreshes the call tree based on the newly selected call tree mode.
 * This slot is connected to the toggled() signal of the call tree mode radio
 * buttons.
 * @param	bOn	true if the button was toggled on
 */
void CallTreeDlg::slotCallTreeModeChanged(bool bOn)
{
	if (bOn) {
		QListView **pOldList, **pNewList;
		QListViewItem *pTempItem, *pNewActiveItem, *pOldActiveItem;
		
		// Setup old and new pointers for the mode switch
		if (m_pCalling->isOn()) {
			pOldList = &m_pCalledTreeSave;
			pNewList = &m_pCallingTreeSave;
			pOldActiveItem = m_pActiveCalledItem;
			pNewActiveItem = m_pActiveCallingItem;
		} else {
			pOldList = &m_pCallingTreeSave;
			pNewList = &m_pCalledTreeSave;
			pOldActiveItem = m_pActiveCallingItem;
			pNewActiveItem = m_pActiveCalledItem;
		}
		
		// Initialize interators 
		QListViewItemIterator itTree(m_pTree);
		QListViewItemIterator itNew(*pNewList);
		// Get the original function name from the current tree
		QString sFunc(itTree.current()->text(0));
		// Hide the tree before switch
		m_pTree->hide();
		// Move current tree to the *pOldList
		for ( pTempItem = itTree.current(); (pTempItem = itTree.current()); 
			++itTree ) {
			m_pTree->takeItem(pTempItem);
			(*pOldList)->insertItem(pTempItem);
		}
		// Move the new tree into visible (but hidden) m_pTree
		for ( pTempItem = itNew.current(); (pTempItem = itNew.current()); 
			++itNew ) {
			(*pNewList)->takeItem(pTempItem);
			m_pTree->insertItem(pTempItem);
		}
		// Switch pointers on active items
		pOldActiveItem = m_pActiveItem;
		m_pActiveItem = pNewActiveItem;
		if (!m_pActiveItem && !m_pTree->childCount()) {
			// If second tree is not initialized yet, then init it.
			pNewActiveItem = new QListViewItem(m_pTree, sFunc);
			m_pActiveItem = pNewActiveItem;
			// Add the immediate functions
			m_pTree->setOpen(m_pActiveItem, true);
			queryCallTree(sFunc);
		}
		// Make updated call tree visible
		m_pTree->show();
	}
}

/**
 * Restores a call tree from the given call tree file.
 * NOTE: The call tree file is deleted when loading is complete.
 * @param	sProjPath	The full path of the project directory
 * @param	sFileName	The name of the call tree file to load
 * @return	true if successful, false otherwise
 */
bool CallTreeDlg::load(const QString& sProjPath, const QString& sFileName)
{
	QListView **pTree = NULL;
	QListViewItem* pItem;
	QString sTemp, sFunc, sFile, sLine, sText;
	int nState, nX, nY, nHeight, nWidth;
	int nDepth = 0;
	bool bNeedToggle = false;
	
	// Try to open the query file for reading
	QFile file(sProjPath + "/" + sFileName);
	if (!file.exists() || !file.open(IO_ReadOnly))
		return false;
	
	{
		// Use a new scope for the QTextStream object, to ensure its 
		// destruction before the file is deleted
		QTextStream str(&file);
		
		// Make sure the file's version is correct
		sTemp = str.readLine();
		if (sTemp != FILE_VERSION) {
			file.remove();
			return false;
		}
		
		// Read window position
		sTemp = str.readLine();
		if (sTemp == QString::null || sTemp.isEmpty())
			return false;
		nX = sTemp.toInt();
		sTemp = str.readLine();
		if (sTemp == QString::null || sTemp.isEmpty())
			return false;
		nY = sTemp.toInt();
		sTemp = str.readLine();
		if (sTemp == QString::null || sTemp.isEmpty())
			return false;
		nWidth = sTemp.toInt();
		sTemp = str.readLine();
		if (sTemp == QString::null || sTemp.isEmpty())
			return false;
		nHeight = sTemp.toInt();
		
		// Reset m_pActiveItem;
		m_pActiveItem = NULL;
		// Read query records
		sTemp = str.readLine();
		nState = 0;
		while (sTemp != QString::null) {
			switch (nState) {
			// Call Tree depth
			case 0:
				if ((sTemp != "Calling:") && 
					(sTemp != "Called:")) {
					nDepth = sTemp.toInt();
					break;
				}
				// Read the call tree function
				sFunc = str.readLine();
				if (sFunc == QString::null || 
					sFunc.isEmpty())
					return false;
				// Check the call tree type 
				if (sTemp == "Calling:") {
					// Now reading the calling tree
					pTree = &m_pCallingTreeSave;
					// Allocate the root item
					m_pActiveCallingItem = new 
						QListViewItem(*pTree, sFunc);
					m_pActiveItem = m_pActiveCallingItem;
				} else {
					// Now reading the called tree
					if (!pTree)
						bNeedToggle = true;
					pTree = &m_pCalledTreeSave;
					// Allocate the root item
					m_pActiveCalledItem = new 
						QListViewItem(*pTree, sFunc);
					m_pActiveItem = m_pActiveCalledItem;
				}
				
				// Set the root item
				m_pActiveItem->setExpandable(true);
				m_pActiveItem->setOpen(true);
				m_sFileName = sFunc;
				m_sFileName.replace(' ', '_');
				m_sFileName += 
					QString::number(++m_iFileNameIndex);
				// Read the call tree first entry depth
				sTemp = str.readLine();
				nDepth = sTemp.toInt();
				break;
				
			// Function name
			case 1:
				sFunc = sTemp;
				break;
				
			// File path
			case 2:
				sFile = sTemp;
				break;
				
			// Line number
			case 3:
				sLine = sTemp;
				break;
				
			// Text string
			case 4:
				if (!m_pActiveItem)
					return false;
				
				sText = sTemp;
				while (nDepth <= m_pActiveItem->depth())
					m_pActiveItem = m_pActiveItem->parent();
				
				if (m_pActiveItem) {
					pItem = new QListViewItem(m_pActiveItem,
						sFunc, sFile, sLine, sText);
					m_pActiveItem->setOpen(true);
				}
				else {
					pItem = new QListViewItem(*pTree,
						sFunc, sFile, sLine, sText);
				}
				
				if (pItem) {
					pItem->setExpandable(true);
					m_pActiveItem = pItem;
				}
				break;
			}
			
			nState = (nState + 1) % 5;
			sTemp = str.readLine();
		}
	}
	
	// Delete the query file
	file.remove();
	
	// Now need to populate m_pTree from the source tree
	m_pTree->hide();
	
	// Initialize interator for m_pTree
	QListViewItemIterator itTree(m_pTree);
	// Clean the cuurent Call tree (just in case)
	for ( pItem = itTree.current(); (pItem = itTree.current());
		 ++itTree ) {
		m_pTree->takeItem(pItem);
		delete pItem;
	}
	
	// Initialize interator for the m_pCallingTreeSave tree
	QListViewItemIterator itSrcTree(m_pCallingTreeSave);
	for ( pItem = itSrcTree.current(); (pItem = itSrcTree.current()); 
		++itSrcTree ) {
		m_pCallingTreeSave->takeItem(pItem);
		m_pTree->insertItem(pItem);
	}
	
	// Restore the root item
	m_pActiveItem = m_pCallingTreeSave->firstChild();
	
	// Update radio button according to the current mode
	if (bNeedToggle)
		m_pCalled->setChecked(true);
	
	// Now need to display m_pTree to user
	m_pTree->show();
	// Some Window Managers could not move window when it is hidden
	show();
	move(nX, nY);
	resize(nWidth, nHeight);
	
	return true;
}

/**
 * Writes the contents of the call tree dialog to a call tree file.
 * This method is called for call trees before the owner project is
 * closed.
 * @param	sProjPath	The full path of the project directory
 * @return	The name of the call tree file generated for this dialog
 *			(or an empty string if no file was written)
 */
void CallTreeDlg::store(const QString& sProjPath)
{
	// Open the query file for writing
	QFile file(sProjPath + "/" + m_sFileName);
	if (!file.open(IO_WriteOnly))
		return;
	
	QTextStream str(&file);
	
	// Write the version string
	str << FILE_VERSION << "\n";
	
	// Write the coordinates
	str << x() << "\n" << y() << "\n";
	// Write the size of the window
	str << width() << "\n" << height() << "\n";
	
	saveSubTreeToStream(m_pCalling->isOn(), str);
	saveSubTreeToStream(!m_pCalling->isOn(), str);
	
	file.close();
	return;
}

/** 
 * Saves current more Call Tree to the stream.
 * @param	bCalling	true if this is a calling-functions sub-tree, false
 *						if this is a called-by sub-tree
 * @param	str			A text stream used to store the sub-tree
 */
void CallTreeDlg::saveSubTreeToStream(bool bCalling, QTextStream& str)
{
	QListView** ppTree;

	// Initialize pointers to the correct QListView object
	if (m_pCalling->isOn()) {
		if (bCalling)
			ppTree = &m_pTree;
		else 
			ppTree = &m_pCalledTreeSave;
	} else {
		if (bCalling)
			ppTree = &m_pCallingTreeSave;
		else 
			ppTree = &m_pTree;
	}
	
	QListViewItemIterator itrTree(*ppTree);
	if (!itrTree.current())
		return;
	
	QString sFunc(itrTree.current()->text(0));
	
	// Write the name, type and text of the Call Tree
	str << (bCalling ? "Calling:\n" : "Called:\n") << sFunc << "\n";
	
	// Write all records
	for(++itrTree; itrTree.current(); ++itrTree) {
		str << itrTree.current()->depth() << "\n" 
			<< itrTree.current()->text(0) << "\n" 
			<< itrTree.current()->text(1) << "\n"
			<< itrTree.current()->text(2) << "\n"
			<< itrTree.current()->text(3) << "\n";
	}
}

/**
 * Copies the text of the requested field (item+column) to the clipboard.
 * This slot is connected to the copy() signal of the QueryResultsMenu object.
 * @param	pItem	The item from which to copy
 * @param	nCol	The index of the item field to copy
 */
void CallTreeDlg::slotCopy(QListViewItem* pItem, int nCol)
{
	QApplication::clipboard()->setText(pItem->text(nCol),
		QClipboard::Clipboard);
}

/**
 * Hides all children of a given item that do not meet the search criteria.
 * An item meets the criteria if any of its children does.
 * This slot is connected to the search() signal emitted by the
 * QueryResultsMenu object.
 * @param	pItem	The root of the subtree to search
 * @param	re		The regular expression used for filtering
 * @param	nCol	The number of the column to which the RE is matched
 */
void CallTreeDlg::slotSearch(QListViewItem* pItem, const QRegExp& re, 
	int nCol)
{
	QListViewItem* pChild;
	
	// Use the tree's root if a NULL item was given
	if (pItem == NULL)
		pItem = m_pTree->firstChild();
	
	// Filter child items
	for (pChild = pItem->firstChild(); pChild != NULL; pChild =
		pChild->nextSibling()) {
		if (pChild->isVisible())
			slotSearch(pChild, re, nCol);
	}
	
	// Check if any of the child items are now visible			
	for (pChild = pItem->firstChild(); pChild != NULL; pChild =
		pChild->nextSibling()) {
		if (pChild->isVisible())
			break;
	}
	
	// An item should be hidden if it doesn't match the search criteria,
	// and neither do any of its children
	if ((pChild == NULL) && (re.search(pItem->text(nCol)) == -1))
		pItem->setVisible(false);
}

/**
 * Makes all tree items visible.
 * This slot is connected to the showAll() signal emitted by the
 * QueryResultsMenu object.
 */
void CallTreeDlg::slotShowAll()
{
	QPtrList<QListViewItem> stack;
	QListViewItem* pItem;
	
	stack.append(m_pTree->firstChild());
	
	// Use a depth-first search to iterate over all tree items
	while (true) {
		pItem = stack.last();
		if (pItem == NULL)
			break;
			
		// Make the current item visible
		pItem->setVisible(true);
		stack.removeLast();
		
		// Do the same for all of its children
		for (pItem = pItem->firstChild(); pItem != NULL; pItem =
			pItem->nextSibling()) {
			stack.append(pItem);
		}
	}
}

/**
 * Deletes the item on which a popup-menu has been invoked.
 * This slot is connected to the remove() signal of the QueryResultsMenu
 * object.
 * @param	pItem	The item to remove
 */
void CallTreeDlg::slotRemoveItem(QListViewItem* pItem)
{
	delete pItem;
}
