/***************************************************************************
                          historypage.h  -  description
                             -------------------
    begin                : Wed Dec 15 2004
    copyright            : (C) 2004 by Elad Lahav
    email                : elad_lahav@users.sf.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <klocale.h>
#include "historypage.h"

/**
 * A list view item for holding position history records.
 * A QListViewItem cannot reference its preceding item, which is required to
 * create a stack-like history list. Therefore a HistoryListItem object
 * stores a pointer to the item just above it in the list. The pointer is
 * persistent, since the list cannot be sorted.
 * @author Elad Lahav
 */

class HistoryListItem : public QListViewItem
{
public:
	/**
	 * Class constructor.
	 * @param	pList	The parent list view
	 * @param	sFile	The file path in this record
	 * @param	sLine	The line number
	 * @param	sText	The contents of the line in the given file
	 */
	HistoryListItem(QListView* pList, QString sFile, QString sLine, 
		QString sText) : 
		QListViewItem(pList, sFile, "", sLine, sText),
		m_pPrevSibling(NULL) {
		HistoryListItem* pNext;
		
		// Mark the new item as the predecessor of its next sibling 
		if ((pNext = (HistoryListItem*)nextSibling()) != NULL)
			pNext->m_pPrevSibling = this;
	}
	
	/** The item immediately above this one in the list. */
	HistoryListItem* m_pPrevSibling;
};

int HistoryPage::s_nMaxPageID = 0;

/**
 * Class constructor.
 * @param	pParent	The parent widget
 * @param	szName	The widget's name
 */
HistoryPage::HistoryPage(QWidget* pParent, const char* szName) :
	QueryPage(pParent, szName),
	m_nPageID(++s_nMaxPageID)
{
	// Disable sorting
	m_pList->setSortColumn(-1);
}

/**
 * Class destructor.
 */
HistoryPage::~HistoryPage()
{
	if (s_nMaxPageID == m_nPageID)
		s_nMaxPageID--;
}

/**
 * Creates a new position history record at the top of the list.
 * @param	sFile	The file name associated with the record
 * @param	nLine	The line number
 * @param	sText	The text of the file at the given line
 */
void HistoryPage::addRecord(const QString& sFile, uint nLine, 
	const QString& sText)
{
	HistoryListItem* pItem, * pNextItem;
	
	pItem = (HistoryListItem*)m_pList->currentItem();
	if (pItem != NULL) {
		// Do not add duplicate items
		if ((pItem->text(0) == sFile) && (pItem->text(2).toUInt() == nLine))
			return;
			
		// Remove all items above the current one, so the new item is added to
		// the top of the list
		pItem = pItem->m_pPrevSibling;
		while (pItem != NULL) {
			pNextItem = pItem;
			pItem = pItem->m_pPrevSibling;
			delete pNextItem;
		}
	}
	
	// Create the new item at the top of the list
	pItem = new HistoryListItem(m_pList, sFile, QString::number(nLine),
		sText);
	m_pList->setSelected(pItem, true);
}

/**
 * Creates a new history item.
 * This version is used when history records are read from a file.
 * @param	sFile	The file name
 * @param	sFunc	The function defining the scope of the result
 * @param	sLine	The line number
 * @param	sText	The contents of the line
 */
void HistoryPage::addRecord(const QString& sFile, const QString&,
	const QString& sLine, const QString& sText)
{
	HistoryListItem* pItem;
		
	pItem = new HistoryListItem(m_pList, sFile, sLine, sText);
	m_pList->setSelected(pItem, true);
}

/**
 * Moves to the previous item in the history, selecting it for display.
 * Note that this function move to the item which chronologically precedes
 * the current one, which, in list view terms, is the next item.
 */
void HistoryPage::selectPrev()
{
	QListViewItem* pItem;

	// Get the current item
	pItem = m_pList->currentItem();
	
	// Select the next item in the list
	if (pItem != NULL && pItem->nextSibling() != NULL) {
		pItem = pItem->nextSibling();
		m_pList->setSelected(pItem, true);
		emit recordSelected(pItem);
	}
}

/**
 * Moves to the next item in the history, selecting it for display.
 * Note that this function move to the item which chronologically succedes
 * the current one, which, in list view terms, is the previous item.
 */
void HistoryPage::selectNext()
{
	HistoryListItem* pItem;

	// Get the current item
	pItem = (HistoryListItem*)m_pList->currentItem();
	
	// Select the previous item in the list
	if (pItem != NULL && pItem->m_pPrevSibling != NULL) {
		pItem = pItem->m_pPrevSibling;
		m_pList->setSelected(pItem, true);
		emit recordSelected(pItem);
	}
}

/**
 * Creates a tab caption for this page, based on the unique page ID.
 * @param	bBrief	true to use brief caption names, false otherwise
 */
QString HistoryPage::getCaption(bool bBrief) const
{
	return (bBrief ? QString(i18n("HIS ")) : QString(i18n("History "))) +
		QString::number(m_nPageID);
}

/**
 * Creates a unique file name for saving the contents of the history page.
 * @return	The unique file name to use
 */
QString HistoryPage::getFileName(const QString&) const
{
	return QString("History_") + QString::number(m_nPageID);
}

/**
 * Deletes the item on which a popup-menu has been invoked.
 * This slot is connected to the remove() signal of the QueryResultsMenu
 * object.
 * @param	pItem	The item to remove
 */
void HistoryPage::slotRemoveItem(QListViewItem* pItem)
{
	HistoryListItem* pCurItem, * pNextItem;
	
	pCurItem = (HistoryListItem*)pItem;
	if ((pNextItem = (HistoryListItem*)pCurItem->nextSibling()) != NULL)
		pNextItem->m_pPrevSibling = pCurItem->m_pPrevSibling;
	
	delete pCurItem;
}
