/***************************************************************************
                          querypage.cpp  -  description
                             -------------------
    begin                : Sat May 24 2003
    copyright            : (C) 2003 by Elad Lahav
    email                : elad_lahav@users.sf.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qapplication.h>
#include <qclipboard.h>
#include <qfile.h>
#include <klocale.h>
#include "querypage.h"
#include "kscopeconfig.h"
#include "cscopefrontend.h"
#include "queryresultsmenu.h"
#include "searchresultsdlg.h"

#define FILE_VERSION	"VERSION=2"

const char* QUERY_TYPES[][2] = {
	{ "References to ", "REF " },
	{ "Definition of ", "DEF " },
	{ "Functions called by ", "<-- " },
	{ "Functions calling ", "-->" },
	{ "Search for ", "TXT " },
	{ "", "" },
	{ "EGrep Search for ", "GRP " },
	{ "Files named ", "FIL " },
	{ "Files #including ", "INC " },
	{ "Query", "Query" }
};

/**
 * Class constructor.
 * @param	pParent	The parent widget
 * @param	szName	The widget's name
 */
QueryPage::QueryPage(QWidget* pParent, const char * szName) :
	QHBox(pParent, szName),
	m_pList(new QListView(this)),
	m_bLocked(false),
	m_nType(CscopeFrontend::None),
	m_progress(m_pList),
	m_bAttached(false),
	m_pLastItem(NULL)
{
	// Create the popup-menu
	m_pQueryMenu = new QueryResultsMenu(this);

	// Initialise the list's columns
	m_pList->setAllColumnsShowFocus(true);
	m_pList->addColumn(i18n("File"));
	m_pList->addColumn(i18n("Function"));
	m_pList->addColumn(i18n("Line"));
	m_pList->addColumn(i18n("Text"));
	m_pList->setColumnAlignment(2, Qt::AlignRight);

	// Set colours and font
	applyPrefs();

	// A record is selected if it is either double-clicked, or the "Enter"
	// key is pressed while the record has the focus
	connect(m_pList, SIGNAL(doubleClicked(QListViewItem*)), this, 
		SIGNAL(recordSelected(QListViewItem*)));
	connect(m_pList, SIGNAL(returnPressed(QListViewItem*)), this, 
		SIGNAL(recordSelected(QListViewItem*)));
		
	// Show the popup-menu when requested
	connect(m_pList, 
		SIGNAL(contextMenuRequested(QListViewItem*, const QPoint&, int)),
		m_pQueryMenu, SLOT(slotShow(QListViewItem*, const QPoint&, int)));
		
	// Handle popup-menu commands
	connect(m_pQueryMenu, SIGNAL(copy(QListViewItem*, int)), this,
		SLOT(slotCopy(QListViewItem*, int)));
	connect(m_pQueryMenu, SIGNAL(search(QListViewItem*, const QRegExp&, int)),
		this, SLOT(slotSearch(QListViewItem*, const QRegExp&, int)));
	connect(m_pQueryMenu, SIGNAL(showAll()), this, SLOT(slotShowAll()));
	connect(m_pQueryMenu, SIGNAL(remove(QListViewItem*)), this,
		SLOT(slotRemoveItem(QListViewItem*)));
}

/**
 * Class destructor.
 */
QueryPage::~QueryPage()
{
}

/**
 * Associates this query page with a Cscope process.
 * @param	pCscope	A Cscope process running a query
 */
void QueryPage::attach(CscopeFrontend* pCscope)
{
	// Make sure sorting is disabled while entries are added
	m_pList->setSorting(100);
		
	// Add records to the page when Cscope outputs them
	connect(pCscope, SIGNAL(dataReady(FrontendToken*)), this,
		SLOT(slotDataReady(FrontendToken*)));
		
	// Report progress information
	connect(pCscope, SIGNAL(progress(int, int)), this,
		SLOT(slotProgress(int, int)));

	// Perform tasks when the query process terminates
	connect(pCscope, SIGNAL(finished(uint)), this,
		SLOT(slotFinished(uint)));
		
	m_bAttached = true;
}

/**
 * Adds a query entry to the list.
 * Called by a CscopeFrontend object, when a new entry was received in its
 * whole from the Cscope back-end process.
 * @param	pToken	The first token in the entry
 */
void QueryPage::slotDataReady(FrontendToken* pToken)
{
	QString sFile, sFunc, sLine, sText;
	
	// Get the file name
	sFile = pToken->getData();
	pToken = pToken->getNext();

	// Get the function name
	sFunc = pToken->getData();
	pToken = pToken->getNext();

	// Get the line number
	sLine = pToken->getData();
	if (!sLine.toInt()) {
		// Line number could not be 0!
		// means that function name was empty
		sLine = sFunc;
		sFunc = "<global>";
	} else 
		pToken = pToken->getNext();

	// Get the line's text
	sText = pToken->getData();
	pToken = pToken->getNext();

	// Add a new item at the end of the list
	m_pLastItem = new QListViewItem(m_pList, m_pLastItem, sFile, sFunc, sLine,
		sText);
}

/**
 * Handles a finished query event, reported by the Cscope frontend object.
 * If no resutls are available, a proper message is displayed. If only one 
 * record was generated by Cscope, it is automatically selected for viewing.
 * @param	nRecords	The number of records the query has generated
 */
void QueryPage::slotFinished(uint nRecords)
{
	// Destroy the progress bar
	m_progress.finished();
		
	// Report a query that has returned an empty record set
	if (nRecords < 1)
		(void)new QListViewItem(m_pList, i18n("No results"));
	
	// Signal that the query process has terminated
	emit queryFinished();
	
	// Auto-select a single record
	if (nRecords == 1)
		emit recordSelected(m_pList->firstChild());
		
	m_bAttached = false;
}

/**
 * Displays search progress information.
 * This slot is connected to the progress() signal emitted by a
 * CscopeFrontend object.
 * @param	nFiles	The number of files scanned
 * @param	nTotal	The total number of files in the project
 */
void QueryPage::slotProgress(int nFiles, int nTotal)
{
	m_progress.setProgress(nFiles, nTotal);
}

/**
 * Sets the list's colours and font, according the user's preferences.
 */
void QueryPage::applyPrefs()
{
	// Apply colour settings
	m_pList->setPaletteBackgroundColor(Config().getColor(
		KScopeConfig::QueryList, KScopeConfig::Background));
	m_pList->setPaletteForegroundColor(Config().getColor(
		KScopeConfig::QueryList, KScopeConfig::Foreground));
	m_pList->setFont(Config().getFont(KScopeConfig::QueryList));
}

/**
 * Restores a locked query from the given query file.
 * NOTE: The query file is deleted when loading is complete.
 * @param	sProjPath	The full path of the project directory
 * @param	sFileName	The name of the query file to load
 * @return	true if successful, false otherwise
 */
bool QueryPage::load(const QString& sProjPath, const QString& sFileName)
{
	QString sTemp, sFile, sFunc, sLine, sText;
	int nState;
	
	// Try to open the query file for reading
	QFile file(sProjPath + "/" + sFileName);
	if (!file.open(IO_ReadOnly))
		return false;
	
	{
		// Use a new scope for the QTextStream object, to ensure its 
		// destruction before the file is deleted
		QTextStream str(&file);
		
		// Make sure the file's version is correct
		sTemp = str.readLine();
		if (sTemp != FILE_VERSION) {
			file.remove();
			return false;
		}

		if (!readHeader(str))
			return false;
		
		// Read query records
		sTemp = str.readLine();
		nState = 0;
		while (sTemp != QString::null) {
			switch (nState) {
			// File path
			case 0:
				sFile = sTemp;
				break;
				
			// Function name
			case 1:
				sFunc = sTemp;
				break;
				
			// Line number
			case 2:
				sLine = sTemp;
				break;
				
			// Text string
			case 3:
				sText = sTemp;
				addRecord(sFile, sFunc, sLine, sText);
				break;
			}
			
			nState = (nState + 1) % 4;
			sTemp = str.readLine();
		}
	}
	
	// Delete the query file
	file.remove();
	
	return true;
}

/**
 * Writes the contents of the page to a query file.
 * This method is called for locked queries before the owner project is
 * closed.
 * @param	sProjPath	The full path of the project directory
 * @return	The name of the query file generated for this page (or an empty
 *			string if no file was written)
 */
QString QueryPage::store(const QString& sProjPath)
{
	QString sFileName;
	QListViewItemIterator itr(m_pList);

	// Get the file name to use
	sFileName = getFileName(sProjPath);
	if (sFileName.isEmpty())
		return sFileName;
		
	// Open the query file for writing
	QFile file(sProjPath + "/" + sFileName);
	if (!file.open(IO_WriteOnly))
		return QString();
	
	QTextStream str(&file);
	
	// Write the version string
	str << FILE_VERSION << "\n";
	
	writeHeader(str);
		
	// Write all records
	for(; itr.current(); ++itr) {
		str << itr.current()->text(0) << "\n" 
			<< itr.current()->text(1) << "\n"
			<< itr.current()->text(2) << "\n"
			<< itr.current()->text(3) << "\n";
	}
	
	return sFileName;
}

/**
 * Resets the query page by deleting all records.
 */
void QueryPage::clear()
{
	m_pList->clear();
	m_nType = CscopeFrontend::None;
	m_sText = QString();
	m_sName = QString();
	m_pLastItem = NULL;
}

/**
 * Associates the page with a query's type and text.
 * @param	nType	The type of the query
 * @param	sText	The text of the query
 */
void QueryPage::setQueryData(uint nType, const QString& sText)
{
	m_nType = nType;
	m_sText = sText;
	m_sName = getCaption();
}

/** 
 * Constructs a caption for this page, based on the query's type and text.
 * @param	bBrief	true to use a shortened version of the caption, false
 *					(default) for the full version
 * @return	The caption for this page
 */
QString QueryPage::getCaption(bool bBrief) const
{
	return QString(QUERY_TYPES[m_nType][bBrief ? 1 : 0] + m_sText);
}

/**
 * Creates a new query result item.
 * @param	sFile	The file name
 * @param	sFunc	The function defining the scope of the result
 * @param	sLine	The line number
 * @param	sText	The contents of the line
 */
void QueryPage::addRecord(const QString& sFile, const QString& sFunc,
	const QString& sLine, const QString& sText)
{
	new QListViewItem(m_pList, sFile, sFunc, sLine, sText);
}

/**
 * Creates a unique file name for saving the contents of the query page.
 * @param	sProjPath	The full path of the project directory
 * @return	The unique file name to use
 */
QString QueryPage::getFileName(const QString& sProjPath) const
{
	QString sFileName, sFileNameBase;
	int i = 0;
	
	// Do nothing if not initialised
	if (m_sName.isEmpty())
		return "";
	
	// Create a unique file name
	sFileNameBase = m_sName;
	sFileNameBase.replace(' ', '_');
	do {
		sFileName = sFileNameBase + QString::number(++i);
	} while (QFile(sProjPath + "/" + sFileName).exists());
	
	return sFileName;
}

/**
 * Reads query parameters from a file.
 * This mehtod is used as part of the loading process.
 * @param	str	A text stream set to the correct place in the file
 * @return	true if successful, false otherwise
 */
bool QueryPage::readHeader(QTextStream& str)
{
	QString sTemp;
	
	// Read the query name
	m_sName = str.readLine();
	if (m_sName == QString::null || m_sName.isEmpty())
		return false;
		
	// Read the query's type
	sTemp = str.readLine();
	if (sTemp == QString::null || sTemp.isEmpty())
		return false;
	
	// Convert the type string to an integer
	m_nType = sTemp.toUInt();
	if (m_nType >= CscopeFrontend::None) {
		m_nType = CscopeFrontend::None;
		return false;
	}		
				
	// Read the query's text
	m_sText = str.readLine();
	if (m_sText == QString::null || m_sText.isEmpty())
		return false;

	return true;
}

/**
 * Writes query parameters to a file.
 * This mehtod is used as part of the storing process.
 * @param	str	A text stream set to the correct place in the file
 */
void QueryPage::writeHeader(QTextStream& str)
{
	str << m_sName << "\n" << m_nType << "\n" << m_sText << "\n";
}

/**
 * Copies the text of the requested field (item+column) to the clipboard.
 * This slot is connected to the copy() signal of the QueryResultsMenu object.
 * @param	pItem	The item from which to copy
 * @param	nCol	The index of the item field to copy
 */
void QueryPage::slotCopy(QListViewItem* pItem, int nCol)
{
	QApplication::clipboard()->setText(pItem->text(nCol),
		QClipboard::Clipboard);
}

/**
 * Hides all items in the page that do not meet the given search criteria.
 * This slot is connected to the search() signal of the QueryResultsMenu
 * object.
 * The search is incremental: only visible items are checked, so that a new
 * search goes over the results of the previous one.
 * @param	re		The pattern to search
 * @param	nCol	The list column to search in
 */
void QueryPage::slotSearch(QListViewItem*, const QRegExp& re, int nCol)
{
	QListViewItem* pItem;
	
	// Iterate over all items in the list
	pItem = m_pList->firstChild();
	while (pItem != NULL) {
		// Filter visible items only
		if (pItem->isVisible() && re.search(pItem->text(nCol)) == -1)
			pItem->setVisible(false);
		
		pItem = pItem->nextSibling();
	}
}

/**
 * Makes all list items visible.
 * This slot is connected to the showAll() signal of the QueryResultsMenu
 * object.
 */
void QueryPage::slotShowAll()
{
	QListViewItem* pItem;
	
	// Iterate over all items in the list
	pItem = m_pList->firstChild();
	while (pItem != NULL) {
		pItem->setVisible(true);
		pItem = pItem->nextSibling();
	}
}

/**
 * Deletes the item on which a popup-menu has been invoked.
 * This slot is connected to the remove() signal of the QueryResultsMenu
 * object.
 * @param	pItem	The item to remove
 */
void QueryPage::slotRemoveItem(QListViewItem* pItem)
{
	delete pItem;
}
