/***************************************************************************
                          querypage.h  -  description
                             -------------------
    begin                : Sat May 24 2003
    copyright            : (C) 2003 by Elad Lahav
    email                : elad_lahav@users.sf.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef QUERYPAGE_H
#define QUERYPAGE_H

#include <qwidget.h>
#include <qhbox.h>
#include <qlistview.h>
#include <qregexp.h>
#include "cscopefrontend.h"

class QueryResultsMenu;

/**
 * Defines a single page in the query widget.
 * Each page displays a list of query result records. When a result record is
 * selected, a signal is emitted notifying the main widget it should display
 * the source file, and select the line, in this record.
 * @author Elad Lahav
 */

class QueryPage : public QHBox
{
   Q_OBJECT

public: 
	QueryPage(QWidget* pParent = 0, const char* szName = 0);
	~QueryPage();
	
	void attach(CscopeFrontend*);
	void applyPrefs();
	bool load(const QString&, const QString&);
	QString store(const QString&);
	void clear();
	
	void setQueryData(uint, const QString&);
	virtual QString getCaption(bool bBrief = false) const;
	
	/**
	 * @return	The list view widget of this page
	 */
	QListView* getList() { return m_pList; }
	
	/**
	 * Locks or unlocks this page.
	 * @param	bLocked	true to lock the page, false to unlock it.
	 */
	void setLocked(bool bLocked) { m_bLocked = bLocked; }
	
	/**
	 * Determines whether this page is locked.
	 * @return	true if the page is locked, false otherwise
	 */
	bool isLocked() { return m_bLocked; }
	
	/**
	 * @return	The type of query whose results are displayed in this page
	 */
	uint getQueryType() const { return m_nType; }
	
	/**
	 * @return	The text given as a parameter to the query whose results are
	 * displayed in this page.
	 */
	const QString& getQueryText() const { return m_sText; }

	/**
	 * @return	true if this page is connected to a still-running Cscope
	 * 			process, false otherwise.
	 */
	bool isAttached() const { return m_bAttached; }
	
public slots:
	void slotDataReady(FrontendToken*);
	void slotFinished(uint);
	void slotProgress(int, int);
	
signals:
	/**
	 * Emitted when the user selects a query result record from the list by
	 * either double-clicking it, or highlighting the record and pressing the
	 * ENTER key.
	 * This signal is also emitted if a query results in a single record.
	 * @param	pItem	The selected record list item
	 */
	void recordSelected(QListViewItem* pItem);
	
	/**
	 * Indicates that a query process has terminated, with all results listed
	 * on this page.
	 */
	void queryFinished();
	
protected:
	/** The list of query result records. */
	QListView* m_pList;
	
	/** Indicates whether this page is locked. A locked page means that new
		queries will be opened in a new page even if this page is the
		current one. Moreover results listed on this page are saved between
		sessions. */
	bool m_bLocked;
	
	/** The type of query whose results are listed on this page. */
	uint m_nType;
	
	/** The text given as a parameter to the query. */
	QString m_sText;
	
	/** A formatted caption for this query, including the type of query and
		its text. */
	QString m_sName;
	
	/** Displays query progress information. */
	CscopeProgress m_progress;
	
	/** Set to true when a running Cscope process uses this page as its
		output target. */
	bool m_bAttached;
	
	/** The last record that was added to the results list.
		Used to add records at the end of the list. */
	QListViewItem* m_pLastItem;
	
	/** A popup-menu for manipulating query result items. */
	QueryResultsMenu* m_pQueryMenu;
	
	virtual void addRecord(const QString&, const QString&, const QString&,
		const QString&);
	virtual QString getFileName(const QString&) const;
	virtual bool readHeader(QTextStream&);
	virtual void writeHeader(QTextStream&);
	
protected slots:
	virtual void slotCopy(QListViewItem*, int);
	virtual void slotSearch(QListViewItem*, const QRegExp&, int);
	virtual void slotShowAll();
	virtual void slotRemoveItem(QListViewItem*);
};

#endif
