/***************************************************************************
 *
 * Copyright (C) 2005 Elad Lahav (elad_lahav@users.sourceforge.net)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ***************************************************************************/

#ifndef PROJECTMANAGER_H
#define PROJECTMANAGER_H

#include <qobject.h>
#include <qstring.h>
#include <qdir.h>
#include <qmap.h>
#include <qptrlist.h>
#include <kconfig.h>

/**
 * Abstract base class for classes that need the list of project files.
 * Objects of classes derived from this one are used as a parameter to
 * ProjectManager::fillList(), which reads all file entries in the project,
 * and calls addItem() for each.
 * Any class that wishes to retrieve the project's file list, should inherit
 * from this class, and implement addItem().
 * @author Elad Lahav
 */

class FileListTarget
{
public:
	/**
	 * Class constructor.
	 */
	FileListTarget() {}

	/**
	 * Class destructor.
	 */
	virtual ~FileListTarget() {}

	/**
	 * Appends a file to the list.
	 * @param	sFilePath	The full path of the file to add
	 */
	virtual void addItem(const QString& sFilePath) = 0;
};

/**
 * Abstract base class for classes that need the list of project files.
 * Objects of classes derived from this one are used as a parameter to
 * ProjectManager::writeList(), which calls getFirstItem() and getNextItem(),
 * and writes the returned values to the project's 'cscope.files' file.
 * Any class that wishes to retrieve the project's file list, should inherit
 * from this class, and implement firstItem() and nextItem().
 * @author Elad Lahav
 */

class FileListSource
{
public:
	/**
	 * Class constructor.
	 */
	FileListSource() {}

	/**
	 * Class destructor.
	 */
	virtual ~FileListSource() {}

	/**
	 * Returns the first file in the list, and initiates a new iteration.
	 * @param	sFilePath	Holds the path of the first file, upon return
	 * @return	true if there are more files, false otherwise
	 */
	virtual bool firstItem(QString& sFilePath) = 0;

	/**
	 * Returns the next file in the list.
	 * @param	sFilePath	Holds the path of the file, upon return
	 * @return	true if there are more files, false otherwise
	 */
	virtual bool nextItem(QString& sFilePath) = 0;
};

/**
 * Defines a cursor location inside a file.
 * This structure is used to store project session information.
 * @author Elad Lahav
 */
struct FileLocation
{
	/**
	 * Struct constructor.
	 * @param	sPath	The full path of the file
	 * @param	nLine	The line position of the cursor
	 * @param	nCol	The column position of the cursor
	 */
	FileLocation(QString sPath, uint nLine, uint nCol) : m_sPath(sPath),
		m_nLine(nLine), m_nCol(nCol) {}
		
	/** The full path of the file. */
	QString m_sPath;
	
	/** The line position of the cursor. */
	uint m_nLine;
	
	/** The column position of the cursor. */
	uint m_nCol;
};

/**
 * A list of file locations used for restoring a session.
 */
typedef QPtrList<FileLocation> FileLocationList;

class FileSemaphore;

/**
 * Maintains a KScope project.
 * A project is a set of source files, on which to run Cscope, as well as
 * other programmes. The project is maintained using a set of configuration
 * files, located in a common directory. These files include:
 * - cscope.proj: A KDE configuration file that stores project parameters
 * - cscope.files: Holds the list of source files included in this project
 * To these cscope adds its symbol reference files.
 * The ProjectManager class is responsible for creating new projects, adding
 * and removing source files, and allowing other objects access to the
 * currently active project (such as filling a list object with the source 
 * files).
 * @author Elad Lahav
 */

class ProjectManager : public QObject
{
	Q_OBJECT
	
public: 
	ProjectManager();
	virtual ~ProjectManager();

	/**
	 * Configurable project options.
	 */
	struct Options {
		/** A list of MIME-types that determines which files are included in
			the project. */
		QStringList slFileTypes;
		
		/** true if the -k option for CScope should be used. */
		bool bKernel;
		
		/** true if Cscope should build an inverted index. */
		bool bInvIndex;
		
		/** true if the -c option for CScope should be used. */
		bool bNoCompress;
		
		/** true if the -D option for CScope should be used. */
		bool bSlowPathDef;
		
		/** The time, in milliseconds, after which the database should be
			automatically rebuilt (-1 if this option is disabled). */
		int nAutoRebuildTime;
		
		/** true to use auto-completion. */
		bool bACEnabled;
		
		/** Minimum number of characters in a symbol for auto-completion. */
		uint nACMinChars;
		
		/** Time interval, in milliseconds, before auto-completion is
			started. */
		uint nACDelay;
		
		/** Maximal number of entries for auto-completion. */
		uint nACMaxEntries;
		
		uint nTabWidth;
	};
	
	bool create(const QString&, const QString&, const Options&);
	bool open(const QString&);
	bool openCscopeOut(const QString&);
	void close();
	int addFiles(const QStringList&);
	void fillList(FileListTarget*);
	void writeList(FileListSource*);
	
	bool isEmpty();
	QString getFileTypes() const;
	void getOptions(Options&) const;
	void setOptions(const Options&);
	void getOpenFiles(FileLocationList&) const;
	void setOpenFiles(FileLocationList&);
	void getBookmarks(FileLocationList&) const;
	void setBookmarks(FileLocationList&);
	QString getLastOpenFile() const;
	void setLastOpenFile(const QString&);
	void getQueryFiles(QStringList&) const;
	void setQueryFiles(QStringList&);
	void getCallTreeFiles(QStringList&) const;
	void setCallTreeFiles(QStringList&);
	QString getRoot() const;
	void setRoot(const QString&);
	void getSymHistory(QStringList&) const;
	void setSymHistory(QStringList&);
	void getMakeParams(QString&, QString&) const;
	void setMakeParams(const QString&, const QString&);
	bool dbExists();
	
	/**
	 * Determines whether a project is open, i.e., a project directory is
	 * associated with this object.
	 * @return	true if the project is open, false otherwise
	 */
	bool isOpen() { return m_bOpen; }

	/**
	 * Determines whether a project is based on a Cscope.out file, and is
	 * therefore considered as a temporary project.
	 * @return	true if this is a temporary project, false otherwise
	 */
	bool isTemporary() { return m_bTempProj; }

	/**
	 * @return	The name of the current project
	 */
	QString getName() const { return m_sName; }
	
	/**
	 * @return	The full path of the project's directory
	 */
	QString getPath() const { return m_dir.absPath(); }
	
	/**
	 * @return	Command-line arguments to pass to a Cscope object, based on
	 * 			project's options
	 */
	uint getArgs() const { return m_nArgs; }
	
	/**
	 * @return	The time, in seconds, to wait before rebuilding the
     *			cross-refernce database.
	 */
	int getAutoRebuildTime() const { return m_nAutoRebuildTime; }
	
	/**
	 * @return	The tab width to use (0 to use the editor's default)
	 */
	uint getTabWidth() const { return m_nTabWidth; }
	
	static void getDefOptions(Options&);

	static bool isGloballyOpen(const QString&);

signals:
	/**
	 * Emitted when a set of files is added to the current project.
	 * @param	slFiles	A list containing the full paths of the new files
	 */
	void filesAdded(const QStringList& slFiles);
	
	/**
	 * Emitted when the list of files in a project is modified.
	 */
	void fileListChanged();
	
private:
	/** The name of the project, as written in the configuration file */
	QString m_sName;

	/** The directory associated with the project */
	QDir m_dir;

	/** The file that holds the paths of all source files in this project
		("cscope.files") */
	QFile* m_pFiles;

	/** The configuration file ("cscope.proj") */
	KConfig* m_pConf;

	/** true if there is an active project, false otherwise */
	bool m_bOpen;
	
	/** true if the current project is a temporary wrapper for a Cscope.out
		file */
	bool m_bTempProj;
	
	/** A list of Cscope command-line arguments based on the project's
		options. */
	uint m_nArgs;

	/** The time, in seconds, to wait before rebuilding the cross-refernce
	 	database. */
	int m_nAutoRebuildTime;
	
	/** If non-zero, overrides the editor's configured tab width. */
	uint m_nTabWidth;
	
	/** A list of symbols previously queried. */
	QStringList m_slSymHistory;
	
	/** A named semaphore to synchronise multiple instances of the same 
		project. */
	FileSemaphore* m_pSem;
	
	static void initConfig(const QString&, QFile**, KConfig**);
	static bool isCscopeOut(const QString&);
	static void writeOptions(KConfig*, const Options&);
};

#endif
