/***************************************************************************
                          cachemanager.cpp  -  description
                             -------------------
    begin                : Fri Jan 25 2002
    copyright            : (C) 2002 by Roberto Virga
    email                : rvirga@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <stdlib.h>

#include <qdatetime.h>
#include <qdir.h>
#include <qfileinfo.h>
#include <qregexp.h>

#include "cachelogdialog.h"
#include "ksetispydoc.h"

#include "cachemanager.h"

const char *Prefix[]    = {"download/", "upload/"};
const char *TestFile[]  = {"work_unit.sah", "result.sah"};

const QStringList ExcludedFiles = QStringList("pid.sah") << "temp.sah" << "wtemp.sah";

CacheManager::CacheManager(QObject *parent, const char *name)
             : DCOPObject("Cache-Interface")
             , QObject(parent, name)
{
  baseURL = clientURL = userAddress = "";
  capacity = 0;

  proxyType = None;
  proxyHost = proxyLogin = proxyPassword = "";
  proxyPort = 0;

  for(uint i = 0; i < 7; i++)
    schedules[i].clear();
  downloadThreshold = 0;

  connections[Download].timer = new QTimer(this);
  connections[Download].process = NULL;
  connections[Download].count = 0;
  connect(connections[Download].timer, SIGNAL(timeout()), this, SLOT(scheduledDownload()));

  connections[Upload].timer = new QTimer(this);
  connections[Upload].process = NULL;
  connections[Upload].count = 0;
  connect(connections[Upload].timer, SIGNAL(timeout()), this, SLOT(scheduledUpload()));

  srand(QTime::currentTime().msec());
}

CacheManager::~CacheManager()
{
  if(connections[Download].process != NULL) {
    connections[Download].process->kill();
    cleanupDir(Download, connections[Download].dir);
  }
  if(connections[Upload].process != NULL)
    connections[Upload].process->kill();
}

KURL CacheManager::getBaseURL() const
{
  return(baseURL);
}

void CacheManager::setBaseURL(const KURL& url)
{
  baseURL = url;
}

KURL CacheManager::getClientURL() const
{
  return(clientURL);
}

void CacheManager::setClientURL(const KURL& url)
{
  clientURL = url;
}

uint CacheManager::getCapacity() const
{
  return(capacity);
}

void CacheManager::setCapacity(uint wus)
{
  capacity = wus;
}

CacheManager::ProxyType CacheManager::getProxyType() const
{
  return(proxyType);
}

void CacheManager::setProxyType(ProxyType type)
{
  proxyType = type;
}

QString CacheManager::getProxyHost() const
{
  return(proxyHost);
}

void CacheManager::setProxyHost(const QString& host)
{
  proxyHost = host;
}

uint CacheManager::getProxyPort() const
{
  return(proxyPort);
}

void CacheManager::setProxyPort(uint port)
{
  proxyPort = port;
}

QString CacheManager::getProxyLogin() const
{
  return(proxyLogin);
}

void CacheManager::setProxyLogin(const QString& login)
{
  proxyLogin = login;
}

QString CacheManager::getProxyPassword() const
{
  return(proxyPassword);
}

void CacheManager::setProxyPassword(const QString& password)
{
  proxyPassword = password;
}

QString CacheManager::getEMail() const
{
  return(userAddress);
}

void CacheManager::setEMail(const QString& address)
{
  userAddress = address;
}

const QValueList<intervalStruct>& CacheManager::getSchedule(uint i) const
{
  return(schedules[i % 7]);
}

void CacheManager::setSchedule(uint i, const QValueList<intervalStruct>& schedule)
{
  schedules[i % 7] = schedule;
}

uint CacheManager::getThreshold() const
{
  return(downloadThreshold);
}

void CacheManager::setThreshold(uint threshold)
{
  if(threshold > 100)
    downloadThreshold = 100;
  else
    downloadThreshold = (threshold / 5) * 5;
}

uint CacheManager::getWUCount(ConnectionType type) const
{
  return(wus[type].count());
}

QDateTime CacheManager::getLastConnect(ConnectionType type) const
{
  return(connections[type].last);
}

QDateTime CacheManager::getNextConnect(ConnectionType type) const
{
  return(connections[type].next);
}

bool CacheManager::isValid() const
{
  if(baseURL.isEmpty() || !baseURL.isValid() || !baseURL.isLocalFile())
    return(false);

  QFileInfo baseDirInfo(baseURL.path(+1));
  if(!baseDirInfo.exists() || !baseDirInfo.isDir())
    return(false);

  return(checkClient());
}

bool CacheManager::checkClient() const
{
  if(clientURL.isEmpty() || !clientURL.isValid() || !clientURL.isLocalFile())
    return(false);

  QFileInfo clientInfo(clientURL.path(-1));
  if(!clientInfo.exists() || !clientInfo.isFile())
    return(false);

  return(clientInfo.isExecutable());
}


QCStringList CacheManager::functions()
{
  QCStringList functions = DCOPObject::functions();

  functions << "int available()";
  functions << "int done()";

  functions << "QDateTime lastDownload()";
  functions << "QDateTime lastUpload()";

  functions << "QDateTime nextDownload()";
  functions << "QDateTime nextUpload()";

  functions << "ASYNC connect()";

  return functions;
}

bool CacheManager::process(const QCString& fun, const QByteArray& data,
                           QCString& replyType, QByteArray& replyData)
{
  if(fun == "available()") {
    replyType = "int";
    QDataStream out(replyData, IO_WriteOnly);
    out << getWUCount(Download);
  } else if(fun == "done()") {
    replyType = "int";
    QDataStream out(replyData, IO_WriteOnly);
    out << getWUCount(Upload);
  } else if(fun == "lastDownload()") {
    replyType = "QDateTime";
    QDataStream out(replyData, IO_WriteOnly);
    out << getLastConnect(Download);
  } else if(fun == "lastUpload()") {
    replyType = "QDateTime";
    QDataStream out(replyData, IO_WriteOnly);
    out << getLastConnect(Upload);
  } else if(fun == "nextDownload()") {
    replyType = "QDateTime";
    QDataStream out(replyData, IO_WriteOnly);
    out << getNextConnect(Download);
  } else if(fun == "nextUpload()") {
    replyType = "QDateTime";
    QDataStream out(replyData, IO_WriteOnly);
    out << getNextConnect(Upload);
  } else if(fun == "connect()") {
    replyType = "void";
    download();
    upload();
  } else
    return DCOPObject::process(fun, data, replyType, replyData);

  return true;
}

void CacheManager::readConfig(KConfig *config)
{
  config->setGroup("CacheManager");

  baseURL = config->readEntry("URL");
  clientURL = config->readEntry("Client URL");
  userAddress = config->readEntry("E-mail");
  capacity = config->readNumEntry("Capacity");

  proxyType = ProxyType(config->readNumEntry("Proxy type", None));
  proxyHost = config->readEntry("Proxy host");
  proxyPort = config->readNumEntry("Proxy port");
  proxyLogin = config->readEntry("Proxy login");
  proxyPassword = config->readEntry("Proxy password");

  for(uint day = 0; day < 7; day++) {
    const QString key = QString("Schedule %1").arg(day);
    const QStringList data = QStringList::split(" ", config->readEntry(key));
    intervalStruct item;

    schedules[day].clear();
    for(uint i = 0; i < data.count(); i++)
    {
      if(i % 3 == 0)
        item.type = intervalStruct::Type(data[i].toInt());
      else if(i % 3 == 1)
        item.start = QTime().addSecs(data[i].toInt());
      else {
        item.stop = QTime().addSecs(data[i].toInt());
        schedules[day] += item;
      }
    }
  }
  downloadThreshold = config->readNumEntry("Download threshold");

  if(this->isValid())
  {
    checkWUs(Download);
    checkWUs(Upload);
  }

  scheduleNextConnection(Download);
  scheduleNextConnection(Upload);
}

void CacheManager::saveConfig(KConfig *config)
{
  config->setGroup("CacheManager");

  config->writeEntry("URL", baseURL.prettyURL(+1, KURL::StripFileProtocol));
  config->writeEntry("Client URL", clientURL.prettyURL(-1, KURL::StripFileProtocol));
  config->writeEntry("E-mail", userAddress);
  config->writeEntry("Capacity", capacity);

  config->writeEntry("Proxy type", proxyType);
  config->writeEntry("Proxy host", proxyHost);
  config->writeEntry("Proxy port", proxyPort);
  config->writeEntry("Proxy login", proxyLogin);
  config->writeEntry("Proxy password", proxyPassword);

  for(uint day = 0; day < 7; day++) {
    const QString key = QString("Schedule %1").arg(day);
    QString data;

    for(uint i = 0; i < schedules[day].count(); i++)
    {
      data += QString(" %1").arg(schedules[day][i].type);
      data += QString(" %1").arg(QTime().secsTo(schedules[day][i].start));
      data += QString(" %1").arg(QTime().secsTo(schedules[day][i].stop));
    }
    data.stripWhiteSpace();
    config->writeEntry(key, data);
  }
  config->writeEntry("Download threshold", downloadThreshold);
}

bool CacheManager::setup()
{
  if(baseURL.isEmpty() || !baseURL.isValid() || !baseURL.isLocalFile())
    return(false);

  QDir dir(baseURL.path(+1));

  if(!dir.exists() && !dir.mkdir(dir.absPath()))
    return(false);

  if(!dir.exists(Prefix[Download]) && !dir.mkdir(Prefix[Download], false))
    return(false);

  for(uint i = 0; i < capacity; i++)
    if(!createDir(Download, i))
      return(false);

  if(!dir.exists(Prefix[Upload]) && !dir.mkdir(Prefix[Upload], false))
    return(false);

  checkWUs(Download);
  checkWUs(Upload);

  scheduleNextConnection(Download);
  scheduleNextConnection(Upload);

  emit updated();

  return(true);
}

bool CacheManager::fetchWU(const QString& url)
{
  if(wus[Download].count() == 0) return(false);

  if(QFile::exists(url + TestFile[Download])) return(false);

  uint from = oldestUsedDir(Download);

  if(!copyDir(Download, from, url))
    return(false);

  // the cache entry has been successfully copied, so the cache entry can be safely cleaned
  cleanupDir(Download, from);
  const int pos = findDir(Download, from);
  if(pos >= 0) wus[Download].remove(wus[Download].at(pos));

  emit updated();

  return(true);
}

bool CacheManager::storeWU(const QString& url)
{
  uint to = smallestUnusedDir(Upload);
  if(checkDir(Upload, to) == DoesNotExists)
    if(!createDir(Upload, to))
      return(false);

  if(!copyDir(url, Upload, to) || !createFile(Upload, to, "stop_after_send.txt"))
  {
    cleanupDir(Upload, to);
    return(false);
  }

  // the result file has been backed up so it can be safely removed
  QFile::remove(url + TestFile[Upload]);

  // remove all useless files too
  for(uint i = 0; i < ExcludedFiles.size(); i++)
    QFile::remove(url + ExcludedFiles[i]);

  cachedWUStruct item;
  item.dir = to;
  item.timestamp = QDateTime::currentDateTime();
  wus[Upload].prepend(item);

  emit updated();

  return(true);
}

void CacheManager::checkWUs(ConnectionType type)
{
  QDir dir(baseURL.path(+1) + Prefix[type]);
  QStringList entries = dir.entryList(QDir::Dirs);
  QRegExp filter("^[1-9][0-9]*$");

  wus[type].clear();
  for(uint i = 0; i < entries.count(); i++) {
    if(filter.search(entries[i], 0) != 0) continue;

    cachedWUStruct info;

    info.dir = uint(entries[i].toInt())-1;
    if(checkDir(type, info.dir, &info.timestamp) == ExistsFull)
      wus[type] += info;
  }
}

CacheManager::Status CacheManager::checkDir(ConnectionType type, uint n, QDateTime *timestamp)
{
  QString dirURL = QString("%1%2%3/").arg(baseURL.path(+1)).arg(Prefix[type]).arg(n+1);
  QFileInfo dirInfo(dirURL);

  if(!dirInfo.exists() || !dirInfo.isDir())
    return(DoesNotExists);

  QString fileName = dirURL + TestFile[type];
  QFileInfo fileInfo(fileName);

  if(!fileInfo.exists())
    return(ExistsEmpty);

  if(timestamp != NULL)
    *timestamp = fileInfo.lastModified();

  return(ExistsFull);
}

void CacheManager::download()
{
  const uint wu_count = wus[Download].count();
  if(wu_count >= capacity) return;

  // ignore the threshold on user-initiated downloads

  startConnection(Download, smallestUnusedDir(Download));
}

void CacheManager::scheduledDownload()
{
  scheduleNextConnection(Download);

  const uint wu_count = wus[Download].count();
  if(wu_count >= capacity) return;

  const double empty = 100.0 - 1e2 * wu_count / capacity;
  if(empty < downloadThreshold) return;

  startConnection(Download, smallestUnusedDir(Download), true);
}

void CacheManager::upload()
{
  const uint wu_count = wus[Upload].count();
  if(wu_count == 0) return;

  startConnection(Upload, oldestUsedDir(Upload));
}

void CacheManager::scheduledUpload()
{
  scheduleNextConnection(Upload);

  const uint wu_count = wus[Upload].count();
  if(wu_count == 0) return;

  startConnection(Upload, oldestUsedDir(Upload), true);
}

void CacheManager::kill(ConnectionType type)
{
  if(connections[type].process != NULL)
    connections[type].process->kill();
}

void CacheManager::scheduleNextConnection(ConnectionType type)
{
  const QDateTime now = QDateTime::currentDateTime();
  const int interval_type = (type == Download) ? intervalStruct::Download : intervalStruct::Upload;
  const int day = now.date().dayOfWeek();
  QDateTime result;

  for(int i = 0; i < 8; i++) {
    const QValueList<intervalStruct> list = schedules[(day + i) % 7];

    for(QValueList<intervalStruct>::ConstIterator it = list.begin(); it != list.end(); ++it) {
      if(i == 0 && now.time() >= (*it).start)
        continue;
      if(!((*it).type & interval_type))
        continue;
      const int range = (*it).start.secsTo((*it).stop);
      result.setDate(now.date().addDays(i));
      result.setTime((*it).start.addSecs(int(range * (rand() / (RAND_MAX+1.0)))));
      break;
    }

    if(result.isValid())
      break;
  }

  connections[type].next = result;

  if(result.isValid())
    connections[type].timer->changeInterval(1000 * now.secsTo(result));

  emit updated();
}

void CacheManager::startConnection(ConnectionType type, uint n, bool force)
{
  const uint wu_count = wus[type].count();

  if(type == Download && wu_count >= capacity)
  {
    const int count = connections[Download].count;

    connections[Download].count = 0;
    if(count > 0)
      emit notify(WUsDownloaded, count);
    return;
  }
  if(type == Upload && wu_count == 0)
  {
    const int count = connections[Upload].count;

    connections[Upload].count = 0;
    if(count > 0)
      emit notify(ResultsUploaded, count);
    return;
  }

  if(connections[type].process != NULL)
    if(force) kill(type);
    else return;

  QDir::setCurrent(QString("%1%2%3").arg(baseURL.path(+1)).arg(Prefix[type]).arg(n+1));

  connections[type].process = new KProcess();
  connections[type].dir = n;

  *connections[type].process << clientURL.path(-1);
  switch(proxyType) {
    case Proxy:
      *connections[type].process << "-proxy";
      *connections[type].process << QString("%1:%2").arg(proxyHost).arg(proxyPort);
      break;
    case SOCKS:
      *connections[type].process << "-socks_server";
      *connections[type].process << QString("%1:%2").arg(proxyHost).arg(proxyPort);
      if(!proxyLogin.isEmpty()) {
        *connections[type].process << "-socks_user";
        *connections[type].process << QString("%1").arg(proxyLogin);
      }
      if(!proxyPassword.isEmpty()) {
        *connections[type].process << "-socks_passwd";
        *connections[type].process << QString("%1").arg(proxyPassword);
      }
    default:
      break;
  }
  *connections[type].process << "-stop_after_xfer";

  connect(connections[type].process, SIGNAL(wroteStdin(KProcess *)),
          this, SLOT(handleStdin(KProcess *)));
  connect(connections[type].process, SIGNAL(receivedStdout(KProcess *, char *, int)),
          this, SLOT(handleStdout(KProcess *, char *, int)));

  if(type == Download)
    connect(connections[type].process, SIGNAL(processExited(KProcess *)),
            this, SLOT(handleDownloadStopped(KProcess *)));
  else
    connect(connections[type].process, SIGNAL(processExited(KProcess *)),
            this, SLOT(handleUploadStopped(KProcess *)));

  connections[type].out = "";
  connections[type].count = 0;

  connections[type].process->start(KProcess::NotifyOnExit, KProcess::All);

  emit startConnect(type, n+1);


}

void CacheManager::handleStdout(KProcess *process, char *out, int outlen)
{
  ConnectionType type = (process == connections[Download].process) ? Download : Upload;

  QString str;
  str.setLatin1(out, outlen);
  connections[type].out = connections[type].out + str;

  emit output(str, type);

  if(connections[type].out.contains("Can't connect")) {
    connections[type].out = "";
    process->kill(SIGTERM);
  } else if(connections[type].out.contains("(1 or 2):")) {
    connections[type].out = "";
    writeLine(process, "2");
  } else if(connections[type].out.contains("Email address:")) {
    connections[type].out = "";
    writeLine(process, userAddress);
  }
}

void CacheManager::handleDownloadStopped(KProcess *)
{
  emit endConnect(Download);

  delete connections[Download].process;
  connections[Download].process = NULL;

  if(checkDir(Download, connections[Download].dir) == ExistsFull) {

    cachedWUStruct item;

    item.dir = connections[Download].dir;
    item.timestamp = connections[Download].last = QDateTime::currentDateTime();

    wus[Download] += item;

    connections[Download].count++;

    startConnection(Download, smallestUnusedDir(Download));

    emit updated();
  }
  else
  {
    cleanupDir(Download, connections[Download].dir);

    const int count = connections[Download].count;

    connections[Download].count = 0;
    if(count > 0)
      emit notify(WUsDownloaded, count);
  }
}

void CacheManager::handleUploadStopped(KProcess *)
{
  emit endConnect(Upload);

  delete connections[Upload].process;
  connections[Upload].process = NULL;

  if(checkDir(Upload, connections[Upload].dir) == ExistsEmpty) {

    const int pos = findDir(Upload, connections[Upload].dir);
    if(pos >= 0) wus[Upload].remove(wus[Upload].at(pos));
    connections[Upload].last = QDateTime::currentDateTime();

    cleanupDir(Upload, connections[Upload].dir);

    connections[Upload].count++;

    startConnection(Upload, oldestUsedDir(Upload));

    emit updated();
  }
  else
  {
    const int count = connections[Upload].count;

    connections[Upload].count = 0;
    if(count > 0)
      emit notify(ResultsUploaded, count);
  }
}

void CacheManager::handleStdin(KProcess *process)
{
  ConnectionType type = (process == connections[Download].process) ? Download : Upload;
  connections[type].in.removeFirst();
}

void CacheManager::writeLine(KProcess *process, const QString& str)
{
  ConnectionType type = (process == connections[Download].process) ? Download : Upload;
  const QString in = str + "\n";
  connections[type].in.append(in.latin1());

  emit input(in, type);

  if(type == Download)
    QTimer::singleShot(1000, this, SLOT(writeDownloadDelayed()));
  else
    QTimer::singleShot(1000, this, SLOT(writeUploadDelayed()));
}

void CacheManager::writeDownloadDelayed()
{
  if(connections[Download].process == NULL) return;
  const char *in = connections[Download].in.at(0);
  connections[Download].process->writeStdin(in, strlen(in));
}

void CacheManager::writeUploadDelayed()
{
  if(connections[Upload].process == NULL) return;
  const char *in = connections[Upload].in.at(0);
  connections[Upload].process->writeStdin(in, strlen(in));
}

int CacheManager::findDir(ConnectionType type, uint n)
{
  const QValueList<cachedWUStruct> list = wus[type];

  for(uint i = 0; i < list.count(); i++)
    if(list[i].dir == n) return(i);

  return(-1);
}

uint CacheManager::smallestUnusedDir(ConnectionType type)
{
  const QValueList<cachedWUStruct> list = wus[type];

  for(uint i = 0; ; i++)
    if(findDir(type, i) < 0)
      return(i);
}

uint CacheManager::oldestUsedDir(ConnectionType type)
{
  const QValueList<cachedWUStruct> list = wus[type];
  QDateTime timestamp;
  int result = -1;

  for(uint i = 0; i < list.count(); i++)
    if(result == -1 || list[i].timestamp <= timestamp) {
      timestamp = list[i].timestamp;
      result = list[i].dir;
    }

  return(result);
}

bool CacheManager::createFile(ConnectionType type, uint n, const QString& name)
{
  const QString path = QString("%1%2%3/").arg(baseURL.path(+1)).arg(Prefix[type]).arg(n+1);
  const QFileInfo pathInfo(path);
  if(!pathInfo.exists() || !pathInfo.isDir()) return(false);

  QFile file(path + name);

  if(!file.exists())
    if(file.open(IO_WriteOnly)) file.close();
    else return(false);

  return(true);
}

bool CacheManager::createDir(ConnectionType type, uint n)
{
  const QString name = Prefix[type] + QString::number(n+1);
  QDir dir(baseURL.path(+1));

  if(!dir.exists(name))
    if(!dir.mkdir(name, false))
      return(false);

  return(true);
}

bool CacheManager::removeFile(ConnectionType type, uint n, const QString& name)
{
  const QString path = QString("%1%2%3/").arg(baseURL.path(+1)).arg(Prefix[type]).arg(n+1);
  const QFileInfo pathInfo(path);
  if(!pathInfo.exists() || !pathInfo.isDir()) return(false);

  return(QFile::remove(path + name));
}

bool CacheManager::removeDir(ConnectionType type, uint n)
{
  const QString name = Prefix[type] + QString::number(n+1);
  QDir dir(baseURL.path(+1));

  cleanupDir(type, n);
  return(dir.remove(name));
}

void CacheManager::cleanupDir(ConnectionType type, uint n)
{
  const QString path = QString("%1%2%3/").arg(baseURL.path(+1)).arg(Prefix[type]).arg(n+1);
  QDir dir(path);
  QStringList entries = dir.entryList(QDir::Files);

  for(uint i = 0; i < entries.count(); i++)
    dir.remove(entries[i]);
}

bool CacheManager::copyFile(const QString& from, const QString& to)
{
  QFile fromFile(from), toFile(to);
  if(!fromFile.exists()) return(false);

  if(!fromFile.open(IO_ReadOnly))
    return(false);
  if(!toFile.open(IO_WriteOnly)) {
    fromFile.close();
    return(false);
  }

  char buffer[10240];
  int len;

  while((len = fromFile.readBlock(buffer, 10240)) > 0) {
    toFile.writeBlock(buffer, len);
    toFile.flush();
  }

  fromFile.close();
  toFile.close();

  return(true);
}

bool CacheManager::copyFile(ConnectionType type, uint n, const QString& name, const QString& to)
{
  return(copyFile(QString("%1%2%3/%4").arg(baseURL.path(+1))
                                      .arg(Prefix[type]).arg(n+1).arg(name),
                  to));
}

bool CacheManager::copyFile(const QString& from, ConnectionType type, uint n, const QString& name)
{
  return(copyFile(from,
                  QString("%1%2%3/%4").arg(baseURL.path(+1))
                                      .arg(Prefix[type]).arg(n+1).arg(name)));
}

bool CacheManager::copyDir(ConnectionType type, uint n, const QString& to)
{
  const QString path = QString("%1%2%3/").arg(baseURL.path(+1)).arg(Prefix[type]).arg(n+1);
  QDir dir(path);

  dir.setNameFilter("*.sah");
  QStringList entries = QDir(path).entryList(QDir::Files);

  for(uint i = 0; i < entries.count(); i++)
    if(!ExcludedFiles.contains(entries[i]))
    {
      if(!copyFile(path + entries[i], to + entries[i]))
        return(false);
    }
    else
      QFile::remove(path + entries[i]);

  return(true);
}

bool CacheManager::copyDir(const QString& from, ConnectionType type, uint n)
{
  const QString path = QString("%1%2%3/").arg(baseURL.path(+1)).arg(Prefix[type]).arg(n+1);
  QDir dir(from);

  dir.setNameFilter("*.sah");
  QStringList entries = dir.entryList(QDir::Files);

  for(uint i = 0; i < entries.count(); i++)
    if(!copyFile(from + entries[i], path + entries[i])) return(false);

  return(true);
}

#include "cachemanager.moc"

