/***************************************************************************
                          cachemanager.h  -  description
                             -------------------
    begin                : Fri Jan 25 2002
    copyright            : (C) 2002 by Roberto Virga
    email                : rvirga@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef CACHEMANAGER_H
#define CACHEMANAGER_H

#include <qdatetime.h>
#include <qptrlist.h>
#include <qstring.h>
#include <qstrlist.h>
#include <qtimer.h>
#include <qvaluelist.h>

#include <dcopobject.h>
#include <kconfig.h>
#include <kprocess.h>
#include <kurl.h>

typedef struct {
  enum Type {Download=1, Upload=2, Both=3} type;
  QTime start, stop;
} intervalStruct;

typedef struct {
  uint dir;
  QDateTime timestamp;
} cachedWUStruct;

typedef struct {
  QTimer *timer;
  QDateTime last, next;
  KProcess *process;
  QDateTime started;
  uint dir;
  QStrList in;
  QString out;
  int count;
} connectionStruct;

class CacheManager : public QObject, virtual public DCOPObject {
  Q_OBJECT
  public:
    enum ProxyType {None, Proxy, SOCKS};
    enum ConnectionType {Download, Upload};
    enum Notify {WUsDownloaded, ResultsUploaded};
    CacheManager(QObject *parent=0, const char *name=0);
    ~CacheManager();
    virtual KURL getBaseURL() const;
    virtual KURL getClientURL() const;
    virtual QString getEMail() const;
    virtual ProxyType getProxyType() const;
    virtual QString getProxyHost() const;
    virtual uint getProxyPort() const;
    virtual QString getProxyLogin() const;
    virtual QString getProxyPassword() const;
    virtual uint getCapacity() const;
    virtual const QValueList<intervalStruct>& getSchedule(uint day) const;
    virtual uint getThreshold() const;
    virtual uint getWUCount(ConnectionType type) const;
    virtual QDateTime getLastConnect(ConnectionType type) const;
    virtual QDateTime getNextConnect(ConnectionType type) const;
    virtual bool isValid() const;
    virtual bool checkClient() const;
    virtual bool setup();
    virtual bool fetchWU(const QString& url);
    virtual bool storeWU(const QString& url);
    virtual QCStringList functions();
    virtual bool process(const QCString& fun, const QByteArray& data,
                         QCString& replyType, QByteArray& replyData);
  public slots:
    virtual void setBaseURL(const KURL& url);
    virtual void setClientURL(const KURL& url);
    virtual void setEMail(const QString& address);
    virtual void setProxyType(ProxyType type);
    virtual void setProxyHost(const QString& host);
    virtual void setProxyPort(uint port);
    virtual void setProxyLogin(const QString& login);
    virtual void setProxyPassword(const QString& password);
    virtual void setCapacity(uint wus);
    virtual void setSchedule(uint day, const QValueList<intervalStruct>& schedule);
    virtual void setThreshold(uint percent);
    virtual void readConfig(KConfig *config);
    virtual void saveConfig(KConfig *config);
    virtual void download();
    virtual void upload();
    virtual void kill(ConnectionType type);
  signals:
    void startConnect(CacheManager::ConnectionType type, uint n);
    void endConnect(CacheManager::ConnectionType type);
    void action(const QString& action, CacheManager::ConnectionType type);
    void input(const QString& input, CacheManager::ConnectionType type);
    void output(const QString& output, CacheManager::ConnectionType type);
    void updated();
    void notify(int notify, int count);
  protected:
    enum Status {DoesNotExists, ExistsEmpty, ExistsFull};
    virtual void checkWUs(ConnectionType type);
    virtual void scheduleNextConnection(ConnectionType type);
    virtual void startConnection(ConnectionType type, uint n, bool force=false);
  private:
    Status checkDir(ConnectionType type, uint dir, QDateTime *timestamp=NULL);
    int findDir(ConnectionType type, uint n);
    uint oldestUsedDir(ConnectionType type);
    uint smallestUnusedDir(ConnectionType type);
    bool createFile(ConnectionType type, uint n, const QString& name);
    bool createDir(ConnectionType type, uint n);
    bool removeFile(ConnectionType type, uint n, const QString& name);
    bool removeDir(ConnectionType type, uint n);
    void cleanupDir(ConnectionType type, uint n);
    bool copyFile(const QString& from, const QString& to);
    bool copyFile(ConnectionType type, uint n, const QString& name, const QString& to);
    bool copyFile(const QString& from, ConnectionType type, uint n, const QString& name);
    bool copyDir(ConnectionType type, uint n, const QString& to);
    bool copyDir(const QString& from, ConnectionType type, uint n);
    KURL baseURL, clientURL;
    QString userAddress;
    ProxyType proxyType;
    QString proxyHost, proxyLogin, proxyPassword;
    uint proxyPort, capacity, completed;
    QValueList<intervalStruct> schedules[7];
    uint downloadThreshold;
    QValueList<cachedWUStruct> wus[2];
    connectionStruct connections[2];
  private slots:
    void scheduledDownload();
    void scheduledUpload();
    void handleStdin(KProcess *process);
    void handleStdout(KProcess *process, char *out, int outlen);
    void handleDownloadStopped(KProcess *process);
    void handleUploadStopped(KProcess *process);
    void writeLine(KProcess *process, const QString& str);
    void writeDownloadDelayed();
    void writeUploadDelayed();
};

#endif
