/***************************************************************************
                          csvlogmonitor.cpp  -  description
                             -------------------
    begin                : Mon Jul 30 2001
    copyright            : (C) 2001 by Roberto Virga
    email                : rvirga@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qapplication.h>
#include <qdatetime.h>
#include <qfileinfo.h>
#include <qregexp.h>

#include <kio/netaccess.h>
#include <ktempfile.h>
#include <kdeversion.h>

#include "csvlogmonitor.h"

const QString SETILog_log_file    = "SETILog.csv";
const QString SETIResult_log_file  = "SETIResult.log";

CSVLogMonitor::CSVLogMonitor(const KURL& logURL, QObject *parent, const char *name)
              : LogMonitor(logURL, parent, name)
{
  QString fileName;

  addFile(SETILog_log_file, SETILog_log_index);
  addFile(SETIResult_log_file, SETIResult_log_index);

  headersDict.setAutoDelete(true);
}

CSVLogMonitor::~CSVLogMonitor()
{
}

const QValueList<log_data> CSVLogMonitor::logData()
{
  return(isOK(SETILog_log_index) ? data : LogMonitor::logData());
}

void CSVLogMonitor::writeSetiData(const KURL& logURL, const seti_data& data)
{
  KTempFile tempFile;
  tempFile.close();

  KURL fileURL;
  QString fileName;

  fileURL = logURL;
  fileURL.addPath(SETILog_log_file);
  if(!fileURL.isLocalFile())
  {
    fileName = tempFile.name();
    // try to download the log file if present
    #if KDE_IS_VERSION(3,2,0)
    KIO::NetAccess::download(fileURL, fileName, qApp->mainWidget());
    #else
    KIO::NetAccess::download(fileURL, fileName);
    #endif
  }
  else
    fileName = fileURL.path(-1);

  {
    QFileInfo fileInfo(fileName);
    fileInfo.refresh();
    bool exists = fileInfo.exists();

    QFile file(fileName);
    if(file.open(IO_ReadWrite))
    {
      QTextStream stream(&file);
      QStringList headers;

      if(!exists || !parseSETILogHeaders(stream, headers))
      {
        headers = defaultSETILogHeaders();
        writeSETILogHeaders(stream, headers);
      }
      else file.at(file.size());

      writeSETILogRow(stream, headers, data);

      file.close();
    }
  }

  if(!fileURL.isLocalFile())
  #if KDE_IS_VERSION(3,2,0)
    KIO::NetAccess::upload(fileName, fileURL, qApp->mainWidget());
  #else
    KIO::NetAccess::upload(fileName, fileURL);
  #endif

  fileURL = logURL;
  fileURL.addPath(SETIResult_log_file);
  if(!fileURL.isLocalFile())
  {
    fileName = tempFile.name();
    // try to download the log file if present
    #if KDE_IS_VERSION(3,2,0)
    KIO::NetAccess::download(fileURL, fileName, qApp->mainWidget());
    #else
    KIO::NetAccess::download(fileURL, fileName);
    #endif
  }
  else
    fileName = fileURL.path(-1);

  {
    QFile file (fileName);
    if(file.open(IO_ReadWrite))
    {
      QTextStream stream(&file);

      file.at(file.size());
      writeSETIResultRow(stream, data);

      file.close();
    }
  }

  if(!fileURL.isLocalFile())
  #if KDE_IS_VERSION(3,2,0)
    KIO::NetAccess::upload(fileName, fileURL, qApp->mainWidget());
  #else
    KIO::NetAccess::upload(fileName, fileURL);
  #endif

  tempFile.unlink();
}

bool CSVLogMonitor::findIntEntry(const QString& key, QStringList& list, const QString&, int& value)
{
  int *pos = headersDict.find(key);
  value = (pos != NULL) ? list[*pos].toInt() : 0;
  return(pos != NULL);  
}

bool CSVLogMonitor::findLongEntry(const QString& key, QStringList& list, const QString&, long& value)
{
  int *pos = headersDict.find(key);
  value = (pos != NULL) ? list[*pos].toInt() : 0L;
  return(pos != NULL);  
}

bool CSVLogMonitor::findFloatEntry(const QString& key, QStringList& list, const QString&, float& value)
{
  int *pos = headersDict.find(key);
  value = (pos != NULL) ? list[*pos].toFloat() : 0.0f;
  return(pos != NULL);  
}

bool CSVLogMonitor::findDoubleEntry(const QString& key, QStringList& list, const QString&, double& value)
{
  int *pos = headersDict.find(key);
  value = (pos != NULL) ? list[*pos].toDouble() : 0.0;
  return(pos != NULL);  
}

bool CSVLogMonitor::findStringEntry(const QString& key, QStringList& list, const QString&, QString& value)
{
  int *pos = headersDict.find(key);
  value = (pos != NULL) ? removeQuotes(list[*pos]) : QString::null;
  return(pos != NULL);
}

bool CSVLogMonitor::parseSETILogFile(const QString& fileName)
{
  QStringList headers, lines;

  data.clear();

  bool read_ok = true;

  QFile file(fileName);
  if(file.open(IO_ReadOnly))
  {
    QTextStream stream(&file);

    if(parseSETILogHeaders(stream, headers))
      while(true)
      {
        QString line = stream.readLine();
        if(!line.isNull()) lines += line;
        else break;
      }
    else
      headers = defaultSETILogHeaders();

    setHeaders(headers);
  } else
    read_ok = false;

  if(!read_ok) return(read_ok);

  const QString sep("=");

  for(QStringList::Iterator line=lines.begin(); line != lines.end(); line++)
  {
    log_data item;

    QStringList fields = QStringList::split(",", *line, true);

    QString done;
    if(!findStringEntry("date", fields, sep, done)) continue;

    const int pos = done.find(' ');
    if(pos < 0) continue;

    const QStringList done_date = QStringList::split('.', done.left(pos));
    if(done_date.count() != 3) continue;
    item.done.setDate(QDate(done_date[0].toInt(), done_date[1].toInt(), done_date[2].toInt()));

    const QStringList done_time = QStringList::split(':', done.mid(pos));
    if(done_time.count() != 3) continue;
    item.done.setTime(QTime(done_time[0].toInt(), done_time[1].toInt(), done_time[2].toInt()));

    seti_data item_data;

    double version;
    if(!findDoubleEntry("version", fields, sep, version)) continue;

    item_data.version.major = int(version);
    item_data.version.minor = int(1e2 * (version - item_data.version.major));

    SetiDataMonitor::parseUserInfoLines(this, fields, item_data.user);
    SetiDataMonitor::parseWorkUnitLines(this, fields, item_data.wu);
    SetiDataMonitor::parseStateLines(this, fields, item_data.state);

    item.name = item_data.wu.name;
    item.start.ra = item_data.wu.start.ra;
    item.start.dec = item_data.wu.start.dec;
    item.angle_range = item_data.wu.angle_range;
    item.teraFLOPs = SetiDataMonitor::teraFLOPs(item_data);
    item.cpu = item_data.state.cpu;
    item.progress = item_data.state.progress;

    item.spikes.best_score = item_data.state.best_spike.score;
    item.gaussians.best_score = item_data.state.best_gaussian.score;
    item.pulses.best_score = item_data.state.best_pulse.score;
    item.triplets.best_score = item_data.state.best_triplet.score;
    item.spikes.returned = item.gaussians.returned = item.pulses.returned = item.triplets.returned = 0;

    item.interesting = SetiDataMonitor::isInteresting(item_data.state.best_gaussian);

    data += item;
  }

  return(true);
}

bool CSVLogMonitor::parseSETIResultFile(const QString& fileName)
{
  bool read_ok;
  const QStringList lines = readFile(fileName, read_ok);
  if(!read_ok) return(false);

  for(QValueList<log_data>::Iterator datum = data.begin(); datum != data.end(); datum++)
  {
    (*datum).spikes.returned = 0;
    (*datum).gaussians.returned = 0;
    (*datum).pulses.returned = 0;
    (*datum).triplets.returned = 0;

    QStringList::ConstIterator line = lines.find("[" + (*datum).name + "]");
    if(line != lines.end()) line++;

    while(line != lines.end())
    {
      if((*line).contains(QRegExp("^spike")))
        (*datum).spikes.returned++;
      else if((*line).contains(QRegExp("^gaussian")))
        (*datum).gaussians.returned++;
      else if((*line).contains(QRegExp("^pulse")))
        (*datum).pulses.returned++;
      else if((*line).contains(QRegExp("^triplet")))
        (*datum).triplets.returned++;
      else break;
      line++;
    }
  }

  return(true);
}

bool CSVLogMonitor::parseFile(int index, const QString& fileName)
{
  if(index == SETILog_log_index)
    return(parseSETILogFile(fileName));
  else if(index == SETIResult_log_index)
    return(parseSETIResultFile(fileName));
  else
    return(DataMonitor::parseFile(index, fileName));
}

void CSVLogMonitor::writeSETILogRow(QTextStream& stream, const QStringList& headers, const seti_data& data)
{
  stream << "\r\n";

  for(QStringList::ConstIterator header=headers.begin(); header != headers.end(); header++)
  {
    if(header != headers.begin())
      stream << ",";

    if(*header == "date")
    {
      const QDateTime now = QDateTime::currentDateTime();
      QString date_string;

      date_string = QString("%1.%2.%3").arg(now.date().year()).arg(now.date().month()).arg(now.date().day());
      date_string += " ";
      date_string += QString("%1:%2:%3").arg(now.time().hour()).arg(now.time().minute()).arg(now.time().second());

      stream << insertQuotes(date_string);
    }
    else if(*header == "id")
      stream << data.user.id;
    else if(*header == "key")
      stream << data.user.key;
    else if(*header == "email_addr")
      stream << insertQuotes(data.user.email);
    else if(*header == "user_name")
      stream << insertQuotes(data.user.name);
    else if(*header == "url")
      stream << insertQuotes(data.user.url);
    else if(*header == "country")
      stream << insertQuotes(data.user.country);
    else if(*header == "postal_code")
      stream << data.user.postal_code;
    else if(*header == "show_name")
      stream << (data.user.show_name ? insertQuotes("yes") : insertQuotes("no"));
    else if(*header == "show_email")
      stream << (data.user.show_email ? insertQuotes("yes") : insertQuotes("no"));
    else if(*header == "venue")
      stream << data.user.venue;
    else if(*header == "register_time")
      stream << insertQuotes(data.user.registered.string);
    else if(*header == "last_wu_time")
      stream << insertQuotes(data.user.last_wu.string);
    else if(*header == "last_result_time")
      stream << insertQuotes(data.user.last_result.string);
    else if(*header == "nwus")
      stream << data.user.n_wus;
    else if(*header == "nresults")
      stream << data.user.n_results;
    else if(*header == "total_cpu")
      stream << formatDouble(data.user.total_cpu, 6);
    else if(*header == "params_index")
      stream << data.user.params_index;
    else if(*header == "task")
      stream << insertQuotes(data.wu.task);
    else if(*header == "version")
      stream << formatDouble(data.version.major + 1e-2 * data.version.minor, 2);
    else if(*header == "name")
      stream << insertQuotes(data.wu.name);
    else if(*header == "data_type")
      stream << insertQuotes(data.wu.data_type);
    else if(*header == "data_class")
      stream << data.wu.data_class;
    else if(*header == "splitter_version")
      stream << insertQuotes(data.wu.splitter_version);
    else if(*header == "start_ra")
      stream << formatDouble(data.wu.start.ra, 3);
    else if(*header == "start_dec")
      stream << formatDouble(data.wu.start.dec, 3);
    else if(*header == "end_ra")
      stream << formatDouble(data.wu.end.ra, 3);
    else if(*header == "end_dec")
      stream << formatDouble(data.wu.end.dec, 3);
    else if(*header == "angle_range")
      stream << formatDouble(data.wu.angle_range, 3);
    else if(*header == "time_recorded")
      stream << insertQuotes(data.wu.recorded.string);
    else if(*header == "subband_center")
      stream << formatDouble(data.wu.subband.center, 2);
    else if(*header == "subband_base")
      stream << formatDouble(data.wu.subband.center, 2);
    else if(*header == "subband_sample_rate")
      stream << formatDouble(data.wu.subband.center, 2);
    else if(*header == "fft_len")
      stream << data.wu.fft_length;
    else if(*header == "ifft_len")
      stream << data.wu.ifft_length;
    else if(*header == "subband_number")
      stream << data.wu.subband.number;
    else if(*header == "receiver")
      stream << insertQuotes(data.wu.receiver);
    else if(*header == "nsamples")
      stream << data.wu.n_samples;
    else if(*header == "tape_version")
      stream << formatDouble(data.wu.tape_version.major + 1e-2 * data.wu.tape_version.minor, 2);
    else if(*header == "ncfft")
      stream << data.state.ncfft;
    else if(*header == "cr")
      stream << formatDouble(data.state.cr, 6, 'e');
    else if(*header == "fl")
      stream << data.state.fl;
    else if(*header == "cpu")
      stream << formatDouble(data.state.cpu, 8);
    else if(*header == "prog")
      stream << formatDouble(data.state.progress, 8);
    else if(*header == "bs_power")
      stream << formatDouble(data.state.best_spike.power, 6);
    else if(*header == "bs_score")
      stream << formatDouble(data.state.best_spike.score, 6);
    else if(*header == "bs_bin")
      stream << data.state.best_spike.bin;
    else if(*header == "bs_fft_ind")
      stream << data.state.best_spike.fft_index;
    else if(*header == "bs_chirp_rate")
      stream << formatDouble(data.state.best_spike.chirp_rate, 6);
    else if(*header == "bs_fft_len")
      stream << data.state.best_spike.fft_length;
    else if(*header == "bg_score")
      stream << formatDouble(data.state.best_gaussian.score, 6);
    else if(*header == "bg_power")
      stream << formatDouble(data.state.best_gaussian.power, 6);
    else if(*header == "bg_chisq")
      stream << formatDouble(data.state.best_gaussian.chi_squared, 6);
    else if(*header == "bg_bin")
      stream << data.state.best_gaussian.bin;
    else if(*header == "bg_fft_ind")
      stream << data.state.best_gaussian.fft_index;
    else if(*header == "bg_chirp_rate")
      stream << formatDouble(data.state.best_gaussian.chirp_rate, 6);
    else if(*header == "bg_fft_len")
      stream << data.state.best_gaussian.fft_length;
    else if(*header == "bp_score")
      stream << formatDouble(data.state.best_pulse.score, 6);
    else if(*header == "bp_power")
      stream << formatDouble(data.state.best_pulse.power, 6);
    else if(*header == "bp_mean")
      stream << formatDouble(data.state.best_pulse.mean, 6);
    else if(*header == "bp_period")
      stream << formatDouble(data.state.best_pulse.period, 6);
    else if(*header == "bp_freq_bin")
      stream << data.state.best_pulse.frequency_bin;
    else if(*header == "bp_time_bin")
      stream << formatDouble(data.state.best_pulse.time_bin, 6);
    else if(*header == "bp_chirp_rate")
      stream << formatDouble(data.state.best_pulse.chirp_rate, 6);
    else if(*header == "bp_fft_len")
      stream << data.state.best_pulse.fft_length;
    else if(*header == "bt_score")
      stream << formatDouble(data.state.best_triplet.score, 6);
    else if(*header == "bt_power")
      stream << formatDouble(data.state.best_triplet.power, 6);
    else if(*header == "bt_mean")
      stream << formatDouble(data.state.best_triplet.mean, 6);
    else if(*header == "bt_period")
      stream << formatDouble(data.state.best_triplet.period, 6);
    else if(*header == "bt_bperiod")
      stream << formatDouble(data.state.best_triplet.bperiod, 6);
    else if(*header == "bt_tpotind0_0")
      stream << data.state.best_triplet.tpotind[0][0];
    else if(*header == "bt_tpotind0_1")
      stream << data.state.best_triplet.tpotind[0][1];
    else if(*header == "bt_tpotind1_0")
      stream << data.state.best_triplet.tpotind[1][0];
    else if(*header == "bt_tpotind1_1")
      stream << data.state.best_triplet.tpotind[2][1];
    else if(*header == "bt_tpotind2_0")
      stream << data.state.best_triplet.tpotind[2][0];
    else if(*header == "bt_tpotind2_1")
      stream << data.state.best_triplet.tpotind[2][1];
    else if(*header == "bt_freq_bin")
      stream << data.state.best_triplet.frequency_bin;
    else if(*header == "bt_time_bin")
      stream << formatDouble(data.state.best_triplet.time_bin, 6);
    else if(*header == "bt_chirp_rate")
      stream << formatDouble(data.state.best_triplet.chirp_rate, 6);
    else if(*header == "bt_scale")
      stream << formatDouble(data.state.best_triplet.scale, 6);
    else if(*header == "bt_fft_len")
      stream << data.state.best_triplet.fft_length;
  }
}

void CSVLogMonitor::writeSETIResultRow(QTextStream& stream, const seti_data& data)
{
  stream << "[" << data.wu.name << "]";

  double wu_time = 0.0;
  SetiDataMonitor::convertToDate(data.wu.recorded.string, &wu_time);

  for(uint i = 0; i < data.output.spikes.count(); i++)
  {
    stream << "\r\n";
    stream << "spike" << QString::number(i+1).rightJustify(2, '0') << "=";

    stream << " power=" << formatDouble(data.output.spikes[i].power, 6 ,'e');
    stream << " ra= " << formatDouble(data.output.spikes[i].index.ra, 2);
    stream << " dec= " << formatDouble(data.output.spikes[i].index.dec, 2);
    stream << " time= " << formatDouble(convertTime(data.output.spikes[i].index.time, wu_time), 2);
    stream << " freq=" << formatDouble(data.output.spikes[i].frequency, 2);
    stream << " fft_len=" << data.output.spikes[i].fft_length;
    stream << " chirp_rate=" << formatDouble(data.output.spikes[i].chirp_rate, 6, 'e');
  }

  for(uint i = 0; i < data.output.gaussians.count(); i++)
  {
    stream << "\r\n";
    stream << "gaussian" << QString::number(i+1).rightJustify(2, '0') << "=";

    stream << " peak=" << formatDouble(data.output.gaussians[i].peak, 6 ,'e');
    stream << " mean=" << formatDouble(data.output.gaussians[i].mean, 6 ,'e');
    stream << " ra= " << formatDouble(data.output.gaussians[i].index.ra, 2);
    stream << " dec= " << formatDouble(data.output.gaussians[i].index.dec, 2);
    stream << " time= " << formatDouble(convertTime(data.output.gaussians[i].index.time, wu_time), 2);
    stream << " freq=" << formatDouble(data.output.gaussians[i].frequency, 2);
    stream << " sigma=" << formatDouble(data.output.gaussians[i].sigma, 2);
    stream << " chisqr=" << formatDouble(data.output.gaussians[i].chi_squared, 2);
    stream << " fft_len=" << data.output.gaussians[i].fft_length;
    stream << " chirprate=" << formatDouble(data.output.gaussians[i].chirp_rate, 6, 'e');
    stream << " maxpow=" << formatDouble(data.output.gaussians[i].max_power, 2);
    stream << " pot=";
    for(uint j = 0; j < data.output.gaussians[i].pot.count(); j++)
      stream << QString::number(data.output.gaussians[i].pot[j], 16).rightJustify(2, '0');
  }

  for(uint i = 0; i < data.output.pulses.count(); i++)
  {
    stream << "\r\n";
    stream << "pulse" << QString::number(i+1).rightJustify(2, '0') << "=";

    stream << " power=" << formatDouble(data.output.pulses[i].power, 6 ,'e');
    stream << " mean=" << formatDouble(data.output.pulses[i].mean, 6 ,'e');
    stream << " period=" << formatDouble(data.output.pulses[i].period, 6 ,'e');
    stream << " ra= " << formatDouble(data.output.pulses[i].index.ra, 2);
    stream << " dec= " << formatDouble(data.output.pulses[i].index.dec, 2);
    stream << " time= " << formatDouble(convertTime(data.output.pulses[i].index.time, wu_time), 2);
    stream << " freq=" << formatDouble(data.output.pulses[i].frequency, 2);
    stream << " fft_len=" << data.output.pulses[i].fft_length;
    stream << " chirp_rate=" << formatDouble(data.output.pulses[i].chirp_rate, 6, 'e');
    stream << " snr=" << formatDouble(data.output.pulses[i].snr, 6 ,'e');
    stream << " thresh=" << formatDouble(data.output.pulses[i].threshold, 6 ,'e');
    stream << " len_prof=" << data.output.pulses[i].prof.count();
    stream << " prof=";
    for(uint j = 0; j < data.output.pulses[i].prof.count(); j++)
      stream << QString::number(data.output.pulses[i].prof[j], 16).rightJustify(2, '0');
  }

  for(uint i = 0; i < data.output.triplets.count(); i++)
  {
    stream << "\r\n";
    stream << "triplet" << QString::number(i+1).rightJustify(2, '0') << "=";

    stream << " power=" << formatDouble(data.output.triplets[i].power, 6 ,'e');
    stream << " mean=" << formatDouble(data.output.triplets[i].mean, 6 ,'e');
    stream << " period=" << formatDouble(data.output.triplets[i].period, 6 ,'e');
    stream << " ra= " << formatDouble(data.output.triplets[i].index.ra, 2);
    stream << " dec= " << formatDouble(data.output.triplets[i].index.dec, 2);
    stream << " time= " << formatDouble(convertTime(data.output.triplets[i].index.time, wu_time), 2);
    stream << " freq=" << formatDouble(data.output.triplets[i].frequency, 2);
    stream << " fft_len=" << data.output.triplets[i].fft_length;
    stream << " chirp_rate=" << formatDouble(data.output.triplets[i].chirp_rate, 6, 'e');
  }

  stream << "\r\n" << "\r\n";
}

bool CSVLogMonitor::parseSETILogHeaders(QTextStream& stream, QStringList& headers)
{
  QString line = stream.readLine();

  if(!line.isEmpty())
  {
    headers = QStringList::split(QRegExp("[\",]"), line);
    return(true);
  }
  else return(false);
}

QStringList CSVLogMonitor::defaultSETILogHeaders()
{
  QStringList out;

  out << "date";

  // from user_info.sah
  out << "id";
  out << "key";
  out << "email_addr";
  out << "user_name";
  out << "url";
  out << "country";
  out << "postal_code";
  out << "show_name";
  out << "show_email";
  out << "venue";
  out << "register_time";
  out << "last_wu_time";
  out << "last_result_time";
  out << "nwus";
  out << "nresults";
  out << "total_cpu";
  out << "params_index";

  // from work_unit.sah
  out << "task";
  out << "version";
  out << "name";
  out << "data_type";
  out << "data_class";
  out << "splitter_version";
  out << "start_ra";
  out << "start_dec";
  out << "end_ra";
  out << "end_dec";
  out << "angle_range";
  out << "time_recorded";
  out << "subband_center";
  out << "subband_base";
  out << "subband_sample_rate";
  out << "fft_len";
  out << "ifft_len";
  out << "subband_number";
  out << "receiver";
  out << "nsamples";
  out << "tape_version";

  // from state.sah
  out << "ncfft";
  out << "cr";
  out << "fl";
  out << "cpu";
  out << "prog";
  out << "bs_power";
  out << "bs_score";
  out << "bs_bin";
  out << "bs_fft_ind";
  out << "bs_chirp_rate";
  out << "bs_fft_len";
  out << "bg_score";
  out << "bg_power";
  out << "bg_chisq";
  out << "bg_bin";
  out << "bg_fft_ind";
  out << "bg_chirp_rate";
  out << "bg_fft_len";
  out << "bp_score";
  out << "bp_power";
  out << "bp_mean";
  out << "bp_period";
  out << "bp_freq_bin";
  out << "bp_time_bin";
  out << "bp_chirp_rate";
  out << "bp_fft_len";
  out << "bt_score";
  out << "bt_power";
  out << "bt_mean";
  out << "bt_period";
  out << "bt_bperiod";
  out << "bt_tpotind0_0";
  out << "bt_tpotind0_1";
  out << "bt_tpotind1_0";
  out << "bt_tpotind1_1";
  out << "bt_tpotind2_0";
  out << "bt_tpotind2_1";
  out << "bt_freq_bin";
  out << "bt_time_bin";
  out << "bt_chirp_rate";
  out << "bt_scale";
  out << "bt_fft_len";

  return(out);
}

void CSVLogMonitor::setHeaders(const QStringList& headers)
{
  headersDict.clear();
  for(uint i = 0; i < headers.count(); i++)
    headersDict.insert(headers[i], new int(i));
}

void CSVLogMonitor::writeSETILogHeaders(QTextStream& stream, const QStringList& headers)
{
  for(QStringList::ConstIterator header=headers.begin(); header != headers.end(); header++)
  {
    if(header != headers.begin()) stream << ",";
    stream << insertQuotes(*header);
  }
}

QString CSVLogMonitor::removeQuotes(const QString& string)
{
  QString out = string;
  return(out.replace(QRegExp("^\""), "").replace(QRegExp("\"$"), ""));
}

QString CSVLogMonitor::insertQuotes(const QString& string)
{
  return(!string.isEmpty() ? ("\"" + string + "\"") : QString::null);
}

double CSVLogMonitor::convertTime(const QTime& time, double wu_time)
{
  return(wu_time + 1e-3 * time.msec() + time.second() + 60.0 * (time.minute() + 60.0 * time.hour()));
}

#include "csvlogmonitor.moc"

