/***************************************************************************
                          intervalselectorwidget.cpp  -  description
                             -------------------
    begin                : Sat Jan 26 2002
    copyright            : (C) 2002 by Roberto Virga
    email                : rvirga@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <kiconloader.h>

#include "intervalwidget.h"

#include "intervalselectorwidget.h"

const QString SchedulePixmap = "schedule";

const int border = 10;
const int hour = 20;

IntervalSelectorWidget::IntervalSelectorWidget(QWidget *parent, const char *name)
                        :QFrame(parent, name)
{
  const QPixmap grid = KIconLoader().loadIcon(SchedulePixmap, KIcon::User);

  setFrameStyle(Panel);
  setFrameShadow(Sunken);
  setBackgroundPixmap(grid);
  setFixedSize(grid.size());

  intervals.setAutoDelete(false);
}

IntervalSelectorWidget::~IntervalSelectorWidget()
{
}

QValueList<intervalStruct> IntervalSelectorWidget::getSchedule()
{
  QValueList<intervalStruct> result;
  for(uint i = 0; i < intervals.count(); i++) {
    intervalStruct intervalData;
    const IntervalWidget *interval = (IntervalWidget *) intervals.at(i);
    const QRect r = interval->geometry();

    intervalData.type = intervalStruct::Type(interval->getType());
    intervalData.start = QTime().addSecs(pixelsToSecs(r.left()));
    intervalData.stop = QTime().addSecs(pixelsToSecs(r.right()+1)-1);

    result.append(intervalData);
  }
  return(result);
}

void IntervalSelectorWidget::setSchedule(const QValueList<intervalStruct>& schedule)
{
  const QRect r = this->rect();

  intervals.clear();
  for(uint i = 0; i < schedule.count(); i++) {
    const int x = secsToPixels(QTime().secsTo(schedule[i].start));
    const int y = (r.bottom() - r.top())/2;
    const int w = secsToPixels(QTime().secsTo(schedule[i].stop)) - x + 1;
    const int h = r.height()/2;

    IntervalWidget *interval = new IntervalWidget(this);
    interval->setGeometry(x, y, w, h);
    interval->setType(schedule[i].type);
    addInterval(interval);
  }
}

void IntervalSelectorWidget::mouseDoubleClickEvent(QMouseEvent *e)
{
  const QRect r = this->rect();
  const QPoint p = e->pos();

  int x = p.x();

  const int minX = getMinX(p.x());
  const int maxX = getMaxX(p.x());

  if(maxX - minX + 1 < hour)
    return;
  else if(maxX - p.x() + 1 < hour)
    x = maxX - hour + 1;

  const int y = (r.bottom() - r.top())/2;
  const int w = hour;
  const int h = r.height()/2;

  if(p.y() < y || p.y() > r.bottom())
    return;

  IntervalWidget *interval = new IntervalWidget(this);
  interval->setGeometry(x, y, w, h);
  addInterval(interval);
}

int IntervalSelectorWidget::getMinX(int x)
{
  int i = findIntervalBefore(x);
  return(i >= 0 ? intervals.at(i)->geometry().right()+1 : border-1);
}

int IntervalSelectorWidget::getMaxX(int x)
{
  int i = findIntervalAfter(x);
  return(i >= 0 ? intervals.at(i)->geometry().left()-1 : this->width()-border-2);
}

int IntervalSelectorWidget::getMinSize()
{
  return(hour);
}

int IntervalSelectorWidget::findIntervalBefore(int x)
{
  unsigned int i = 0;
  while(i < intervals.count() && x >= intervals.at(i)->geometry().right())
    i++;
  return(i-1);
}

int IntervalSelectorWidget::findIntervalAfter(int x)
{
  unsigned int i = 0;
  while(i < intervals.count() && x >= intervals.at(i)->geometry().left())
    i++;
  return(i == intervals.count() ? -1 : int(i));
}

void IntervalSelectorWidget::addInterval(QWidget *interval)
{
  int i = findIntervalAfter(interval->geometry().right());

  if(i == -1) i = intervals.count();
  intervals.insert(i, interval);
  interval->show();
  update();
}

void IntervalSelectorWidget::removeInterval(QWidget *interval)
{
  intervals.removeRef(interval);

  interval->hide();
  delete interval;
  
  update();
}

int IntervalSelectorWidget::pixelsToSecs(int pixels)
{
  return((pixels-border+1) * 3600 / hour);
}

int IntervalSelectorWidget::secsToPixels(int secs)
{
  return(secs * hour / 3600 + border-1);
}

#include "intervalselectorwidget.moc"

