/***************************************************************************
                          ksetispy.cpp  -  description
                             -------------------
    begin                : Tue May 29 09:02:53 EDT 2001
    copyright            : (C) 2001 by Roberto Virga
    email                : rvirga@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qaccel.h>
#include <qiconset.h>
#include <qlayout.h>
#include <qpopupmenu.h>
#include <qwidgetlist.h>
#include <qwidgetstack.h>

#include <kapplication.h>
#include <kconfig.h>
#include <kiconloader.h>
#include <klocale.h>

#include "aboutview.h"
#include "cacheview.h"
#include "clientview.h"
#include "groupstatsview.h"
#include "ksetispytray.h"
#include "processorview.h"
#include "progressview.h"
#include "performanceview.h"
#include "resultsview.h"
#include "setupview.h"
#include "userstatsview.h"
#include "workunitview.h"

#include "ksetispy.h"

const int N_VIEWS = 11;

const int StartupViewPos = 9;
const int AboutViewPos = 10;

const viewDataStruct ViewIcons[N_VIEWS] =
{
  {"ksetispy_progress", "ksetispy_progress"},
  {"ksetispy_performance", "ksetispy_performance"},
  {"ksetispy_results", "ksetispy_results"},
  {"ksetispy_work_unit", "ksetispy_work_unit"},
  {"ksetispy_user_stats", "ksetispy_user_stats"},
  {"ksetispy_group_stats", "ksetispy_group_stats"},
  {"ksetispy_client", "ksetispy_client"},
  {"ksetispy_processor", "ksetispy_processor"},
  {"ksetispy_cache", "ksetispy_cache"},
  {"ksetispy_setup", "ksetispy_setup"},
  {"ksetispy_about", "ksetispy_about"}
};

const viewDataStruct ViewNames[N_VIEWS] =
{
  {I18N_NOOP("Progress"), I18N_NOOP("Progress")},
  {I18N_NOOP("Performance"), I18N_NOOP("Performance")},
  {I18N_NOOP("Results"), I18N_NOOP("Results")},
  {I18N_NOOP("Work Unit"), I18N_NOOP("Work Unit")},
  {I18N_NOOP("User Statistics"), I18N_NOOP("User Stats")},
  {I18N_NOOP("Group Statistics"), I18N_NOOP("Group Stats")},
  {I18N_NOOP("Client"), I18N_NOOP("Client")},
  {I18N_NOOP("Processor"), I18N_NOOP("Processor")},
  {I18N_NOOP("Work Unit Cache"), I18N_NOOP("Cache")},
  {I18N_NOOP("Setup"), I18N_NOOP("Setup")},
  {I18N_NOOP("About KSetiSpy"), I18N_NOOP("About")}
};

KSetiSpy::KSetiSpy(QWidget *parent, const char *name)
         : DCOPObject("Views-Interface")
         , KMainWindow(parent, name, WStyle_Customize | WStyle_NormalBorder)
{
  kdoc = new KSetiSpyDoc(this);
  KSetiSpyView::setDocument(kdoc);

  forceQuit = false;

  initToolBar();
  initViews();
  initTray();

  readConfig();

  updateContent();
  connect(kdoc, SIGNAL(updated()), this, SLOT(updateContent()));
}

KSetiSpy::~KSetiSpy()
{
}

void KSetiSpy::readConfig()
{
  KConfig *config = kapp->config();
  bool minimized = false;

  config->setGroup("KSetiSpy");

  SetupView *setup = (SetupView *) child("KSetiSpy::SetupView", "SetupView");

  const bool firstTime = config->readBoolEntry("First time", true);

  const bool rememberGeometry = config->readBoolEntry("Remember window positions", false);
  setup->setRememberGeometry(rememberGeometry);

  setup->setTrayFlags(config->readNumEntry("Tray flags", 2));
  setTrayFlags(setup->trayFlags());

  config->setGroup("Main Window");

  if(rememberGeometry)
  {
    const QRect rect = geometry();
    minimized = config->readBoolEntry("Minimized", false);
    setGeometry(config->readRectEntry("Geometry", &rect));
  }

  showView(config->readNumEntry("Active View", StartupViewPos));

  if(!firstTime)
    toolbar->applySettings(config, "Main Window Toolbar");

  kdoc->readConfig();

  QWidgetStack *views = (QWidgetStack *) child("KSetiSpy::views", "QWidgetStack");
  for(int i = 0; i < N_VIEWS; i++)
  {
    KSetiSpyView *view = (KSetiSpyView *) views->widget(i);
    view->readConfig(rememberGeometry);
  }

  KSetiSpyTray *tray = (KSetiSpyTray *) child("KSetiSpy::tray", "KSetiSpyTray");
  tray->readConfig();

  if(minimized)
  {
    const int flags = setup->trayFlags();

    if(flags & SetupView::MinimizeToTray) hide();
    else showMinimized();
  }
  else showNormal();
}

void KSetiSpy::saveConfig()
{
  KConfig *config = kapp->config();

  config->setGroup("KSetiSpy");

  SetupView *setup = (SetupView *) child("KSetiSpy::SetupView", "SetupView");

  config->writeEntry("First time", false);
  config->writeEntry("Remember window positions", setup->rememberGeometry());
  config->writeEntry("Tray flags", setup->trayFlags());

  config->setGroup("Main Window");

  if(setup->rememberGeometry())
  {
    bool minimized;
    const int flags = setup->trayFlags();

    if(flags & SetupView::MinimizeToTray) minimized = !isVisible();
    else minimized = isMinimized();

    config->writeEntry("Minimized", minimized);
    config->writeEntry("Geometry", geometry());
  }

  config->writeEntry("Active View", currentView());

  toolbar->saveSettings(config, "Main Window Toolbar");

  kdoc->saveConfig();

  QWidgetStack *views = (QWidgetStack *) child("KSetiSpy::views", "QWidgetStack");
  for(int i = 0; i < N_VIEWS; i++)
  {
    KSetiSpyView *view = (KSetiSpyView *) views->widget(i);
    view->saveConfig(true);
  }

  KSetiSpyTray *tray = (KSetiSpyTray *) child("KSetiSpy::tray", "KSetiSpyTray");
  tray->saveConfig();
}

int KSetiSpy::currentView()
{
  QWidgetStack *views = (QWidgetStack *) child("KSetiSpy::views", "QWidgetStack");
  return(views->id(views->visibleWidget()));
}

void KSetiSpy::showView(int index)
{
  QLabel *headerIcon = (QLabel *) child("KSetiSpy::header_icon", "QLabel");
  QLabel *headerText = (QLabel *) child("KSetiSpy::header_text", "QLabel");
  QWidgetStack *views = (QWidgetStack *) child("KSetiSpy::views", "QWidgetStack");
  KSetiSpyView *view = (KSetiSpyView *) views->widget(index);

  headerIcon->setPixmap(KIconLoader().loadIcon(ViewIcons[index].panel,
                        KIcon::NoGroup, KIcon::SizeLarge));
  headerText->setText(i18n(ViewNames[index].panel));
  views->raiseWidget(view);
  view->updateContent();
}

QCStringList KSetiSpy::functions()
{
  QCStringList functions = DCOPObject::functions();

  functions << "int current()";
  functions << "void setCurrent(int index)";

  functions << "QString name(int index)";
  functions << "QStringList content(int index)";
  
  return functions;
}

bool KSetiSpy::process(const QCString& fun, const QByteArray& data,
                       QCString& replyType, QByteArray& replyData)
{
  if(fun == "current()") {
    replyType = "int";
    QDataStream out(replyData, IO_WriteOnly);
    out << currentView();
  } else if(fun == "setCurrent(int)") {
    QDataStream in(data, IO_ReadOnly);
    int index;
    in >> index;

    replyType = "void";
    if(index >= 0 && index < N_VIEWS)
      showView(index);
  } else if(fun == "name(int)") {
    QDataStream in(data, IO_ReadOnly);
    int index;
    in >> index;

    replyType = "QString";
    QDataStream out(replyData, IO_WriteOnly);
    QString name;

    if(index >= 0 && index < N_VIEWS)
      name = i18n(ViewNames[index].panel);
    out << name;
  } else if(fun == "content(int)") {
    QDataStream in(data, IO_ReadOnly);
    int index;
    in >> index;

    replyType = "QStringList";
    QDataStream out(replyData, IO_WriteOnly);
    QWidgetStack *views = (QWidgetStack *) child("KSetiSpy::views", "QWidgetStack");
    QStringList fields;

    if(index >= 0 && index < N_VIEWS) {
      KSetiSpyView *view = (KSetiSpyView *) views->widget(index);
      fields = view->text();
    }
    out << fields;
  } else
    return DCOPObject::process(fun, data, replyType, replyData);

  return true;
}

bool KSetiSpy::event(QEvent *e)
{
  switch(e->type()) {
    case QEvent::Show:
      {
        KSetiSpyTray *tray = (KSetiSpyTray *) child("KSetiSpy::tray", "KSetiSpyTray");
        SetupView *setup = (SetupView *) child("KSetiSpy::SetupView", "SetupView");
        const int flags = setup->trayFlags();

        if(flags & SetupView::RunInTray) tray->show();
        else tray->hide();

        toolbar->show();
      }
      break;
    case QEvent::Hide:
      {
        KSetiSpyTray *tray = (KSetiSpyTray *) child("KSetiSpy::tray", "KSetiSpyTray");
        SetupView *setup = (SetupView *) child("KSetiSpy::SetupView", "SetupView");
        const int flags = setup->trayFlags();

        if(flags & SetupView::MinimizeToTray)
        {
          tray->show();
          hideAllWindows();
        }
      }
      break;
    default:
      break;
  }

  return(KMainWindow::event(e));
}

bool KSetiSpy::queryClose()
{
  SetupView *setup = (SetupView *) child("KSetiSpy::SetupView", "SetupView");
  const int flags = setup->trayFlags();

  if(!forceQuit && (flags & SetupView::CloseToTray)) {
    hide();
    return(false);
  } else if(!forceQuit)
    saveConfig();

  return(true);
}

void KSetiSpy::updateContent()
{
  QString caption = kdoc->currentProfile();

  KLocale *locale = KGlobal::locale();

  if(!caption.isEmpty())
  {
    const SetiClientMonitor::State state = kdoc->setiMonitorState();
    if(state >= SetiClientMonitor::Idle)
    {
      const seti_data *data = kdoc->setiMonitor()->setiData();
      caption += " " + QString("[%1%]").arg(locale->formatNumber(data->state.progress * 1e2, 2));
    }
  }

  setCaption(caption);
}

void KSetiSpy::initToolBar()
{
  toolbar = new KToolBar(this, QMainWindow::Left, "KSetiSpy::toolbar");
  toolbar->setTitle(i18n("Main Toolbar"));
  toolbar->setIconText(KToolBar::IconTextRight);
  toolbar->setIconSize(16);
  toolbar->setFullSize(true);

  for (int i = 0; i < N_VIEWS; i++)
    toolbar->insertButton(ViewIcons[i].toolbar, i, true, i18n(ViewNames[i].toolbar));
}

void KSetiSpy::initViews()
{
  QWidget *central = new QWidget(this);
  setCentralWidget(central);

  QGridLayout *grid = new QGridLayout(central, 2, 2);
  grid->setMargin(8);
  grid->setSpacing(8);

  grid->addWidget(new QLabel(central, "KSetiSpy::header_icon"), 0, 0);

  QFont headerFont = this->font();
  headerFont.setPointSize(20);
  headerFont.setBold(true);

  QLabel *headerText = new QLabel(central, "KSetiSpy::header_text");
  headerText->setFont(headerFont);
  grid->addWidget(headerText, 0, 1);

  QWidgetStack *views = new QWidgetStack (central, "KSetiSpy::views");
  grid->addMultiCellWidget(views, 1, 1, 0, 1);

  views->addWidget(new ProgressView(views, "KSetiSpy::ProgressView"), 0);
  views->addWidget(new PerformanceView(views, "KSetiSpy::PerformanceView"), 1);
  views->addWidget(new ResultsView(views, "KSetiSpy::ResultsView"), 2);
  views->addWidget(new WorkUnitView(views, "KSetiSpy::WorkUnitView"), 3);
  views->addWidget(new UserStatsView(views, "KSetiSpy::UserStatsView"), 4);
  views->addWidget(new GroupStatsView(views, "KSetiSpy::GroupStatsView"), 5);
  views->addWidget(new ClientView(views, "KSetiSpy::ClientView"), 6);
  views->addWidget(new ProcessorView(views, "KSetiSpy::ProcessorView"), 7);
  views->addWidget(new CacheView(views, "KSetiSpy::CacheView"), 8);
  views->addWidget(new SetupView(views, "KSetiSpy::SetupView"), 9);
  views->addWidget(new AboutView(views, "KSetiSpy::AboutView"), 10);

  grid->setColStretch(1, 1);
  grid->setRowStretch(1, 1);

  showView(StartupViewPos);

  QAccel *accelerator = new QAccel(this);
  int pageUp = accelerator->insertItem(Key_PageUp);
  int pageDown = accelerator->insertItem(Key_PageDown);

  accelerator->connectItem(pageUp, this, SLOT(showPreviousView()));
  accelerator->connectItem(pageDown, this, SLOT(showNextView()));

  connect(toolbar, SIGNAL(clicked(int)), this, SLOT(showView(int)));

  SetupView *setup = (SetupView *) child("KSetiSpy::SetupView", "SetupView");
  connect(setup, SIGNAL(updatedTrayFlags(int)), this, SLOT(setTrayFlags(int)));
}

void KSetiSpy::initTray()
{
  KSetiSpyTray *tray = new KSetiSpyTray(kdoc, this, "KSetiSpy::tray");
  connect(tray, SIGNAL(about()), this, SLOT(showAboutView()));
  connect(tray, SIGNAL(quit()), this, SLOT(exit()));
}

void KSetiSpy::setTrayFlags(int flags)
{
  KSetiSpyTray *tray = (KSetiSpyTray *) child("KSetiSpy::tray", "KSetiSpyTray");
  if((flags & SetupView::RunInTray) || (isHidden() && (flags & SetupView::MinimizeToTray))) tray->show();
  else tray->hide();
}

void KSetiSpy::showPreviousView()
{
  showView((currentView() + N_VIEWS - 1) % N_VIEWS);
}

void KSetiSpy::showNextView()
{
  showView((currentView() + 1) % N_VIEWS);
}

void KSetiSpy::showAboutView()
{
  AboutView *view = (AboutView *) child("KSetiSpy::AboutView", "AboutView");
  QWidgetStack *views = (QWidgetStack *) child("KSetiSpy::views", "QWidgetStack");

  if(!isVisible()) show();
  showView(views->id(view));
}

void KSetiSpy::hideAllWindows()
{
  KSetiSpyTray *tray = (KSetiSpyTray *) child("KSetiSpy::tray", "KSetiSpyTray");
  QWidgetList *list = KApplication::topLevelWidgets();
  QWidgetListIt it(*list);
  QWidget *w;

  while((w=it.current()) != 0) {
    ++it;
    if(w != tray && w->isVisible())
      w->hide();
   }

  delete list;

  this->hide();
}

void KSetiSpy::exit()
{
  saveConfig();
  forceQuit = true;
  kapp->quit();
}

#include "ksetispy.moc"

