/***************************************************************************
                          showrecord.h  -  description
                             -------------------
    begin                : Mon Dec 3 2001
    copyright            : (C) 2001 by Eggert Ehmke
    email                : eggert.ehmke@berlin.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef SHOWRECORD_H
#define SHOWRECORD_H

//C++ header
#include <iostream>

//Qt headers
#include <qdom.h>
#include <qptrlist.h>

//KDE headers
#include <kstandarddirs.h>
#include <kdebug.h>

//KShowmail headers
#include "kshowmailview.h"
#include "showrecordelem.h"
#include "types.h"

using namespace Types;
using namespace std;

//forward class declarations
class ShowRecordElem;
class KshowmailView;

/**
 * @short List which contains all mails of an account.
 *
 * The mails are objects of ShowRecordElem.
 * Inherits QPtrList.
 *
 * @see ShowRecordElem
 * @author Eggert Ehmke
 * @author Ulrich Weigelt
 */
class ShowRecord: public QPtrList<ShowRecordElem>
{

  public:

    /**
     * Returned by showSelectedHeaders() if the user has
     * always clicked OK.
     */
    static const int continueShowHeaders;

    /**
     * Returned by showSelectedHeaders() if the user has
     * clicked Cancel.
     */
    static const int cancelShowHeaders;

    /**
     * Constructor
     */
    ShowRecord();

    /**
     * Destructor
     * Does nothing.
     */
    ~ShowRecord();

    /**
     * Saves all stored mails into the given DOM document inside the
     * given account element.
     * @param doc DOM document in that all options are stored
     * @param parent account element
     */
    void saveOptions( QDomDocument& doc, QDomElement& parent );

    /**
     * Reads out all mails stored inside the given account element,
     * creates objects from class ShowRecordElem and stores them in
     * this list. All old items will be removed.
     * @param parent account element (DOM element)
     */
    void readStoredMails( QDomElement& parent );

    /**
     * Apply the filters to the mails in this list.
     * @see ShowRecordElem::applyFilters
     */
    void applyFilters();

    /**
     * Returns whether there are mails in this list which are selected
     * in the list view.
     * @return TRUE - there are selected mails
     * @return FALSE - there aren't selected mails
     */
    bool hasSelectedMails();

    /**
     * Returns the numbers of all selected mails.
     * @return numbers of selected mails.
     */
    Types::MailNumberList_Type getSelectedMails();

    /**
     * Removes the mail which has the given number.
     * @param number number of the mail which will be removed
     */
    void removeMail( int number );

    /**
     * Returns the subjects of the selected mails.
     * @return subjects of selected mails
     */
    QStringList getSelectedSubjects() const;

    /**
     * Returns the sender of the mail with the given number.
     * @param number mail number
     * @return sender
     */
    QString getSenderOf( int number ) const;

    /**
     * Returns the date of sent of the mail with the given number.
     * @param number mail number
     * @return date
     */
    QString getDateOf( int number ) const;

    /**
     * Returns the size of the mail with the given number.
     * @param number mail number
     * @return size
     */
    QString getSizeOf( int number ) const;

    /**
     * Returns the subject of the mail with the given number.
     * @param number mail number
     * @return subject
     */
    QString getSubjectOf( int number ) const;

    /**
     * Decodes the given mail body by the appropriate mail object.
     * The mail object (ShowRecordElem) has to do this job, because
     * it has stored the mail header and therefore it knows the content
     * type and encoding of the mail.
     * It decodes just the text or html part of the body. The rest of it
     * will be rejected.
     * @param body the encoded mail (including header)
     * @param number number of the downloaded mail
     * @param preferHTML decode HTML part if present
     * @return decoded mail body
     */
    QString decodeMailBody( QByteArray body, int number, bool preferHTML ) const;

    /**
     * Returns whether there is mail with the given uid in the list.
     * @param uid UID of the mail
     * @return TRUE - mail is in the list
     * @return FALSE - mail is not in the list
     */
    bool hasMail( QString uid );

    /**
     * Returns whether the given mail is new.
     * @param uid UID of the mail
     * @return TRUE - mail is new
     * @return FALSE - mail is not new or mail doesn't exist
     */
    bool isNew( QString uid ) const;

    /**
     * Appends a new mail to the list.
     * @param number number of the mail on the server
     * @param uid Unique ID of the mail
     * @param isNew TRUE - mail is new; FALSE - mail is not new
     */
    void appendNewMail( int number, QString uid, bool isNew );

    /**
     * Just for debugging.
     * Prints a list of all contained mails to stdout.
     */
    void printMailList();

    /**
     * Sets the size of a mail.
     * @param number number of the mail
     * @param size mail size
     */
    void setSize( int number, long size );

    /**
     * Returns the numbers of mails which are marked as new.
     * @return numbers of the new mails
     */
    Types::MailNumberList_Type getNewMails();

    /**
     * Sets the header of the given mail.
     * @param number number of the mail
     * @param subject the subject
     */
    void setHeader( int number, QString header );

    /**
     * Returns a list of the UIDs of all old mails.
     * @return UIDs of all old mails
     */
    QStringList getUIDsOfOldMails();

    /**
     * Returns the header of the mail with the given UID.
     * @param uid UID
     * @return mail header
     */
    QString getHeaderOf( QString uid );

    /**
     * Sets the header of the mail with the given UID.
     * @param uid UID
     * @param header mail header
     */
    void setHeader( QString uid, QString header );

    /**
     * Returns the number of new mails.
     * @return number of new mails
     */
    int getNumberNewMails();

    /**
     * Returns the number of mails.
     * @return number of mails
     */
    int getNumberMails();

    /**
     * Returns the total size of all mails.
     * @return total size
     */
    long getTotalSize();

    /**
     * Creates for every mail a list view item and insert it
     * into the mail list view.
     * @param view pointer to the application view
     * @param account account name
     */
    void fillMailListView( KshowmailView* view, QString& account );

    /**
     * Shows the headers of all selected mails.
     * @param account account name
     * @return continueShowHeaders - the user has always clicked OK. The invoking function will show the next header
     * @return cancelShowHeaders - the user has clicked Cancel. The invoking function will not show the next header.
     */
    int showSelectedHeaders( QString& account );


};

#endif
