/***************************************************************************
                          showrecord.h  -  description
                             -------------------
    begin                : Thu Dec 28 2000
    copyright            : (C) 2000-2001 by Eggert Ehmke
    email                : eggert.ehmke@berlin.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef SHOWRECORDELEM_H
#define SHOWRECORDELEM_H

//Qt headers
#include <qdom.h>
#include <qstring.h>
#include <qlistview.h>
#include <qdatetime.h>

//KDE headers
#include <klocale.h>
#include <kglobal.h>
#include <kdebug.h>
#include <mimelib/datetime.h>
#include <kmdcodec.h>

//KShowmail headers
#include "showlistviewitem.h"
#include "decodeRFC2047.h"
#include "filterelem.h"
#include "filter.h"
#include "logger.h"
#include "constants.h"
#include "kshowmailview.h"
#include "showheaderdialog.h"

using namespace Constants;

//forward class declarations
class KshowmailView;
class FilterElem;

/**
 * @short Represents a mail.
 *
 * It contains UID, Subject, From, To and date of the mail. Subject, From and To
 * will be read out from the given header. It also contains pointer to the
 * corresponding list view entry and maybe to a matching filter.
 * All mail objects are held by ShowRecord.
 * @see ShowRecord
 * @see ConfigElem
 * @author Eggert Ehmke
 */

class ShowRecordElem
{

  public:

    /**
     * Returned by showHeader(), if the user has
     * clicked OK.
     */
    static const int continueShowHeaders;

    /**
     * Returned by showHeader(), if the user has
     * clicked Cancel.
     */
    static const int cancelShowHeaders;

    /**
     * Constructor
     */
	ShowRecordElem ();

    /**
     * Constructor
     * @param number number of the mail on the server
     * @param uid Unique ID of the mail
     * @param isNew TRUE - mail is new; FALSE - mail is not new
     */
    ShowRecordElem( int number, QString& uid, bool isNew );

    /**
     * Sets the mail header and extracts From, To, Subject, Date and Content Type.
     * All parts of the header are coded according RFC 2047.
     * @param header the mail header
     */
    void setHeader( const QString& header );

    /**
     * Returns the whole header.
     * @return the mail header
     */
    QString header() const;

    /**
     * Returns the sender address.
     * @return sender address
     */
    QString from() const;

    /**
     * Returns the addressee.
     * @return the addressee
     */
    QString to() const;

    /**
     * Returns the subject.
     * @return mail subject
     */
    QString subject() const;

    /**
     * Returns the stored content type.
     * @return content type
     */
    QString content() const;


    /**
     * Returns the date on which the mail was sent, formated to the current
     * locale's conventions.
     * @return the date
     */
    QString date() const;

    /**
     * Returns the date on which the mail was sent.
     * The format corresponds to the ISO 8601 extended specification for
     * representations of dates and times, which is YYYY-MM-DDTHH:MM:SS
     * @return the date
     */
    QString strUnixTime () const;

    /**
     * Sets the unique mail ID.
     * @param uid the ID
     */
    void setUIDL( const QString& uid );

    /**
     * Returns the unique ID of this mail.
     * @return UID
     */
    QString uidl() const;

    /**
     * Sets the size of the mail.
     * @param size Size
     */
    void setSize( int size );

    /**
     * Returns the mail size as numeric value.
     * @return size of the mail
     */
    int size() const;

    /**
     * Returns the mail size as string.
     * The String has a length of 8 and the number is right-aligned to allow sorting.
     */
    QString strSize () const;

    /**
     * Returns the mails size as string with prefix
     */
    QString strSizePrefix() const;

    /**
     * Sets whether the mail is new or not.
     * @param isnew TRUE - mail is new; FALSE - mail is not new
     */
    void setNew( bool isnew );

    /**
     * Returns whether the mail is new or not.
     * @return TRUE - mail is new
     * @return FALSE - mail is not new
     */
    bool isNew() const;

    /**
     * Returns whether the mail is new or not.
     * @return localized String "new", if the mail is new
     * @return localized String "old", if the mail is not new
     */
    QString state() const;

    /**
     * Sets the number of this mail.
     * @param n mail number
     */
    void setNumber( int n );

    /**
     * Returns the number of this mail.
     * @return mail number
     */
    int number() const;

    /**
     * Sets the pointer to the corresponding list view item.
     * @see m_pItem
     * @see ShowListViewItem
     * @param item pointer to the corresponding item
     */
    void setViewItem( ShowListViewItem* item );

    /**
     * Returns the pointer to the corresponding list view item.
     * @see m_pItem
     * @see ShowListViewItem
     * @return pointer to the item
     */
    ShowListViewItem* viewItem() const;

    /**
     * Searches for a matching filer in the filter list and when
     * it has found one, will store a pointer to the filter and
     * will highlight the corresponding list view entry. This is the only
     * thing which a filter produces.
     * @see m_pFilter
     * @see Filter::matches
     */
    void applyFilters();

    /**
     * Read a mail (header, size, uid and number) from the given DOM element
     * and store it in this mail object.
     * The element has to contains the attributes "uidl", "size", "number" and
     * a child node called "header".
     * @param elem a DOM element which contains a mail
     */
    void readOptions( QDomElement& elem );

    /**
     * Creates a new DOM element in the given DOM document and add it to given
     * DOM parent element
     * @param doc the DOM document in that all options are stored
     * @param parent the parent element (account) of the mail
     */
    void saveOptions( QDomDocument& doc, QDomElement& parent );

    /**
     * Returns whether the mail is selected in the list view.
     * @return TRUE - mail is selected
     * @return FALSE - mail is not selected
     */
    bool isSelected() const;

    /**
     * Decodes the given mail body.
     * The mail object has to do this job, because
     * it has stored the mail header and therefore it knows the content
     * type and encoding of the mail.
     * It decodes just the text or html part of the body. The rest of it
     * will be rejected.
     * @param body the encoded mail (including header)
     * @param preferHTML decode HTML part if present
     * @return decoded mail body
     */
    QString decodeMailBody( QByteArray body, bool preferHTML ) const;

    /**
     * Shows the header in a dialog box.
     * @param account account name
     * @return continueShowHeaders - the user has OK clicked. The invoking function will show the next header
     * @return cancelShowHeaders - the user has Cancel clicked. The invoking function will not show the next header.
     */
    int showHeader( QString& account );

  private:

    /**
     * Mail header; coded according RFC 2047
     */
    QCString m_header;

    /**
     * The sender address; coded according RFC 2047
     */
    QCString m_from;

    /**
     * The addressee; coded according RFC 2047
     */
    QCString m_to;

    /**
     * The subject; coded according RFC 2047
     */
    QCString m_subject;

    /**
     * The date on which the mail was sent
     */
    QDateTime m_unixDate;

    /**
     * The content type
     */
    QCString m_content;

    /**
     * The unique ID of the mail.
     */
    QString m_uid;

    /**
     * Size of the mail
     */
    int m_size;

    /**
     * It is set to TRUE when the mail is new.
     */
    bool m_new;

    /**
     * The mail number of the account
     */
    int m_nNumber;

    /**
     * Pointer to the corresponding list view entry. Used to check
     * whether the entry is highlighted and we have to process this mail.
     * @see ShowListViewItem
     */
    ShowListViewItem* m_pItem;

    /**
     * This is a pointer to the filter, which matches this mail.
     * If it is NULL, no filter matches. Will be set by applyFilters().
     * This pointer will just be used to count the deleted mails by this
     * filter( see implementation of setDeleted() ).
     */
    FilterElem* m_pFilter;

    /**
     * Searches in the header for a line which starts with the
     * given item.
     * For example:
     * scanHeader( "From") will find the line
     * "From: Ulrich Weigelt <ulrich.weigelt@gmx.de>"
     * and returns "Ulrich Weigelt <ulrich.weigelt@gmx.de>"
     * Returns an empty string (""), if nothing was found.
     * @param item the search item
     * @return the content of the found line
     */
    QCString scanHeader( const QString& item ) const;

    /**
     * Sets the given string as sender address
     * @param from sender address
     */
    void setFrom( const QCString& from );

    /**
     * Sets the given string as recipient address
     * @param to recipient address
     */
    void setTo( const QCString& to );

    /**
     * Sets the given string as subject
     * @param subject the mail subject
     */
    void setSubject( const QCString& subject );

    /**
     * Converts the given string to a date-time value and stores it.
     * @param date the date
     */
    void setDate( const QCString& date );

    /**
     * Sets the given string as content type.
     * @param content the content type
     */
    void setContent( const QCString& content );

    /**
     * Sets the pointer to a matching filter. Used by applyFilters().
     * @see applyFilters
     * @param filer pointer to the filter
     */
    void setFilter( FilterElem* filter );

    /**
     * Returns the boundary, if the mail has a multi part body.
     * Otherwise it returns an empty string.
     * @return boundary
     */
    QString getBoundary() const;

    /**
     * Returns the char set of the content (e.g. iso-8859-1).
     * If no char set is denoted, it will returns an empty string.
     * @return charset
     */
    QString getCharset() const;

    /**
     * Returns the content transfer encoding of the content
     * (e.g. quoted-printable).
     * If no encoding is denoted, it will returns an empty string.
     * @return content transfer encoding
     */
    QString getTransferEncoding() const;

};

#endif
