/*
	appscheduler.cpp - applications scheduler
	Copyright (C) 2003  Konrad Twardowski <kdtonline@poczta.onet.pl>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "appscheduler.h"
#include "configuration.h"
#include "kshutdowndcopapi.h"
#include "miscutils.h"
#include "msystemtray.h"

#include <qtimer.h>

#include <kdebug.h>
#include <klocale.h>

/* constructor: AppInfo */

AppInfo::AppInfo(const int id, const QString &name, const QString &description,
	const int action)
	: QObject()
{
	_action = action;
	_id = id;
	_appId = name;
	_description = description;
	_name = name;
}

/* destructor: AppInfo */

AppInfo::~AppInfo()
{
}

/* constructor: AppScheduler */

AppScheduler::AppScheduler()
	:
	QObject(),
	// init
	_newId(mainWindow->winId())
{
	// init timer
	_removeDeadTasksTimer = new QTimer(this);
	connect(
		_removeDeadTasksTimer, SIGNAL(timeout()),
		this, SLOT(removeDeadTasksSlot()));

	// init list
	_list = new QValueList<AppInfo *>();
}

/* destructor: AppScheduler */

AppScheduler::~AppScheduler()
{
	if (_list)
	{
		delete _list;
		_list = 0;
	}
}

/* public: AppScheduler */

bool AppScheduler::isEnabled() const
{
	return
		kshutdownrc->schedEnabled &&
		!mainWindow->isRestricted("kshutdown_sched");
}

MMainWindow::What AppScheduler::translateAction(const int action) const
{
	switch (action)
	{
		case ACTION_SHUT_DOWN:
			return MMainWindow::What_ShutDown;

		case ACTION_REBOOT:
			return MMainWindow::What_Reboot;

		case ACTION_LOGOUT:
			return MMainWindow::What_Logout;

		default:
			return MMainWindow::What_Nothing;
	}
}

// DCOP

bool AppScheduler::activateAction(int id)
{
	kdDebug() << "activateAction: id = \"" << id << "\"" << endl;

	if (!isEnabled())
	{
		showStatusInfo();

		return FALSE;
	}

	QValueList<AppInfo *>::iterator it = findId(id); // find task by id
	if (it == 0) // task not found
	{
		// inform user
		QString text = i18n("The task is not registered!");
		text += " (id = " + QString::number(id) + ")";
		mainWindow->tray()->popupMessage(
			MSystemTray::Reason_Error,
			mainWindow->caption(),
			text
		);

		return FALSE; // error
	}

// TODO: 0.7: priority
	MMainWindow::What actionToExecute = translateAction((*it)->getAction());

	if (actionToExecute == MMainWindow::What_Nothing)
		return FALSE; // error

	QString desc = (*it)->getDescription(); // save before unregister!
	unregisterTask(id); // unregister task, deref.

	// ok, but wait for other registered tasks
	if (_list->count() != 0)
		return TRUE; // ok

	// no more tasks, so we can shut down, reboot or logout
	mainWindow->setCancelled(FALSE);

	// no warning message, immediate action execute
	if (!kshutdownrc->warningMessageEnabled)
		return mainWindow->execAction(actionToExecute, FALSE);

	// warning message
	// - create non-modal dialog
	// - after specified time, action will be continued
	// - user can cancel the action
	mainWindow->messageDialogAccepted(FALSE, actionToExecute, desc);

	return TRUE; // ok
}

int AppScheduler::registerTask(const QString &name, const QString &description,
	int action)
{
	kdDebug() << "registerTask: action = \"" << action << "\"" << endl;

	if (!isEnabled())
	{
		showStatusInfo();

		return 0;
	}

	if (
		(action != ACTION_SHUT_DOWN) &&
		(action != ACTION_REBOOT) &&
		(action != ACTION_LOGOUT)
	)
	{
		QString s = QString(i18n("Invalid action: %1"))
			.arg(action);
		mainWindow->tray()->popupMessage(
			MSystemTray::Reason_Error,
			mainWindow->caption(),
			name + ": " + s
		);

		return 0;
	}

	_newId++;
	QString desc = description + " (" +
		mainWindow->getActionName(translateAction(action)) + ")";
	_list->append(new AppInfo(
		_newId,
		name,
		desc,
		action)
	);

	// inform user
	QString message = name + ": " + desc;
	mainWindow->tray()->popupMessage(
		MSystemTray::Reason_Info,
		mainWindow->caption(),
		message,
		20
	);
	MiscUtils::customMessage(message);

	// update info
	mainWindow->updateSchedulerTab(TRUE);

	return _newId;
}

bool AppScheduler::unregisterTask(int id)
{
	if (!isEnabled())
	{
		showStatusInfo();

		return FALSE;
	}

	// find task
	QValueList<AppInfo *>::iterator it = findId(id);
	if (it != 0)
	{
		_list->remove(it);
		// update info
		mainWindow->updateSchedulerTab();

		return TRUE;
	}

	return FALSE;
}

/* private: AppScheduler */

QValueList<AppInfo *>::iterator AppScheduler::findId(int id)
{
	QValueList<AppInfo *>::iterator it = _list->begin();
	QValueList<AppInfo *>::iterator e = _list->end();
	for (; it != e; ++it)
	{
		if ((*it)->getId() == id)
			return it; // found
	}

	return 0; // not found
}

void AppScheduler::showStatusInfo()
{
	// inform user
	mainWindow->tray()->popupMessage(
		MSystemTray::Reason_Error,
		mainWindow->caption(),
		i18n("The scheduler is disabled!")
	);
}

/* private slots: AppScheduler */

void AppScheduler::removeDeadTasksSlot()
{
#if 0
	if (!isEnabled())
		return;

	QValueList<AppInfo *>::iterator it = _list->begin();
	QValueList<AppInfo *>::iterator e = _list->end();
	DCOPClient *client = kapp->dcopClient();
	QCString cs;
	for (; it != e; ++it)
	{
		cs = (*it)->getAppId();
		if ((*it)->getId() && !client->isApplicationRegistered(cs))
		{
			kdDebug() << "Removing dead task: " << cs << endl;
			(*it)->setId(0);
			(*it)->setDescription(
				(*it)->getDescription() + " (" + i18n("Error") + ")"
			);
		}
	}
	mainWindow->updateSchedulerTab();
	_removeDeadTasksTimer->start(REMOVE_DEAD_TASKS_TIMEOUT);
#endif
}
