/*
	extras.h - extras
	Copyright (C) 2004  Konrad Twardowski <kdtonline@poczta.onet.pl>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef __EXTRAS_H__
#define __EXTRAS_H__

#include <qobject.h>

class KPopupMenu;

/** @short Extras - user commands stored in "extras" directory. */
class Extras: public QObject
{
	Q_OBJECT
public:
	/**
	 * Constructor.
	 * @param parent A parent object
	 * @param name A class name
	 */
	Extras(QObject *parent = 0, const char *name = 0);

	/** Destructor. */
	virtual ~Extras();

	/** Returns an action description. */
	inline QString actionDescription() const { return _actionDescription; }

	/**
	 * Executes an "extras" action previously selected from the menu.
	 * @return @c FALSE if error; otherwise @c TRUE
	 */
	bool execAction();

	/** Returns a file name to execute. */
	inline QString fileToExecute() const { return _fileToExecute; }

	/**
	 * Returns a full description of the action,
	 * or @c QString::null if no action.
	 */
	QString getActionName() const;

	/** Returns an "extras" popup menu. */
	inline KPopupMenu *menu() const { return _menu; }

	/** Launches Konqueror which allows you to modify the "extras" files. */
	void modify();

	/**
	 * Sets action to @p path (file to execute) with description @p description.
	 */
	void setAction(const QString &path, const QString &description);
private:
	int _modifyID; // "Modify..." item
	KPopupMenu *_menu; // a popup menu associated with "Extras..." button
	// a list of menu items associated with "extras" files
	QMap<int, QString> *_idToFileMap;
	QString
		_actionDescription,
		_fileToExecute,
		_userExtrasDir;

	// reads "extras" directory and creates popup menu items
	void buildMenu(KPopupMenu *parentMenu, const QString &subdir);

	/**
	 * Executes an "extras" file associated with the menu item.
	 * @param path A full path name
	 * @return @c FALSE if error; otherwise @c TRUE
	 */
	bool execFile(const QString &path);

	QMap<int, QString> *idToFileMap();

	// initializes and returns the user directory that contains "extras" items
	// (.desktop files or other executables)
	QString userExtrasDir();
private slots:
	/**
	 * Called when a menu item is selected.
	 * @param id A menu item ID.
	 * This menu ID is mapped to a .desktop file.
	 */
	void activatedSlot(int id);

	void modifySlot();

	/**
	 * Called before the popup menu is displayed.
	 * The menu items are initialized here.
	 */
	void showMenuSlot();
};

#endif // __EXTRAS_H__
