/*
	miscutils.h - Misc Utils
	Copyright (C) 2004  Konrad Twardowski <kdtonline@poczta.onet.pl>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef __MISCUTILS_H__
#define __MISCUTILS_H__

#include "mmainwindow.h"

#include <kprocess.h>

/** @short Misc. functions. */
class MiscUtils
{
public:
	/** Closes the CD-ROM tray (if this option is enabled). */
	static void closeCDTray();

	/**
	 * Displays a confirmation message box to confirm the selected action.
	 * @param action An action to confirm
	 * @return @c TRUE if action has been confirmed,
	 * or if @c kshutdownrc->confirmAction is @c FALSE; otherwise @c FALSE
	 */
	static bool confirmAction(const MMainWindow::What action);

	/**
	 * Displays a custom message.
	 * @param text A text to display
	 * @param testCommand A command to execute (used internally in settings dialog to test entered command)
	 */
	static void customMessage(const QString &text, const QString &testCommand = QString::null);

	/**
	 * Converts seconds to time string.
	 * @param secs Seconds to convert
	 * @param format A date/time format
	 * @return Converted seconds or ">24", if @p secs >= 86400
	 */
	static QString formatDateTime(const int secs,
		const QString &format = QString::null);

	/**
	 * Converts seconds to time string.
	 * @param dt A date/time to convert
	 * @return Converted seconds or ">24", if @p secs >= 86400
	 */
	static QString formatDateTime(const QDateTime &dt);

	/** Returns @c TRUE if @p action is not disabled by KIOSK. */
	static bool isActionEnabled(const MMainWindow::What action);

	/** Launches KShutDown help and optionally opens @p anchor. */
	static void launchHelp(const QString &anchor = "");

	/**
	 * A sound notification.
	 * @param name An event name
	 * @code
	 * playSound("kshutdown-one");
	 * @endcode
	 */
	static void playSound(const QString &name);

	/**
	 * Executes a command.
	 * @param command A command to execute
	 * @return @c TRUE if successfull; otherwise @c FALSE
	 */
	static bool runCommand(const QString &command);

	/**
	 * Executes a shell command before the specified action.
	 * @param configEntry A configuration entry (e.g. <i>"Shut Down"</i>)
	 */
	static void runCommandBeforeAction(const QString &configEntry);

	/**
	 * Executes a shell command.
	 * @param command A shell command to execute
	 * @param mode A run mode
	 * @param pause A timeout in seconds (no timeout by default)
	 * @return @c TRUE if successfull; otherwise @c FALSE
	 */
	static bool runShellCommand(const QString &command, const KProcess::RunMode mode, const int pause = -1);

	/** Sets "tool tip text" and "what's this text" of @p widget to @p value. */
	static void setHint(QWidget *widget, const QString &value);

	/**
	 * Displays a "KRun" error message in a passive message.
	 * @param command A command that failed
	 */
	static void showRunErrorMessage(const QString &command);

	/**
	 * Displays an information if KShutDown is running in "Test Mode".
	 * @param message A message to display
	 */
	static void showTestMessage(const QString &message);
};

#endif // __MISCUTILS_H__
